/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <klocalizedstring.h>

#include "skgbankincludes.h"
#include "skgimportexportmanager.h"
#include "skgtestmacro.h"
#include <utility>

struct test {
    QString fileName;
    QString password;
    QMap<QString, double> expectedAccountAmount;
};

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    {
        // Test export ledger
        SKGDocumentBank document1;
        SKGError err;
        SKGTESTERROR(QLatin1String("DOC.load"), document1.load(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportskg/all_types.skg"), true)

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_LEDGE"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportledger/test-obfuscated.ledger"));
            SKGTESTERROR(QLatin1String("LEDGE.exportFile"), imp1.exportFile(), true)
        }
    }

    QVector<test> listTests;
    {
        test t1;
        t1.fileName = QLatin1String("demo.ledger");

        QMap<QString, double> accounts;
        accounts[QLatin1String("MasterCard")] = -20;
        accounts[QLatin1String("Checking")] = -4124;
        accounts[QLatin1String("Savings")] = -5200;
        accounts[QLatin1String("Mortgage:Principal")] = -800; //Expected : 200
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = QLatin1String("drewr.ledger");

        QMap<QString, double> accounts;
        accounts[QLatin1String("Checking")] = 1366;
        accounts[QLatin1String("Checking:Business")] = 30;
        accounts[QLatin1String("Savings")] = -5200;
        accounts[QLatin1String("MasterCard")] = -20;
        accounts[QLatin1String("Mortgage:Principal")] = -800; //Expected : 200
        accounts[QLatin1String("Tithe")] = -243.6;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = QLatin1String("sample.ledger");

        QMap<QString, double> accounts;
        accounts[QLatin1String("Bank:Checking")] = 1480;
        accounts[QLatin1String("Brokerage")] = 50;
        accounts[QLatin1String("MasterCard")] = -70;
        accounts[QLatin1String("Taxes")] = -2;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = QLatin1String("sebastien.ledger");

        QMap<QString, double> accounts;
        accounts[QLatin1String("CMB:Compte Cheques")] = 3313.19;
        accounts[QLatin1String("CMB:LIVRET A de Mme")] = 5650.00;
        accounts[QLatin1String("CMB:LIVRET A de Mr")] = 5650.00;
        accounts[QLatin1String("Soldes d'Ouvertures")] = -12000.00;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = QLatin1String("transfer.ledger");

        QMap<QString, double> accounts;
        accounts[QLatin1String("Payable:hcoop.net")] = -72355001;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = QLatin1String("wow.ledger");

        QMap<QString, double> accounts;
        accounts[QLatin1String("Tajer")] = 1973592.662; //Expected : "Orb of Deception" 1
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    for (const auto& t : std::as_const(listTests)) {
        // Test import ledger
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_LEDGER"), err)
            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportledger/" % t.fileName));
            SKGTESTERROR(t.fileName % QLatin1String(".importFile"), imp1.importFile(), true)
        }

        // document1.dump(DUMPOPERATION | DUMPACCOUNT | DUMPUNIT | DUMPPAYEE | DUMPCATEGORY);

        QStringList keys = t.expectedAccountAmount.keys();
        for (const auto& k : std::as_const(keys)) {
            SKGAccountObject account;
            SKGTESTERROR(t.fileName % QLatin1String(".getObjectByName(") % k % ")", SKGNamedObject::getObjectByName(&document1, QLatin1String("v_account"), k, account), true)
            SKGTEST(t.fileName % "." % k % QLatin1String(".getCurrentAmount"), SKGServices::toCurrencyString(account.getCurrentAmount(), QLatin1String(""), 2), SKGServices::toCurrencyString(t.expectedAccountAmount[k], QLatin1String(""), 2))
        }
    }
    // End test
    SKGENDTEST()
}
