/*
 *   Copyright (C) 2000 Matthias Elter <elter@kde.org>
 *   Copyright (C) 2001-2002 Raffaele Sandrini <sandrini@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */
#include "kmenuedit.h"

#include <QSplitter>

#include <QAction>
#include <KActionCollection>
#include <KActionMenu>
#include <QIcon>
#include <KLocalizedString>
#include <KXMLGUIFactory>
#include <KMessageBox>
#include <KService>
#include <KStandardAction>
#include <KStandardShortcut>
#include <sonnet/configdialog.h>

#include "treeview.h"
#include "basictab.h"
#include "preferencesdlg.h"
#include "kmenueditadaptor.h"
#include "configurationmanager.h"

KMenuEdit::KMenuEdit ()
    : KXmlGuiWindow(nullptr)
{
    // dbus
    (void)new KmenueditAdaptor(this);
    QDBusConnection::sessionBus().registerObject(QStringLiteral("/KMenuEdit"), this);

    m_showHidden = ConfigurationManager::getInstance()->hiddenEntriesVisible();

    // setup GUI
    setupActions();
    slotChangeView();
}

KMenuEdit::~KMenuEdit()
{
    ConfigurationManager::getInstance()->setSplitterSizes(m_splitter->sizes());
}

void KMenuEdit::setupActions()
{
    QAction *action = actionCollection()->addAction(NEW_SUBMENU_ACTION_NAME);
    action->setIcon(QIcon::fromTheme(QStringLiteral("menu_new")));
    action->setText(i18n("&New Submenu..."));
    actionCollection()->setDefaultShortcut(action, QKeySequence(Qt::CTRL + Qt::SHIFT + Qt::Key_N));
    action = actionCollection()->addAction(NEW_ITEM_ACTION_NAME);
    action->setIcon(QIcon::fromTheme(QStringLiteral("document-new")));
    action->setText(i18n("New &Item..."));
    actionCollection()->setDefaultShortcuts(action, KStandardShortcut::openNew());
    action = actionCollection()->addAction(NEW_SEPARATOR_ACTION_NAME);
    action->setIcon(QIcon::fromTheme(QStringLiteral("menu_new_sep")));
    action->setText(i18n("New S&eparator"));
    actionCollection()->setDefaultShortcut(action, QKeySequence(Qt::CTRL + Qt::Key_I));

    // "sort selection" menu
    KActionMenu *sortMenu = new KActionMenu(QIcon::fromTheme(QStringLiteral("view-sort-ascending")), i18n("&Sort"), this);
    sortMenu->setDelayed(false);
    actionCollection()->addAction(SORT_ACTION_NAME, sortMenu);
    action = actionCollection()->addAction(SORT_BY_NAME_ACTION_NAME);
    action->setText(i18n("&Sort selection by Name"));
    sortMenu->addAction(action);
    action = actionCollection()->addAction(SORT_BY_DESCRIPTION_ACTION_NAME);
    action->setText(i18n("&Sort selection by Description"));
    sortMenu->addAction(action);
    sortMenu->addSeparator();
    action = actionCollection()->addAction(SORT_ALL_BY_NAME_ACTION_NAME);
    action->setText(i18n("&Sort all by Name"));
    sortMenu->addAction(action);
    action = actionCollection()->addAction(SORT_ALL_BY_DESCRIPTION_ACTION_NAME);
    action->setText(i18n("&Sort all by Description"));
    sortMenu->addAction(action);

    // move up/down
    action = actionCollection()->addAction(MOVE_UP_ACTION_NAME);
    action->setIcon(QIcon::fromTheme(QStringLiteral("go-up")));
    action->setText(i18n("Move &Up"));
    action = actionCollection()->addAction(MOVE_DOWN_ACTION_NAME);
    action->setIcon(QIcon::fromTheme(QStringLiteral("go-down")));
    action->setText(i18n("Move &Down"));

    actionCollection()->addAction(KStandardAction::Save, this, SLOT(slotSave()));
    actionCollection()->addAction(KStandardAction::Quit, this, SLOT(close()));
    actionCollection()->addAction(KStandardAction::Cut);
    actionCollection()->addAction(KStandardAction::Copy);
    actionCollection()->addAction(KStandardAction::Paste);

    action = new QAction(i18n("Restore to System Menu"), this);
    actionCollection()->addAction(QStringLiteral("restore_system_menu"), action);
    connect(action, &QAction::triggered, this, &KMenuEdit::slotRestoreMenu);

    KStandardAction::preferences(this, SLOT(slotConfigure()), actionCollection());
}

void KMenuEdit::slotConfigure()
{
    PreferencesDialog dialog(this);
    if (dialog.exec()) {
        bool newShowHiddenValue = ConfigurationManager::getInstance()->hiddenEntriesVisible();
        if (newShowHiddenValue != m_showHidden) {
            m_showHidden = newShowHiddenValue;
            m_tree->updateTreeView(m_showHidden);
            m_basicTab->updateHiddenEntry(m_showHidden);
        }
    }
}

void KMenuEdit::setupView()
{
    m_splitter = new QSplitter(this);
    m_splitter->setOrientation(Qt::Horizontal);
    m_tree = new TreeView(actionCollection(), this);
    m_splitter->addWidget(m_tree);
    m_basicTab = new BasicTab;
    m_splitter->addWidget(m_basicTab);

    connect(m_tree, SIGNAL(entrySelected(MenuFolderInfo*)),
            m_basicTab, SLOT(setFolderInfo(MenuFolderInfo*)));
    connect(m_tree, SIGNAL(entrySelected(MenuEntryInfo*)),
            m_basicTab, SLOT(setEntryInfo(MenuEntryInfo*)));
    connect(m_tree, &TreeView::disableAction,
            m_basicTab, &BasicTab::slotDisableAction);

    connect(m_basicTab, SIGNAL(changed(MenuFolderInfo*)),
            m_tree, SLOT(currentDataChanged(MenuFolderInfo*)));

    connect(m_basicTab, SIGNAL(changed(MenuEntryInfo*)),
            m_tree, SLOT(currentDataChanged(MenuEntryInfo*)));

    connect(m_basicTab, &BasicTab::findServiceShortcut,
            m_tree, &TreeView::findServiceShortcut);

    // restore splitter sizes
    QList<int> sizes = ConfigurationManager::getInstance()->getSplitterSizes();
    if (sizes.isEmpty()) {
        sizes << 1 << 3;
    }
    m_splitter->setSizes(sizes);
    m_tree->setFocus();

    setCentralWidget(m_splitter);
}

void KMenuEdit::selectMenu(const QString &menu)
{
    m_tree->selectMenu(menu);
}

void KMenuEdit::selectMenuEntry(const QString &menuEntry)
{
    m_tree->selectMenuEntry(menuEntry);
}

void KMenuEdit::slotChangeView()
{
    guiFactory()->removeClient(this);

    delete m_actionDelete;

    m_actionDelete = actionCollection()->addAction(DELETE_ACTION_NAME);
    m_actionDelete->setIcon(QIcon::fromTheme(QStringLiteral("edit-delete")));
    m_actionDelete->setText(i18n("&Delete"));
    actionCollection()->setDefaultShortcut(m_actionDelete, QKeySequence(Qt::Key_Delete));

    if (!m_splitter) {
        setupView();
    }
    setupGUI(KXmlGuiWindow::ToolBar|Keys|Save|Create, QStringLiteral("kmenueditui.rc"));

    m_tree->setViewMode(m_showHidden);
    m_basicTab->updateHiddenEntry(m_showHidden);
}

void KMenuEdit::slotSave()
{
    m_tree->save();
}

bool KMenuEdit::queryClose()
{
    if (!m_tree->dirty()) {
        return true;
    }

    int result;
    result = KMessageBox::warningYesNoCancel(this,
                                             i18n("You have made changes to the menu.\n"
                                                  "Do you want to save the changes or discard them?"),
                                             i18n("Save Menu Changes?"),
                                             KStandardGuiItem::save(), KStandardGuiItem::discard());

    switch (result) {
    case KMessageBox::Yes:
        return m_tree->save();

    case KMessageBox::No:
        return true;

    default:
        break;
    }
    return false;
}

void KMenuEdit::slotRestoreMenu()
{
    m_tree->restoreMenuSystem();
}

void KMenuEdit::restoreSystemMenu()
{
    m_tree->restoreMenuSystem();
}
