/**************************************************************************
              kstplotdialog_i.cpp - plot dialog: inherits designer dialog
                             -------------------
    begin                :  2000
    copyright            : (C) 2000-2002 by Barth Netterfield
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for Qt
#include <qcheckbox.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qfontdatabase.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qstring.h>
#include <qtabwidget.h>
#include <qtextstream.h>
#include <qwidget.h>

// include files for KDE
#include <kcolorbutton.h>
#include <kdebug.h>
#include <kdualcolorbutton.h>
#include <kfontcombo.h>
#include <klistbox.h>
#include <klocale.h>
#include <kmdimainfrm.h>
#include <kmessagebox.h>

// application specific includes
#include "kst.h"
#include "kstdatacollection.h"
#include "kstdebug.h"
#include "kstlegend.h"
#include "kstplotdialog_i.h"
#include "kstsettings.h"
#include "kstviewwindow.h"
#include "plotlistbox.h"

struct LegendLayout {
  QString Label;
  KstLegendLayoutType type;
};

struct LegendAlignment {
  QString Label;
  KstLegendAlignmentType type;
};

const LegendAlignment LegendAlignments[] = {
  { i18n("Right"), AlignmentRight },
  { i18n("Center"), AlignmentCentre },
  { i18n("Left"), AlignmentLeft }
};

const LegendLayout LegendLayouts[] = {
  { i18n("Top left"), LeftTop },
  { i18n("Top center"), CentreTop },
  { i18n("Top right"), RightTop },
  { i18n("Middle left"), LeftCentre },
  { i18n("Middle center"), CentreCentre },
  { i18n("Middle right"), RightCentre },
  { i18n("Bottom left"), LeftBottom },
  { i18n("Bottom center"), CentreBottom },
  { i18n("Bottom right"), RightBottom }
};

const unsigned int numLegendLayouts = sizeof( LegendLayouts ) / sizeof( LegendLayout );
const unsigned int numLegendAlignments = sizeof( LegendAlignments ) / sizeof( LegendAlignment );

/*
 *  Constructs a KstPlotDialogI which is a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'
 *
 *  The dialog will by default be modeless, unless you set 'modal' to
 *  TRUE to construct a modal dialog.
 */
KstPlotDialogI::KstPlotDialogI(KstDoc *in_doc, QWidget* parent,
                               const char* name, bool modal, WFlags fl)
: KstPlotDialog(parent, name, modal, fl ) {
  unsigned int i;

  doc = in_doc;
  setScalarDestTopLabel();

  connect(Select, SIGNAL(activated(int)), this, SLOT(update(int)));
  connect(TabWidget, SIGNAL(currentChanged(QWidget*)), this, SLOT(newTab()));
  connect(New, SIGNAL(clicked()), this, SLOT(new_I()));
  connect(Edit, SIGNAL(clicked()), this, SLOT(edit_I()));
  connect(Delete, SIGNAL(clicked()), this, SLOT(delete_I()));

  connect(DisplayedCurveList, SIGNAL(clicked(QListBoxItem*)),
          this, SLOT(updateButtons()));
  connect(AvailableCurveList, SIGNAL(clicked(QListBoxItem*)),
          this, SLOT(updateButtons()));
  connect(DisplayedCurveList, SIGNAL(selectionChanged()),
          this, SLOT(updateButtons()));
  connect(AvailableCurveList, SIGNAL(selectionChanged()),
          this, SLOT(updateButtons()));

  /* adding/removing curves */
  connect(CurveUnplot, SIGNAL(clicked()),
          this, SLOT(removeDisplayedCurve()));
  connect(CurvePlot, SIGNAL(clicked()),
          this, SLOT(addDisplayedCurve()));

  connect(_newWindow, SIGNAL(clicked()), this, SLOT(newWindow()));

  /* labels */
  connect(TopLabelText, SIGNAL(textChanged(const QString &)),
          this, SLOT(updateSampTopLabel()));
  connect(XAxisText, SIGNAL(textChanged(const QString &)),
          this, SLOT(updateSampXLabel()));
  connect(YAxisText, SIGNAL(textChanged(const QString &)),
          this, SLOT(updateSampYLabel()));
  connect(TopLabelFontSize, SIGNAL(valueChanged(int)),
          this, SLOT(updateSampTopLabel()));
  connect(XLabelFontSize, SIGNAL(valueChanged(int)),
          this, SLOT(updateSampXLabel()));
  connect(YLabelFontSize, SIGNAL(valueChanged(int)),
          this, SLOT(updateSampYLabel()));
  connect(NumberFontSize, SIGNAL(valueChanged(int)),
          this, SLOT(updateSampNumLabel()));
  connect(FontComboBox, SIGNAL(activated(int)),
          this, SLOT(updateLabels()));
  connect(FontComboBox, SIGNAL(highlighted(int)),
          this, SLOT(updateLabels()));
  connect(FontComboBox, SIGNAL(textChanged(const QString &)),
          this, SLOT(updateLabels()));
  connect(AutoLabel, SIGNAL(clicked()),
          this, SLOT(applyAutoLabels()));

  connect(ScalarList, SIGNAL(activated(int)),
          this, SLOT(insertCurrentScalar()));

  connect(YAxisText, SIGNAL(selectionChanged()),
          this, SLOT(setScalarDestYLabel()));
  connect(YAxisText, SIGNAL(textChanged(const QString &)),
          this, SLOT(setScalarDestYLabel()));
  connect(XAxisText, SIGNAL(selectionChanged()),
          this, SLOT(setScalarDestXLabel()));
  connect(XAxisText, SIGNAL(textChanged(const QString &)),
          this, SLOT(setScalarDestXLabel()));
  connect(TopLabelText, SIGNAL(selectionChanged()),
          this, SLOT(setScalarDestTopLabel()));
  connect(TopLabelText, SIGNAL(textChanged(const QString &)),
          this, SLOT(setScalarDestTopLabel()));

  SampleLabel = new KstLabel("\0", CxCy, 0.0, 0.0, 0.0, true);
  QFontDatabase qfd;
  FontComboBox->setFonts(qfd.families());

  connect(_window, SIGNAL(activated(int)), this, SLOT(updatePlotList()));

  /* Legend */
  connect(LegendFontSize, SIGNAL(valueChanged(int)),
          this, SLOT(updateSampLegendLabel()));
  connect(LegendFontComboBox, SIGNAL(textChanged(const QString &)),
          this, SLOT(updateSampLegendLabel()));
  connect(LegendFontComboBox, SIGNAL(activated(int)),
          this, SLOT(updateSampLegendLabel()));
  connect(LegendFontComboBox, SIGNAL(highlighted(int)),
          this, SLOT(updateSampLegendLabel()));
//   connect(LegendFontComboBox, SIGNAL(textChanged(const QString &)),
//           this, SLOT(updateSampLegendLabels()));
  connect(legendColors, SIGNAL(fgChanged(const QColor &)),
          this, SLOT(updateSampLegendLabel()));
  connect(legendColors, SIGNAL(bgChanged(const QColor &)),
          this, SLOT(updateSampLegendLabel()));
  connect(legendInFront, SIGNAL(clicked()),
          this, SLOT(updateSampLegendLabel()));

  LegendFontComboBox->setFonts(qfd.families());
  LegendPosition->hide();
  LegendPositionLabel->hide();
  LegendPosition->setEditable(false);
  for (i = 0; i < numLegendLayouts; i++) {
    LegendPosition->insertItem(LegendLayouts[i].Label);
  }

  LegendAlignment->setEditable(false);
  for (i = 0; i < numLegendAlignments; i++) {
    LegendAlignment->insertItem(LegendAlignments[i].Label);
  }

  radioButtonThisPlot->setChecked(true);
  radioButtonLegendThisPlot->setChecked(true);
  
  /* Plot Markers */
  connect(AddPlotMarker, SIGNAL(clicked()),
          this, SLOT(addPlotMarker()));
  connect(RemovePlotMarker, SIGNAL(clicked()),
          this, SLOT(removePlotMarker()));
  connect(RemoveAllPlotMarkers, SIGNAL(clicked()),
          this, SLOT(removeAllPlotMarkers()));
  connect(PlotMarkerList, SIGNAL(clicked(QListBoxItem*)),
          this, SLOT(updateButtons()));
  connect(PlotMarkerList, SIGNAL(selectionChanged()),
          this, SLOT(updateButtons()));
  connect(NewPlotMarker, SIGNAL(returnPressed()),
          this, SLOT(addPlotMarker()));
  connect(NewPlotMarker, SIGNAL(textChanged(const QString &)),
          this, SLOT(updateButtons()));
  connect(UseCurve, SIGNAL(clicked()),
          this, SLOT(updateButtons()));
}

KstPlotDialogI::~KstPlotDialogI() {
  delete SampleLabel;
  SampleLabel = 0L;
}

void KstPlotDialogI::show_I(const QString& window, const QString& plot) {
  _windowName = window;
  _plotName = plot;

  show();
  raise();
  update();
}

void KstPlotDialogI::show_I() {
  show();
  raise();
  update();
}

void KstPlotDialogI::newTab() {
  updateLabels();
  updateSampLegendLabel();
}

void KstPlotDialogI::updateWindow() {
  updateWindowList();
}

void KstPlotDialogI::update(int new_index) {
  Q_UNUSED(new_index)

  if (!isShown()) {
    return;
  }

  updateScalarCombo();
  updateWindowList();
  updatePlotList();
  updateCurveLists();
  updatePlotMarkers();

  int i_plot;
  if (Select->count() > 0) {
    i_plot = Select->currentItem();
  } else {
    i_plot = -1;
  }

  KstApp *app = KstApp::inst();
  KMdiChildView *c = app->findWindow(_window->currentText());

  if (c) {
    KstTopLevelViewPtr view = static_cast<KstViewWindow*>(c)->view();
    _reGrid->setChecked(view->onGrid());
    _plotColumns->setValue(view->columns());
  }

  if (i_plot >= 0 && c) {
    KstViewObjectPtr obj = static_cast<KstViewWindow*>(c)->view()->findChild(Select->currentText());
    Kst2DPlotPtr plot = dynamic_cast<Kst2DPlot*>(obj.data());
    if (!plot) {
      return;
    }

    /* insert the current plot name in the plot name edit box */
    Name->setText(plot->tagName());

    /* file the log axis check boxes */
    XIsLog->setChecked(plot->isXLog());
    YIsLog->setChecked(plot->isYLog());

    double xmin, ymin, xmax, ymax;
    plot->getScale(xmin, ymin, xmax, ymax);

    XFixedMin->setText(QString::number(xmin));
    XFixedMax->setText(QString::number(xmax));
    YFixedMin->setText(QString::number(ymin));
    YFixedMax->setText(QString::number(ymax));

    XACRange->setText(QString::number(xmax - xmin));
    YACRange->setText(QString::number(ymax - ymin));

    if (plot->getXScaleMode() == AUTO) {
      XAuto->setChecked(true);
    } else if (plot->getXScaleMode() == AC) {
      XAC->setChecked(true);
    } else if (plot->getXScaleMode() == FIXED) {
      XFixed->setChecked(true);
    } else if (plot->getXScaleMode() == AUTOUP) {
      XAutoUp->setChecked(true);
    } else if (plot->getXScaleMode() == NOSPIKE) {
      XNoSpikes->setChecked(true);
    } else {
      XAuto->setChecked(true);
    }

    if (plot->getYScaleMode() == AUTO) {
      YAuto->setChecked(true);
    } else if (plot->getYScaleMode() == AC) {
      YAC->setChecked(true);
    } else if (plot->getYScaleMode() == FIXED) {
      YFixed->setChecked(true);
    } else if (plot->getYScaleMode() == AUTOUP) {
      YAutoUp->setChecked(true);
    } else if (plot->getYScaleMode() == NOSPIKE) {
      YNoSpikes->setChecked(true);
    } else {
      YAuto->setChecked(true);
    }

    NumberFontSize->setValue(plot->XTickLabel->size());

    XAxisText->setText(plot->XLabel->text());
    XLabelFontSize->setValue(plot->XLabel->size());

    YAxisText->setText(plot->YLabel->text());
    YLabelFontSize->setValue(plot->YLabel->size());

    TopLabelText->setText(plot->TopLabel->text());
    TopLabelFontSize->setValue(plot->TopLabel->size());

    FontComboBox->setCurrentFont(plot->TopLabel->fontName());

    updateLabels();

    /* initialize all the legend settings for the current plot */
    LegendFontComboBox->setCurrentFont(plot->Legend->fontName());
    LegendFontSize->setValue(plot->Legend->size());
    ShowLegend->setChecked(plot->Legend->getShow());

    legendInFront->setChecked(plot->Legend->getFront());
    legendColors->setForeground(plot->Legend->getColorForeground());
    legendColors->setBackground(plot->Legend->getColorBackground());

    updateSampLegendLabel();

    /*initialize the plot color widget */
    plotColors->setForeground(plot->foregroundColor());
    plotColors->setBackground(plot->backgroundColor());
  }

  //UpdateDelay->setValue(doc->delay());

  updateButtons();
}

void KstPlotDialogI::fill2DPlotList(Kst2DPlotList& plots) {
  KstApp *app = KstApp::inst();
  KMdiIterator<KMdiChildView*> *it = app->createIterator();
  while (it->currentItem()) {
    KstViewWindow *c = dynamic_cast<KstViewWindow*>(it->currentItem());
    if (c) {
      KstPlotBaseList allplots = c->view()->findChildrenType<KstPlotBase>(true);
      for (KstPlotBaseList::Iterator i = allplots.begin(); i != allplots.end(); ++i) {
        Kst2DPlotPtr plot = dynamic_cast<Kst2DPlot*>((*i).data());
        if (plot) {
          plots += plot;
        }
      }
    }
    it->next();
  }
  app->deleteIterator(it);
}


void KstPlotDialogI::applyLabels(Kst2DPlotPtr plot) {
  Kst2DPlotList plots;

  plot->XLabel->setText(XAxisText->text());
  plot->YLabel->setText(YAxisText->text());
  plot->TopLabel->setText(TopLabelText->text());

  if (radioButtonThisPlot->isChecked()) {
    plots += plot;
  } else if (radioButtonAllPlotsInWindow->isChecked()) {
    KMdiChildView *c = KstApp::inst()->findWindow(_window->currentText());
    if (c) {
      plots  = static_cast<KstViewWindow*>(c)->view()->findChildrenType<Kst2DPlot>(true);
    }
  } else {
    fill2DPlotList(plots);
  }

  for (uint i = 0; i < plots.size(); i++) {
    plot = plots[i];

    plot->XLabel->setFontName(FontComboBox->currentText());
    plot->XLabel->setSize(XLabelFontSize->value());

    plot->YLabel->setFontName(FontComboBox->currentText());
    plot->YLabel->setSize(YLabelFontSize->value());

    plot->TopLabel->setFontName(FontComboBox->currentText());
    plot->TopLabel->setSize(TopLabelFontSize->value());

    plot->XTickLabel->setFontName(FontComboBox->currentText());
    plot->XTickLabel->setSize(NumberFontSize->value());
    plot->XFullTickLabel->setFontName(FontComboBox->currentText());
    plot->XFullTickLabel->setSize(NumberFontSize->value());
    plot->YTickLabel->setFontName(FontComboBox->currentText());
    plot->YTickLabel->setSize(NumberFontSize->value());
    plot->YFullTickLabel->setFontName(FontComboBox->currentText());
    plot->YFullTickLabel->setSize(NumberFontSize->value());
  }
}


void KstPlotDialogI::applyLegends(Kst2DPlotPtr plot) {
  KstLegendLayoutType layout = LeftTop;
  KstLegendAlignmentType alignment = AlignmentRight;
  Kst2DPlotList plots;
  unsigned int i;

  if (radioButtonLegendThisPlot->isChecked()) {
    plots += plot;
  } else if (radioButtonAllPlotsInWindow->isChecked()) {
    KMdiChildView *c = KstApp::inst()->findWindow(_window->currentText());
    if (c) {
      plots = static_cast<KstViewWindow*>(c)->view()->findChildrenType<Kst2DPlot>(true);
    }
  } else {
    fill2DPlotList(plots);
  }

  const QString& layoutName = LegendPosition->currentText();
  for (i = 0; i < numLegendLayouts; i++) {
    if (layoutName == LegendLayouts[i].Label) {
      layout = LegendLayouts[i].type;
      break;
    }
  }

  const QString& alignmentName = LegendAlignment->currentText();
  for (i = 0; i < numLegendAlignments; i++) {
    if (alignmentName == LegendAlignments[i].Label) {
      alignment = LegendAlignments[i].type;
      break;
    }
  }

  for (i = 0; i < plots.size(); i++) {
    plot = plots[i];

    plot->Legend->setFontName(LegendFontComboBox->currentText());
    plot->Legend->setSize(LegendFontSize->value());
    plot->Legend->setShow(ShowLegend->isChecked());
    plot->Legend->setLayout(layout);
    plot->Legend->setAlignment(alignment);
    plot->Legend->setFront(legendInFront->isChecked());
    plot->Legend->setColorForeground(legendColors->foreground());
    plot->Legend->setColorBackground(legendColors->background());
  }
}


void KstPlotDialogI::applyLimits(Kst2DPlotPtr plot) {
  /* do X Scale */
  if (XAC->isChecked()) {
    plot->setXScaleMode(AC);
    plot->setXScale(0, XACRange->text().toDouble());
  } else if (XFixed->isChecked()) {
    plot->setXScaleMode(FIXED);
    plot->setXScale(XFixedMin->text().toDouble(), XFixedMax->text().toDouble());
  } else if (XAutoUp->isChecked()) {
    plot->setXScaleMode(AUTOUP);
  } else if (XAuto->isChecked()) {
    plot->setXScaleMode(AUTO);
  } else if (XNoSpikes->isChecked()) {
    plot->setXScaleMode(NOSPIKE);
  } else {
    KstDebug::self()->log(i18n("Internal error: No X scale type checked in %1.").arg(Select->currentText()), KstDebug::Error);
  }

    /* do Y Scale */
  if (YAC->isChecked()) {
    plot->setYScaleMode(AC);
    plot->setYScale(0, YACRange->text().toDouble());
  } else if (YFixed->isChecked()) {
    plot->setYScaleMode(FIXED);
    plot->setYScale(YFixedMin->text().toDouble(), YFixedMax->text().toDouble());
  } else if (YAutoUp->isChecked()) {
    plot->setYScaleMode(AUTOUP);
  } else if (YAuto->isChecked()) {
    plot->setYScaleMode(AUTO);
  } else if (YNoSpikes->isChecked()) {
    plot->setYScaleMode(NOSPIKE);
  } else {
    KstDebug::self()->log(i18n("Internal error: No Y scale type checked in %1.").arg(Select->currentText()), KstDebug::Error);
  }

  plot->setLog(XIsLog->isChecked(), YIsLog->isChecked());
}

void KstPlotDialogI::applyPlotColors(Kst2DPlotPtr plot) {
  plot->setForegroundColor(plotColors->foreground());
  plot->setBackgroundColor(plotColors->background());
}


void KstPlotDialogI::new_I() {
  KMdiChildView *c = KstApp::inst()->findWindow(_window->currentText());
  if (!c) {
    QString name = KstApp::inst()->newWindow();
    if (name.isEmpty()) {
      return;
    }
    c = KstApp::inst()->findWindow(name);
    _window->insertItem(name);
    _window->setCurrentText(name);
  }

  //check plot name
  if (!checkPlotName()) {
    return;
  }

  Kst2DPlotPtr plot = static_cast<KstViewWindow*>(c)->view()->createPlot<Kst2DPlot>(Name->text());

  if (_reGrid->isChecked()) {
    static_cast<KstViewWindow*>(c)->view()->cleanup(_plotColumns->value());
  }

  KstBaseCurveList curves = kstObjectSubList<KstDataObject, KstBaseCurve>(KST::dataObjectList);
  /* add the curves */
  plot->Curves.clear();
  for (unsigned i = 0; i < DisplayedCurveList->count(); i++) {
    KstBaseCurveList::Iterator it = curves.findTag(DisplayedCurveList->text(i));
    if (it != curves.end()) {
      plot->Curves.append(*it);
    }
  }
  curves.clear();
  //also add the images
  KstImageList images = kstObjectSubList<KstDataObject, KstImage>(KST::dataObjectList);
  plot->removeAllImages();
  for (unsigned i = 0; i < DisplayedCurveList->count(); i++) {
    KstImageList::Iterator it = images.findTag(DisplayedCurveList->text(i));
    if (it != images.end()) {
      plot->addImage(*it);
    }
  }
  images.clear();

  applyLabels(plot);
  applyLegends(plot);
  //plot->GenerateDefaultLabels();

  applyLimits(plot);

//   plot->setForegroundColor(Colors->foreground());
//   plot->setBackgroundColor(Colors->background());
  applyPlotColors(plot);
  applyPlotMarkers(plot);


  static_cast<KstViewWindow*>(c)->view()->paint(P_PLOT);

  //doc->setDelay(UpdateDelay->value());

  _plotName = Name->text();
  update();

  emit docChanged();
}

void KstPlotDialogI::edit_I() {
  int index = Select->currentItem();
  if (index < 0) {
    KMessageBox::sorry(0L, i18n("You need to select an active plot to edit."));
    return;
  }

  //check the plot name
  if (Name->text() != Select->currentText()) {
    if (!checkPlotName()) {
    return;
    }
  }

  KMdiChildView *c = KstApp::inst()->findWindow(_window->currentText());
  if (!c) {
    return;
  }

  Kst2DPlotList plots = static_cast<KstViewWindow*>(c)->view()->findChildrenType<Kst2DPlot>(true);

  if (index >= (int)plots.count()) {
    new_I();
  } else {
    Kst2DPlotPtr plot = *plots.at(index);

    plot->setTagName(Name->text());
    /* add the curves */
    plot->Curves.clear();
    KstBaseCurveList curves = kstObjectSubList<KstDataObject, KstBaseCurve>(KST::dataObjectList);
    for (unsigned i = 0; i < DisplayedCurveList->count(); i++) {
      KstBaseCurveList::Iterator it = curves.findTag(DisplayedCurveList->text(i));
      if (it != curves.end()) {
        plot->Curves.append(*it);
      }
    }
    curves.clear();
    //also add the images
    KstImageList images = kstObjectSubList<KstDataObject, KstImage>(KST::dataObjectList);
    plot->removeAllImages();
    for (unsigned i = 0; i < DisplayedCurveList->count(); i++) {
      KstImageList::Iterator it = images.findTag(DisplayedCurveList->text(i));
      if (it != images.end()) {
        plot->addImage(*it);
      }
    }
    images.clear();

    applyLabels(plot);
    applyLegends(plot);
    applyLimits(plot);
    applyPlotColors(plot);
    applyPlotMarkers(plot);

//  FIXME: why is this commented out?
//     if (ApplyToAll->isChecked()) {
//       for (unsigned i = 0; i < KST::plotList.count(); i++) {
//         KST::plotList.at(i)->setForegroundColor(Colors->foreground());
//         KST::plotList.at(i)->setBackgroundColor(Colors->background());
//       }
//     } else {
//       plot->setForegroundColor(Colors->foreground());
//       plot->setBackgroundColor(Colors->background());
//     }

    if (_reGrid->isChecked()) {
      static_cast<KstViewWindow*>(c)->view()->cleanup(_plotColumns->value());
    }

    plot->setDirty();
    static_cast<KstViewWindow*>(c)->view()->paint(P_PLOT);

    _plotName = Name->text().stripWhiteSpace();
    update();
    emit docChanged();
  }
}

void KstPlotDialogI::delete_I() {
  int index = Select->currentItem();
  if (index < 0) {
    KMessageBox::sorry(0L, i18n("You need to select an active plot to delete."));
    return;
  }

  KMdiChildView *c = KstApp::inst()->findWindow(_window->currentText());
  if (!c) {
    return;
  }

  KstViewObjectPtr obj = static_cast<KstViewWindow*>(c)->view()->findChild(Select->currentText());
  static_cast<KstViewWindow*>(c)->view()->removeChild(obj);

  if (_reGrid->isChecked()) {
    static_cast<KstViewWindow*>(c)->view()->cleanup(_plotColumns->value());
  }

  update();

  static_cast<KstViewWindow*>(c)->view()->paint(P_PLOT);

  emit docChanged();
}

void KstPlotDialogI::updateSampNumLabel() {
  if (!FontComboBox->currentText().isEmpty()) {
    QPixmap pix(SampleNumLabel->size());
    QPainter p(&pix);

    p.fillRect(p.window(), KstSettings::globalSettings()->backgroundColor);
    p.setPen(KstSettings::globalSettings()->foregroundColor);

    SampleLabel->setFontName(FontComboBox->currentText());
    SampleLabel->setSize(NumberFontSize->value());
    SampleLabel->setText("1.234E-5");
    SampleLabel->draw(p,SampleNumLabel->width()/2,
                      SampleNumLabel->height()/2,true);

    SampleNumLabel->setPixmap(pix);

    QFont font(FontComboBox->currentText(), NumberFontSize->value()+12);
    SampleNumLabel->setFont(font);
  } else {
    QFont font;
    SampleNumLabel->setFont(font);
  }
}

void KstPlotDialogI::updateSampTopLabel() {
  if (!FontComboBox->currentText().isEmpty()) {
    QPixmap pix(SampleTopLabel->size());
    QPainter p(&pix);

    p.fillRect(p.window(), KstSettings::globalSettings()->backgroundColor);
    p.setPen(KstSettings::globalSettings()->foregroundColor);

    SampleLabel->setFontName(FontComboBox->currentText());
    SampleLabel->setSize(TopLabelFontSize->value());
    SampleLabel->setText(TopLabelText->text());
    SampleLabel->draw(p,SampleTopLabel->width()/2,
                      SampleTopLabel->height()/2,true);

    SampleTopLabel->setPixmap(pix);
  } else {
    QFont font;
    SampleTopLabel->setFont(font);
  }
}

void KstPlotDialogI::updateSampXLabel() {
  if (!FontComboBox->currentText().isEmpty()) {
    QPixmap pix(SampleXLabel->size());
    QPainter p(&pix);

    p.fillRect(p.window(), KstSettings::globalSettings()->backgroundColor);
    p.setPen(KstSettings::globalSettings()->foregroundColor);

    SampleLabel->setFontName(FontComboBox->currentText());
    SampleLabel->setSize(XLabelFontSize->value());
    SampleLabel->setText(XAxisText->text());
    SampleLabel->draw(p,SampleXLabel->width()/2,
                      SampleXLabel->height()/2,true);

    SampleXLabel->setPixmap(pix);
  } else {
    QFont font;
    SampleXLabel->setFont(font);
  }
}

void KstPlotDialogI::updateSampYLabel() {
  if (!FontComboBox->currentText().isEmpty()) {
    QPixmap pix(SampleYLabel->size());
    QPainter p(&pix);

    p.fillRect(p.window(), KstSettings::globalSettings()->backgroundColor);
    p.setPen(KstSettings::globalSettings()->foregroundColor);

    SampleLabel->setFontName(FontComboBox->currentText());
    SampleLabel->setSize(YLabelFontSize->value());
    SampleLabel->setText(YAxisText->text());
    SampleLabel->draw(p,SampleYLabel->width()/2,
                      SampleYLabel->height()/2,true);

    SampleYLabel->setPixmap(pix);
   } else {
    QFont font;
    SampleYLabel->setFont(font);
  }
}

void KstPlotDialogI::updateLabels() {
  updateSampYLabel();
  updateSampXLabel();
  updateSampTopLabel();
  updateSampNumLabel();
}


void KstPlotDialogI::updateSampLegendLabel() {
  if (!LegendFontComboBox->currentText().isEmpty()) {
    QPixmap pix(SampleLegendLabel->size());
    QPainter p(&pix);

    p.fillRect(p.window(), KstSettings::globalSettings()->backgroundColor);
    p.setPen(KstSettings::globalSettings()->foregroundColor);
    if (legendInFront->isChecked()) {
      p.fillRect(p.window(), legendColors->background());
      p.setPen(legendColors->foreground());
    } else {
      p.fillRect(p.window(), KstSettings::globalSettings()->backgroundColor);
      p.setPen(KstSettings::globalSettings()->foregroundColor);
    }

    SampleLabel->setFontName(LegendFontComboBox->currentText());
    SampleLabel->setSize(LegendFontSize->value());
    SampleLabel->setText(i18n("Curve name"));
    SampleLabel->draw(p, SampleLegendLabel->width()/2,
                      SampleLegendLabel->height()/2, true);

    SampleLegendLabel->setPixmap(pix);
  } else {
    QFont font;
    SampleLegendLabel->setFont(font);
  }
}

void KstPlotDialogI::updatePlotMarkers() {
  KstApp* app = KstApp::inst();

  PlotMarkerList->clear();

  KMdiChildView *c = app->findWindow(_window->currentText());
  if (Select->count() > 0 && c) {
    KstViewObjectPtr obj = static_cast<KstViewWindow*>(c)->view()->findChild(Select->currentText());
    Kst2DPlotPtr plot = dynamic_cast<Kst2DPlot*>(obj.data());

    if (plot) {
      for (QValueList<double>::ConstIterator it = plot->plotMarkers().begin(); it != plot->plotMarkers().end(); it++) {
        PlotMarkerList->insertItem(QString::number(*it, 'g', 64));
      }
      //update the auto-generation settings
      KstBaseCurveList curves = kstObjectSubList<KstDataObject, KstBaseCurve>(KST::dataObjectList);
      CurveCombo->clear();
      for (KstBaseCurveList::iterator curves_iter = curves.begin(); curves_iter != curves.end(); curves_iter++) {
        (*curves_iter)->readLock();
        CurveCombo->insertItem((*curves_iter)->tagName());
        (*curves_iter)->readUnlock();
      }
      if (plot->hasCurveToMarkers()) {
        UseCurve->setChecked(true);
        Both->setChecked(false);
        Falling->setChecked(false);
        Rising->setChecked(false);
        if (plot->curveToMarkersFallingDetect()) {
          if (plot->curveToMarkersRisingDetect()) {
            Both->setChecked(true);
          } else {
            Falling->setChecked(true);
          }
        } else {
          Rising->setChecked(true);
        }
        int curveComboIndex;
        for (curveComboIndex = 0; curveComboIndex < CurveCombo->count(); curveComboIndex++) {
          if (CurveCombo->text(curveComboIndex) == plot->curveToMarkers()->tagName()) {
            break;
          }
        }
        CurveCombo->setCurrentItem(curveComboIndex);
      }
    }
  }

}


void KstPlotDialogI::removeDisplayedCurve() {
  uint count = DisplayedCurveList->count();

  if (count > 0) {
    for (int i = count-1; i >= 0; i--) {
      if (DisplayedCurveList->isSelected(i)) {
        AvailableCurveList->insertItem(DisplayedCurveList->text(i));
        DisplayedCurveList->removeItem(i);
      }
    }
    updateButtons();
  }
}


void KstPlotDialogI::removeDisplayedCurve(QListBoxItem* item){
  int index = DisplayedCurveList->index(item);
  if (index != -1) {
    QString str = DisplayedCurveList->text(index);
    DisplayedCurveList->removeItem(index);
    AvailableCurveList->insertItem(str);
    updateButtons();
  }
}


void KstPlotDialogI::addDisplayedCurve() {
  uint count = AvailableCurveList->count();

  if (count > 0) {
    for (int i = count-1; i >= 0; i--) {
      if (AvailableCurveList->isSelected(i)) {
        DisplayedCurveList->insertItem(AvailableCurveList->text(i));
        AvailableCurveList->removeItem(i);
      }
    }
    updateButtons();
  }
}


void KstPlotDialogI::addDisplayedCurve(QListBoxItem* pItem){
  int index = AvailableCurveList->index(pItem);
  if (index != -1) {
    QString str = AvailableCurveList->text(index);
    AvailableCurveList->removeItem(index);
    DisplayedCurveList->insertItem(str);
    updateButtons();
  }
}


void KstPlotDialogI::applyAutoLabels() {
  KMdiChildView *c = KstApp::inst()->findWindow(_window->currentText());
  if (!c) {
    return;
  }

  KstViewObjectPtr obj = static_cast<KstViewWindow*>(c)->view()->findChild(Select->currentText());
  Kst2DPlotPtr plot = dynamic_cast<Kst2DPlot*>(obj.data());
  if (plot) {
    plot->GenerateDefaultLabels();
    update();
  }
}


void KstPlotDialogI::updateButtons() {
  bool selected = false;
  uint count = AvailableCurveList->count();

  for (uint i = 0; i < count; i++) {
    if (AvailableCurveList->isSelected(i)) {
      selected = true;
    }
  }
  if (selected && !CurvePlot->isEnabled()) {
    CurvePlot->setEnabled(true);
  } else if (!selected && CurvePlot->isEnabled()) {
    CurvePlot->setEnabled(false);
  }

  selected = false;
  count = DisplayedCurveList->count();
  for (uint i = 0; i < count; i++) {
    if (DisplayedCurveList->isSelected(i)) {
      selected = true;
    }
  }
  if (selected && !CurveUnplot->isEnabled()) {
    CurveUnplot->setEnabled(true);
  } else if (!selected && CurveUnplot->isEnabled()) {
    CurveUnplot->setEnabled(false);
  }

  /** updates for Plot Markers tab **/
  AddPlotMarker->setEnabled(!NewPlotMarker->text().isEmpty());

  selected = false;
  count = PlotMarkerList->count();
  for (uint i = 0; i < count; i++) {
    if (PlotMarkerList->isSelected(i)) {
      selected = true;
    }
  }
  RemovePlotMarker->setEnabled(selected);
  RemoveAllPlotMarkers->setEnabled(count > 0);

  //updates for auto-generation marker curve section
  CurveCombo->setEnabled(UseCurve->isChecked());
  Rising->setEnabled(UseCurve->isChecked());
  Falling->setEnabled(UseCurve->isChecked());
  Both->setEnabled(UseCurve->isChecked());

}

void KstPlotDialogI::updateCurveLists() {
  KstApp* app = KstApp::inst();
  KstBaseCurveList curves = kstObjectSubList<KstDataObject, KstBaseCurve>(KST::dataObjectList);
  KstImageList images = kstObjectSubList<KstDataObject, KstImage>(KST::dataObjectList);

  DisplayedCurveList->clear();
  AvailableCurveList->clear();

  KMdiChildView *c = app->findWindow(_window->currentText());
  if (Select->count() && c) {
    KstViewObjectPtr obj = static_cast<KstViewWindow*>(c)->view()->findChild(Select->currentText());
    Kst2DPlotPtr plot = dynamic_cast<Kst2DPlot*>(obj.data());
    if (!plot) {
      return;
    }
    for (KstBaseCurveList::iterator it = curves.begin(); it != curves.end(); it++) {
      (*it)->readLock();
      if (plot->Curves.find(*it) == plot->Curves.end()) {
        AvailableCurveList->insertItem((*it)->tagName());
      } else {
        DisplayedCurveList->insertItem((*it)->tagName());
      }
      (*it)->readUnlock();
    }
    //add images as well
    for (KstImageList::iterator it = images.begin(); it != images.end(); it++) {
      (*it)->readLock();
      if (plot->hasImage(*it)) {
        DisplayedCurveList->insertItem((*it)->tagName());
      } else {
        AvailableCurveList->insertItem((*it)->tagName());
      }
      (*it)->readUnlock();
    }
  } else {
    for (KstBaseCurveList::iterator it = curves.begin(); it != curves.end(); it++) {
      (*it)->readLock();
      AvailableCurveList->insertItem((*it)->tagName());
      (*it)->readUnlock();
    }
    //add images as well
    for (KstImageList::iterator it = images.begin(); it != images.end(); it++) {
      (*it)->readLock();
      AvailableCurveList->insertItem((*it)->tagName());
      (*it)->readUnlock();
    }
  }
}

void KstPlotDialogI::updateScalarCombo() {
  ScalarList->clear();
  KST::scalarList.lock().readLock();
  for (int index = 0; index < (int)KST::scalarList.count(); index++) {
    KstScalarPtr s = KST::scalarList[index];
    s->readLock();
    ScalarList->insertItem(s->tagLabel());
    s->readUnlock();
  }
  KST::scalarList.lock().readUnlock();
}

void KstPlotDialogI::updateWindowList() {
  KstApp *app = KstApp::inst();
  KMdiIterator<KMdiChildView*> *it = app->createIterator();
  QString oldText;

  if (_window->count()) {
    oldText = _window->currentText();
  }

  _window->clear();

  while (it->currentItem()) {
    KstViewWindow *v = dynamic_cast<KstViewWindow*>(it->currentItem());
    if (v) {
      _window->insertItem(v->caption());
      if (!oldText.isEmpty() && oldText == v->caption()) {
        _window->setCurrentItem(_window->count() - 1);
      }
      if (!_windowName.isEmpty() && _windowName == v->caption()) {
        _window->setCurrentItem(_window->count() - 1);
        oldText = QString::null;
      }
    }
    it->next();
  }

  app->deleteIterator(it);

  _windowName = QString::null;
}

void KstPlotDialogI::updatePlotList() {
  KstApp *app = KstApp::inst();
  QString oldName;

  if (Select->count()) {
    oldName = Select->currentText();
  }

  Select->clear();

  KMdiChildView *c = app->findWindow(_window->currentText());
  if (c) {
    Kst2DPlotList plots = static_cast<KstViewWindow*>(c)->view()->findChildrenType<Kst2DPlot>(true);

    for (Kst2DPlotList::Iterator i = plots.begin(); i != plots.end(); ++i) {
      Select->insertItem((*i)->tagName());
      if (!oldName.isEmpty() && oldName == (*i)->tagName()) {
        Select->setCurrentItem(Select->count() - 1);
      }
      if (!_plotName.isEmpty() && _plotName == (*i)->tagName()) {
        Select->setCurrentItem(Select->count() - 1);
        oldName = QString::null;
      }
    }
  }

  _plotName = QString::null;
}

void KstPlotDialogI::addPlotMarker() {
  //silently do nothing if there is no text, to be consistent
  //with "Add" button disabled
  if (NewPlotMarker->text().isEmpty()) {
    return;
  }

  bool ok;
  double newMarkerVal = NewPlotMarker->text().toDouble(&ok);

  if (ok) {
    uint i = 0;
    QString stringnum;
    stringnum.setNum(newMarkerVal, 'g', 64);
    while (i < PlotMarkerList->count() && PlotMarkerList->text(i).toDouble() < newMarkerVal) {
      i++;
    }
    if (i == PlotMarkerList->count()) {
      PlotMarkerList->insertItem(stringnum, -1);
      NewPlotMarker->clear();
      return;
    }
    if (newMarkerVal != PlotMarkerList->text(i).toDouble()) {
      PlotMarkerList->insertItem(stringnum, i);
      NewPlotMarker->clear();
      return;
    }
    KMessageBox::sorry(this,
                       i18n("A plot marker with equal (or very close) value already exists."),
                       i18n("Kst"));
    return;
  }
  KMessageBox::sorry(this,
                     i18n("The text you have entered is not a valid number. Please enter a valid number."),
                     i18n("Kst"));
}

void KstPlotDialogI::removePlotMarker() {
  uint count = PlotMarkerList->count();
  if (count > 0) {
    for (int i = count-1; i >= 0; i--) {
      if (PlotMarkerList->isSelected(i)) {
        PlotMarkerList->removeItem(i);
      }
    }
    updateButtons();
  }
}

void KstPlotDialogI::removeAllPlotMarkers() {
  uint count = PlotMarkerList->count();
  if (count > 0) {
    for (int i = count-1; i >= 0; i--) {
      PlotMarkerList->removeItem(i);
    }
    updateButtons();
  }
}

void KstPlotDialogI::applyPlotMarkers(Kst2DPlotPtr plot) {
  QValueList<double> newMarkers;
  for (unsigned i = 0; i < PlotMarkerList->count(); i++) {
    newMarkers.append(PlotMarkerList->text(i).toDouble());
  }
  plot->setPlotMarkerList(newMarkers);

  //apply the autogeneration settings
  if (UseCurve->isChecked()) {
    KstBaseCurveList curves = kstObjectSubList<KstDataObject, KstBaseCurve>(KST::dataObjectList);
    KstBaseCurveList::iterator curves_iter = curves.findTag(CurveCombo->currentText());
    bool falling = Falling->isChecked(), rising = Rising->isChecked();
    if (Both->isChecked()) {
      falling = true;
      rising = true;
    }
    plot->setCurveToMarkers(*curves_iter, rising, falling);
  } else {
    plot->removeCurveToMarkers();
  }
}

void KstPlotDialogI::newWindow() {
  QString name = KstApp::inst()->newWindow();
  if (!name.isEmpty()) {
    _window->setCurrentText(name);
    updatePlotList();
  }
}

bool KstPlotDialogI::checkPlotName() {
  KstApp *app = KstApp::inst();
  KMdiIterator<KMdiChildView*> *iter;
  QString name = Name->text();
  //check the name
  KstViewObjectPtr rc;
  iter = app->createIterator();
  while (iter->currentItem()) {
    KMdiChildView *childview = iter->currentItem();
    KstViewWindow *viewwindow = dynamic_cast<KstViewWindow*>(childview);
    if (viewwindow) {
      rc = viewwindow->view()->findChild(name);
      if (rc) {
        QString message = i18n("Could not create a new plot.\n"
            "%1: not a unique plot name.\n"
            "Change it to a unique name.").arg(Name->text());

        KMessageBox::sorry(this, message);
        Name->setFocus();
        app->deleteIterator(iter);
        return false;
      }
    }
    iter->next();
  }
  app->deleteIterator(iter);
  return true;
}

#include "kstplotdialog_i.moc"
// vim: et ts=2 sw=2
