/***************************************************************************
                   kstdatasource.cpp  -  abstract data source
                             -------------------
    begin                : Thu Oct 16 2003
    copyright            : (C) 2003 The University of Toronto
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kstdatasource.h"

#include <assert.h>

#include <kdebug.h>
#include <klibloader.h>
#include <klocale.h>
#include <kservice.h>
#include <kservicetype.h>

#include <qdeepcopy.h>
#include <qfile.h>
#include <qstylesheet.h>

#include "kstdatacollection.h"
#include "kstdebug.h"
#include "stdinsource.h"


// Eventually this will move to another file but I leave it here until then
// to avoid confusion between the function plugins and Kst applicaton plugins.
namespace KST {
  class Plugin : public KstShared {
    public:
      Plugin(KService::Ptr svc) : KstShared(), service(svc), _lib(0L) {
        assert(service);
        _plugLib = service->property("X-Kst-Plugin-Library").toString();
        //kdDebug() << "Create plugin " << (void*)this << " " << service->property("Name").toString() << endl;
      }

      virtual ~Plugin() {
        //kdDebug() << "Destroy plugin " << (void*)this << " " << service->property("Name").toString() << endl;
        if (_lib) {
          _lib->unload();
        }
      }

      KstDataSource *create(const QString& filename, const QString& type = QString::null) const {
        KstDataSource *(*sym)(const QString&, const QString&) = (KstDataSource*(*)(const QString&, const QString&))symbol("create");
        if (sym) {
          //kdDebug() << "Trying to create " << filename << " type=" << type << " with " << service->property("Name").toString() << endl;
          KstDataSource *ds = (sym)(filename, type);
          if (ds) {
            ds->_source = service->property("Name").toString();
          }
          //kdDebug() << (ds ? "SUCCESS" : "FAILED") << endl;
          return ds;
        }

        return 0L;
      }

      QStringList fieldList(const QString& filename, const QString& type = QString::null) const {
        QStringList (*sym)(const QString&, const QString&) = (QStringList(*)(const QString&, const QString&))symbol("fieldList");
        if (sym) {
          return (sym)(filename, type);
        }

        // fallback incase the helper isn't implemented
        //  (note: less efficient now)
        KstDataSourcePtr ds = create(filename, type);
        if (ds) {
          return ds->fieldList();
        }

        return QStringList();
      }

      int understands(const QString& filename) const {
        int (*sym)(const QString&) = (int(*)(const QString&))symbol("understands");
        if (sym) {
          //kdDebug() << "Checking if " << service->property("Name").toString() << " understands " << filename << endl;
          int rc = (sym)(filename);
          //kdDebug() << "result: " << rc << endl;
          return rc;
        }

        return false;
      }

      bool provides(const QString& type) const {
        QStringList (*sym)() = (QStringList(*)())symbol("provides");
        if (sym) {
          //kdDebug() << "Checking if " << service->property("Name").toString() << " provides " << type << endl;
          bool rc = ((sym)()).contains(type);
          //kdDebug() << "result: " << rc << endl;
          return rc;
        }

        return false;
      }

      KService::Ptr service;

    private:
      void *symbol(const QString& sym) const {
        if (!loadLibrary()) {
          return 0L;
        }

        // FIXME: might be a good idea to cache this per-symbol

        return _lib->symbol(QFile::encodeName(sym + "_" + _plugLib));
      }

      bool loadLibrary() const {
        assert(service);
        if (_lib) {
          return true;
        }

        QCString libname = QFile::encodeName(QString("kstdata_") + _plugLib);
        _lib = KLibLoader::self()->library(libname);
        if (!_lib) {
          KstDebug::self()->log(i18n("Error loading data-source plugin [%1]: %2").arg(libname).arg(KLibLoader::self()->lastErrorMessage()), KstDebug::Error);
        }
        return _lib != 0L;
      }

      QString _plugLib;
      // mutable so we can lazy load the library, but at the same time
      // use const iterators and provide a nice const interface
      mutable KLibrary *_lib;
  };

  typedef QValueList<KstSharedPtr<KST::Plugin> > PluginInfoList;
}


static KST::PluginInfoList pluginInfo;


// Scans for plugins and stores the information for them in "pluginInfo"
static void scanPlugins() {
  KST::PluginInfoList tmpList;

  KstDebug::self()->log(i18n("Scanning for data-source plugins."));

  KService::List sl = KServiceType::offers("Kst Data Source");
  for (KService::List::ConstIterator it = sl.begin(); it != sl.end(); ++it) {
    for (KST::PluginInfoList::ConstIterator i2 = pluginInfo.begin(); i2 != pluginInfo.end(); ++i2) {
      if ((*i2)->service == *it) {
        tmpList.append(*i2);
        continue;
      }
    }

    KstSharedPtr<KST::Plugin> p = new KST::Plugin(*it);
    tmpList.append(p);
  }

  // This cleans up plugins that have been uninstalled and adds in new ones.
  // Since it is a shared pointer it can't dangle anywhere.
  pluginInfo.clear();
  pluginInfo = QDeepCopy<KST::PluginInfoList>(tmpList);
}


QStringList KstDataSource::pluginList() {
  QStringList rc;

  if (pluginInfo.isEmpty()) {
    scanPlugins();
  }

  for (KST::PluginInfoList::ConstIterator it = pluginInfo.begin(); it != pluginInfo.end(); ++it) {
    rc += (*it)->service->property("Name").toString();
  }

  return rc;
}


namespace {
class PluginSortContainer {
  public:
    KstSharedPtr<KST::Plugin> plugin;
    int match;
    int operator<(const PluginSortContainer& x) const {
      return match > x.match; // yes, this is by design.  biggest go first
    }
    int operator==(const PluginSortContainer& x) const {
      return match == x.match;
    }
};
}

static KstDataSourcePtr findPluginFor(const QString& filename, const QString& type) {
  KstDataSourcePtr plugin;

  scanPlugins();

  KST::PluginInfoList info = QDeepCopy<KST::PluginInfoList>(pluginInfo);

  if (!type.isEmpty()) {
    for (KST::PluginInfoList::ConstIterator it = info.begin(); it != info.end(); ++it) {
      if ((*it)->provides(type)) {
        plugin = (*it)->create(filename, type);
        break;
      }
    }

    return plugin;
  }

  QValueList<PluginSortContainer> bestPlugins;

  for (KST::PluginInfoList::ConstIterator it = info.begin(); it != info.end(); ++it) {
    PluginSortContainer psc;
    if ((psc.match = (*it)->understands(filename)) > 0) {
      psc.plugin = *it;
      bestPlugins.append(psc);
    }
  }

  qHeapSort(bestPlugins);

  for (QValueList<PluginSortContainer>::Iterator i = bestPlugins.begin(); i != bestPlugins.end(); ++i) {
      plugin = (*i).plugin->create(filename);
      if (plugin) {
        break;
      }
  }

  return plugin;
}


KstDataSourcePtr KstDataSource::loadSource(const QString& filename, const QString& type) {
  if (filename == "stdin" || filename == "-") {
    return new KstStdinSource;
  }

  if (filename.isEmpty()) {
    return 0L;
  }

  return findPluginFor(filename, type);
}


QStringList KstDataSource::fieldListForSource(const QString& filename, const QString& type) {
  if (filename.isEmpty() || filename == "stdin" || filename == "-") {
    return QStringList();
  }

  scanPlugins();

  KST::PluginInfoList info = QDeepCopy<KST::PluginInfoList>(pluginInfo);

  if (!type.isEmpty()) {
    for (KST::PluginInfoList::ConstIterator it = info.begin(); it != info.end(); ++it) {
      if ((*it)->provides(type)) {
        return (*it)->fieldList(filename, type);
      }
    }
  }

  QValueList<PluginSortContainer> bestPlugins;

  for (KST::PluginInfoList::ConstIterator it = info.begin(); it != info.end(); ++it) {
    PluginSortContainer psc;
    if ((psc.match = (*it)->understands(filename)) > 0) {
      psc.plugin = *it;
      bestPlugins.append(psc);
    }
  }

  qHeapSort(bestPlugins);

  QStringList rc;
  for (QValueList<PluginSortContainer>::Iterator i = bestPlugins.begin(); i != bestPlugins.end(); ++i) {
      rc = (*i).plugin->fieldList(filename);
      if (!rc.isEmpty()) {
        break;
      }
  }

  return rc;
}


KstDataSourcePtr KstDataSource::loadSource(QDomElement& e) {
  QString filename, type;

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "filename") {
        filename = e.text();
      } else if (e.tagName() == "type") {
        type = e.text();
      }
    }
    n = n.nextSibling();
  }

  if (filename.isEmpty()) {
    return 0L;
  }

  if (filename == "stdin" || filename == "-") {
    return new KstStdinSource;
  }

  return findPluginFor(filename, type);
}


KstDataSource::KstDataSource(const QString& filename, const QString& type)
: KstObject(), _filename(filename) {
  Q_UNUSED(type)
  _valid = false;
  d = new KstDataSourcePrivate;
  d->numFrames = new KstScalar(filename + i18n("-frames"));
}

KstDataSource::~KstDataSource() {
  KST::scalarList.lock().writeLock();
  KST::scalarList.remove(d->numFrames);
  KST::scalarList.lock().writeUnlock();
  d->numFrames = 0;
  delete d;
}


KstObject::UpdateType KstDataSource::update(int u) {
  Q_UNUSED(u)
  return KstObject::NO_CHANGE;
}

void KstDataSource::updateNumFramesScalar() {
  d->numFrames->setValue(frameCount());
}

int KstDataSource::readField(double *v, const QString& field, int s, int n) {
  Q_UNUSED(v)
  Q_UNUSED(field)
  Q_UNUSED(s)
  Q_UNUSED(n)
  return -1;
}


bool KstDataSource::isValid() const {
  return _valid;
}


bool KstDataSource::isValidField(const QString& field) const {
  Q_UNUSED(field)
  return false;
}


int KstDataSource::samplesPerFrame(const QString &field) {
  Q_UNUSED(field)
  return 0;
}


int KstDataSource::frameCount(const QString& field) const {
  Q_UNUSED(field)
  return 0;
}


QString KstDataSource::fileName() const {
  return _filename;
}


QStringList KstDataSource::fieldList() const {
  return _fieldList;
}


QString KstDataSource::fileType() const {
  return QString::null;
}


void KstDataSource::save(QTextStream &ts) {
  ts << " <filename>" << QStyleSheet::escape(_filename) << "</filename>" << endl;
  ts << " <type>" << QStyleSheet::escape(fileType()) << "</type>" << endl;
}


void KstDataSource::virtual_hook(int id, void *data) {
  Q_UNUSED(id)
  Q_UNUSED(data)
  //KstObject::virtual_hook(id, data);
}


void *KstDataSource::bufferMalloc(size_t size) {
  return KST::malloc(size);
}


void KstDataSource::bufferFree(void *ptr) {
  return ::free(ptr);
}


void *KstDataSource::bufferRealloc(void *ptr, size_t size) {
  return KST::realloc(ptr, size);
}


QGuardedPtr<QWidget> KstDataSource::configWidget() const {
  return 0L;
}


bool KstDataSource::fieldListIsComplete() const {
  return true;
}


bool KstDataSource::isEmpty() const {
  return true;
}


bool KstDataSource::reset() {
  return false;
}

// vim: ts=2 sw=2 et
