/***************************************************************************
                          itemdlg.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qfileinfo.h>
#include <qheader.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <kapplication.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klistview.h>
#include <klocale.h>
#include "arrayitem.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "funitem.h"
#include "itemdlg.h"
#include "kplchecklistitem.h"
#include "kpldoc.h"
#include "splfitdlg.h"
#include "splineitem.h"
#include "utils.h"

NewItemDlg::NewItemDlg(QWidget* _parent, int* i) :
 KDialog(_parent, 0, true), ityp(i)
{
  setCaption(i18n("New item"));
  QVBoxLayout* vbox = new QVBoxLayout(this, marginHint(), spacingHint());
  vbox->addWidget(bg = new QButtonGroup(8, Qt::Vertical, this));
  QPushButton* b = new QPushButton(i18n("Frame"), bg);
  int w = b->sizeHint().width();
  b = new QPushButton(i18n("Array"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Function"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Parametric Function"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Spline"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("3D Array"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("3D Function"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Image"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Legend"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Text"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Line"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Arrow"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Arc"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Rectangle"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Ellipse"), bg);
  w = QMAX(w, b->sizeHint().width());
  b = new QPushButton(i18n("Scale Bar"), bg);
  w = QMAX(w, b->sizeHint().width());
  connect(bg, SIGNAL(clicked(int)), SLOT(slotClicked(int)));
  QHBoxLayout* hbox = new QHBoxLayout(vbox, spacingHint());
  hbox->addWidget(help = new QPushButton(i18n("&Help"), this));
  connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
  hbox->addItem(new QSpacerItem(10, 10));
  hbox->addWidget(b = new QPushButton(i18n("&Cancel"), this));
  b->setDefault(true);
  connect(b, SIGNAL(clicked()), SLOT(reject()));
  resize(w + w + 4 * marginHint() + spacingHint(), 200);
}

NewItemDlg::~NewItemDlg()
{
}

void NewItemDlg::keyPressEvent(QKeyEvent* e)
{
  if ((e->state() == 0) && (e->key() == Key_F1)) {
       help->animateClick();
       e->accept();
  } else
    KDialog::keyPressEvent(e);
}

void NewItemDlg::slotClicked(int id)
{
  const int types[] = {KplItem::Frame, KplItem::Array, KplItem::Function,
                       KplItem::ParFunction, KplItem::Spline, KplItem::Array3D,
                       KplItem::Function3D, KplItem::Image, KplItem::Legend,
                       KplItem::Text, KplItem::Line, KplItem::Arrow,
                       KplItem::Arc, KplItem::Rectangle, KplItem::Ellipse,
                       KplItem::ScaleBar};
  *ityp = types[id];
  accept();
}

void NewItemDlg::slotHelp()
{
  kapp->invokeHelp("SEC-NEWITEM");
}

ItemDlg::ItemDlg(KplDoc* model) : m(model)
{
  popupMenu = new QPopupMenu();
  popupMenu->setCheckable(true);
  popupMenu->insertItem(i18n("&Edit"), this, SLOT(slotEdit()));
  popupMenu->insertItem(i18n("&Delete"), this, SLOT(slotDelete()));
  popupMenu->insertItem(i18n("&Copy"), this, SLOT(slotCopy()));
  idActive = popupMenu->insertItem(i18n("&Active"), this,
                                   SLOT(slotToggleState()));
  itemList = new KListView(this);
  itemList->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  itemList->addColumn(i18n("Active"));
  itemList->addColumn(i18n("Type"));
  itemList->addColumn(i18n("Settings"));
  itemList->setRootIsDecorated(true);
  itemList->setSorting(-1);
  itemList->header()->setClickEnabled(false);
  itemList->setDragEnabled(true);
  itemList->setAcceptDrops(true);
  itemList->setSelectionModeExt(KListView::FileManager);
  connect(itemList, SIGNAL(contextMenuRequested(QListViewItem*,
    const QPoint&, int)), SLOT(slotPopup(QListViewItem*, const QPoint&)));
  connect(itemList, SIGNAL(moved()), SLOT(slotMoved()));
  connect(itemList, SIGNAL(returnPressed(QListViewItem*)),
          SLOT(slotList(QListViewItem*)));
  connect(itemList, SIGNAL(selectionChanged()),
          SLOT(updWin()));
  connect(itemList, SIGNAL(currentChanged(QListViewItem*)),
          SLOT(slotSelectionChanged(QListViewItem*)));
  connect(itemList, SIGNAL(executed(QListViewItem*)),
          SLOT(slotExecuted(QListViewItem*)));
  QHBoxLayout* hbox = new QHBoxLayout(this, marginHint(), spacingHint());
  QVBoxLayout* vbox = new QVBoxLayout(hbox, spacingHint());
  vbox->addItem(new QSpacerItem(10, 10));
  vbox->addWidget(up = new QPushButton(this));
  up->setIconSet(BarIconSet("up"));
  connect(up, SIGNAL(clicked()), SLOT(slotMoveUp()));
  vbox->addWidget(down = new QPushButton(this));
  down->setIconSet(BarIconSet("down"));
  connect(down, SIGNAL(clicked()), SLOT(slotMoveDown()));
  vbox->addItem(new QSpacerItem(10, 10));
  vbox = new QVBoxLayout(hbox, spacingHint());
  vbox->addWidget(itemList);
  QHBoxLayout* hbox2 = new QHBoxLayout(vbox, spacingHint());
  hbox2->addWidget(appendItems = new QCheckBox(i18n("Append items"), this));
  hbox2->addItem(new QSpacerItem(10, 10, QSizePolicy::Expanding));
  vbox = new QVBoxLayout(hbox, spacingHint());
  QPushButton* b = new QPushButton(i18n("&New"), this);
  vbox->addWidget(b);
  connect(b, SIGNAL(clicked()), SLOT(slotNew()));
  vbox->addWidget(edit = new QPushButton(i18n("&Edit"), this));
  connect(edit, SIGNAL(clicked()), SLOT(slotEdit()));
  vbox->addWidget(fit = new QPushButton(i18n("&Fit"), this));
  connect(fit, SIGNAL(clicked()), SLOT(slotFit()));
  vbox->addWidget(del = new QPushButton(i18n("&Delete"), this));
  connect(del, SIGNAL(clicked()), SLOT(slotDelete()));
  vbox->addWidget(copy = new QPushButton(i18n("&Copy"), this));
  connect(copy, SIGNAL(clicked()), SLOT(slotCopy()));
  vbox->addWidget(help = new QPushButton(i18n("&Help"), this));
  connect(help, SIGNAL(clicked()), SLOT(slotHelp()));
  vbox->addWidget(b = new QPushButton(i18n("C&lose"), this));
  connect(b, SIGNAL(clicked()), SLOT(hide()));
  updItemList();
#if KDE_VERSION >= 0x030200
  QTimer::singleShot(0, this, SLOT(setPosition()));
#else
  Utils::setGeometry(this, "ItemDialog");
#endif
}

ItemDlg::~ItemDlg()
{
  delete popupMenu;
}

void ItemDlg::updItemList()
{
  QToolTip::remove(this);
  setPlainCaption(m->URL().fileName() + " - " + i18n("Items"));
  int current = lItem.findRef((KplCheckListItem*) itemList->currentItem());
  itemList->clear();
  up->setEnabled(false);
  down->setEnabled(false);
  lItem.clear();
  int nItems = m->items()->count();
  bool funcs = false, arrays = false;
  if (nItems) {
    KplCheckListItem* _parent = 0;
    for (KplItem* item = m->items()->first(); item;
         item = m->items()->next()) {
      KplItem::ItemTypes typ = item->iType();
      bool framelike = (typ == KplItem::Frame) || (typ == KplItem::Array3D) ||
                        (typ == KplItem::Function3D);
      if (_parent)
        if (framelike)
          lItem.append(new KplCheckListItem(itemList, _parent));
        else
          lItem.append(new KplCheckListItem(_parent, lItem.getLast()));
      else
        if (item == m->items()->getFirst())
          lItem.append(new KplCheckListItem(itemList));
        else
          lItem.append(new KplCheckListItem(itemList, lItem.getLast()));
      if (framelike) {
        _parent = lItem.current();
        _parent->setOpen((typ == KplItem::Frame) ?
                         !((FrameItem*) item)->isCollapsed() : true);
      }
      item->setText(lItem.current(), &arrays, &funcs);
      lItem.current()->setOn(item->isActive());
      connect(lItem.current(), SIGNAL(stateChanged(KplCheckListItem*, bool)),
              SLOT(slotStateChanged(KplCheckListItem*, bool)));
      connect(lItem.current(), SIGNAL(openChanged(KplCheckListItem*, bool)),
              SLOT(slotOpenChanged(KplCheckListItem*, bool)));
    }
    KplCheckListItem* itm = lItem.at(QMAX(0, QMIN(current, nItems - 1)));
    itemList->setSelected(itm, true);
    itemList->setCurrentItem(itm);
    itemList->ensureItemVisible(itm);
    slotSelectionChanged(itm);
  }
  edit->setEnabled(nItems);
  updWin();
  itemList->setFocus();
}

void ItemDlg::showPopup(int it, const QPoint& p)
{
  itemList->clearSelection();
  QListViewItem* item = lItem.at(it);
  itemList->setCurrentItem(item);
  itemList->setSelected(item, true);
  slotPopup(item, p);
}

void ItemDlg::moveItem(int di)
{
  int iSource = lItem.findRef((KplCheckListItem*) itemList->currentItem());
  int iDest = iSource + di;
  m->moveItem(iSource, iDest);
  itemList->clearSelection();
  QListViewItem* item = lItem.at(iDest);
  itemList->setCurrentItem(item);
  itemList->setSelected(item, true);
  if (!iDest)
    slotSelectionChanged(item);
}

void ItemDlg::keyPressEvent(QKeyEvent* e)
{
  if ((e->state() == 0) && (e->key() == Key_F1)) {
       help->animateClick();
       e->accept();
  } else
    KDialog::keyPressEvent(e);
}

void ItemDlg::slotList(QListViewItem* it)
{
  int i = lItem.findRef((KplCheckListItem*) it);
  m->items()->at(i)->editItem(this, m, i);
}

void ItemDlg::slotMoved()
{
  QValueList<int> list;
  QListViewItemIterator it(itemList);
  while (it.current()) {
    list.append(lItem.findRef((KplCheckListItem*) it.current()));
    it++;
  }
  m->orderItems(list);
}

void ItemDlg::slotNew()
{
  int ityp;
  NewItemDlg dlg(this, &ityp);
  if (dlg.exec()) {
    int it = m->items()->count();
    if (!appendItems->isChecked() && it)
      it = lItem.findRef((KplCheckListItem*) itemList->currentItem());
    m->newItem((KplItem::ItemTypes) ityp, it);
    if ((ityp == KplItem::Array) || (ityp == KplItem::Function) ||
        (ityp == KplItem::ParFunction) || (ityp == KplItem::Spline))
      for (int i = it - 1; i >= 0; i--) {
        ityp = m->items()->at(i)->iType();
        if (ityp == KplItem::Frame)
          break;
        if ((ityp == KplItem::Array) || (ityp == KplItem::Function) ||
            (ityp == KplItem::ParFunction) || (ityp == KplItem::Spline)) {
          m->items()->at(it)->normalize(((ScaledItem*)m->items()->at(i))->fx,
                                        ((ScaledItem*)m->items()->at(i))->fy);
          break;
        }
      }
    if (!m->items()->at(it)->editItem(this, m, it)) {
      m->setModified();
      m->backupItems();
    }
  }
}

void ItemDlg::slotEdit()
{
  if (QListViewItem* it = itemList->currentItem())
    slotList(it);
}

void ItemDlg::slotFit()
{
  QPtrList<ArrayItem> arr;
  QPtrList<FunItem> fun;
  QPtrList<SplineItem> spl;
  for (unsigned i = 0; i < m->items()->count(); i++)
    if (defSel || itemList->isSelected(lItem.at(i))) {
      switch (m->items()->at(i)->iType()) {
        case KplItem::Array:
          arr.append((ArrayItem*) m->items()->at(i));
          break;
        case KplItem::Function:
          fun.append((FunItem*) m->items()->at(i));
          break;
        case KplItem::Spline:
          spl.append((SplineItem*) m->items()->at(i));
          break;
        default:
          ;
      }
    }
  if (fun.count()) {
    emit displayMessage(i18n("Fitting function parameters..."));
    FitDlg dlg(this, m, &arr, &fun, FitDlg::ShowDlg);
    dlg.exec();
  } else {
    emit displayMessage(i18n("Fitting spline..."));
    SplFitDlg dlg(this, m, arr.first(), &spl, FitDlg::ShowDlg);
    dlg.exec();
  }
  updItemList();
  emit displayMessage(i18n("Ready."));
}

void ItemDlg::slotDelete()
{
  bool b = false;
  for (int i = m->items()->count() - 1; i >= 0; i--)
    if (itemList->isSelected(lItem.at(i))) {
      m->items()->remove(i);
      b = true;
    }
  if (b) {
    m->setModified();
    m->backupItems();
  }
}

void ItemDlg::slotCopy()
{
  bool b = false;
  if (appendItems->isChecked()) {
    for (unsigned i = 0; i < m->items()->count(); i++)
      if (itemList->isSelected(lItem.at(i))) {
        m->items()->append(m->items()->at(i)->copy());
        b = true;
      }
  } else
    for (int i = m->items()->count() - 1; i >= 0; i--)
      if (itemList->isSelected(lItem.at(i))) {
        m->items()->insert(i, m->items()->at(i)->copy());
        b = true;
      }
  if (b) {
    m->setModified();
    m->backupItems();
  }
}

void ItemDlg::slotHelp()
{
  kapp->invokeHelp("SEC-ITEMS");
}

void ItemDlg::slotPopup(QListViewItem* item, const QPoint& p)
{
  if (item) {
    itemList->setSelected(item, true);
    popupMenu->setItemChecked(idActive, ((KplCheckListItem*) item)->isOn());
    popupMenu->exec(p);
  }
}

void ItemDlg::slotExecuted(QListViewItem* item)
{
  ((KplCheckListItem*) item)->setOn(!((KplCheckListItem*) item)->isOn());
}

void ItemDlg::slotStateChanged(KplCheckListItem* it, bool bState)
{
  m->items()->at(lItem.findRef(it))->setActive(bState);
  m->setModified();
  m->backupItems(false);
}

void ItemDlg::slotOpenChanged(KplCheckListItem* it, bool open)
{
  KplItem* item = m->items()->at(lItem.findRef(it));
  if (item->iType() == KplItem::Frame) {
    ((FrameItem*) item)->setCollapsed(!open);
    m->setModified();
    m->backupItems(false);
  }
}

void ItemDlg::slotToggleState()
{
  if (QListViewItem* item = itemList->currentItem())
    slotExecuted(item);
}

void ItemDlg::slotSelectionChanged(QListViewItem* it)
{
  int i = lItem.findRef((KplCheckListItem*) it);
  up->setEnabled(i > 0);
  down->setEnabled(i < ((int) m->items()->count() - 1));
}

void ItemDlg::slotMoveUp()
{
  moveItem(-1);
}

void ItemDlg::slotMoveDown()
{
  moveItem(+1);
}

void ItemDlg::setPosition()
{
  Utils::setGeometry(this, "ItemDialog");
}

void ItemDlg::updWin()
{
  int nSel = 0;
  int nArr = 0;
  int nFun = 0;
  int nSpl = 0;
  int nSelArr = 0;
  int nSelFun = 0;
  int nSelSpl = 0;
  int kDeg = 0;
  bool splValid = true;
  for (unsigned i = 0; i < m->items()->count(); i++) {
    KplCheckListItem* itm = lItem.at(i);
    bool sel = itemList->isSelected(itm);
    if (sel)
      nSel++;
    switch (m->items()->at(i)->iType()) {
      case KplItem::Array:
        nArr++;
        if (sel)
          nSelArr++;
        break;
      case KplItem::Function:
        nFun++;
        if (sel)
          nSelFun++;
        break;
      case KplItem::Spline:
        nSpl++;
        if (sel) {
          if (nSelSpl) {
            if (((SplineItem*) m->items()->at(i))->kdeg != kDeg)
              splValid = false;
          } else
            kDeg = ((SplineItem*) m->items()->at(i))->kdeg;
          nSelSpl++;
        }
        break;
      default:
        ;
    }
  }
  defSel = ((nArr == 1) && ((nFun + nSpl) == 1));
  fit->setEnabled(defSel ||
                  (nSelArr && nSelFun && (!nSelSpl) && (nSelArr == nSelFun)) ||
                  (nSelArr && nSelSpl && splValid && (!nSelFun) &&
                   (nSelArr == 1)));
  copy->setEnabled(nSel);
  del->setEnabled(nSel);
}
