
/***************************************************************************
 *                                                                         *
 *   KNetLoad is copyright (c) 1999-2000, Markus Gustavsson                *
 *                         (c) 2002, Ben Burton                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "devicedialog.h"
#include "knetdock.h"
#include "knetload.h"
#include "knetproc.h"
#include "scaledialog.h"

#include <kaction.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>

#define DEFAULT_SCALE 128000

// Stock device names are not i18n()ed since they're literal
// interface names.
const char* stockDevice[] = {
    "lo", "eth0", "ippp0", "ppp0", 0
};

const char* stockDeviceIconOff[] = {
    "devlo", "deveth0", "devippp0", "devppp0", 0
};

const char* stockDeviceIconOn[] = {
    "devloon", "deveth0on", "devippp0on", "devppp0on", 0
};

KNetLoad::KNetLoad(QWidget *parent, const char *name) :
        StatPopup(false, parent, name) {
    // Create the raw network data reader.
    proc = new KNetProc();

    // Set up actions and read the config file.
    setupActions();

    // Create system tray windows.
    dock[0] = new KNetDock(0, true, this); // In
    dock[1] = new KNetDock(1, false, this); // Out

    // Initialise the pop-up window.
    readPopupState();

    // Off we go!
    requestResize();
    if (isActive())
        startUpdates();
}

KNetLoad::~KNetLoad() {
    delete proc;
}

void KNetLoad::setDevice(const QString& newDevice) {
    proc->setDevice(newDevice);
    clearHistory();
    updateDeviceMenus();

    requestResize();

    config->setGroup("General Options");
    config->writeEntry("Device", newDevice);
    config->sync();
}

void KNetLoad::setDevice(int deviceIndex) {
    // Is it one of the stock devices?
    if (deviceIndex >= 0 && stockDevice[deviceIndex]) {
        setDevice(stockDevice[deviceIndex]);
        return;
    }

    // It's a custom device.  Open a dialog.
    DeviceDialog dlg(proc->getDevice(), firstDock());
    if (dlg.exec()) {
        QString newDevice = dlg.getDevice().stripWhiteSpace();
        if (newDevice.isEmpty())
            KMessageBox::error(firstDock(),
                i18n("The device name cannot be empty."));
        else {
            setDevice(newDevice);
            return;
        }
    }

    // If we fell through, update the menus anyway in case we
    // inadvertently changed a checked state.
    updateDeviceMenus();
}

void KNetLoad::setScaleIn(int scale) {
    if (scale <= 0) {
        // Select a non-standard scale.
        ScaleDialog dlg(scaleIn / 1000,
            i18n("Select Scale (In)"), firstDock());
        if (dlg.exec()) {
            scale = dlg.getScale();
            if (scale <= 0)
                return;
            // Convert to bits per second and fall through.
            scale *= 1000;
        } else
            return;
    }

    scaleIn = scale;
    updateScaleInMenus();

    config->setGroup("General Options");
    config->writeEntry("ScaleIn", scale);
    config->sync();
}

void KNetLoad::setScaleOut(int scale) {
    // Note that, for scale (out), scale == 0 means to use the same as
    // scale (in).
    if (scale < 0) {
        // Select a non-standard scale.
        ScaleDialog dlg(scaleOut ? scaleOut / 1000 : scaleIn / 1000,
            i18n("Select Scale (Out)"), firstDock());
        if (dlg.exec()) {
            scale = dlg.getScale();
            if (scale <= 0)
                return;
            // Convert to bits per second and fall through.
            scale *= 1000;
        } else
            return;
    }

    scaleOut = scale;
    updateScaleOutMenus();

    config->setGroup("General Options");
    config->writeEntry("ScaleOut", scale);
    config->sync();
}

QString KNetLoad::dockName(int which) const {
    return (which == 0 ? i18n("In") : i18n("Out"));
}

QColor KNetLoad::defaultDockColor(int which) const {
    return (which == 0 ? QColor(0, 0, 255) : QColor(255, 0, 255));
}

void KNetLoad::setupCustomActions() {
    // There are no device actions; instead menu items are used
    // directly.
    proc->setDevice(config->readEntry("Device", "eth0"));

    // There are no scaling actions; instead menu items are used
    // directly.
    scaleIn = config->readNumEntry("ScaleIn", DEFAULT_SCALE);
    scaleOut = config->readNumEntry("ScaleOut", 0 /* same as in */);
}

void KNetLoad::insertCustomItems(KPopupMenu* menu) {
    // Device menu.  The menu item IDs are indices into the
    // stockDevice[] array.
    KPopupMenu* deviceMenu = new KPopupMenu(menu);
    deviceMenus.append(deviceMenu);
    deviceMenu->setCheckable(true);
    for (int i = 0; stockDevice[i]; i++)
        deviceMenu->insertItem(SmallIcon(stockDeviceIconOff[i]),
            stockDevice[i], i);
    deviceMenu->insertItem(SmallIcon("devother"), i18n("Other..."));
    updateDeviceMenu(deviceMenu);
    connect(deviceMenu, SIGNAL(activated(int)), this, SLOT(setDevice(int)));
    menu->insertItem(SmallIcon("devselect"), i18n("&Device"), deviceMenu);

    // Scale menus.  The menu item IDs are the scales themselves (in bits
    // per second), to make event handling sane.
    KPopupMenu* scaleInMenu = new KPopupMenu(menu);
    scaleInMenus.append(scaleInMenu);
    scaleInMenu->setCheckable(true);
    scaleInMenu->insertItem(i18n("28.8KBit/s"), 28800);
    scaleInMenu->insertItem(i18n("33.6KBit/s"), 33600);
    scaleInMenu->insertItem(i18n("56KBit/s"), 56000);
    scaleInMenu->insertItem(i18n("64KBit/s"), 64000);
    scaleInMenu->insertItem(i18n("128KBit/s"), 128000);
    scaleInMenu->insertItem(i18n("256KBit/s"), 256000);
    scaleInMenu->insertItem(i18n("512KBit/s"), 512000);
    scaleInMenu->insertItem(i18n("1MBit/s"), 1000000);
    scaleInMenu->insertItem(i18n("2MBit/s"), 2000000);
    scaleInMenu->insertItem(i18n("10MBit/s"), 10000000);
    scaleInMenu->insertItem(i18n("Other..."));
    updateScaleInMenu(scaleInMenu);
    connect(scaleInMenu, SIGNAL(activated(int)), this, SLOT(setScaleIn(int)));
    menu->insertItem(SmallIcon("scaling"), i18n("&Scale (In)"), scaleInMenu);

    KPopupMenu* scaleOutMenu = new KPopupMenu(menu);
    scaleOutMenus.append(scaleOutMenu);
    scaleOutMenu->setCheckable(true);
    scaleOutMenu->insertItem(i18n("28.8KBit/s"), 28800);
    scaleOutMenu->insertItem(i18n("33.6KBit/s"), 33600);
    scaleOutMenu->insertItem(i18n("56KBit/s"), 56000);
    scaleOutMenu->insertItem(i18n("64KBit/s"), 64000);
    scaleOutMenu->insertItem(i18n("128KBit/s"), 128000);
    scaleOutMenu->insertItem(i18n("256KBit/s"), 256000);
    scaleOutMenu->insertItem(i18n("512KBit/s"), 512000);
    scaleOutMenu->insertItem(i18n("1MBit/s"), 1000000);
    scaleOutMenu->insertItem(i18n("2MBit/s"), 2000000);
    scaleOutMenu->insertItem(i18n("10MBit/s"), 10000000);
    scaleOutMenu->insertItem(i18n("Other..."));
    scaleOutMenu->insertSeparator();
    scaleOutMenu->insertItem(i18n("Same as for in"), 0);
    updateScaleOutMenu(scaleOutMenu);
    connect(scaleOutMenu, SIGNAL(activated(int)), this, SLOT(setScaleOut(int)));
    menu->insertItem(SmallIcon("scaling"), i18n("&Scale (Out)"), scaleOutMenu);

    // The final separator.
    menu->insertSeparator();
}

void KNetLoad::takeReadingInternal() {
    proc->readLoad();

    // Rates in bits per second.
    bitRateIn = proc->recentBytesIn() * 8.0 /
        (((float) getSpeed()) / 1000.0);
    bitRateOut = proc->recentBytesOut() * 8.0 /
        (((float) getSpeed()) / 1000.0);

    // Convert to percentages.
    upper[0] = (int) (100.0 * bitRateIn / ((float) scaleIn));
    upper[1] = (int) (100.0 * bitRateOut /
        ((float) (scaleOut ? scaleOut : scaleIn)));

    if (upper[0] < 0)
        upper[0] = 0;
    if (upper[1] < 0)
        upper[1] = 0;
    if (upper[0] > 100)
        upper[0] = 100;
    if (upper[1] > 100)
        upper[1] = 100;

    if (isVisible())
        fullReading.sprintf(i18n(
            "Current In: %.0fKBit/s, Total In: %.2fMB.\n"
            "Current Out: %.0fKBit/s, Total Out: %.2fMB."),
            bitRateIn / 1000.0, proc->totalMbIn(),
            bitRateOut / 1000.0, proc->totalMbOut());
}

void KNetLoad::updateDeviceMenu(KPopupMenu* menu) {
    const QString& dev = proc->getDevice();

    // Update the checked/unchecked states of menu items.
    bool found = false;
    int id;
    int otherId = -1;
    for (unsigned index = 0; index < menu->count(); index++) {
        id = menu->idAt(index);

        if (id >= 0 && dev == menu->text(id)) {
            // This is our device.
            found = true;
            if (! menu->isItemChecked(id)) {
                menu->setItemChecked(id, true);
                menu->changeItem(id, SmallIcon(stockDeviceIconOn[id]),
                    stockDevice[id]);
            }
        } else {
            // This is not our device.
            if (menu->isItemChecked(id)) {
                menu->setItemChecked(id, false);
                menu->changeItem(id, SmallIcon(stockDeviceIconOff[id]),
                    stockDevice[id]);
            }
            if (id < 0)
                if (! menu->text(id).isEmpty())
                    otherId = id;
        }
    }

    // Update the "other" item text and checked state.
    if (found) {
        menu->changeItem(otherId, SmallIcon("devother"), i18n("Other..."));
        menu->setItemChecked(otherId, false);
    } else {
        menu->changeItem(otherId, SmallIcon("devotheron"),
            QString(i18n("Other (%1)...")).arg(dev));
        menu->setItemChecked(otherId, true);
    }
}

void KNetLoad::updateDeviceMenus() {
    for (KPopupMenu* menu = deviceMenus.first(); menu;
            menu = deviceMenus.next())
        updateDeviceMenu(menu);
}

void KNetLoad::updateScaleInMenu(KPopupMenu* menu) {
    // Update the checked/unchecked states of menu items.
    bool found = false;
    int id;
    int otherId = -1;
    for (unsigned index = 0; index < menu->count(); index++) {
        id = menu->idAt(index);

        if (id == scaleIn) {
            menu->setItemChecked(id, true);
            found = true;
        } else {
            menu->setItemChecked(id, false);
            if (id < 0)
                if (! menu->text(id).isEmpty())
                    otherId = id;
        }
    }

    // Update the "other" item text.
    if (found)
        menu->changeItem(otherId, i18n("Other..."));
    else {
        QString text;
        text.sprintf(i18n("Other (%.0fKBit/s)..."), ((float) scaleIn) / 1000);
        menu->changeItem(otherId, text);
    }

    // Check the "other" item if necessary.
    if (! found)
        menu->setItemChecked(otherId, true);
}

void KNetLoad::updateScaleInMenus() {
    for (KPopupMenu* menu = scaleInMenus.first(); menu;
            menu = scaleInMenus.next())
        updateScaleInMenu(menu);
}

void KNetLoad::updateScaleOutMenu(KPopupMenu* menu) {
    // Update the checked/unchecked states of menu items.
    bool found = false;
    int id;
    int otherId = -1;
    for (unsigned index = 0; index < menu->count(); index++) {
        id = menu->idAt(index);

        if (id == scaleOut) {
            menu->setItemChecked(id, true);
            found = true;
        } else {
            menu->setItemChecked(id, false);
            if (id < 0)
                if (! menu->text(id).isEmpty())
                    otherId = id;
        }
    }

    // Update the "other" item text.
    if (found)
        menu->changeItem(otherId, i18n("Other..."));
    else {
        QString text;
        text.sprintf(i18n("Other (%.0fKBit/s)..."), ((float) scaleOut) / 1000);
        menu->changeItem(otherId, text);
    }

    // Check the "other" item if necessary.
    if (! found)
        menu->setItemChecked(otherId, true);
}

void KNetLoad::updateScaleOutMenus() {
    for (KPopupMenu* menu = scaleOutMenus.first(); menu;
            menu = scaleOutMenus.next())
        updateScaleOutMenu(menu);
}

