/***************************************************************************
                          threeddlg.cpp  -  description
                             -------------------
    begin                : Thu Apr 25 2002
    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qfile.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <kapp.h>
#if KDE_VERSION_MAJOR < 3
#include <qmultilineedit.h>
#endif
#include <kcolorbtn.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "arrayitem.h"
#include "editdlg.h"
#include "editparameterdlg.h"
#include "edittextdlg.h"
#include "funcdlg.h"
#include "kpldoc.h"
#include "kpldoubleedit.h"
#include "kplgraph.h"
#include "kplspinbox.h"
#include "threeddlg.h"
#include "threeditem.h"
#include "utils.h"

ThreeDDlg::ThreeDDlg(QWidget* _parent, KplDoc* model, ThreeDItem *td0,
                     const QString& caption, const QString anchor,
                     const QString& page1) :
 KDialogBase(Tabbed, caption, Help | Ok | Apply | Cancel | User1, Ok, _parent,
             0, true, true, i18n("&Export")), m(model), td(td0)
{
  setHelp(anchor);
  frame1 = addPage(page1);
  vbox1 = new QVBoxLayout(frame1, marginHint(), spacingHint());
  g1 = new QGroupBox(0, Qt::Vertical, i18n("Range"), frame1);
  QFrame* frame = addPage(i18n("Scaling"));
  QGridLayout* grid = new QGridLayout(frame, 3, 3, spacingHint());
  QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Window"), frame);
  grid->addMultiCellWidget(g, 0, 0, 0, 1);
  QGridLayout* grid2 = new QGridLayout(g->layout(), 2, 7, spacingHint());
  grid2->addWidget(new QLabel(i18n("Left margin"), g), 0, 0);
  grid2->addWidget(ex0 = new KplDoubleEdit(td->x0, 0.0, 200.0, g), 0, 1);
  grid2->addWidget(new QLabel("cm", g), 0, 2);
  grid2->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 3);
  grid2->addWidget(new QLabel(i18n("Width"), g), 0, 4);
  grid2->addWidget(exl = new KplDoubleEdit(td->w, 0.5, 200.0, g), 0, 5);
  grid2->addWidget(new QLabel("cm", g), 0, 6);
  grid2->addWidget(new QLabel(i18n("Bottom margin"), g), 1, 0);
  grid2->addWidget(ey0 = new KplDoubleEdit(td->y0, 0.0, 200.0, g), 1, 1);
  grid2->addWidget(new QLabel("cm", g), 1, 2);
  grid2->addWidget(new QLabel(i18n("Height"), g), 1, 4);
  grid2->addWidget(eyl = new KplDoubleEdit(td->h, 0.5, 200.0, g), 1, 5);
  grid2->addWidget(new QLabel("cm", g), 1, 6);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Reference point"),
                  frame), 0, 2);
  grid2 = new QGridLayout(g->layout(), 2, 3, spacingHint());
  grid2->addWidget(new QLabel("x", g), 0, 0);
  grid2->addWidget(eXref = new KplDoubleEdit(td->xref, 0.0, 200.0, g), 0, 1);
  grid2->addWidget(new QLabel("cm", g), 0, 2);
  grid2->addWidget(new QLabel("y", g), 1, 0);
  grid2->addWidget(eYref = new KplDoubleEdit(td->yref, 0.0, 200.0, g), 1, 1);
  grid2->addWidget(new QLabel("cm", g), 1, 2);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("3D box"), frame), 1, 0);
  grid2 = new QGridLayout(g->layout(), 3, 3, spacingHint());
  grid2->addWidget(new QLabel("x", g), 0, 0);
  grid2->addWidget(eBoxx = new KplDoubleEdit(td->boxx, 0.5, 200.0, g), 0, 1);
  grid2->addWidget(new QLabel("cm", g), 0, 2);
  grid2->addWidget(new QLabel("y", g), 1, 0);
  grid2->addWidget(eBoxy = new KplDoubleEdit(td->boxy, 0.5, 200.0, g), 1, 1);
  grid2->addWidget(new QLabel("cm", g), 1, 2);
  grid2->addWidget(new QLabel("z", g), 2, 0);
  grid2->addWidget(eBoxz = new KplDoubleEdit(td->boxz, 0.5, 200.0, g), 2, 1);
  grid2->addWidget(new QLabel("cm", g), 2, 2);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Rotation"), frame), 1, 1);
  grid2 = new QGridLayout(g->layout(), 2, 3, spacingHint());
  grid2->addWidget(new QLabel("phi", g), 0, 0);
  grid2->addWidget(ePhi = new KplDoubleEdit(td->phi, -360.0, 360.0, g), 0, 1);
  grid2->addWidget(new QLabel("", g), 0, 2);
  grid2->addWidget(new QLabel("theta", g), 1, 0);
  grid2->addWidget(eTheta = new KplDoubleEdit(td->theta, -360.0, 360.0, g),
                   1, 1);
  grid2->addWidget(new QLabel("", g), 1, 2);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Automatic"),
                  frame), 1, 2);
  QVBoxLayout* vbox2 = new QVBoxLayout(g->layout(), spacingHint());
  vbox2->addWidget(autoNorm = new QCheckBox(i18n("Normalization"), g));
  autoNorm->setChecked(m->options()->autoNorm &&
                       (!(td->logx || td->logy || td->logz)));
  bAuto = new QPushButton(i18n("Autoscale"), g);
  vbox2->addWidget(bAuto);
  grid->addMultiCellWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Axes"),
                                             frame), 2, 2, 0, 1);
  grid2 = new QGridLayout(g->layout(), 3, 8, spacingHint());
  grid2->addWidget(new QLabel("xtic", g), 0, 0);
  grid2->addWidget(exTic = new KplDoubleEdit(td->xtic, g), 0, 1);
  grid2->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 2);
  grid2->addWidget(new QLabel("mticx", g), 0, 3);
  grid2->addWidget(emTicx = new KplSpinBox(0, 10, 1, g), 0, 4);
  emTicx->setValue(td->mticx);
  grid2->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 5);
  grid2->addWidget(new QLabel("ndigx", g), 0, 6);
  grid2->addWidget(enDigx = new KplSpinBox(-1, 8, 1, g), 0, 7);
  enDigx->setValue(td->ndigx);
  enDigx->setEnabled(!td->logx);
  grid2->addWidget(new QLabel("ytic", g), 1, 0);
  grid2->addWidget(eyTic = new KplDoubleEdit(td->ytic, g), 1, 1);
  grid2->addWidget(new QLabel("mticy", g), 1, 3);
  grid2->addWidget(emTicy = new KplSpinBox(0, 10, 1, g), 1, 4);
  emTicy->setValue(td->mticy);
  grid2->addWidget(new QLabel("ndigy", g), 1, 6);
  grid2->addWidget(enDigy = new KplSpinBox(-1, 8, 1, g), 1, 7);
  enDigy->setValue(td->ndigy);
  enDigy->setEnabled(!td->logy);

  grid2->addWidget(new QLabel("ztic", g), 2, 0);
  grid2->addWidget(ezTic = new KplDoubleEdit(td->ztic, g), 2, 1);
  grid2->addWidget(new QLabel("mticz", g), 2, 3);
  grid2->addWidget(emTicz = new KplSpinBox(0, 10, 1, g), 2, 4);
  emTicz->setValue(td->mticz);
  grid2->addWidget(new QLabel("ndigz", g), 2, 6);
  grid2->addWidget(enDigz = new KplSpinBox(-1, 8, 1, g), 2, 7);
  enDigz->setValue(td->ndigz);
  enDigz->setEnabled(!td->logz);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Normalization"),
                  frame), 2, 2);
  grid2 = new QGridLayout(g->layout(), 3, 2, spacingHint());
  grid2->addWidget(new QLabel("x", g), 0, 0);
  grid2->addWidget(efx = new KplDoubleEdit(td->fx, g), 0, 1);
  grid2->addWidget(new QLabel("y", g), 1, 0);
  grid2->addWidget(efy = new KplDoubleEdit(td->fy, g), 1, 1);
  grid2->addWidget(new QLabel("z", g), 2, 0);
  grid2->addWidget(efz = new KplDoubleEdit(td->fz, g), 2, 1);
  frame = addPage(i18n("Representation"));
  grid = new QGridLayout(frame, 3, 2, spacingHint());
  grid->addWidget(bg1 = new QButtonGroup(0, Qt::Vertical, i18n("Axis mode"),
                                         frame), 0, 0);
  QVBoxLayout* vbox = new QVBoxLayout(bg1->layout(), 8);
  vbox->addWidget(new QRadioButton(i18n("No Tics"), bg1));
  vbox->addWidget(new QRadioButton(i18n("Tics"), bg1));
  vbox->addWidget(new QRadioButton(i18n("Tics with numbers"), bg1));
  ((QRadioButton*) bg1->find(td->gridmode))->setChecked(true);
  grid->addWidget(bg2 = new QButtonGroup(0, Qt::Vertical, i18n("3D mode"),
                                         frame), 0, 1);
  vbox = new QVBoxLayout(bg2->layout(), 8);
  vbox->addWidget(new QRadioButton("x", bg2));
  vbox->addWidget(new QRadioButton("z", bg2));
  vbox->addWidget(new QRadioButton(i18n("x and z"), bg2));
  ((QRadioButton*) bg2->find(td->mode3D))->setChecked(true);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Colors"), frame),
                  1, 0);
  grid2 = new QGridLayout(g->layout(), 3, 2, spacingHint());
  grid2->addWidget(new QLabel(i18n("Frame"), g), 0, 0);
  grid2->addWidget(colFrame = new KColorButton(QColor(td->colFrame), g), 0, 1);
  grid2->addWidget(new QLabel(i18n("Tics"), g), 1, 0);
  grid2->addWidget(colGrid = new KColorButton(QColor(td->colGrid), g), 1, 1);
  grid2->addWidget(new QLabel(i18n("Data"), g), 2, 0);
  grid2->addWidget(colData = new KColorButton(QColor(td->colData), g), 2, 1);
  grid->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("General"), frame),
                  1, 1);
  vbox = new QVBoxLayout(g->layout(), spacingHint());
  vbox->addWidget(modeFrame = new QCheckBox(i18n("Show frame"), g));
  modeFrame->setChecked(td->showFrame);
  QHBoxLayout* hbox = new QHBoxLayout(g->layout(), 6);
  hbox->addWidget(new QLabel(i18n("Line width"), g));
  hbox->addWidget(eRelSiz = new KplSpinBox(10, 800, 1, g));
  eRelSiz->setValue(qRound(100 * td->relsiz));
  eRelSiz->setSuffix(" %");
  grid->addMultiCellWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Labels"),
                                             frame), 2, 2, 0, 1);
  grid2 = new QGridLayout(g->layout(), 3, 6, spacingHint());
  grid2->addWidget(new QLabel("iex", g), 0, 0);
  grid2->addWidget(ex = new KplSpinBox(-300, 300, 1, g), 0, 1);
  ex->setValue(td->iex);
  grid2->addItem(new QSpacerItem(20, 10, QSizePolicy::Minimum), 0, 2);
  grid2->addWidget(new QLabel(i18n("x axis"), g), 0, 3);
  grid2->addWidget(eLetX = new QLineEdit(td->sx, g), 0, 4);
  QPushButton* b = new QPushButton("...", g);
  grid2->addWidget(b, 0, 5);
  b->setFixedWidth(28);
  connect(b, SIGNAL(clicked()), SLOT(slotEditLetX()));
  grid2->addWidget(new QLabel("iey", g), 1, 0);
  grid2->addWidget(ey = new KplSpinBox(-300, 300, 1, g), 1, 1);
  ey->setValue(td->iey);
  grid2->addWidget(new QLabel(i18n("y axis"), g), 1, 3);
  grid2->addWidget(eLetY = new QLineEdit(td->sy, g), 1, 4);
  grid2->addWidget(b = new QPushButton("...", g), 1, 5);
  b->setFixedWidth(28);
  connect(b, SIGNAL(clicked()), SLOT(slotEditLetY()));
  grid2->addWidget(new QLabel("iez", g), 2, 0);
  grid2->addWidget(ez = new KplSpinBox(-300, 300, 1, g), 2, 1);
  ez->setValue(td->iez);
  grid2->addWidget(new QLabel(i18n("z axis"), g), 2, 3);
  grid2->addWidget(eLetZ = new QLineEdit(td->sz, g), 2, 4);
  grid2->addWidget(b = new QPushButton("...", g), 2, 5);
  b->setFixedWidth(28);
  connect(b, SIGNAL(clicked()), SLOT(slotEditLetZ()));
  grid2->addWidget(new QLabel(i18n("Heading"), g), 3, 3);
  grid2->addWidget(eLetH = new QLineEdit(td->sh, g), 3, 4);
  grid2->addWidget(b = new QPushButton("...", g), 3, 5);
  b->setFixedWidth(28);
  connect(b, SIGNAL(clicked()), SLOT(slotEditLetH()));
  connect(autoNorm, SIGNAL(toggled(bool)), SLOT(slotNormToggled(bool)));
}

ThreeDDlg::~ThreeDDlg()
{
}

void ThreeDDlg::initRange(QGridLayout* grid, int row)
{
  char frm = m->options()->format;
  int prec = m->options()->prec;
  grid->addWidget(new QLabel("xmin", g1), row, 0);
  grid->addWidget(exMin = new KplDoubleEdit(td->xmi, g1, frm, prec), row, 1);
  int w = exMin->fontMetrics().width(m->number(-1.0e-123 / 3.0)) + 3;
  exMin->setMinimumWidth(w);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), row, 2);
  grid->addWidget(new QLabel("xmax", g1), row, 3);
  grid->addWidget(exMax = new KplDoubleEdit(td->xma, g1, frm, prec), row, 4);
  exMax->setMinimumWidth(w);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), row, 5);
#if KDE_VERSION_MAJOR > 2
  grid->addWidget(new QLabel(QString::fromUtf8("\316\224x"), g1), row, 6);
#else
  QHBoxLayout* hbox = new QHBoxLayout(1);
  grid->addLayout(hbox = new QHBoxLayout(1), row, 6);
  QLabel* l = new QLabel("D", g1);
  l->setFont(QFont("Symbol", 12, QFont::Normal, false, QFont::AnyCharSet));
  hbox->addWidget(l);
  hbox->addWidget(new QLabel("x", g1));
  hbox->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding));
#endif
  grid->addWidget(exStep = new KplDoubleEdit(td->dx, g1, frm, prec), row, 7);
  exStep->setMinimumWidth(w);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), row, 8);
  grid->addWidget(logX = new QCheckBox("log", g1), row++, 9);
  logX->setChecked(td->logx);
  grid->addWidget(new QLabel("ymin", g1), row, 0);
  grid->addWidget(eyMin = new KplDoubleEdit(td->ymi, g1), row, 1);
  grid->addWidget(new QLabel("ymax", g1), row, 3);
  grid->addWidget(eyMax = new KplDoubleEdit(td->yma, g1), row, 4);
  grid->addWidget(logY = new QCheckBox("log", g1), row++, 9);
  logY->setChecked(td->logy);
  grid->addWidget(new QLabel("zmin", g1), row, 0);
  grid->addWidget(ezMin = new KplDoubleEdit(td->zmi, g1, frm, prec), row, 1);
  ezMin->setMinimumWidth(w);
  grid->addWidget(new QLabel("zmax", g1), row, 3);
  grid->addWidget(ezMax = new KplDoubleEdit(td->zma, g1, frm, prec), row, 4);
  ezMax->setMinimumWidth(w);
#if KDE_VERSION_MAJOR > 2
  grid->addWidget(new QLabel(QString::fromUtf8("\316\224z"), g1), row, 6);
#else
  grid->addLayout(hbox = new QHBoxLayout(1), row, 6);
  l = new QLabel("D", g1);
  l->setFont(QFont("Symbol", 12, QFont::Normal, false, QFont::AnyCharSet));
  hbox->addWidget(l);
  hbox->addWidget(new QLabel("z", g1));
  hbox->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding));
#endif
  grid->addWidget(ezStep = new KplDoubleEdit(td->dz, g1, frm, prec), row, 7);
  ezStep->setMinimumWidth(w);
  grid->addWidget(logZ = new QCheckBox("log", g1), row, 9);
  logZ->setChecked(td->logz);
  connect(logX, SIGNAL(toggled(bool)), SLOT(slotLogToggled(bool)));
  connect(logY, SIGNAL(toggled(bool)), SLOT(slotLogToggled(bool)));
  connect(logZ, SIGNAL(toggled(bool)), SLOT(slotLogToggled(bool)));
  connect(bAuto, SIGNAL(clicked()), SLOT(slotAutoScale()));
}

void ThreeDDlg::getValues()
{
  td->xmi = exMin->value();
  td->xma = exMax->value();
  td->dx = exStep->value();
  td->logx = logX->isChecked();
  td->ymi = eyMin->value();
  td->yma = eyMax->value();
  td->logy = logY->isChecked();
  td->zmi = ezMin->value();
  td->zma = ezMax->value();
  td->dz = ezStep->value();
  td->logz = logZ->isChecked();
  td->x0 = ex0->value();
  td->w = exl->value();
  td->y0 = ey0->value();
  td->h = eyl->value();
  td->fx = efx->value();
  td->fy = efy->value();
  td->fz = efz->value();
  td->boxx = eBoxx->value();
  td->boxy = eBoxy->value();
  td->boxz = eBoxz->value();
  td->xtic = exTic->value();
  td->ytic = eyTic->value();
  td->ztic = ezTic->value();
  td->mticx = emTicx->interpretedValue();
  td->mticy = emTicy->interpretedValue();
  td->mticz = emTicz->interpretedValue();
  td->ndigx = enDigx->interpretedValue();
  td->ndigy = enDigy->interpretedValue();
  td->ndigz = enDigz->interpretedValue();
  td->phi = ePhi->value();
  td->theta = eTheta->value();
  td->xref = eXref->value();
  td->yref = eYref->value();
  td->gridmode = bg1->id(bg1->selected());
  td->mode3D = bg2->id(bg2->selected());
  td->showFrame = modeFrame->isChecked();
  td->relsiz = 0.01 * eRelSiz->interpretedValue();
  td->colFrame = colFrame->color().rgb();
  td->colGrid = colGrid->color().rgb();
  td->colData = colData->color().rgb();
  td->iex = ex->interpretedValue();
  td->iey = ey->interpretedValue();
  td->iez = ez->interpretedValue();
  td->sx = eLetX->text();
  td->sy = eLetY->text();
  td->sz = eLetZ->text();
  td->sh = eLetH->text();
}

void ThreeDDlg::autoScale(double xmin, double xmax, double ymin, double ymax,
                          double zmin, double zmax)
{
  double w = exl->value();
  double h = eyl->value();
  double r = h / w;
  double phi = 0.01745329251994 * ePhi->value();
  double theta = 0.01745329251994 * eTheta->value();
  double ctheta = cos(theta);
  double xr = fabs(sin(phi)) + fabs(cos(phi));
  double b = 0.7 * QMIN(w / xr, h / (r * fabs(ctheta) +
                                     xr * fabs(sin(theta))));
  eBoxx->setValue(b);
  eBoxz->setValue(b);
  eBoxy->setValue(b * r);
  eXref->setValue(0.5 * w);
  eYref->setValue(0.5 * (h - b * r * ctheta));
  int iex, iey, iez;
  double fx, fy, fz;
  if (autoNorm->isChecked()) {
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), &iex, &fx);
    Utils::expo(QMAX(fabs(ymin), fabs(ymax)), &iey, &fy);
    Utils::expo(QMAX(fabs(zmin), fabs(zmax)), &iez, &fz);
  } else {
    iex = iey = iez = 0;
    fx = fy = fz = 1.0;
  }
  efx->setValue(fx);
  ex->setValue(iex);
  efy->setValue(fy);
  ey->setValue(iey);
  efz->setValue(fz);
  ez->setValue(iez);
  int mtic, ndig;
  double vmin, vman, vtic;
  Utils::autoSc(&vmin, &vman, &vtic, &mtic, &ndig, xmin, xmax, fx, 0.25,
                logX->isChecked());
  exMin->setValue(vmin / fx);
  exMax->setValue(vman / fx);
  exTic->setValue(vtic);
  emTicx->setValue(mtic);
  enDigx->setValue(ndig);
  Utils::autoSc(&vmin, &vman, &vtic, &mtic, &ndig, ymin, ymax, fy, 0.25,
                logY->isChecked());
  eyMin->setValue(vmin / fy);
  eyMax->setValue(vman / fy);
  eyTic->setValue(vtic);
  emTicy->setValue(mtic);
  enDigy->setValue(ndig);
  Utils::autoSc(&vmin, &vman, &vtic, &mtic, &ndig, zmin, zmax, fz, 0.25,
                logZ->isChecked());
  ezMin->setValue(vmin / fz);
  ezMax->setValue(vman / fz);
  ezTic->setValue(vtic);
  emTicz->setValue(mtic);
  enDigz->setValue(ndig);
}

void ThreeDDlg::slotLogToggled(bool state)
{
  const QCheckBox* b = (const QCheckBox*) sender();
  if ((b == logX) || (b == logY) || (b == logZ)) {
    KplSpinBox* nd;
    KplDoubleEdit *mi, *ma;
    if (b == logX) {
      nd = enDigx;
      mi = exMin;
      ma = exMax;
    } else
      if (b == logY) {
        nd = enDigy;
        mi = eyMin;
        ma = eyMax;
      } else {
        nd = enDigz;
        mi = ezMin;
        ma = ezMax;
      }
    nd->setEnabled(!state);
    double bottom = state ? 1.0e-300 : -1.0e300;
    mi->setBottom(bottom);
    ma->setBottom(bottom);
  }
  if (state)
    autoNorm->setChecked(false);
}

void ThreeDDlg::slotNormToggled(bool state)
{
  if (state) {
    logX->setChecked(false);
    logY->setChecked(false);
    logZ->setChecked(false);
  }
}

void ThreeDDlg::slotEditLetX()
{
  EditTextDlg::editText(this, eLetX);
}

void ThreeDDlg::slotEditLetY()
{
  EditTextDlg::editText(this, eLetY);
}

void ThreeDDlg::slotEditLetZ()
{
  EditTextDlg::editText(this, eLetZ);
}

void ThreeDDlg::slotEditLetH()
{
  EditTextDlg::editText(this, eLetH);
}

Fun3DDlg::Fun3DDlg(QWidget* _parent, KplDoc* model, Fun3DItem *fd0) :
 ThreeDDlg(_parent, model, fd0, i18n("3D function"), "SEC-FUNC3D",
           i18n("Function")), fd(fd0)
{
  fdt = new Fun3DItem(*fd);
  QGroupBox* g = new QGroupBox(0, Qt::Vertical, "y(x,z)", frame1);
  vbox1->addWidget(g);
  QGridLayout* grid = new QGridLayout(g->layout(), 3, 2, spacingHint());
  grid->addWidget(new QLabel(i18n("Library"), g), 0, 0);
  QHBoxLayout* hbox = new QHBoxLayout();
  grid->addLayout(hbox, 0, 1);
  hbox->addWidget(fileName = new QLineEdit(fd->pathy.isLocalFile() ?
                                           fd->pathy.path() :
                                           fd->pathy.url(), g));
  QPushButton* b = new QPushButton("...", g);
  b->setFixedWidth(28);
  hbox->addWidget(b);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  grid->addWidget(new QLabel(i18n("Function"), g), 1, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 1, 1);
  hbox->addWidget(func = new QLineEdit(fd->namey, g));
  hbox->addWidget(b = new QPushButton("...", g));
  b->setFixedWidth(28);
  connect(b, SIGNAL(clicked()), SLOT(slotFunc()));
  grid->addWidget(new QLabel(i18n("Parameter"), g), 2, 0);
  grid->addLayout(hbox = new QHBoxLayout(), 2, 1);
  hbox->addWidget(b = new QPushButton(i18n("Edit"), g));
  connect(b, SIGNAL(clicked()), SLOT(slotEditParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Load"), g));
  connect(b, SIGNAL(clicked()), SLOT(slotGetParameter()));
  hbox->addWidget(b = new QPushButton(i18n("Save"), g));
  connect(b, SIGNAL(clicked()), SLOT(slotSaveParameter()));
  vbox1->addWidget(g1);
  initRange(new QGridLayout(g1->layout(), 3, 10, spacingHint()));
  Utils::setSize(this, "Func3DDialog");
}

Fun3DDlg::~Fun3DDlg()
{
  Utils::saveSize(this, "Func3DDialog");
  delete fdt;
}

bool Fun3DDlg::getTemp()
{
  fdt->pathy = fileName->text();
  fdt->namey = func->text();
  if (fdt->hmody)
    lt_dlclose(fdt->hmody);
  if (Fun3DItem::getFunc3DAddr(fdt->pathy.path(), fdt->namey, &fdt->hmody,
                           &fdt->fkty))
    return false;
  return true;
}

void Fun3DDlg::getValues(bool ok)
{
  if (getTemp()) {
    *fd = *fdt;
    ThreeDDlg::getValues();
    m->setModified();
    m->backupItems();
    if (ok)
      accept();
    else
      *fdt = *fd;
  }
}

void Fun3DDlg::slotFile()
{
  m->getFile(fileName);
}

void Fun3DDlg::slotFunc()
{
  FuncDlg::getFunc(this, fileName, func, true);
}

void Fun3DDlg::slotGetParameter()
{
  m->getPar(this, fdt->py);
}

void Fun3DDlg::slotEditParameter()
{
  EditParameterDlg dlg(this, m, fdt->py, fileName->text(), func->text(), true,
                       true);
  connect(&dlg, SIGNAL(applyClicked()), SLOT(slotApply()));
  dlg.exec();
}

void Fun3DDlg::slotSaveParameter()
{
  m->saveFunPar(this, fdt->py);
}

void Fun3DDlg::slotExportValues()
{
  if (getTemp()) {
    KURL url;
    if (m->getWriteURL(this, url, "*.dat\n*")) {
      QFile f(url.isLocalFile() ? url.path() : m->tmpFile());
      if (f.open(IO_WriteOnly)) {
        QTextStream ts(&f);
        fdt->exportTable(ts, m);
        f.close();
        m->setCurrentDir(url);
        if (!url.isLocalFile())
          m->copyTmp(f.name(), url);
      } else
        KMessageBox::error(this, i18n("while trying to open"));
    }
  }
}

void Fun3DDlg::slotAutoScale()
{
  if (getTemp()) {
    double ymi, yma;
    fdt->xmi = exMin->value();
    fdt->xma = exMax->value();
    fdt->zmi = ezMin->value();
    fdt->zma = ezMax->value();
    int nv = fdt->calcTable();
    Utils::minMaxFile(&ymi, &yma, fdt->yv[0], nv);
    autoScale(fdt->xmi, fdt->xma, ymi, yma, fdt->zmi, fdt->zma);
  }
}

void Fun3DDlg::slotOk()
{
  getValues(true);
}

void Fun3DDlg::slotApply()
{
  getValues(false);
}

void Fun3DDlg::slotUser1()
{
  if (getTemp()) {
    KURL url;
    if (m->getWriteURL(this, url, "*.dat\n*")) {
      QFile f(url.isLocalFile() ? url.path() : m->tmpFile());
      if (f.open(IO_WriteOnly)) {
        QTextStream ts(&f);
        fdt->exportTable(ts, m);
        f.close();
        m->setCurrentDir(url);
        if (!url.isLocalFile())
          m->copyTmp(f.name(), url);
      } else
        KMessageBox::error(this, i18n("while trying to open"));
    }
  }
}

Array3DDlg::Array3DDlg(QWidget* _parent, KplDoc* model, Array3DItem *ad0) :
 ThreeDDlg(_parent, model, ad0, i18n("3D array"), "SEC-ARRAY3D",
           i18n("Array")), ad(ad0), adt(0)
{
  QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Data"), frame1);
  vbox1->addWidget(g);
  QGridLayout* grid = new QGridLayout(g->layout(), 2, 2, spacingHint());
  grid->addWidget(new QLabel(i18n("File"), g), 0, 0);
  QHBoxLayout* hbox = new QHBoxLayout();
  grid->addLayout(hbox, 0, 1);
  hbox->addWidget(fileName = new QLabel(g));
  if (!ad->internal)
    fileName->setText(ad->url.isLocalFile() ?  ad->url.path() : ad->url.url());
  fileName->setFrameStyle(QFrame::Panel | QFrame::Sunken);
#if QT_VERSION >= 0x030100
  fileName->setSizePolicy(QSizePolicy::MinimumExpanding,
                          QSizePolicy::Minimum);
#else
  fileName->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::Minimum));
#endif
  QPushButton* b = new QPushButton("...", g);
  b->setFixedWidth(28);
  hbox->addWidget(b);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  hbox->addWidget(bReload = new QPushButton(g));
  bReload->setPixmap(BarIcon("reload"));
  bReload->setDisabled(fileName->text().isEmpty());
  connect(bReload, SIGNAL(clicked()), SLOT(slotReload()));
  grid->addLayout(hbox = new QHBoxLayout(), 1, 1);
  hbox->addWidget(internal = new QCheckBox(i18n("Internal data"), g));
  internal->setChecked(ad->internal);
  hbox->addWidget(edit = new QPushButton(i18n("&Edit"), g));
  edit->setEnabled(ad->internal);
  connect(edit, SIGNAL(clicked()), SLOT(slotEdit()));
  hbox->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding));
  vbox1->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Columns"),
                                     frame1));
  grid = new QGridLayout(g->layout(), 3, 8, spacingHint());
  grid->addWidget(new QLabel(i18n("x column"), g), 0, 0);
  grid->addWidget(ix = new KplSpinBox(0, 0, 1, g), 0, 1);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 2);
  grid->addWidget(new QLabel(i18n("y column"), g), 0, 3);
  grid->addWidget(iy = new KplSpinBox(0, 0, 1, g), 0, 4);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 5);
  grid->addWidget(new QLabel(i18n("z column"), g), 0, 6);
  grid->addWidget(iz = new KplSpinBox(0, 0, 1, g), 0, 7);
  grid->addWidget(new QLabel(i18n("Error column"), g), 1, 0);
  grid->addWidget(ie = new KplSpinBox(0, 0, 1, g), 1, 1);
  ie->setEnabled(ad->errbars);
  grid->addMultiCellWidget(err = new QCheckBox(i18n("Error bar"), g),
                           1, 1, 2, 4);
  err->setChecked(ad->errbars);
  grid->addWidget(new QLabel(i18n("Smoothing factor"), g), 2, 0);
  grid->addWidget(eFactor = new KplDoubleEdit(ad->smf, g), 2, 1);
  vbox1->addWidget(g1);
  grid = new QGridLayout(g1->layout(), 3, 10, spacingHint());
  grid->addWidget(new QLabel(i18n("Start index"), g1), 0, 0);
  grid->addWidget(iStart = new KplSpinBox(0, 0, 1, g1), 0, 1);
  grid->addWidget(new QLabel(i18n("Number of points"), g1), 0, 3);
  grid->addWidget(n = new KplSpinBox(0, 0, 1, g1), 0, 4);
  setSpinBoxes(ad);
  initRange(grid, 1);
  connect(internal, SIGNAL(toggled(bool)), SLOT(slotInternal(bool)));
  connect(err, SIGNAL(toggled(bool)), ie, SLOT(setEnabled(bool)));
  connect(iStart, SIGNAL(valueChanged(int)), SLOT(slotStartChanged(int)));
  connect(n, SIGNAL(valueChanged(int)), SLOT(slotNChanged(int)));
  Utils::setSize(this, "Array3DDialog");
}

Array3DDlg::~Array3DDlg()
{
  Utils::saveSize(this, "Array3DDialog");
  delete adt;
}

void Array3DDlg::setSpinBoxes(Array3DItem* _ad)
{
  int iColMax = QMAX(0, _ad->ncols - 1);
  ix->setRange(0, iColMax);
  ix->setValue(_ad->ix);
  iy->setRange(0, iColMax);
  iy->setValue(_ad->iy);
  iz->setRange(0, iColMax);
  iz->setValue(_ad->iz);
  ie->setRange(0, iColMax);
  ie->setValue(_ad->ie);
  iStart->setRange(0, QMAX(0, _ad->nrows - 1));
  iStart->setValue(_ad->istart);
  n->setRange(0, _ad->nrows);
  n->setValue(_ad->n);
}

bool Array3DDlg::load(KURL &url, int idec)
{
  bool success = !url.isEmpty();
  if (success) {
    delete adt;
    adt = new Array3DItem;
    adt->url = url;
    adt->idec = idec;
    success = readData();
  }
  return success;
}

void Array3DDlg::getValues(bool ok)
{
  if (adt) {
    *ad = *adt;
    delete adt;
    adt = 0;
    ArrayItem::freeX(&ad->yv);
  }
  ad->ix = ix->interpretedValue();
  ad->iy = iy->interpretedValue();
  ad->iz = iz->interpretedValue();
  ad->ie = ie->interpretedValue();
  ad->smf = eFactor->value();
  ad->istart = iStart->interpretedValue();
  ad->url = fileName->text();
  ad->errbars = err->isChecked();
  ad->internal = internal->isChecked();
  ad->istart = QMAX(ad->istart, 0);
  ad->n = QMIN(n->interpretedValue(), ad->nrows - ad->istart);
  ThreeDDlg::getValues();
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
  else
    setSpinBoxes(ad);
}

bool Array3DDlg::readData(bool fromBuffer)
{
  if (fromBuffer)
    adt->nrows = ArrayItem::readFile(&buf, &adt->ncols, &adt->xv);
  else
    adt->nrows = ArrayItem::readFile(adt->url, &adt->ncols, &adt->xv,
                                     adt->idec);
  if (adt->nrows) {
    int maxCol = adt->ncols - 1;
    adt->ix = QMIN(ad->ix, maxCol);
    adt->iy = QMIN(ad->iy, maxCol);
    adt->iz = QMIN(ad->iz, maxCol);
    adt->ie = QMIN(ad->ie, maxCol);
    adt->istart = QMIN(ad->istart, adt->nrows - 1);
    adt->n = adt->nrows - adt->istart;
    if (ad->n && (!fromBuffer))
      adt->n = QMIN(ad->n, adt->n);
    setSpinBoxes(adt);
    return adt->nrows;
  } else {
    delete adt;
    adt = 0;
    KMessageBox::error(this, i18n("while reading the data"));
    return 0;
  }
}

void Array3DDlg::slotStartChanged(int ia)
{
  int nmax = ad->nrows - ia;
  if (n->interpretedValue() > nmax)
    n->setValue(nmax);
}

void Array3DDlg::slotNChanged(int na)
{
  int imin = ad->nrows - na;
  if (iStart->interpretedValue() > imin)
    iStart->setValue(imin);
}

void Array3DDlg::slotFile()
{
  int idec = adt ? adt->idec : ad->idec;
  KURL url = m->getReadURL(idec, (adt ? adt : ad)->url.fileName());
  if (load(url, idec)) {
    fileName->setText(url.isLocalFile() ? url.path() : url.url());
    m->setCurrentDir(url);
    bReload->setDisabled(fileName->text().isEmpty());
    disconnect(internal, SIGNAL(toggled(bool)), this,
               SLOT(slotInternal(bool)));
    internal->setChecked(false);
    connect(internal, SIGNAL(toggled(bool)), SLOT(slotInternal(bool)));
  }
}

void Array3DDlg::slotReload()
{
  KURL url(fileName->text());
  load(url, adt ? adt->idec : ad->idec);
}

void Array3DDlg::slotInternal(bool state)
{
  if (state) {
    Array3DItem* a2 = 0;
    if (adt)
      a2 = new Array3DItem(*adt);
    delete adt;
    adt = new Array3DItem;
    Array3DItem* src = a2 ? a2 : ad;
    adt->url = KURL();
    adt->nrows = src->nrows;
    adt->ncols = 4;
    adt->istart = iStart->interpretedValue();
    adt->n = n->interpretedValue();
    int dim = adt->ncols * adt->nrows;
    if (dim) {
      double* x1 = new double[dim];
      memset(x1, 0, dim * sizeof(double));
      adt->xv = new double*[adt->ncols];
      for (int j = 0; j < adt->ncols; j++)
        adt->xv[j] = &x1[j * adt->nrows];
      if (src->ncols > 4) {
        int siz = adt->nrows * sizeof(double);
        memcpy(adt->xv[0], src->xv[ix->interpretedValue()], siz);
        memcpy(adt->xv[1], src->xv[iy->interpretedValue()], siz);
        memcpy(adt->xv[2], src->xv[iz->interpretedValue()], siz);
        memcpy(adt->xv[3], src->xv[ie->interpretedValue()], siz);
        adt->iy = 1;
        adt->iz = 2;
        adt->ie = 3;
      } else {
        memcpy(adt->xv[0], src->xv[0],
               src->ncols * src->nrows * sizeof(double));
        adt->ix = ix->interpretedValue();
        adt->iy = iy->interpretedValue();
        adt->iz = iz->interpretedValue();
        adt->ie = ie->interpretedValue();
      }
    }
    setSpinBoxes(adt);
    fileName->clear();
    adt->idec = 0;
    delete a2;
  } else {
    Array3DItem* src = adt ? adt : ad;
    KURL url;
    if (m->getWriteURL(this, url, "*.dat\n*")) {
      QFile f(url.isLocalFile() ? url.path() : m->tmpFile());
      bool success = f.open(IO_WriteOnly);
      if (success) {
        QTextStream t(&f);
        for (int i = 0; i < src->nrows; i++)
          for (int j = 0; j < src->ncols; j++)
            t << m->number(src->xv[j][i])
              << (j < (src->ncols - 1) ? " " : "\n");
        f.close();
        fileName->setText(url.isLocalFile() ? url.path() : url.url());
        m->setCurrentDir(url);
        if (!url.isLocalFile())
          m->copyTmp(f.name(), url);
      } else {
        KMessageBox::error(this, i18n("while trying to open file"));
        internal->setChecked(true);
        return;
      }
    } else {
      internal->setChecked(true);
      return;
    }
  }
  edit->setEnabled(state);
  bReload->setDisabled(fileName->text().isEmpty());
}

void Array3DDlg::slotEdit()
{
  QTextStream t(&buf, IO_WriteOnly);
  Array3DItem* src = adt ? adt : ad;
  for (int i = 0; i < src->nrows; i++)
    for (int j = 0; j < src->ncols; j++)
      t << m->number(src->xv[j][i])
        << (j < (src->ncols - 1) ? m->separator() : QString("\n"));
#if KDE_VERSION_MAJOR > 2
  EditDlg dlg(this, &buf, m->options()->prec + 4);
#else
  int sav = QMultiLineEdit::defaultTabStop();
  QMultiLineEdit::setDefaultTabStop(m->options()->prec + 4);
  EditDlg dlg(this, &buf);
#endif
  connect(&dlg, SIGNAL(applyClicked()), SLOT(slotEditApply()));
  if (dlg.exec()) {
    delete adt;
    adt = new Array3DItem;
    readData(true);
  }
#if KDE_VERSION_MAJOR < 3
  QMultiLineEdit::setDefaultTabStop(sav);
#endif
}

void Array3DDlg::slotEditApply()
{
  delete adt;
  adt = new Array3DItem;
  if (readData(true))
    slotApply();
}

void Array3DDlg::slotAutoScale()
{
  Array3DItem* a = adt ? adt : ad;
  if (a->xv) {
    double xmi, xma, ymi, yma, zmi, zma;
    int istart = iStart->interpretedValue();
    int nv = n->interpretedValue();
    Utils::minMaxFile(&xmi, &xma, &a->xv[ix->interpretedValue()][istart], nv);
    Utils::minMaxFile(&ymi, &yma, &a->xv[iy->interpretedValue()][istart], nv);
    Utils::minMaxFile(&zmi, &zma, &a->xv[iz->interpretedValue()][istart], nv);
    autoScale(xmi, xma, ymi, yma, zmi, zma);
  }
}

void Array3DDlg::slotOk()
{
  getValues(true);
}

void Array3DDlg::slotApply()
{
  getValues(false);
}

void Array3DDlg::slotUser1()
{
  KURL url(fileName->text());
  if (load(url, adt ? adt->idec : ad->idec)) {
    if (m->getWriteURL(this, url, "*.dat\n*")) {
      QFile f(url.isLocalFile() ? url.path() : m->tmpFile());
      if (f.open(IO_WriteOnly)) {
        QTextStream ts(&f);
        int iColMax = QMAX(0, adt->ncols - 1);
        adt->ix = QMIN(ix->interpretedValue(), iColMax);
        adt->iy = QMIN(iy->interpretedValue(), iColMax);
        adt->iz = QMIN(iz->interpretedValue(), iColMax);
        adt->ie = QMIN(ie->interpretedValue(), iColMax);
        adt->smf = eFactor->value();
        adt->istart = QMIN(iStart->interpretedValue(), adt->nrows - 1);
        adt->errbars = err->isChecked();
        adt->istart = QMAX(adt->istart, 0);
        adt->n = QMIN(n->interpretedValue(), adt->nrows - adt->istart);
        adt->xmi = exMin->value();
        adt->xma = exMax->value();
        adt->dx = exStep->value();
        adt->logx = logX->isChecked();
        adt->zmi = ezMin->value();
        adt->zma = ezMax->value();
        adt->dz = ezStep->value();
        adt->logz = logZ->isChecked();
        adt->exportTable(ts, m);
        f.close();
        m->setCurrentDir(url);
        if (!url.isLocalFile())
          m->copyTmp(f.name(), url);
      } else
        KMessageBox::error(this, i18n("while trying to open"));
    }
  }
}
