/***************************************************************************
                          threeditem.cpp  -  description
                             -------------------
    begin                : Tue Dec 25 2001
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <float.h>
#include <math.h>
#include <dlfcn.h>
#include <qurl.h>
#include <qfileinfo.h>
#include <ksimpleconfig.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "threeditem.h"
#include "arrayitem.h"
#include "kgraph.h"
#include "utils.h"
#include "kplchecklistitem.h"
#include "kpldoc.h"
#include "fitpack.h"
#include "framedlg.h"
#include "threeddlg.h"

static int sortCol;

static int cmpasc(const void *e1, const void *e2)
{
  double a1 = (*((double **) e1))[sortCol];
  double a2 = (*((double **) e2))[sortCol];
  if (a1 < a2)
    return -1;
  if (a1 > a2)
    return 1;
  return 0;
}

ThreeDItem::P3d::P3d(int n3d) : nRes(n3d), scal((n3d - 1) / 15.0),
 length(15.0), height(15.0), phi(0.0), theta(0.0), a11(1.0), a13(0.0),
 a21(0.0), a22(1.0), a23(0.0), yMin(new double[n3d]), yMax(new double[n3d])
{
}

ThreeDItem::P3d::~P3d()
{
  delete [] yMax;
  delete [] yMin;
}

ThreeDItem::ThreeDItem() : showFrame(true), logz(false), logxo(false),
 logzo(false), ndigz(-1), mticz(5), iez(0), mode3D(2), colData(0), zmi(0.0),
 zma(15.0), dx(0.0), dz(0.0), ztic(5.0), phi(20.0), theta(-20.0), boxx(7.5),
 boxy(7.5), boxz(7.5), xref(7.5), yref(4.0), fx(1.0), fy(1.0), fz(1.0),
 xmio(0.0), xmao(0.0), dxo(0.0), zmio(0.0), zmao(0.0), dzo(0.0), yv(0)
{
  w = h = 15.0;
}

ThreeDItem::ThreeDItem(const ThreeDItem& a) :
 FrameItem(a), showFrame(a.showFrame), logz(a.logz), logxo(a.logxo),
 logzo(a.logzo), ndigz(a.ndigz), mticz(a.mticx), iez(a.iez), mode3D(a.mode3D),
 nx(a.nx), nz(a.nz), colData(a.colData), zmi(a.zmi), zma(a.zma), dx(a.dx),
 dz(a.dz), ztic(a.ztic), phi(a.phi), theta(a.theta), boxx(a.boxx),
 boxy(a.boxy), boxz(a.boxz), xref(a.xref), yref(a.yref), fx(a.fx),
 fy(a.fy), fz(a.fz), xmio(a.xmio), xmao(a.xmao), dxo(a.dxo), zmio(a.zmio),
 zmao(a.zmao), dzo(a.dzo), yv(a.yv), sz(a.sz)
{
  yv = ArrayItem::copyX(a.yv, a.nx, a.nz);
}

ThreeDItem::ThreeDItem(Kpl::AutoStruct* aut) : FrameItem(aut), showFrame(true),
 logz(false), logxo(false), logzo(false), ndigz(-1), mticz(5), iez(0),
 mode3D(2), colData(aut->colData), zmi(0.0), zma(15.0), dx(0.0), dz(0.0),
 ztic(5.0), phi(20.0), theta(-20.0), boxx(7.5), boxy(7.5), boxz(7.5),
 xref(7.5), yref(4.0), fx(1.0), fy(1.0), fz(1.0), xmio(0.0), xmao(0.0),
 dxo(0.0), zmio(0.0), zmao(0.0), dzo(0.0), yv(0)
{
}

ThreeDItem::ThreeDItem(KSimpleConfig* plo, Kpl::AutoStruct* aut) :
 FrameItem(plo, aut), logxo(false), logzo(false), xmio(0.0), xmao(0.0),
 dxo(0.0), zmio(0.0), zmao(0.0), dzo(0.0), yv(0)
{
  showFrame = plo->readBoolEntry("frame", true);
  logz = plo->readBoolEntry("logz", false);
  ndigz = plo->readNumEntry("ndigz", -1);
  mticz = plo->readNumEntry("mticz", 5);
  iez = plo->readNumEntry("iez");
  mode3D = plo->readNumEntry("mode3d", 2);
  QString s = plo->readEntry("coldata", "");
  if (!s.isEmpty()) {
    unsigned u;
    sscanf(s.latin1(), "%x", &u);
    colData = KplGraph::rgbQt(u);
  } else
    colData = aut->colData;
  zmi = plo->readDoubleNumEntry("zmi");
  zma = plo->readDoubleNumEntry("zma", zmi + 1.0);
  dx = plo->readDoubleNumEntry("dx");
  dz = plo->readDoubleNumEntry("dz");
  ztic = plo->readDoubleNumEntry("ztic", 5.0);
  phi = plo->readDoubleNumEntry("phi", 20.0);
  theta = plo->readDoubleNumEntry("theta", -20.0);
  boxx = plo->readDoubleNumEntry("boxx", 7.5);
  boxy = plo->readDoubleNumEntry("boxy", 7.5);
  boxz = plo->readDoubleNumEntry("boxz", 7.5);
  xref = plo->readDoubleNumEntry("xref", 7.5);
  yref = plo->readDoubleNumEntry("yref", 4.0);
  fx = plo->readDoubleNumEntry("fx", 1.0);
  fy = plo->readDoubleNumEntry("fy", 1.0);
  fz = plo->readDoubleNumEntry("fz", 1.0);
  sz = plo->readEntry("sz", "");
}

ThreeDItem::ThreeDItem(bool act, bool lgx, bool lgy, bool lgz, bool frame,
                       int ndx, int ndy, int ndz, int mtx, int mty, int mtz,
                       int gm, int gm3, int ex, int ey, int ez,
                       const QString& col_f, const QString& col_g,
                       const QString& col_d, double x_0, double wx, double y_0,
                       double hy, double xmin, double xmax, double ymin,
                       double ymax, double zmin, double zmax, double d_x,
                       double d_z, double xt, double yt, double zt,
                       double relSize, double phi_, double theta_, double b_x,
                       double b_y, double b_z, double xr, double yr,
                       double f_x, double f_y, double f_z, const QString& sX,
                       const QString& sY, const QString& sZ, const QString& sH,
                       Kpl::AutoStruct* aut) :
 FrameItem(act, lgx, lgy, ndx, ndy, mtx, mty, gm, ex, ey, col_f, col_g, x_0,
           wx, y_0, hy, xmin, xmax, ymin, ymax, xt, yt, relSize, sX, sY, sH,
           aut),
 showFrame(frame), logz(lgz), logxo(false), logzo(false), ndigz(ndz),
 mticz(mtz), iez(ez), mode3D(gm3), zmi(zmin), zma(zmax), dx(d_x), dz(d_z),
 ztic(zt), phi(phi_), theta(theta_), boxx(b_x), boxy(b_y), boxz(b_z),
 xref(xr), yref(yr), fx(f_x), fy(f_y), fz(f_z), xmio(0.0), xmao(0.0), dxo(0.0),
 zmio(0.0), zmao(0.0), dzo(0.0), yv(0), sz(sZ), p3d(2000)
{
  unsigned u;
  sscanf(col_d.latin1(), "%x", &u);
  colData = KplGraph::rgbQt(u);
}

ThreeDItem::~ThreeDItem()
{
  ArrayItem::freeX(&yv);
}

ThreeDItem& ThreeDItem::operator=(const ThreeDItem& a)
{
  if (this != &a) {
    *(FrameItem*)this = a;
    showFrame = a.showFrame;
    logz = a.logz;
    logxo = a.logxo;
    logzo = a.logzo;
    ndigz = a.ndigz;
    mticz = a.mticx;
    iez = a.iez;
    mode3D = a.mode3D;
    colData = a.colData;
    zmi = a.zmi;
    zma = a.zma;
    dx = a.dx;
    dz = a.dz;
    xmio = a.xmio;
    xmao = a.xmao;
    dxo = a.dxo;
    zmio = a.zmio;
    zmao = a.zmao;
    dzo = a.dzo;
    ztic = a.ztic;
    phi = a.phi;
    theta = a.theta;
    boxx = a.boxx;
    boxy = a.boxy;
    boxz = a.boxz;
    xref = a.xref;
    yref = a.yref;
    fx = a.fx;
    fy = a.fy;
    fz = a.fz;
    ArrayItem::freeX(&yv);
    nx = a.nx;
    nz = a.nz;
    yv = ArrayItem::copyX(a.yv, a.nx, a.nz);
    sz = a.sz;
  }
  return *this;
}

void ThreeDItem::draw(KplGraph* g)
{
  if (!g->type())
    bRect.setRect(0, 0, 0, 0);
  if (w && h && active) {
    g->format(xf, yf);
    g->setRelSize(relsiz);
    g->Window(x0, w, y0, h);
    if (!g->type()) {
      KGraph* gr = (KGraph*) g;
      bRect.setCoords(gr->minx, gr->maxy, gr->maxx, gr->miny);
    }
    g->scale(0.0, w, 0.0, h, false, false);
    g->setDig(ndigx, ndigy);
    g->setColFrame(colFrame);
    g->setColData(colData);
    g->raster(xtic, ytic, mticx, mticy,
              showFrame ? KplGraph::FrameOnly : KplGraph::Nothing,
              xTop, yRight);
    g->letH(sh, xTop);
    g->setSymbol(1);
    scalx = boxx / (logx ? log10(xma / xmi) : (fx * (xma - xmi)));
    scaly = boxy / (logy ? log10(yma / ymi) : (fy * (yma - ymi)));
    scalz = boxz / (logz ? log10(zma / zmi) : (fz * (zma - zmi)));
    init3d();
    rot3d(phi, theta);
    plar3d(g);
    rast3d(g);
    g->setColData(colData);
  }
}

void ThreeDItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                          KplDoc* m) const
{
  FrameItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("frame", showFrame);
  if (logz)
    plo->writeEntry("logz", logz);
  plo->writeEntry("ndigz", ndigz);
  plo->writeEntry("mticz", mticz);
  plo->writeEntry("iez", iez);
  plo->writeEntry("mode3d", mode3D);
  QString s;
  plo->writeEntry("coldata", s.sprintf("%x", KplGraph::rgbQt1(colData)));
  plo->writeEntry("zmi", zmi);
  plo->writeEntry("zma", zma);
  plo->writeEntry("dx", dx);
  plo->writeEntry("dz", dz);
  plo->writeEntry("ztic", ztic);
  plo->writeEntry("phi", phi);
  plo->writeEntry("theta", theta);
  plo->writeEntry("boxx", boxx);
  plo->writeEntry("boxy", boxy);
  plo->writeEntry("boxz", boxz);
  plo->writeEntry("xref", xref);
  plo->writeEntry("yref", yref);
  plo->writeEntry("fx", fx);
  plo->writeEntry("fy", fy);
  plo->writeEntry("fz", fz);
  plo->writeEntry("sz", sz);
}

void ThreeDItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  for (int ix = 0; ix < nx; ix++)
    for (int iz = 0; iz < nz; iz++)
      ts << m->number(xmi + ix * xStep) << m->separator()
         << m->number(zmi + iz * zStep) << m->separator()
         << m->number(yv[ix][iz]) << "\n";
}

void ThreeDItem::rot3d(double phi, double theta)
{
  if ((phi != p3d.phi) || (theta != p3d.theta)) {
    p3d.phi = phi;
    p3d.theta = theta;
    phi *= 0.01745329251994;
    theta *= 0.01745329251994;
    p3d.a13 = -sin(phi);
    p3d.a11 = cos(phi);
    p3d.a22 = cos(theta);
    double stheta = sin(theta);
    p3d.a21 = -stheta * p3d.a13;
    p3d.a23 = stheta * p3d.a11;
  }
}

void ThreeDItem::init3d()
{
  p3d.length = w;
  p3d.height = h;
  p3d.scal = (p3d.nRes - 1) / w;
}

double ThreeDItem::xFrom3d(double x, double z)
{
  return p3d.a11 * x + p3d.a13 * z + p3d.xOff;
}

double ThreeDItem::yFrom3d(double x, double y, double z)
{
  return p3d.a21 * x + p3d.a23 * z + p3d.a22 * y + p3d.yOff;
}

void ThreeDItem::line3d(KplGraph* g, double x1, double y1, double z1,
                        double x2, double y2, double z2)
{
  double x[2], y[2];
  x1 *= fx;
  x2 *= fx;
  y1 *= fy;
  y2 *= fy;
  z1 *= fz;
  z2 *= fz;
  x[0] = xFrom3d(x1, z1);
  y[0] = yFrom3d(x1, y1, z1);
  x[1] = xFrom3d(x2, z2);
  y[1] = yFrom3d(x2, y2, z2);
  g->plArray(x, y, 1.0, 1.0, 2);
}

void ThreeDItem::plot3d(KplGraph* g, bool xArr, bool yArr, bool zArr,
                        const double* xData, const double* yData,
                        const double* zData, double xScale, double yScale,
                        double zScale, bool firstLine, int nPnts,
                        double xOff, double yOff, double* vtx)
{
  bool xOutLeftOld = false;
  bool xOutRightOld = false;
  bool xOutOld = false;
  bool lowOld = false;
  bool highOld = false;
  int jOld = -1;
  double xOld = 0.0;
  double yOld = 0.0;
  QArray<double> xBuf(nPnts);
  QArray<double> yBuf(nPnts);
  int nBuf = 0;
  if (firstLine)
    for (int i = 0; i < p3d.nRes; i++) {
      p3d.yMin[i] = DBL_MAX;
      p3d.yMax[i] = -DBL_MAX;
    }
  for (int i = 0; i < nPnts; i++) {
    double x3d = xArr ? xData[i] * xScale : xData[0] + i * xScale;
    double y3d = yArr ? yData[i] * yScale : yData[0] + i * yScale;
    double z3d = zArr ? zData[i] * zScale : zData[0] + i * zScale;
    double x = p3d.a11 * x3d + p3d.a13 * z3d + xOff;
    double y0 = p3d.a21 * x3d + p3d.a23 * z3d + yOff;
    double y = y0 + p3d.a22 * y3d;
    bool xOutLeft = x < 0.0;
    bool xOutRight = x > p3d.length;
    bool xOut = xOutLeft || xOutRight;
    bool yOut = (y < 0.0) || (y > p3d.height);
    bool low, high;
    if (i) { // not first point
      if (!((xOutLeft && xOutLeftOld) || (xOutRight && xOutRightOld))) {
        double xStart = xOutOld ? (xOutLeftOld ? 0.0 : p3d.length) : xOld;
        int jStart = qRound(p3d.scal * xStart);
        int jEnd = xOut ? (xOutLeft ?
                           0 : (p3d.nRes - 1)) : qRound(p3d.scal * x);
        if ((fabs(x - xStart) * p3d.scal) < 0.5)
          jEnd = jStart;
        bool vertical = (jEnd == jStart);
        int mask = 0;
        double yStart = yOld;
        double slope = 1.0;
        double yMinOld = 0.0;
        double yMaxOld = 0.0;
        double ymi1 = 0.0;
        double yma1 = 0.0;
        if (vertical) {
          mask = jStart;
          ymi1 = yMinOld = p3d.yMin[mask];
          yma1 = yMaxOld = p3d.yMax[mask];
          jStart = qRound(p3d.scal * yOld);
          jEnd = qRound(p3d.scal * y);
        } else {
          slope = (y - yOld) / (x - xOld);
          yStart += slope * (xStart - xOld);
        }
        slope /= p3d.scal;
        int inc = (jEnd > jStart) ? 1 : -1;
        for (int j = jStart; ; j += inc) {
          double yj = yStart + (j - jStart) * slope;
          if (!vertical) {
            mask = j;
            yMinOld = p3d.yMin[mask];
            yMaxOld = p3d.yMax[mask];
            ymi1 = p3d.yMin[j - inc];
            yma1 = p3d.yMax[j - inc];
          }
          p3d.yMin[mask] = QMIN(yj, yMinOld);
          p3d.yMax[mask] = QMAX(yj, yMaxOld);
          if ((j == jOld) && (!vertical)) {
            low = lowOld;
            high = highOld;
          } else {
            low = (yj <= yMinOld);
            high = (yj >= yMaxOld);
          }
          if ((low || high) && (!yOut)) { // visible
            if (nBuf)
              if ((low ^ lowOld) && (high ^ highOld)) {
                if (nBuf > 1) {
                  yBuf[nBuf - 1] = lowOld ? ymi1 : yma1;
                  g->plArray(xBuf, yBuf, 1.0, 1.0, nBuf);
                  xBuf[0] = xBuf[nBuf - 1];
                  yBuf[0] = low ? ymi1 : yma1;
                }
                nBuf = 1;
              } else {
                if ((nBuf > 1) && (j != jStart)) // replace point
                  nBuf--;
              }
            xBuf[nBuf] = vertical ? x : (xStart + (j - jStart) / p3d.scal);
            yBuf[nBuf++] = yj;
          } else { // not visible
            if (nBuf > 1) {
              yBuf[nBuf - 1] = lowOld ? ymi1 : yma1;
              g->plArray(xBuf, yBuf, 1.0, 1.0, nBuf);
            }
            nBuf = 0;
          }
          lowOld = low;
          highOld = high;
          jOld = mask;
          if (j == jEnd)
            break;
        }
      }
      if (vtx) {
        vtx[4] = vtx[12] = x;
        vtx[5] = y;
        vtx[13] = y0;
        if (firstLine) {
          vtx[2] = vtx[10] = x;
          vtx[3] = y;
          vtx[11] = y0;
        }
      }
    } else { // first point
      if (vtx) {
        vtx[6] = vtx[14] = x;
        vtx[7] = y;
        vtx[15] = y0;
        if (firstLine) {
          vtx[0] = vtx[8] = x;
          vtx[1] = y;
          vtx[9] = y0;
        }
      }
    }
    xOld = x;
    yOld = y;
    xOutLeftOld = xOutLeft;
    xOutRightOld = xOutRight;
    xOutOld = xOut;
  }
  if (nBuf > 1)
    g->plArray(xBuf, yBuf, 1.0, 1.0, nBuf);
}

void ThreeDItem::frame3d(KplGraph* g, int ihcor)
{
  int i = QMIN(QMAX(2 * ihcor - 1, 1), 7);
  double x[5], y[5];
/* the vertices which may be hidden are drawn by a call to plot3d */
  x[0] = vertex[i - 1];
  y[0] = vertex[i];
  x[1] = x[3] = vertex[i + 7];
  y[1] = y[3] = vertex[i + 8];
  i -= 2;
  if (i == -1)
    i = 7;
  x[2] = vertex[i + 7];
  y[2] = vertex[i + 8];
  i += 4;
  if (i > 7)
    i -= 8;
  x[4] = vertex[i + 7];
  y[4] = vertex[i + 8];
  double z = 0.0;
  double phi = p3d.phi;
  double theta= p3d.theta;
  rot3d(0.0, 0.0);
  plot3d(g, true, true, false, x, y, &z, 1.0, 1.0, 0.0, false, 3, 0.0, 0.0, 0);
  plot3d(g, true, true, false, &x[3], &y[3], &z, 1.0, 1.0, 0.0, false, 2, 0.0,
         0.0, 0);
  rot3d(phi, theta);
  x[0] = vertex[i - 1];
  y[0] = vertex[i];
  i -= 2;
  for (int j = 1; j < 4; j++) {
    i += 2;
    if (i == 9)
      i = 1;
    x[j] = vertex[i + 7];
    y[j] = vertex[i + 8];
  }
  x[4] = vertex[i - 1];
  y[4] = vertex[i];
  g->plArray(x, y, 1.0, 1.0, 5);
  i -= 2;
  if (i == -1)
    i = 7;
  x[0] = vertex[i - 1];
  y[0] = vertex[i];
  x[1] = vertex[i + 7];
  y[1] = vertex[i + 8];
  g->plArray(x, y, 1.0, 1.0, 2);
}

void ThreeDItem::axis3d(KplGraph* g, double rmi, double rma, double tic,
                        int mtic, double f, double scal, double xa, double za,
                        double da2, int ndig, int ia, bool log)
{
  int io = (ia == 1) ? 3 : 2;
  int kl = (ia == 1) ? 1 : 2;
  double xr, yr, rl2;
  double rl = 0.0;
  double mant = 0.0;
  double sav = g->setRelSize(1.0);
  double d2 = 0.0125 * p3d.length;
  double xArr[2], yArr[2];
  g->setColData(colGrid);
  if (gridmode == KplGraph::AxesWithLabels) {
    g->cm2r(0.0, 0.0, 1.0, 1.0, &xr, &yr);
    switch (ia) {
      case 0:
        if (fabs(p3d.a21) > 0.8 * fabs(p3d.a11))
          io = (((p3d.a23 * p3d.a22 * zStep < 0.0) ? -1 : 1) *
                p3d.a13 >= 0.0) ? 3 : 1;
        break;
      case 2:
        if (fabs(p3d.a23) > 0.8 * fabs(p3d.a13))
          io = (((p3d.a21 * p3d.a22 * xStep < 0.0) ? -1 : 1) *
                p3d.a11 >= 0.0) ? 3 : 1;
        break;
    }
  }
  if (log) {
    if (tic < 2.5)
      tic = 2.0;
    else
      if (tic < 5.5)
        tic = 3.0;
      else
        if (tic < 99.0)
          tic = 10.0;
  }
  bool e = log && (tic > 9.9) && ((rma > 0.999999e3) || (rmi < 1.000001e-3));
  for (double r = rmi; r < (rma + 0.00001 * (rma - rmi));) {
    g->setRelSize(0.707 * sav);
    if (log) {
      rl = log10(r);
      rl2 = rl;
      mant = KGraph::incLog(&rl2, tic);
    }
    if (mtic > 1) {
      double d;
      double rm = 0.0;
      if (log)
        rm = rl;
      d = log ? pow(10.0, rm - mant) : (tic / (mtic * f));
      for (int i = 1;; i++) {
        if (log) {
          rm = log10(pow(10.0, rm) + d);
          if ((rm >= rl2) || (rm >= log10(rma)))
            break;
        } else {
          rm = r + i * d;
          if (rm >= rma)
            break;
        }
        double r2 = rm * scal;
        switch (ia) {
          case 0:
            line3d(g, r2, 0.0, za, r2, 0.0, za + 0.5 * da2);
            break;
          case 1:
            xArr[0] = xFrom3d(xa, za);
            xArr[1] = xArr[0] + 0.5 * da2;
            yArr[0] = yArr[1] = yFrom3d(xa, f * (r2 - scal * (log ?
                                                 log10(rmi) : rmi)), za);
            g->plArray(xArr, yArr, 1.0, 1.0, 2);
            break;
          case 2:
            line3d(g, xa, 0.0, r2, xa + 0.5 * da2, 0.0, r2);
        }
      }
    }
    double r2 = (log ? rl : r) * scal;
    if (r > rmi)
      switch (ia) {
        case 0:
          if (r < rma)
            line3d(g, r2, 0.0, za, r2, 0.0, za + da2);
          break;
        case 1: {
            double xArr[2], yArr[2];
            xArr[0] = xFrom3d(xa, za);
            xArr[1] = xArr[0] + da2;
            yArr[0] = yArr[1] = yFrom3d(xa, f * (r2 - scal * (log ?
                                                 log10(rmi) : rmi)), za);
            g->plArray(xArr, yArr, 1.0, 1.0, 2);
          }
          break;
        case 2:
          if (r < rma)
            line3d(g, xa, 0.0, r2, xa + da2, 0.0, r2);
      }
    g->setRelSize(sav);
    if (gridmode == KplGraph::AxesWithLabels) {
      QString label;
      double r2 = QMIN(r, rma);
      int l = e ? (label.sprintf("10#nu%d", qRound(log10(r2))).length() - 3) :
                  g->prenum(f * r2, ndig, label);
      if (io != 2)
        kl = QMAX(kl, l);
      r2 = f * scal * (log ? rl : r);
      switch (ia) {
        case 0: {
            double z = fz * (za - 2.5 * da2);
            g->textcm(label, xFrom3d(r2, z) - xr,
                      yFrom3d(r2, 0.0, z) - 0.0125 * p3d.length - yr, io);
          }
          break;
        case 1:
          g->textcm(label, xFrom3d(xa, za) - d2 - xr,
                    yFrom3d(xa, r2 - f * scal * (log ? log10(rmi) : rmi), za) -
                    d2 - yr, io);
          break;
        case 2:
          double x = fx * (xa - 2.5 * da2);
          g->textcm(label, xFrom3d(x, r2) - xr,
                    yFrom3d(x, 0.0, r2) - d2 - yr, io);
      }
    }
    r = log ? pow(10.0, rl2) : (r + tic / f);
  }
  g->setRelSize(sav);
  if (gridmode == KplGraph::AxesWithLabels) {
    double phi = 1.570796326795;
    switch (ia) {
      case 0:
        if (p3d.a11)
          phi = atan(p3d.a21 / p3d.a11) + 6.28318530718;
        break;
      case 2:
        if (p3d.a13)
          phi = atan(p3d.a23 / p3d.a13) + 6.28318530718;
    }
    while (phi > 1.57097)
      phi -= 3.14159265359;
    g->setDir(phi * 57.29577951308);
    double r;
    if (ia == 1)
      r = log ? log10(rma / rmi) : (rma - rmi);
    else
      r = log ? log10(rmi * rma) : (rmi + rma);
    r *= 0.5 * f * scal;
    double d = 2 * (kl + 2) * d2;
    double r2;
    switch (ia) {
      case 0:
        r2 = fz * za;
        if (p3d.a22 >= 0.0)
          d = -d;
        g->textcm(iex ?
                  ("10#nu" + QString::number(iex) + "#nn " + sx) : sx,
                  xFrom3d(r, r2) - d * sin(phi)  - xr,
                  yFrom3d(r, 0.0, r2) + d * cos(phi) - yr, 2);
        break;
      case 1:
        g->textcm(iey ?
                  ("10#nu" + QString::number(iey) + "#nn " + sy) : sy,
                  xFrom3d(xa, za) - 1.5 * (kl + 2) * d2 - xr,
                  yFrom3d(xa, r, za) - yr, 2);
        break;
      case 2:
        r2 = fx * xa;
        if (p3d.a22 <= 0.0)
          d = -d;
        g->textcm(iez ?
                  ("10#nu" + QString::number(iez) + "#nn " + sz) : sz,
                  xFrom3d(r2, r) + d * sin(phi) - xr,
                  yFrom3d(r2, 0.0, r) - d * cos(phi) - yr, 2);
    }
    g->setDir(0.0);
  }
}

void ThreeDItem::rast3d(KplGraph* g)
{
  double sav = g->setRelSize(1.0);
  g->setRelSize(1.414 * sav);
  g->setColData(colFrame);
  frame3d(g, (p3d.a11 * p3d.a22 < 0.0) ? 3 : 4);
  if (gridmode) {
    double xa, za;
    int ix, iz;
    xa = logx ? log10(xmi) : xmi;
    double xe = xa + (nx - 1) * xStep;
    if (p3d.a21 * p3d.a22 * xStep < 0.0) {
      xa = xe * scalx;
      ix = -1;
    } else {
      xa *= scalx;
      ix = 1;
    }
    za = logz ? log10(zmi) : zmi;
    double ze = za + (nz - 1) * zStep;
    if (p3d.a23 * p3d.a22 * zStep < 0.0) {
      za = ze * scalz;
      iz = -1;
    } else {
      za *= scalz;
      iz = 1;
    }
    double xl = scalx * ((p3d.a11 < 0.0) ? xe : (logx ? log10(xmi) : xmi));
    double zl = scalz * ((p3d.a13 < 0.0) ? ze : (logz ? log10(zmi) : zmi));
    line3d(g, xl, 0.0, zl, xl,
           (logy ? log10(yma / ymi) : (yma - ymi)) * scaly, zl);
    double dx1 = 0.04 * nx * xStep * scalx;
    double dx2 = ix * dx1;
    double a = p3d.a11 * p3d.a11 + p3d.a21 * p3d.a21;
    if (a > 0.0)
      dx2 /= sqrt(a);
    double dz2 = 0.04 * iz * nz * zStep * scalz;
    a = p3d.a13 * p3d.a13 + p3d.a23 * p3d.a23;
    if (a > 0.0)
      dz2 /= sqrt(a);
    g->setRelSize(sav);
    axis3d(g, xmi, xma, xtic, mticx, fx, scalx, xa, za, dz2, ndigx, 0, logx);
    axis3d(g, zmi, zma, ztic, mticz, fz, scalz, xa, za, dx2, ndigz, 2, logz);
    axis3d(g, ymi, yma, ytic, mticy, fy, scaly, fx * xl, fz * zl, fx * dx1,
           ndigy, 1, logy);
  }
  g->setRelSize(sav);
}

void ThreeDItem::plar3d(KplGraph* g)
{
  double sav = g->setRelSize(1.0);
  g->setRelSize(0.707 * sav);
  double x = 0.5 * scalx * fx * (logx ? log10(xma * xmi) : (xma + xmi));
  double z = 0.5 * scalz * fz * (logz ? log10(zma * zmi) : (zma + zmi));
  p3d.xOff = xref - p3d.a11 * x - p3d.a13 * z;
  p3d.yOff = yref - p3d.a21 * x - p3d.a23 * z;
  if (mode3D != 1) {
    x = fx * scalx * (logx ? log10(xmi) : xmi);
    QArray<double> y(nx);
    bool dec = (p3d.a11 * p3d.a22 * zStep > 0.0);
    for (int nline = 0; nline < nz; nline++) {
      int lz = dec ? (nz - 1 - nline) : nline;
      z = scalz * fz * ((logz ? log10(zmi) : zmi) + zStep * lz);
      for (int i = 0; i < nx; i++) {
        double ya = QMIN(QMAX(yv[i][lz], ymi), yma);
        y[i] = logy ? log10(ya / ymi) : (fy * (ya - ymi));
      }
      plot3d(g, false, true, false, &x, y, &z, fx * xStep * scalx, scaly,
             0.0, nline == 0, nx, p3d.xOff, p3d.yOff, vertex);
    }
  }
  if (mode3D != 0) {
    z = fz * (logz ? log10(zmi) : zmi) * scalz;
    QArray<double> y(nz);
    bool dec = (p3d.a13 * p3d.a22 * xStep < 0.0);
    for (int nline = 0; nline < nx; nline++) {
      int lx = dec ? (nx - 1 - nline) : nline;
      x = scalx * fx * ((logx ? log10(xmi) : xmi) + xStep * lx);
      for (int i = 0; i < nz; i++) {
        double ya = QMIN(QMAX(yv[lx][i], ymi), yma);
        y[i] = logy ? log10(ya / ymi) : (fy * (ya - ymi));
      }
      plot3d(g, false, true, false, &x, y, &z, 0.0, scaly,
             fz * zStep * scalz, nline == 0, nz, p3d.xOff, p3d.yOff, vertex);
    }
  }
  g->setRelSize(sav);
}

Fun3DItem::Fun3DItem() : fkty(0), fktyo(0), hmody(0)
{
  init();
}

Fun3DItem::Fun3DItem(const Fun3DItem& f) : ThreeDItem(f), fkty(f.fkty),
 fktyo(f.fktyo), namey(f.namey), pathy(f.pathy), hmody(f.hmody)
{
  memcpy(py, f.py, sizeof(py));
  memcpy(pyo, f.pyo, sizeof(pyo));
  if (f.hmody)
    getFunc3DAddr(pathy.path(), namey, &hmody, &fkty);
}

Fun3DItem::Fun3DItem(Kpl::AutoStruct* aut) : ThreeDItem(aut), fkty(0), fktyo(0),
 hmody(0)
{
  init();
}

Fun3DItem::Fun3DItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                     const KURL& uPlo) : ThreeDItem(plo, aut), fktyo(0)
{
  memset(pyo, 0, sizeof(pyo));
  QStringList list = plo->readListEntry("p", ' ');
  int cnt = list.count();
  for (int i = 0; i < KPL_NPMAX; i++)
    py[i] = (i < cnt) ? list[i].toDouble() : 0.0;
  namey = plo->readEntry("name", "");
  QString s = plo->readEntry("path", "");
  pathy = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  getFunc3DAddr(pathy.path(), namey, &hmody, &fkty);
}

Fun3DItem::Fun3DItem(bool act, bool lgx, bool lgy, bool lgz, bool frame,
                     int ndx, int ndy, int ndz, int mtx, int mty, int mtz,
                     int gm, int gm3, int ex, int ey, int ez,
                     const QString& col_f, const QString& col_g,
                     const QString& col_d, double x_0, double wx, double y_0,
                     double hy, double xmin, double xmax, double ymin,
                     double ymax, double zmin, double zmax, double d_x,
                     double d_z, double xt, double yt, double zt,
                     double relSize, double phi_, double theta_, double b_x,
                     double b_y, double b_z, double xr, double yr, double f_x,
                     double f_y, double f_z, const QString& sX,
                     const QString& sY, const QString& sZ, const QString& sH,
                     const QString& name, const KURL& path,
                     Kpl::AutoStruct* aut) :
 ThreeDItem(act, lgx, lgy, lgz,frame, ndx, ndy, ndz, mtx, mty, mtz, gm, gm3,
            ex, ey, ez, col_f, col_g, col_d, x_0, wx, y_0, hy, xmin, xmax,
            ymin, ymax, zmin, zmax, d_x, d_z, xt, yt, zt, relSize, phi_,
            theta_, b_x, b_y, b_z, xr, yr, f_x, f_y, f_z, sX, sY, sZ, sH, aut),
 fkty(0), fktyo(0), namey(name), pathy(path), hmody(0)
{
  init();
  getFunc3DAddr(pathy.path(), namey, &hmody, &fkty);
}

Fun3DItem::~Fun3DItem()
{
  if (hmody)
    dlclose(hmody);
}

Fun3DItem& Fun3DItem::operator=(const Fun3DItem& f)
{
  if (this != &f) {
    *(ThreeDItem*)this = f;
    fkty = f.fkty;
    fktyo = f.fktyo;
    memcpy(py, f.py, sizeof(py));
    memcpy(pyo, f.pyo, sizeof(pyo));
    namey = f.namey;
    pathy = f.pathy;
    if (hmody)
      dlclose(hmody);
    hmody = 0;
    if (f.hmody)
      getFunc3DAddr(pathy.path(), namey, &hmody, &fkty);
  }
  return *this;
}

KplItem::ItemTypes Fun3DItem::iType() const
{
  return Function3D;
}

void Fun3DItem::draw(KplGraph* g)
{
  if (fkty && active) {
    calcTable();
    ThreeDItem::draw(g);
  }
}

void Fun3DItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                         KplDoc* m) const
{
  ThreeDItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("Type", "FUN3DITEM");
  QStrList list;
  for (int i = 0; i < KPL_NPMAX; i++)
    list.insert(i, m->number(py[i]));
  plo->writeEntry("p", list, ' ');
  plo->writeEntry("name", namey);
  plo->writeEntry("path", Utils::relPath(url, pathy, _abs));
}

void Fun3DItem::setText(KplCheckListItem* it, bool*, bool*) const
{
  it->setText(1, i18n("3D function"));
  QFileInfo fi(pathy.path());
  it->setText(2, QString("y = ") + fi.baseName() + "." + namey + "(x,z)");
}

int Fun3DItem::editItem(QWidget* parent, KplDoc* m, int)
{
  Fun3DDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* Fun3DItem::copy() const
{
  return new Fun3DItem(*this);
}

void Fun3DItem::setPar(int i, double value, bool)
{
  py[i] = value;
}

void Fun3DItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  calcTable();
  ThreeDItem::exportTable(ts, m);
}

bool Fun3DItem::getFunc3DAddr(const QString& path, const QString& name,
                              void** hmod,
                              double (**fkt)(double, double, const double*))
{
  *hmod = 0;
  *fkt = 0;
  if (path.isEmpty() || name.isEmpty()) {
    KMessageBox::sorry(0, i18n("no library or no function specified!"));
    return true;
  }
  if (!(*hmod = dlopen(path, RTLD_NOW))) {
    KMessageBox::error(0, i18n("while trying to open") + "\n" + path +
                       "\n" + dlerror());
    return true;
  } else {
    *fkt = (double (*)(double, double, const double *)) dlsym(*hmod, name);
    if (const char* error = dlerror()) {
      KMessageBox::error(0, i18n("while trying to determine the address of") +
                         "\n" + name + "\n" + error);
      dlclose(*hmod);
      *hmod = 0;
      return true;
    } else
      return false;
  }
}

void Fun3DItem::init()
{
  memset(py, 0, sizeof(py));
  memset(pyo, 0, sizeof(pyo));
}

int Fun3DItem::calcTable() const
{
  bool newv = !yv;
  if (dx) {
    nx = int(fabs((xma - xmi) / dx)) + 1;
    xStep = logx ? (log10(xma / xmi) / (nx - 1)) : dx;
  } else {
    nx = 41;
    xStep = 0.025 * (logx ? log10(xma / xmi) : (xma - xmi));
  }
  if (dz) {
    nz = int(fabs((zma - zmi) / dz)) + 1;
    zStep = logz ? (log10(zma / zmi) / (nz - 1)) : dz;
  } else {
    nz = 41;
    zStep = 0.025 * (logz ? log10(zma / zmi) : (zma - zmi));
  }
  int n = nx * nz;
  bool pchanged = false;
  for (int i = 0; i < KPL_NPMAX; i++)
    if ((pchanged = (py[i] != pyo[i])))
      break;
  if (pchanged || newv || (xmi != xmio) || (xma != xmao) ||
      (dx != dxo) || (logx != logxo) || (zmi != zmio) || (zma != zmao) ||
      (dz != dzo) || (logz != logzo) || (fkty != fktyo)) {
    ArrayItem::freeX(&yv);
    double* y1 = new double[n];
    yv = new double*[nx];
    for (int ix = 0; ix < nx; ix++) {
      yv[ix] = &y1[ix * nz];
      for (int iz = 0; iz < nz; iz++) {
        double xa = (logx ? log10(xmi) : xmi) + ix * xStep;
        double za = (logz ? log10(zmi) : zmi) + iz * zStep;
        yv[ix][iz] = ((*fkty)((logx ? pow(10.0, xa) : xa),
                              (logz ? pow(10.0, za) : za), py));
      }
    }
  }
  if (pchanged)
    for (int i = 0; i < KPL_NPMAX; i++)
      pyo[i] = py[i];
  fktyo = fkty;
  xmio = xmi;
  xmao = xma;
  dxo = dx;
  logxo = logx;
  zmio = zmi;
  zmao = zma;
  dzo = dz;
  logzo = logz;
  return n;
}

Array3DItem::Array3DItem() : internal(false), ix(0), iy(0), iz(0), ie(0),
 istart(0), n(0), errbars(0), nrows(0), ncols(0), ixo(0), iyo(0), izo(0),
 ieo(0), istarto(0), no(0), errbarso(0), smf(0.0), smfo(0.0), xv(0)
{
}

Array3DItem::Array3DItem(const Array3DItem& a) : ThreeDItem(a),
 internal(a.internal), ix(a.ix), iy(a.iy), iz(a.iz), ie(a.ie),
 istart(a.istart), n(a.n), errbars(a.errbars), nrows(a.nrows), ncols(a.ncols),
 ixo(a.ixo), iyo(a.iyo), izo(a.izo), ieo(a.ieo), istarto(a.istarto), no(a.no),
 errbarso(a.errbarso), smf(a.smf), smfo(a.smfo), xv(a.xv), url(a.url)
{
  xv = ArrayItem::copyX(a.xv, a.ncols, a.nrows);
}

Array3DItem::Array3DItem(Kpl::AutoStruct* aut) : ThreeDItem(aut),
 internal(false), ix(aut->ixAuto), iy(aut->iyAuto), iz(2), ie(aut->ieAuto),
 istart(0), n(0), errbars(aut->autoErr), nrows(0), ncols(0), ixo(0), iyo(0),
 izo(0), ieo(0), istarto(0), no(0), errbarso(0), smf(0.0), smfo(0.0), xv(0)
{
}

Array3DItem::Array3DItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                         const KURL& uPlo) : ThreeDItem(plo, aut),
 ix(0), iy(0), iz(0), ie(0), istart(0), n(0), errbars(0), ixo(0), iyo(0),
 izo(0), ieo(0), istarto(0), no(0), errbarso(0), smfo(0.0), xv(0)
{
  QString s = plo->readEntry("path", "");
  url = QUrl::isRelativeUrl(s) ? (uPlo.directory(false) + s) : s;
  internal = plo->readNumEntry("internal", false);
  if (internal) {
    n = nrows = plo->readNumEntry("n");
    ncols = 3;
    double* x1 = new double[ncols * nrows];
    xv = new double*[ncols];
    for (int j = 0; j < ncols; j++)
      xv[j] = &x1[j * nrows];
    for (int i = 0; i < nrows; i++) {
      QString s;
      QStringList list = plo->readListEntry(s.sprintf("r%i", i), ' ');
      int cnt = list.count();
      for (int j = 0; j < ncols; j++)
        xv[j][i] = (j < cnt) ? list[j].toDouble() : 0.0;
    }
  } else {
    nrows = ArrayItem::readFile(url, &ncols, &xv);
    if (!nrows) {
      KMessageBox::error(0, url.url());
        return;
    }
    istart = QMIN(plo->readNumEntry("istart"), nrows - 1);
    int nmax =  nrows - istart;
    n = QMIN(plo->readNumEntry("n", nmax), nmax);
  }
  int icolmax = ncols - 1;
  ix = plo->readNumEntry("ix", QMIN(aut->ixAuto, icolmax));
  iy = plo->readNumEntry("iy", QMIN(aut->iyAuto, icolmax));
  iz = plo->readNumEntry("iz", QMIN(1, icolmax));
  ie = plo->readNumEntry("ie", QMIN(aut->ieAuto, icolmax));
  errbars = plo->readNumEntry("errbars", aut->autoErr);
  smf = plo->readDoubleNumEntry("smooth", 1.0);
}

Array3DItem::Array3DItem(bool act, bool lgx, bool lgy, bool lgz, bool frame,
                     int ndx, int ndy, int ndz, int mtx, int mty, int mtz,
                     int gm, int gm3, int ex, int ey, int ez,
                     const QString& col_f, const QString& col_g,
                     const QString& col_d, double x_0, double wx, double y_0,
                     double hy, double xmin, double xmax, double ymin,
                     double ymax, double zmin, double zmax, double d_x,
                     double d_z, double xt, double yt, double zt,
                     double relSize, double phi_, double theta_, double b_x,
                     double b_y, double b_z, double xr, double yr, double f_x,
                     double f_y, double f_z, const QString& sX,
                     const QString& sY, const QString& sZ, const QString& sH,
                     int i_x, int i_y, int i_z, int i_e, int i_start, int np,
                     int errb, double _smf, const KURL& u,
                     Kpl::AutoStruct* aut, bool _int) :
 ThreeDItem(act, lgx, lgy, lgz,frame, ndx, ndy, ndz, mtx, mty, mtz, gm, gm3,
            ex, ey, ez, col_f, col_g, col_d, x_0, wx, y_0, hy, xmin, xmax,
            ymin, ymax, zmin, zmax, d_x, d_z, xt, yt, zt, relSize, phi_,
            theta_, b_x, b_y, b_z, xr, yr, f_x, f_y, f_z, sX, sY, sZ, sH, aut),
 internal(_int), ix(i_x), iy(i_y), iz(i_z), ie(i_e), istart(0), n(0),
 errbars(errb), smf(_smf), xv(0), url(u)
{
  if (internal) {
    n = nrows = np;
    ncols = 3;
    int dim = ncols * nrows;
    double* x1 = new double[dim];
    memset(x1, 0, dim * sizeof(double));
    xv = new double*[ncols];
    for (int j = 0; j < ncols; j++)
      xv[j] = &x1[j * nrows];
  } else {
    nrows = ArrayItem::readFile(url, &ncols, &xv);
    if (!nrows) {
      KMessageBox::error(0, url.url());
        return;
    }
    istart = QMIN(i_start, nrows - 1);
    n = QMIN(np, nrows - istart);
  }
}

Array3DItem::~Array3DItem()
{
  ArrayItem::freeX(&xv);
}

Array3DItem& Array3DItem::operator=(const Array3DItem& a)
{
  if (this != &a) {
    *(ThreeDItem*)this = a;
    internal = a.internal;
    ix = a.ix;
    iy = a.iy;
    iz = a.iz;
    ie = a.ie;
    istart = a.istart;
    n = a.n;
    errbars = a.errbars;
    nrows = a.nrows;
    ncols = a.ncols;
    ixo = a.ixo;
    iyo = a.iyo;
    izo = a.izo;
    ieo = a.ieo;
    istarto = a.istarto;
    no = a.no;
    errbarso = a.errbarso;
    smf = a.smf;
    smfo = a.smfo;
    url = a.url;
    ArrayItem::freeX(&xv);
    xv = ArrayItem::copyX(a.xv, a.ncols, a.nrows);
  }
  return *this;
}

KplItem::ItemTypes Array3DItem::iType() const
{
  return Array3D;
}

void Array3DItem::draw(KplGraph* g)
{
  if (xv && active) {
    calcTable();
    ThreeDItem::draw(g);
  }
}

void Array3DItem::writePlo(KSimpleConfig* plo, const KURL& uPlo, bool _abs,
                           KplDoc* m) const
{
  ThreeDItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("Type", "ARRAY3DITEM");
  plo->writeEntry("internal", internal);
  plo->writeEntry("ix", ix);
  plo->writeEntry("iy", iy);
  plo->writeEntry("iz", iz);
  plo->writeEntry("ie", ie);
  plo->writeEntry("errbars", errbars);
  plo->writeEntry("smooth", smf);
  plo->writeEntry("n", n);
  if (internal) {
    plo->writeEntry("istart", 0);
    for (int i = 0; i < n; i++) {
      QStrList list;
      for (int j = 0; j < ncols; j++)
        list.insert(j, QString::number(xv[j][i + istart]));
      QString s;
      plo->writeEntry(s.sprintf("r%i", i), list, ' ');
    }
  } else {
    plo->writeEntry("istart", istart);
    plo->writeEntry("path", Utils::relPath(uPlo, url, _abs));
  }
}

void Array3DItem::setText(KplCheckListItem* it, bool*, bool*) const
{
  it->setText(1, i18n("3D array"));
  it->setText(2, i18n("file") + " " + url.fileName());
}

int Array3DItem::editItem(QWidget* parent, KplDoc* m, int)
{
  Array3DDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* Array3DItem::copy() const
{
  return new Array3DItem(*this);
}

void Array3DItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  calcTable();
  ThreeDItem::exportTable(ts, m);
}

void Array3DItem::calcTable() const
{
  bool newv = !yv;
  if (dx) {
    nx = int(fabs((xma - xmi) / dx)) + 1;
    xStep = logx ? (log10(xma / xmi) / (nx - 1)) : dx;
  } else {
    nx = 41;
    xStep = 0.025 * (logx ? log10(xma / xmi) : (xma - xmi));
  }
  if (dz) {
    nz = int(fabs((zma - zmi) / dz)) + 1;
    zStep = logz ? (log10(zma / zmi) / (nz - 1)) : dz;
  } else {
    nz = 41;
    zStep = 0.025 * (logz ? log10(zma / zmi) : (zma - zmi));
  }
  if (newv || (xmi != xmio) || (xma != xmao) || (dx != dxo) ||
      (logx != logxo) || (zmi != zmio) || (zma != zmao) ||
      (dz != dzo) || (logz != logzo) || (ix != ixo) || (iy != iyo) ||
      (iz != izo) || (istart != istarto) || (n != no) || (ie != ieo) ||
      (errbars != errbarso) || (smf != smfo)) {
    ArrayItem::freeX(&yv);
    double* y1 = new double[nx * nz];
    yv = new double*[nx];
    for (int i = 0; i < nx; i++)
      yv[i] = &y1[i * nz];
    int nkx, nkz, ier, nk, k, k1, nest, lwrk;
    double xmin, xmax, zmin, zmax, fp;
    Utils::minMaxFile(&xmin, &xmax, &xv[ix][istart], n);
    Utils::minMaxFile(&zmin, &zmax, &xv[iz][istart], n);
    xmin = QMIN(xmi, xmin);
    xmax = QMAX(xma, xmax);
    if (logx) {
      xmin = log10(xmin);
      xmax = log10(xmax);
    }
    zmin = QMIN(zmi, zmin);
    zmax = QMAX(zma, zmax);
    if (logz) {
      zmin = log10(zmin);
      zmax = log10(zmax);
    }
    QArray<double> xi(nx);
    for (int i = 0; i < nx; i++)
      xi[i] = (logx ? log10(xmi) : xmi) + i * xStep;
    QArray<double> zi(nz);
    for (int i = 0; i < nz; i++)
      zi[i] = (logz ? log10(zmi) : zmi) + i * zStep;
    if (smf == 0.0) {
      double* a1 = new double[3 * n];
      double** a2 = new double*[n];
      for (int i = 0; i < n; i++) {
        a2[i] = &a1[3 * i];
        a2[i][0] = logx ? log10(xv[ix][istart + i]) : xv[ix][istart + i];
        a2[i][1] = logz ? log10(xv[iz][istart + i]) : xv[iz][istart + i];
        a2[i][2] = xv[iy][istart + i];
      }
      sortCol = 1;
      qsort(a2, n, sizeof(double*), cmpasc);
      int ib = 0;
      nkz = 0;
      QArray<double> z;
      QArray<double*> c2;
      while (ib < n) {
        int j = ib + 1;
        while ((j < n) && (a2[j][1] == a2[j - 1][1]))
          j++;
        nkx = j - ib;
        double* b1 = new double[2 * nkx];
        double** b2 = new double*[nkx];
        for (int i = 0; i < nkx; i++) {
          b2[i] = &b1[2 * i];
          b2[i][0] = a2[ib + i][0];
          b2[i][1] = a2[ib + i][2];
        }
        sortCol = 0;
        qsort(b2, nkx, sizeof(double*), cmpasc);
        QArray<double> x(nkx);
        QArray<double> y(nkx);
        QArray<double> wt(nkx);
        for (int i = 0; i < nkx; i++) {
          x[i] = b2[i][0];
          y[i] = b2[i][1];
          wt[i] = 1.0;
        }
        k = QMIN(3, nkx - 1);
        k1 = k + 1;
        nest = nkx + k1;
        QArray<double> t(nest);
        QArray<double> c(nest);
        QArray<int> iwrk(nest);
        lwrk = nkx * k1 + nest * (7 + 3 * k);
        QArray<double> wrk(lwrk);
        FitPack::curfit(0, nkx, x, y, wt, xmin, xmax, k, 0.0, nest, &nk,
                        t.data(), c.data(), &fp, wrk.data(), lwrk,
                        iwrk.data(), &ier);
        if (ier < 4) {
          z.resize(nkz + 1);
          z[nkz] = a2[ib][1];
          c2.resize(nkz + 1);
          c2[nkz] = new double[nx];
          FitPack::splev(t, nk, c, k, xi, c2[nkz++], nx, &ier);
        }
        ArrayItem::freeX(&b2);
        ib += nkx;
      }
      QArray<double> y(nkz);
      QArray<double> wt(nkz);
      for (int i = 0; i < nkz; i++)
        wt[i] = 1.0;
      k = QMIN(3, nkz - 1);
      k1 = k + 1;
      nest = nkz + k1;
      QArray<double> t(nest);
      QArray<double> c(nest);
      QArray<int> iwrk(nest);
      lwrk = nkz * k1 + nest * (7 + 3 * k);
      QArray<double> wrk(lwrk);
      for (int j = 0; j < nx; j++) {
        for (int i = 0; i < nkz; i++)
          y[i] = c2[i][j];
        FitPack::curfit(0, nkz, z, y, wt, zmin, zmax, k, 0.0, nest, &nk,
                        t.data(), c.data(), &fp, wrk.data(), lwrk,
                        iwrk.data(), &ier);
        if (ier < 4)
          FitPack::splev(t, nk, c, k, zi, yv[j], nz, &ier);
      }
      ArrayItem::freeX(&a2);
    } else {
      int iopt = 0;
      k = 3;
      k1 = k + 1;
      nest = QMIN(n, 121) + k1;
      nk = nest - k1;
      int nm = nest - 2 * k1 + 1;
      int nrint = 2 * nm;
      int ib1 = k * nk + k1;
      int ib3 = k1 * nk + 1;
      lwrk = nk * nk * (ib1 + 2 + ib3) +
             2 * (nrint + nest * (k1 + 1) + n * k1) + ib3 + 1;
      QArray<double> x(n);
      QArray<double> z(n);
      QArray<double> wt(n);
      bool errcol = errbars && (ie < ncols);
      for (int i = 0; i < n; i++) {
        x[i] = xv[ix][istart + i];
        if (logx)
           x[i] = log10(x[i]);
        z[i] = xv[iz][istart + i];
        if (logz)
           z[i] = log10(z[i]);
        wt[i] = errcol ? (1.0 / xv[ie][istart + i]) : 1.0;
      }
      QArray<double> tx(nest);
      QArray<double> tz(nest);
      QArray<double> c(nk * nk);
      QArray<double> wrk(lwrk);
      FitPack::surfit(iopt, n, x.data(), z.data(), &xv[iy][istart], wt,
                      xmin, xmax, zmin, zmax, k, k, smf * smf * n, nest, nest,
                      nest, DBL_EPSILON, &nkx, tx.data(), &nkz, tz.data(),
                      c.data(), &fp, wrk.data(), lwrk, &ier);
debug("surfit: ier = " + QString::number(ier) + ", fp = " + QString::number(fp));
      FitPack::bispev(tx, nkx, tz, nkz, c, k, k, xi, nx, zi, nz, yv[0], &ier);
debug("bispev: ier = " + QString::number(ier));
    }
  }
  xmio = xmi;
  xmao = xma;
  dxo = dx;
  logxo = logx;
  zmio = zmi;
  zmao = zma;
  dzo = dz;
  logzo = logz;
  ixo = ix;
  iyo = iy;
  izo = iz;
  ieo = ie;
  istarto = istart;
  no = n;
  errbarso = errbars;
  smfo = smf;
}
