/***************************************************************************
                          splfitdlg.cpp  -  description
                             -------------------
    begin                : Fri Mar 1 2002
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <math.h>
#include <dlfcn.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <klocale.h>
#include <kapp.h>
#include <kmessagebox.h>
#include "kpldoc.h"
#include "kpldoubleedit.h"
#include "kplspinbox.h"
#include "fitdlg.h"
#include "arrayitem.h"
#include "splineitem.h"
#include "funitem.h"
#include "frameitem.h"
#include "fitpack.h"
#include "residualsdlg.h"
#include "kplgraph.h"
#include "utils.h"
#include "splfitdlg.h"

SplFitDlg::SplFitDlg(QWidget* _parent, KplDoc* model, ArrayItem* ad0,
                     SplineItem* fd0, int mode) :
 KDialogBase(Plain, i18n("Smoothing spline fit"),
             Help | Ok | Apply | Cancel | User1 | User2,
             Ok, _parent, 0, true, true, i18n("&Start"), i18n("&Residuals")),
             running(false), dlgMode(mode), m(model), ad(ad0), fd(fd0)
{
  spl = new SplineItem(*fd);
  err = (*m->options()).err;
  int nc = 3;
  if (ad->ie >= ad->ncols) {
    err.fitErrCol = false;
    nc = 2;
  }
  double* tmp = new double[nc * ad->n];
  int j = 0;
  for (int i = 0; i < ad->n; i++) {
    int i2 = ad->istart + i;
    tmp[j++] = ad->x[ad->ix][i2];
    tmp[j++] = ad->x[ad->iy][i2];
    if (nc == 3)
      tmp[j++] = ad->x[ad->ie][i2];
  }
  qsort(tmp, ad->n, nc * sizeof(double), cmpasc);
  xv.resize(ad->n);
  yv.resize(ad->n);
  if (nc == 3)
    ev.resize(ad->n);
  sigv.resize(ad->n);
  j = 0;
  for (int i = 0; i < ad->n; i++) {
    xv[i] = tmp[j++];
    yv[i] = tmp[j++];
    if (nc == 3)
      ev[i] = tmp[j++];
  }
  delete [] tmp;
  if (dlgMode & FitDlg::ShowDlg) {
    Utils::setSize(this, "SplineFitDialog");
    QFrame* frame = plainPage();
    QVBoxLayout* vbox = new QVBoxLayout(frame, 11, spacingHint());
    QHBoxLayout* hbox = new QHBoxLayout(vbox);
    QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Spline"), frame);
    hbox->addWidget(g);
    QGridLayout* grid = new QGridLayout(g->layout(), 5, 2, spacingHint());
    grid->addWidget(new QLabel(i18n("Degree"), g), 0, 0);
    grid->addWidget(sDeg = new KplSpinBox(1, 5, 1, g), 0, 1);
    sDeg->setValue(spl->kdeg);
    grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 0, 2);
    grid->addWidget(new QLabel(i18n("Smoothing factor"), g), 0, 3);
    grid->addWidget(eFactor = new KplDoubleEdit(1.0, g), 0, 4);
    grid->addWidget(new QLabel("xmin", g), 1, 0);
    double xlim = xv[0];
    grid->addWidget(exMin = new KplDoubleEdit(QMIN(spl->xmin, xlim), -1.0e300,
                                              xlim, g, "%.6lg"), 1, 1);
    grid->addWidget(new QLabel("xmax", g), 1, 3);
    xlim = xv[ad->n - 1];
    grid->addWidget(exMax = new KplDoubleEdit(QMAX(spl->xmax, xlim), xlim,
                                              1.0e300, g, "%.6lg"), 1, 4);
    g = new QGroupBox(0, Qt::Horizontal, i18n("Data errors"), frame);
    QHBoxLayout* hbox2 = new QHBoxLayout(g->layout(), spacingHint());
    hbox2->addWidget(new QLabel(i18n("Array"), g));
    QLabel* l = new QLabel(ad->url.fileName() + " " + QString::number(ad->ix) +
                           " " + QString::number(ad->iy) + " " +
                           QString::number(ad->ie), g);
    hbox2->addWidget(l);
    l->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    hbox2->addWidget(errCol = new QCheckBox(i18n("Error column"), g));
    errCol->setChecked(err.fitErrCol);
    errCol->setEnabled(nc == 3);
    hbox2->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding));
    hbox2->addWidget(errMod = new QPushButton(i18n("Model"), g));
    errMod->setEnabled(!errCol->isChecked());
    hbox->addWidget(g);
    vbox->addWidget(results = new QListBox(frame));
    results->setMinimumHeight(50);
    enableButton(User2, false);
#if KDE_VERSION == 303
    connect(this, SIGNAL(helpClicked()), SLOT(slotHelp2()));
#else
    setHelp("SEC-SPLINE-FIT");
#endif
    connect(errMod, SIGNAL(clicked()), SLOT(slotErrMod()));
    connect(errCol, SIGNAL(toggled(bool)), SLOT(errColToggled(bool)));
  }
}

SplFitDlg::~SplFitDlg()
{
  if (dlgMode & FitDlg::ShowDlg)
    Utils::saveSize(this, "SplineFitDialog");
  delete spl;
}

int SplFitDlg::cmpasc(const void *e1, const void *e2)
{
  double a1 = *((double*) e1);
  double a2 = *((double*) e2);
  if (a1 < a2)
    return -1;
  if (a1 > a2)
    return 1;
  return 0;
}

void SplFitDlg::getValues(bool ok)
{
  *fd = *spl;
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
}

void SplFitDlg::errColToggled(bool state)
{
  errMod->setEnabled(!state);
}

void SplFitDlg::slotOk()
{
  getValues(true);
}

void SplFitDlg::slotApply()
{
  getValues(false);
}

void SplFitDlg::slotHelp2()
{
#if KDE_VERSION == 303
  kapp->startServiceByDesktopName("konqueror",
                                  QString("help:/kpl/spline-fit.html"),
                                  0, 0, 0, "", true);
#endif
}

void SplFitDlg::slotUser1()
{
  enableButton(User2, false);
  fit(sDeg->interpretedValue(), eFactor->value(), exMin->value(),
  exMax->value(), errCol->isChecked(), err.errModPath, err.errModName);
}

int SplFitDlg::fit(int k, double f, double xmin, double xmax, bool errcol,
                   const QString& errModPath, const QString& errModName)
{
  int ier = 0;
  double (*fktErrMod)(double, const double*);
  void* hErrMod;
  if (!errcol) {
    if (FunItem::getFuncAddr(errModPath, errModName, &hErrMod,
                             &fktErrMod))
    return 5;
  }
  QArray<double> w(ad->n);
  int nest = ad->n + k + 1;
  QArray<double> t(nest);
  QArray<double> c(nest);
  int nk;
  for (int i = 0; i < ad->n; i++) {
    sigv[i] = errcol ? ev[i] : (fktErrMod(yv[i], err.pErrMod));
    if (sigv[i] <= 0.0) {
      ier = 4;
      break;
    }
    w[i] = 1.0 / sigv[i];
  }
  if (!errcol)
    dlclose(hErrMod);
  if (!ier) {
    int lwrk = ad->n * (k + 1) + nest * (7 + 3 * k);
    QArray<double> wrk(lwrk);
    QArray<int> iwrk(nest);
    FitPack::curfit(0, ad->n, xv, yv, w, xmin, xmax, k, f * f * ad->n, nest,
                    &nk, t.data(), c.data(), &m->chisq, wrk.data(), lwrk,
                    iwrk.data(), &ier);
  }
  QString s;
  if (dlgMode & FitDlg::ShowDlg) {
    results->clear();
    if (ier > 0) {
      switch (ier) {
        case 1:
          s = i18n("curfit: The required storage space exceeds the available "
                   "storage space");
          break;
        case 2:
          s = i18n("curfit: A theoretically impossible result was found during "
                   "the iteration process");
          break;
        case 3:
          s = i18n("curfit: The maximal number of iterations maxit has been "
                   "reached");
          break;
        case 4:
          s = i18n("Data error <= 0. No output generated");
          break;
        case 10:
          s = i18n("Input error. No output generated");
      }
      results->insertItem(s);
    }
  }
  if (ier < 4) {
    spl->nk = nk;
    spl->kdeg = k;
    spl->nderiv = QMIN(spl->nderiv, k);
    spl->xmin = QMAX(fd->xmin, xmin);
    spl->xmax = QMIN(fd->xmax, xmax);
    spl->t.duplicate(t, nk);
    spl->c.duplicate(c, nk - k - 1);
    spl->xv.detach();
    spl->xv.resize(0);
    fv.resize(ad->n);
    FitPack::splev(t, nk, c, k, xv, fv.data(), ad->n, &ier);
    double avgErr = 0.0;
    for (int i = 0; i < ad->n; i++) {
      double e = yv[i] - fv[i];
      avgErr += e * e;
      fv[i] = e;
    }
    avgErr = sqrt(avgErr / ad->n);
    if (dlgMode & FitDlg::ShowDlg) {
      results->insertItem(s.sprintf(i18n("Number of knots = %i    "
                                         "chi-square = %.4g    "
                                         "Average error = %.3g"),
                                          nk, m->chisq, avgErr));
      enableButton(User2, true);
    }
  }
  return ier;
}

void SplFitDlg::slotErrMod()
{
  ErrModDlg dlg(this, m, &err);
  dlg.exec();
}

void SplFitDlg::slotUser2()
{
  QList<KplItem> items;
  items.setAutoDelete(true);
  FrameItem* fd = new FrameItem();
  fd->gridmode = KplGraph::GridWithLabels;
  fd->sx = "x";
  fd->sy = i18n("Residuals");
  items.append(fd);
  ArrayItem* ar = new ArrayItem(true, ad->symb, "0", 1.0, 1.0, 0, 1, 2,
                                0, xv.size(), 1, "", true);
  ar->color = ad->color;
  for (unsigned i = 0; i < xv.size(); i++) {
    ar->x[0][i] = xv[i];
    ar->x[1][i] = fv[i];
    ar->x[2][i] = sigv[i];
  }
  items.append(ar);
  double fx, fy, xmin, xmax, ymin, ymax;
  ar->expoItem(&fd->iex, &fd->iey, &fx, &fy);
  ar->minMax(&xmin, &xmax, &ymin, &ymax);
  Utils::autoSc(&fd->xmi, &fd->xma, &fd->xtic, &fd->mticx, &fd->ndigx,
                xmin, xmax, fx, 0.2, false);
  Utils::autoSc(&fd->ymi, &fd->yma, &fd->ytic, &fd->mticy, &fd->ndigy,
                ymin, ymax, fy, 0.3, false);
  ar->normalize(fx, fy);
  ResidualsDlg dlg(this, m, &items);
  dlg.exec();
}
