/***************************************************************************
                    herschelSPIRE.cpp  - Herschel SPIRE Format file data source
                             -------------------
    begin                : May 12 2010
    copyright            : (C) 2010 The University of British Columbia
    email                :
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "herschelSPIRE.h"
#include "herschelSPIREconfig.h"

#include <ctype.h>
#include <math.h>
#include <stdlib.h>

#include <qcheckbox.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qspinbox.h>
#include <qstylesheet.h>

#include <kdebug.h>

#include "kststring.h"

#define TIME_FIELD QString("signal/sampleTime")

class HERSCHELSPIRESource::Config {
  public:
    Config() {
      _checkFilename = true;
    }

    void read(KConfig *cfg, const QString& fileName = QString::null) {
      Q_UNUSED( fileName );

      cfg->setGroup("HerschelSPIRE General");
      _checkFilename = cfg->readBoolEntry("Check Filename", true);
    }

    bool _checkFilename;

    void save(QTextStream& str, const QString& indent) {
      if (_checkFilename) {
        str << indent << "<checkfilename/>" << endl;
      }
    }

    void load(const QDomElement& e) {
      _checkFilename = false;

      QDomNode n = e.firstChild();
      while (!n.isNull()) {
        QDomElement e = n.toElement();
        if (!e.isNull()) {
          if (e.tagName() == "checkfilename") {
            _checkFilename = true;
          }
        }
        n = n.nextSibling();
      }
   }
};


HERSCHELSPIRESource::HERSCHELSPIRESource( KConfig *cfg, const QString& filename, const QString& type, const QDomElement& e)
: KstDataSource( cfg, filename, type ), _config(0L)
{
  _fields.setAutoDelete( true );
  _files.setAutoDelete( true );

  if( type.isEmpty( ) || type == "HERSCHELSPIRE" )
  {
    if( initialize( ) )
    {
      _config = new HERSCHELSPIRESource::Config;
      _config->read(cfg, filename);
      if (!e.isNull()) {
        _config->load(e);
      }

      _valid = true;
    }
  }
}


HERSCHELSPIRESource::~HERSCHELSPIRESource( )
{
  delete _config;
  _config = 0L;
}


bool HERSCHELSPIRESource::isHistoryTable( fitsfile* ffits )
{
  char  value[FLEN_VALUE];
  char  comment[FLEN_COMMENT];
  bool  isHistory = false;
  int   iStatus = 0;

  if( fits_read_key( ffits, TSTRING, "EXTNAME", value, comment, &iStatus ) == 0 )
  {
    QString section = value;

    section.stripWhiteSpace();
    if( section.find( "History" ) == 0 )
    {
      isHistory = true;
    }
  }

  return isHistory;
}


bool HERSCHELSPIRESource::isValidFilename( const QString& filename, Config *config )
{
  bool ok = false;

  if( !config || config->_checkFilename )
  {
    if( filename.right( 8 ).lower( ) == ".fits.gz" ) 
    {
      ok  = true;
    }
    else if( filename.right( 5 ).lower( ) == ".fits" ) 
    {
      ok = true;
    }
  }

  return ok;
}


bool HERSCHELSPIRESource::reset( )
{
  return true;
}


QString HERSCHELSPIRESource::units( const QString& fieldName )
{
  QString strUnits;

  if( !fieldName.isEmpty() )
  {
    field *fld = 0L;

    fld = _fields.find( fieldName );
    if( fld != 0L )
    {
      strUnits = fld->units;
    }
  }

  return strUnits;
}

void HERSCHELSPIRESource::addToMetadata( fitsfile *ffits, int &iStatus )
{
  int iResult;
  int keysexist;
  int morekeys;

  iResult = fits_get_hdrspace( ffits, &keysexist, &morekeys, &iStatus );
  if( iResult == 0 )
  {
    QString strKey;
    char keyname[FLEN_KEYWORD];
    char value[FLEN_VALUE];
    char comment[FLEN_COMMENT];
    int keynum;
    int hdu;

    fits_get_hdu_num( ffits, &hdu );

    for( keynum=1; keynum<=keysexist; ++keynum )
    {
      iResult = fits_read_keyn( ffits, keynum, keyname, value, comment, &iStatus );
      if( iResult == 0 )
      {
        strKey.sprintf("%02d_%03d %s", hdu, keynum, keyname);

        KstString *metaString;
        KstObjectTag newTag( strKey, tag( ) );
        QString str;

        if( strlen( comment ) > 0 )
        {
          if( strlen( value ) > 0 )
          {
            str.sprintf( "%s / %s", value, comment );
          }
          else
          {
            str.sprintf( "%s", comment );
          }
        }
        else if( strlen( value ) > 0 )
        {
          str.sprintf( "%s", value );
        }

        metaString = new KstString( newTag, this, str );
        _metaData.insert( keyname, metaString );
      }
    }
  }
}


void HERSCHELSPIRESource::addToFieldList( fitsfile *ffits, const int iNumCols, int &iStatus )
{
  QString str;
  char charTemplate[ FLEN_CARD ];
  char charName[ FLEN_CARD ];
  char headerName[ FLEN_CARD ];
  char units[ FLEN_CARD ];
  char ttype[ FLEN_CARD ];
  char dtype[ FLEN_CARD ];
  char disp[ FLEN_CARD ];
  double dScale;
  double dZero;
  long lNull;
  long lRepeat;
  long lWidth;
  int iHDUNumber;
  int iTypeCode;
  int iColNumber;
  int iResult;
  int table;
  int col;

  table = fits_get_hdu_num( ffits, &iHDUNumber );

  iResult = fits_read_key_str( ffits, "EXTNAME", headerName, NULL, &iStatus );
  iStatus = 0;

  for( col=0; col<iNumCols; col++ )
  {
    iResult = fits_get_coltype( ffits, col+1, &iTypeCode, &lRepeat, &lWidth, &iStatus );
    if( iResult == 0 )
    {
      if( iTypeCode != TSTRING )
      {
        sprintf( charTemplate, "%d", col+1 );

        if( fits_get_colname( ffits, CASEINSEN, charTemplate, charName, &iColNumber, &iStatus ) == 0 )
        {
          if( lRepeat == 1 )
          {
            QString strName = charName;
            QString strPrefix;
            QString strSuffix;

            if( strName.startsWith( "PSW", FALSE ) ||
                strName.startsWith( "PMW", FALSE ) ||
                strName.startsWith( "PLW", FALSE ) )
            {
              strName = strName.left( 3 ) + QString("/") + strName.right( strName.length( ) - 3 );
            }

            str = QString( "%1/%2" ).arg( headerName ).arg( strName );

            field *fld = new field;

            iResult = fits_get_bcolparms( ffits, col+1, ttype, units, dtype, &lRepeat, &dScale, &dZero, &lNull, disp, &iStatus );
            if( iResult == 0 ) {
              fld->units = units;
            } else {
              fld->units = QString::null;
            }
            fld->tableName = headerName;
            fld->columnName = charName;

            _fields.insert( str, fld );
            _fieldList.append( str );
          }
        }
      }
    }
  }
}


long HERSCHELSPIRESource::getNumFrames( fitsfile* ffits, int iNumHeaderDataUnits )
{
  long lNumRows = 0;

  if( iNumHeaderDataUnits > 1 )
  {
    int iHDUType;
    int iResult = 0;
    int iStatus = 0;

    if( fits_movabs_hdu( ffits, 2, &iHDUType, &iStatus ) == 0 )
    {
      if( fits_get_hdu_type( ffits, &iHDUType, &iStatus ) == 0 )
      {
        if( iHDUType == BINARY_TBL )
        {
          iResult = fits_get_num_rows( ffits, &lNumRows, &iStatus );
        }
      }
    }
  }

  return lNumRows;
}


bool HERSCHELSPIRESource::initFile( const QString& filename, bool bAddMetadata )
{
  QString   prefixNew;
  QString   str;
  fitsfile* ffits;
  field*    fld;
  bool      bRetVal = false;
  int       iResult = 0;
  int       iStatus = 0;

  fld = new field;
  _fields.insert( "INDEX", fld );
  _fieldList.prepend( "INDEX" );

  iResult = fits_open_file( &ffits, filename.ascii( ), READONLY, &iStatus );
  if( iResult == 0 )
  {
    int iNumHeaderDataUnits;
    if( fits_get_num_hdus( ffits, &iNumHeaderDataUnits, &iStatus ) == 0 )
    {
      long lNumRows;
      int iHDUType;
      int i;

      _numFrames = getNumFrames( ffits, iNumHeaderDataUnits );
      if( _numFrames > 0 )
      {
        fits_movabs_hdu( ffits, 1, &iHDUType, &iStatus );

        //
        // add the fields and metadata...
        //
        for( i=0; i<iNumHeaderDataUnits; i++ )
        {
          if( iStatus == 0 )
          {
            //
            // create the field entries...
            //
            fits_get_hdu_type( ffits, &iHDUType, &iStatus );
            if( iStatus == 0 )
            {
              if( bAddMetadata && i == 0 )
              {
                addToMetadata( ffits, iStatus );
                iStatus = 0;
              }

              if( iHDUType == BINARY_TBL )
              {
                if( !isHistoryTable( ffits ) )
                {
                  int iNumCols;

                  iResult = fits_get_num_cols( ffits, &iNumCols, &iStatus );
                  if( iResult == 0 )
                  {
                    iResult = fits_get_num_rows( ffits, &lNumRows, &iStatus );
                    if( iResult == 0 )
                    {
                      addToFieldList( ffits, iNumCols, iStatus );
                    }
                  }
                }
              }
            }

            fits_movrel_hdu( ffits, 1, &iHDUType, &iStatus);
          }
        }

        bRetVal = true;
      }
    }

    iStatus = 0;

    fits_close_file( ffits, &iStatus );
  }

  return bRetVal;
}


void HERSCHELSPIRESource::addTableFile( const QString& filename, fitsfile* ffits, const QString& date ) 
{
  char headerName[ FLEN_CARD ];
  int iStatus = 0;
  long int rows;
  fileList* folderFields;
  folderField fields;

  fits_read_key_str( ffits, "EXTNAME", headerName, NULL, &iStatus );
  fits_get_num_rows( ffits, &rows, &iStatus );

  if( iStatus == 0 && rows > 0 )
  {
    folderFields = _files.find( headerName );
    if( folderFields == 0L )
    {
      folderFields = new fileList;

      fields.frameLo = -1;
      fields.frames  = rows;
      fields.file    = filename;
      fields.date    = date;

      folderFields->append( fields );

      _files.insert( headerName, folderFields );
    }
    else
    {
      fields.frameLo  = -1;
      fields.frames   = rows;
      fields.file     = filename;
      fields.date     = date;

      folderFields->append( fields );
    }
  }
}

bool HERSCHELSPIRESource::initFolderFile( const QString& filename, bool bAddMetadata )
{
  QString   prefixNew;
  QString   str;
  fitsfile* ffits;
  bool      bRetVal = false;
  int       iResult = 0;
  int       iStatus = 0;

  iResult = fits_open_file( &ffits, filename.ascii( ), READONLY, &iStatus );
  if( iResult == 0 )
  {
    int iNumHeaderDataUnits;
    if( fits_get_num_hdus( ffits, &iNumHeaderDataUnits, &iStatus ) == 0 )
    {
      QString strDate;
      int iHDUType;
      int i;

      _numFrames = getNumFrames( ffits, iNumHeaderDataUnits );
      if( _numFrames > 0 )
      {
        fits_movabs_hdu( ffits, 1, &iHDUType, &iStatus );

        for( i=0; i<iNumHeaderDataUnits; i++ )
        {
          if( iStatus == 0 )
          {
            if( i == 0 )
            {
              //
              // get the date of each file, as we will need this later for sorting...
              //

              char headerDate[ FLEN_CARD ];

              fits_read_key_str( ffits, "DATE-OBS", headerDate, NULL, &iStatus );
              if( iStatus != 0 ) 
              {
                iStatus = 0;

                fits_read_key_str( ffits, "DATE_OBS", headerDate, NULL, &iStatus );
              }

              if( iStatus == 0 )
              {
                strDate = headerDate;
              }
            }

            fits_get_hdu_type( ffits, &iHDUType, &iStatus );
            if( iStatus == 0 )
            {
              if( bAddMetadata && i == 0 )
              {
                addToMetadata( ffits, iStatus );
                iStatus = 0;
              }

              if( iHDUType == BINARY_TBL )
              {
                if( !isHistoryTable( ffits ) )
                {
                  addTableFile( filename, ffits, strDate );

                  bRetVal = true;
                }
              }
            }

            fits_movrel_hdu( ffits, 1, &iHDUType, &iStatus);
          }
        }
      }
    }

    iStatus = 0;

    fits_close_file( ffits, &iStatus );
  }

  return bRetVal;
}


bool HERSCHELSPIRESource::initFolder( )
{
  QDir            folder( _filename, "*.fits *.fits.gz", QDir::Name | QDir::IgnoreCase, QDir::Files | QDir::Readable );
  QStringList     files;
  QString         pathname;
  bool            bRetVal = true;

  files = folder.entryList( QDir::Files | QDir::Readable, QDir::Time );
  if( files.size() > 0 )
  {
    bool bFirst = true;

    for( QStringList::ConstIterator it = files.begin(); it != files.end(); ++it )
    {
      if( isValidFilename( *it, 0L ) )
      {
        pathname = folder.path() + separator() + *it;

        if( bFirst )
        {
          initFile( pathname, true );

          bFirst = false;
        }

        initFolderFile( pathname, false );
      }
    }

    //
    // now we sort for each of the tables,
    //  using the date (DATE-OBS or DATE_OBS)
    //  from the primary header of each file.
    //  Then we determine the value of
    //  of frameLo for each file's table...
    //

    QDictIterator<fileList> itFiles( _files );
    fileList::Iterator it;

    for( ; itFiles.current( ); ++itFiles )
    {
      qHeapSort( *(itFiles.current( )) );

      long frameLo = 0;

      for( it = itFiles.current()->begin(); it != itFiles.current()->end(); ++it )
      {
        (*it).frameLo = frameLo;

        frameLo += (*it).frames;
      }
    }

    _numFrames = frameCount(TIME_FIELD);
  }

  return bRetVal;
}


bool HERSCHELSPIRESource::initFile( )
{
  bool bRetVal = false;

  if( initFile( _filename, true ) )
  {
    updateNumFramesScalar( );

    bRetVal = true;
  }

  return bRetVal;
}


bool HERSCHELSPIRESource::initialize( )
{
  bool bRetVal = true;

  _numFrames = 0;

  if( !_filename.isNull( ) && !_filename.isEmpty( ) )
  {
    QFileInfo fileInfo( _filename );

    if( fileInfo.isFile( ) )
    {
      bRetVal = initFile( );
      if( bRetVal )
      {
        _isSingleFile = true;
      }
    }
    else if( fileInfo.isDir( ) )
    {
      bRetVal = initFolder( );
      if( bRetVal )
      {
        _isSingleFile = false;
      }
    }
  }

  return bRetVal;
}


KstObject::UpdateType HERSCHELSPIRESource::update( int u )
{
  if (KstObject::checkUpdateCounter(u)) {
    return lastUpdateResult();
  }

  KstObject::UpdateType updateType =  KstObject::NO_CHANGE;

  return setLastUpdateResult(updateType);
}


int HERSCHELSPIRESource::readFileFrames( const QString& filename, field *fld, double *v, int s, int n )
{
  double    dNan = strtod( "nan", NULL );
  fitsfile* ffits;
  int       iRead = -1;
  int       iStatus = 0;
  int       iAnyNull;
  int       iResult = 0;
  int       iColumn = 0;

  iResult = fits_open_file( &ffits, filename.ascii( ), READONLY, &iStatus );
  if( iResult == 0 )
  {
    if( fits_movnam_hdu( ffits, BINARY_TBL, (char*)fld->tableName.ascii(), 0, &iStatus ) == 0 )
    {
      _valid = true;

      iResult = fits_get_colnum( ffits, CASEINSEN, (char*)fld->columnName.latin1(), &iColumn, &iStatus );
      if( iResult == 0 )
      {
        iResult = fits_read_col( ffits, TDOUBLE, iColumn, s+1, 1, n, &dNan, v, &iAnyNull, &iStatus );
        if( iResult == 0 )
        {
          iRead = n;
        }
      }

      iStatus = 0;
    }

    fits_close_file( ffits, &iStatus );
  }

  return iRead;
}


int HERSCHELSPIRESource::readFolderFrames( field *fld, double *v, int s, int n )
{
  int iRead = 0;

  if( !fld->tableName.isEmpty( ) )
  {
    fileList*   folderFields;
    double*     vNew = v;
    int         sNew = s;
    int         nNew = n;
    int         iReadSub = 0;

    folderFields = _files.find( fld->tableName );
    if( folderFields != 0L )
    {
      for( fileList::ConstIterator it = folderFields->begin(); it != folderFields->end(); ++it )
      {
        if( (*it).frameLo < s + n && (*it).frameLo + (*it).frames > s )
        {
          sNew = s - (*it).frameLo;
          if( sNew < 0 )
          {
            sNew = 0;
          }

          if( (*it).frameLo - s > 0 )
          {
            vNew = v + ( (*it).frameLo - s );
          }
          else
          {
            vNew = v;
          }

          nNew = ( v - vNew ) + n;
          if( sNew + nNew > (*it).frames )
          {
            nNew = (*it).frames - sNew;
          }

          if( nNew > 0 )
          {
            iReadSub = readFileFrames( (*it).file, fld, vNew, sNew, nNew );
            if( iReadSub > 0 )
            {
              iRead += iReadSub;
            }

            if( iRead == n )
            {
              break;
            }
          }
        }
      }
    }
  }

  return iRead;
}


int HERSCHELSPIRESource::readField( double *v, const QString& fieldName, int s, int n )
{
  int       i;
  int       iRead = -1;

  if( n < 0 ) 
  {
    //
    // n < 0 means read one sample, not frame - irrelevent here
    //

    n = 1;
  }

  if( fieldName == "INDEX" )
  {
    for( i = 0; i < n; ++i )
    {
      v[i] = (double)( s + i );
    }

    iRead =  n;
  }
  else
  {
    field *fld = 0L;

    fld = _fields.find( fieldName );
    if( fld != 0L ) 
    {
      _valid = false;

      if( !_filename.isNull( ) && !_filename.isEmpty( ) )
      {
        if( _isSingleFile )
        {
          iRead = readFileFrames( _filename, fld, v, s, n );
        }
        else
        {
          iRead = readFolderFrames( fld, v, s, n );
        }
      }
    }
  }

  return iRead;
}


bool HERSCHELSPIRESource::isValidField( const QString& field ) const
{
  bool bRetVal = false;

  if( field == "INDEX" )
  {
    bRetVal = true;
  }
  else
  {
    if( _fields.find( field ) != 0L )
    {
      bRetVal = true;
    }
  }

  return bRetVal;
}


int HERSCHELSPIRESource::samplesPerFrame( const QString& fieldName )
{
  Q_UNUSED( fieldName )

  int rc = 1;

  return rc;
}


int HERSCHELSPIRESource::frameCount( const QString& fieldName ) const
{
  int rc = 0;

  if( _isSingleFile )
  {
    rc = _numFrames;
  }
  else
  {
    field* fld;

    if( !fieldName.isEmpty() )
    {
      fileList* folderFields = 0L;

      if( fieldName == "INDEX" )
      {
        if( _files.count() > 0 )
        {
          QDictIterator<fileList> it( _files );

          folderFields = it.current();
        }
      }
      else
      {
        fld = _fields.find( fieldName );
        if( fld != 0L )
        {
          folderFields = _files.find( fld->tableName );
        }
      }

      if( folderFields != 0L )
      {
        for( fileList::ConstIterator it = folderFields->begin(); it != folderFields->end(); ++it )
        {
          rc += (*it).frames;
        }
      }
    }
  }

  return rc;
}


bool HERSCHELSPIRESource::isEmpty( ) const 
{
  return _fields.isEmpty();
}


QString HERSCHELSPIRESource::fileType( ) const
{
  return "HERSCHELSPIRE";
}


void HERSCHELSPIRESource::save( QTextStream& ts, const QString& indent )
{
  KstDataSource::save( ts, indent );
  _config->save(ts, indent);
}


bool HERSCHELSPIRESource::supportsHierarchy( ) const
{
  return true;
}


bool HERSCHELSPIRESource::checkValidHerschelSPIREFolder( const QString& filename )
{
  QDir folder( filename, "*.fits *.fits.gz", QDir::Name | QDir::IgnoreCase, QDir::Files | QDir::Readable );
  QStringList files;
  QString pathname;
  bool ok = false;

  files = folder.entryList( );
  if( files.size() > 0 )
  {
    for (QStringList::ConstIterator it = files.begin(); it != files.end(); ++it) 
    {
      pathname = folder.path() + separator() + *it;

      if( checkValidHerschelSPIREFile( pathname, 0L ) )
      {
        ok = true;

        break;
      }
    }
  }

  return ok;
}


bool HERSCHELSPIRESource::checkValidHerschelSPIREFile( const QString& filename, Config *cfg )
{
  bool ok = false;
  fitsfile* ffits;
  int iStatus = 0;

  //
  // determine if it is a Herschel SPIRE Format file...
  //
  if( isValidFilename( filename, cfg ) )
  {
    if( fits_open_file( &ffits, filename.ascii( ), READONLY, &iStatus ) == 0 )
    {
      int iNumHeaderDataUnits;

      if( fits_get_num_hdus( ffits, &iNumHeaderDataUnits, &iStatus ) == 0 )
      {
        char  value[FLEN_VALUE];
        char  comment[FLEN_COMMENT];
        int   iHDUType;
        int   iValue;
        int   i;

        //
        // the primary header should never have any data...
        //
        if( fits_get_hdu_type( ffits, &iHDUType, &iStatus ) == 0 )
        {
          if( iHDUType == IMAGE_HDU )
          {
            if( fits_read_key( ffits, TLOGICAL, "SIMPLE", &iValue, comment, &iStatus ) == 0 )
            {
              if( iValue != 0 )
              {
                if( fits_read_key( ffits, TLOGICAL, "EXTEND", &iValue, comment, &iStatus ) == 0 )
                {
                  if( iValue != 0 )
                  {
                    if( fits_read_key( ffits, TINT, "NAXIS", &iValue, comment, &iStatus ) == 0 )
                    {
                      if( iValue == 0 )
                      {
                        if( fits_read_key( ffits, TSTRING, "INSTRUME", value, comment, &iStatus ) == 0 )
                        {
                          QString spire = "SPIRE";
                          QString read = value;

                          read.stripWhiteSpace( );

                          if( read.compare( spire ) == 0 )
                          {
                            ok = true;
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }

        if( ok && iStatus == 0 )
        {
          if( iNumHeaderDataUnits > 1 )
          {
            long rowsCompare = -1;
            long rows;
            int cols;

            for( i=0; i<iNumHeaderDataUnits-1 && ok; i++ )
            {
              if( fits_movrel_hdu( ffits, 1, &iHDUType, &iStatus ) == 0 )
              {
                bool historyHeader = false;

                if( iStatus == 0 && iHDUType == BINARY_TBL )
                {
                  if( isHistoryTable( ffits ) )
                  {
                    historyHeader = true;
                  }

                  //
                  // all non-History tables should have the same number of rows...
                  //
                  if( ok )
                  {
                    if( !historyHeader )
                    {
                      bool okCols = false;

                      if( fits_get_num_cols( ffits, &cols, &iStatus ) == 0 )
                      {
                        if( cols > 0 )
                        {
                          okCols = true;
                        }
                      }

                      if( okCols )
                      {
                        if( fits_get_num_rows( ffits, &rows, &iStatus ) == 0 )
                        {
                          if( rowsCompare == -1 )
                          {
                            rowsCompare = rows;
                          }
                          else if( rowsCompare == rows )
                          {
                            ok = true;
                          }
                          else
                          {
                            ok = false;
                          }
                        }
                        else
                        {
                          ok = false;
                        }
                      }
                    }
                    else
                    {
                      ok = true;
                    }
                  }
                }
              }
              else
              {
                ok = false;
              }
            }
          }
          else
          {
            ok = false;
          }
        }
      }

      if( iStatus != 0 ) 
      {
        ok = false;
      }

      iStatus = 0;

      fits_close_file( ffits, &iStatus );
    }
  }

  return ok;
}


QString HERSCHELSPIRESource::configuration(QString setting) 
{
  if (setting.lower() == "checkfilename") 
  {
    if (_config->_checkFilename) 
    {
      return QString("true");
    }
    else
    {
      return QString("false");
    }
  }

  return QString();
}


bool HERSCHELSPIRESource::setConfiguration(QString setting, const QString &value) {
  if (setting.lower() == "checkfilename") {
    if (value.lower() == "true") {
      _config->_checkFilename = true;
      return true;
    } else if (value.lower() == "false") {
      _config->_checkFilename = false;
      return true;
    }
  }

  return false;
}


bool HERSCHELSPIRESource::supportsTimeConversions() const {
  return _fieldList.contains(TIME_FIELD);
}


int HERSCHELSPIRESource::sampleForOBT(double timeIDEF, bool *ok) {
  double    value;
  int       indexLo = 0;
  int       indexHi = _numFrames - 1;
  int       index;
  int       sample = -1;

  while( indexHi > indexLo )
  {
    index = ( indexLo + indexHi ) / 2;

    if( readField( &value, TIME_FIELD, index, 1 ) == 1 )
    {
      if( value == timeIDEF )
      {
        sample = index;

        if (ok)
        {
          *ok = true;
        }

        break;
      }
      else if( value > timeIDEF )
      {
        indexHi = index;
      }
      else
      {
        indexLo = index;
      }

      if( ( indexLo + indexHi ) / 2 == indexLo )
      {
        sample = indexLo;

        if (ok)
        {
          *ok = true;
        }

        break;
      }
    }
    else
    {
      break;
    }
  }

  return sample;
}


int HERSCHELSPIRESource::sampleForTime(const KST::ExtDateTime& time, bool *ok) {
  int sample = -1;

  if (!_valid)
  {
    if (ok)
    {
      *ok = false;
    }

    return sample;
  }

  double timeIDEF;

  //
  // need to convert from KST::ExtDateTime to TAI in seconds...
  //
  timeIDEF = extDateTimeUTCToTime_t(time) + ( 86400.0 * ( 365.0 * 12.0 + 3.0 ) );
  sample = sampleForOBT( timeIDEF, ok );

  return sample;
}


int HERSCHELSPIRESource::sampleForTime(double ms, bool *ok) {
  int sample = -1;

  if (!_valid)
  {
    if (ok)
    {
      *ok = false;
    }

    return sample;
  }

  double timeZero;

  if( readField( &timeZero, TIME_FIELD, 0, 1 ) == 1 )
  {
    double timeIDEF;

    //
    // need to convert from ms time to TAI in seconds...
    //
    timeIDEF = timeZero + ( ms / 1000.0 );

    sample = sampleForOBT( timeIDEF, ok );
  }

  return sample;
}


KST::ExtDateTime HERSCHELSPIRESource::timeForSample(int sample, bool *ok) {
  KST::ExtDateTime t;

  if (!_valid)
  {
    if (ok)
    {
      *ok = false;
    }

    return t;
  }

  double time;

  if( readField( &time, TIME_FIELD, sample, 1 ) == 1 )
  {
    if( time == time )
    {
      t.setTime_t( (int)(time - ( 86400.0 * (365.0 * 12.0 + 3.0))));

      if (ok)
      {
        *ok = true;
      }
    }
  }

  return t;
}


double HERSCHELSPIRESource::relativeTimeForSample(int sample, bool *ok) {
  double timeRelativeMS = -1.0;

  if (!_valid)
  {
    if (ok)
    {
      *ok = false;
    }

    return timeRelativeMS;
  }

  double value;
  double valueZero;

  if( readField( &valueZero, TIME_FIELD, 0, 1 ) == 1 )
  {
    if( readField( &value, TIME_FIELD, sample, 1 ) == 1 )
    {
      timeRelativeMS = ( value - valueZero ) * 1000.0;

      if (ok)
      {
        *ok = true;
      }
    }
  }

  return timeRelativeMS;
}


class ConfigWidgetHerschelSPIRE : public KstDataSourceConfigWidget {
  public:
    ConfigWidgetHerschelSPIRE() : KstDataSourceConfigWidget() {
      QGridLayout *layout = new QGridLayout(this, 1, 1);
      _ac = new HerschelSPIREConfig(this);
      layout->addWidget(_ac, 0, 0);
      layout->activate();
    }

    virtual ~ConfigWidgetHerschelSPIRE() {}

    virtual void setConfig(KConfig *cfg) {
      KstDataSourceConfigWidget::setConfig(cfg);
    }

    virtual void load() {
      _cfg->setGroup("HerschelSPIRE General");
      _ac->_checkFilename->setChecked(_cfg->readBoolEntry("Check Filename", true));
    }

    virtual void save() {
      _cfg->setGroup("HerschelSPIRE General");
      _cfg->writeEntry("Check Filename", (int)_ac->_checkFilename->isChecked());
    }

    HerschelSPIREConfig *_ac;
};


extern "C" {
  KstDataSource *create_herschelSPIRE( KConfig *cfg, const QString& filename, const QString& type )
  {
    return new HERSCHELSPIRESource( cfg, filename, type );
  }

  KstDataSource *load_herschelSPIRE( KConfig *cfg, const QString& filename, const QString& type, const QDomElement& e )
  {
    return new HERSCHELSPIRESource(cfg, filename, type, e);
  }

  QStringList provides_herschelSPIRE( )
  {
    QStringList rc;

    rc += "HERSCHELSPIRE";

    return rc;
  }

  bool supportsTime_herschelSPIRE( KConfig*, const QString& filename )
  {
    Q_UNUSED(filename);

    return true;
  }

  bool supportsHierarchy_herschelSPIRE( )
  {
    return true;
  }

  int understands_herschelSPIRE( KConfig* cfg, const QString& filename )
  {
    HERSCHELSPIRESource::Config config;
    QFileInfo fileinfo( filename );
    int       iRetVal = 0;

    config.read(cfg, filename);

    if( fileinfo.isFile( ) )
    {
      if( HERSCHELSPIRESource::checkValidHerschelSPIREFile( filename, &config ) )
      {
        iRetVal = 99;
      }
    }
    else if( fileinfo.isDir( ) )
    {
      if( HERSCHELSPIRESource::checkValidHerschelSPIREFolder( filename ) )
      {
        iRetVal = 99;
      }
    }

    return iRetVal;
  }

  QWidget *widget_herschelSPIRE(const QString& filename) {
    Q_UNUSED(filename)

    return new ConfigWidgetHerschelSPIRE;
  }
}

KST_KEY_DATASOURCE_PLUGIN(herschelSPIRE)

