/*
     File: khronopanel.cpp
     Khrono : Watch, timer and countdown utility for KDE 2
     Copyright (C) 1999,2000,2001 Axel von Arnim <vonarnim@club-internet.fr>

     This program is free software; you can redistribute it and/or
     modify it under the terms of the GNU General Public License
     as published by the Free Software Foundation; either
     version 2 of the License, or (at your option) any later
     version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "khronopanel.h"
#include "khronodisplay.h"
#include "khronotimedisplay.h"
#include "khronotimer.h"
#include <qcolor.h>
#include <qpixmap.h>
#include <qtooltip.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <stdlib.h>
#include <qframe.h>
#include <kstddirs.h>
#include <kapp.h>
//#include <arts/soundserver.h>
#include <kaudioplayer.h>
#include <kaboutapplication.h>
#include "main.h"

#define DEF_WIDTH 300
#define DEF_HEIGHT 180

// *** Constructeurs *** //

KhronoPanel::KhronoPanel( QWidget *parent, const char *name )
    : QWidget( parent, name )
{
  config_=0;
  //modeStr=new (char*)[3];
  //modeStr[0]="Time";
  //modeStr[1]="Chrono";
  //modeStr[2]="Countdown";
  alarmFile=new QString(locate("data", QString(APP_NAME)+"/sounds/alarm.wav"));
  alarmRepeat=1;
  colors=new QString(ThemeDialog::DEFAULT);
  pxPlay=new QPixmap(locate("data", QString(APP_NAME)+"/pics/play.gif"));
  pxStop=new QPixmap(locate("data", QString(APP_NAME)+"/pics/stop.gif"));
  pxPause=new QPixmap(locate("data", QString(APP_NAME)+"/pics/pause.gif"));
  pxMode=new QPixmap(locate("data", QString(APP_NAME)+"/pics/mode.gif"));
  pxLap=new QPixmap(locate("data", QString(APP_NAME)+"/pics/lap.gif"));
  pxSet=new QPixmap(locate("data", QString(APP_NAME)+"/pics/set.gif"));
  pxReset=new QPixmap(locate("data", QString(APP_NAME)+"/pics/reset.gif"));
  pxAbout=new QPixmap(locate("data", QString(APP_NAME)+"/pics/about.gif"));
  pxTheme=new QPixmap(locate("data", QString(APP_NAME)+"/pics/theme.gif"));
  display=new KhronoDisplay( this, "display" );
  buttonPanel=new QWidget( this, "button_panel" );
  buttonPanel->setMaximumHeight(30);
  modeButton=new QPushButton( "Mode", buttonPanel, "mode_button" );
  if( !pxMode->isNull() ) modeButton->setPixmap(*pxMode);
  QToolTip::add( modeButton, "Mode" );
  startStopButton=new QPushButton( "Start", buttonPanel, "start_button" );
  if( !pxPlay->isNull() ) startStopButton->setPixmap(*pxPlay);
  QToolTip::add( startStopButton, "Start/Stop" );
  startStopButton->setToggleButton(TRUE);
  startStopButton->setEnabled(FALSE);
  pauseButton=new QPushButton( "Pause", buttonPanel, "pause_button" );
  if( !pxPause->isNull() ) pauseButton->setPixmap(*pxPause);
  QToolTip::add( pauseButton, "Pause" );
  pauseButton->setToggleButton(TRUE);
  setButton=new QPushButton( "Set", buttonPanel, "set_button" );
  if( !pxSet->isNull() ) setButton->setPixmap(*pxSet);
  QToolTip::add( setButton, "Set" );
  setButton->setEnabled(FALSE);
  resetButton=new QPushButton( "Reset", buttonPanel, "reset_button" );
  if( !pxReset->isNull() ) resetButton->setPixmap(*pxReset);
  QToolTip::add( resetButton, "Reset" );
  resetButton->setEnabled(FALSE);
  lapButton=new QPushButton( "Lap", buttonPanel, "lap_button" );
  if( !pxLap->isNull() ) lapButton->setPixmap(*pxLap);
  QToolTip::add( lapButton, "Lap" );
  lapButton->setEnabled(TRUE);
  aboutButton=new QPushButton( "About", buttonPanel, "about_button" );
  if( !pxAbout->isNull() ) aboutButton->setPixmap(*pxAbout);
  QToolTip::add( aboutButton, "About" );
  helpMenu=new QPopupMenu(0, "help_menu");
  helpMenu->insertItem("Help", this, SLOT(help()));
  helpMenu->insertItem("About", this, SLOT(about()));
  themeButton=new QPushButton( "Theme", buttonPanel, "theme_button" );
  if( !pxTheme->isNull() ) themeButton->setPixmap(*pxTheme);
  QToolTip::add( themeButton, "Theme" );
  buttonLayout=new QHBoxLayout( buttonPanel );
  buttonLayout->addWidget( modeButton );
  buttonLayout->addWidget( startStopButton );
  buttonLayout->addWidget( pauseButton );
  buttonLayout->addWidget( lapButton );
  buttonLayout->addWidget( resetButton );
  buttonLayout->addWidget( setButton );
  buttonLayout->addWidget( themeButton );
  buttonLayout->addWidget( aboutButton );
  mainLayout=new QVBoxLayout( this );
  mainLayout->addWidget( display );
  mainLayout->addWidget( buttonPanel );
  mainLayout->activate();
  
  QObject::connect(modeButton, SIGNAL(clicked()),
                   display->getTimeDisplay()->getTimer(), SLOT(switchMode()));
  QObject::connect(display->getTimeDisplay()->getTimer(), SIGNAL(modeSwitched(int)),
                   this, SLOT(setMode(int)));
  QObject::connect(startStopButton, SIGNAL(clicked()),
                   display->getTimeDisplay()->getTimer(), SLOT(startStop()));
  QObject::connect(display->getTimeDisplay()->getTimer(), SIGNAL(started(bool)),
                   this, SLOT(setStart(bool)));
  QObject::connect(display->getTimeDisplay()->getTimer(), SIGNAL(paused(bool)),
                   this, SLOT(setPause(bool)));
  QObject::connect(setButton, SIGNAL(clicked()),
                   this, SLOT(setCountdown()));
  QObject::connect(resetButton, SIGNAL(clicked()),
                   display, SLOT(reset()));
  QObject::connect(lapButton, SIGNAL(clicked()),
                   this, SLOT(lap()));
  QObject::connect(pauseButton, SIGNAL(clicked()),
                   display->getTimeDisplay()->getTimer(), SLOT(pause()));
  QObject::connect(aboutButton, SIGNAL(clicked()),
                   this, SLOT(helpPopup()));
  QObject::connect(themeButton, SIGNAL(clicked()),
                   this, SLOT(setTheme()));
  QObject::connect(this, SIGNAL(themeChanged(QColorGroup*)),
                   display, SLOT(setTheme(QColorGroup*)));
  QObject::connect(display->getTimeDisplay()->getTimer(), SIGNAL(alarm()),
                   this, SLOT(alarm()));
  resize(DEF_WIDTH, DEF_HEIGHT);
}

KhronoDisplay* KhronoPanel::getDisplay()
{
  return display;
}

void KhronoPanel::setConfig(KConfig* config)
{
  config_=config;
  QString _cols=config_->readEntry(QString("theme.colors"), "Default");
  if( colors!=0 ) delete colors;
  colors=new QString(_cols);
  display->setTheme(ThemeDialog::getThemeValue(colors));
  QSize size=config_->readSizeEntry(QString("size"), new QSize(DEF_WIDTH, DEF_HEIGHT));
  resize(size);
  int mode=config_->readNumEntry(QString("mode"), TIME);
  for(int i=0; i<mode; i++)
    display->getTimeDisplay()->getTimer()->switchMode();
  display->getTimeDisplay()->getTimer()->
  setCountdown(config_->readNumEntry(QString("countdown.days"), 0),
               config_->readNumEntry(QString("countdown.hours"), 0),
               config_->readNumEntry(QString("countdown.minutes"), 0),
	         config_->readNumEntry(QString("countdown.seconds"), 0));
  QString *_alarmFile=new QString(config_->readEntry(QString("theme.alarm.file"), ""));
  if( *_alarmFile!=QString("") ) {
    if( alarmFile!=0 ) delete alarmFile;
    alarmFile=_alarmFile;
  }
  alarmRepeat=config_->readNumEntry(QString("theme.alarm.repeat"), 1);
}

void KhronoPanel::setMode(int mode) {
  //modeButton->setText(modeStr[mode]);
  saveMode(mode);
  if( mode==COUNTDOWN ) setButton->setEnabled(TRUE);
  else setButton->setEnabled(FALSE);
  if( mode==TIME ) {
    resetButton->setEnabled(FALSE);
    startStopButton->setOn(FALSE);
    startStopButton->setEnabled(FALSE);
    if( !pxPlay->isNull() ) startStopButton->setPixmap(*pxPlay);
    else startStopButton->setText("Start");
  }
  else {
    resetButton->setEnabled(TRUE);
    startStopButton->setEnabled(TRUE);
  }
}

void KhronoPanel::setStart(bool start) {
  if( start==FALSE ) {
    if( startStopButton->isOn() ) startStopButton->toggle();
    if( !pxPlay->isNull() ) startStopButton->setPixmap(*pxPlay);
    else startStopButton->setText("Start");
  }
  else {
    if( !startStopButton->isOn() ) startStopButton->toggle();
    if( !pxStop->isNull() ) startStopButton->setPixmap(*pxStop);
    else startStopButton->setText("Stop");
  }
}

void KhronoPanel::setPause(bool pause) {
  if( pause==FALSE ) {
    if( pauseButton->isOn() ) pauseButton->toggle();
  }
  else {
    if( !pauseButton->isOn() ) pauseButton->toggle();
  }
}

void KhronoPanel::lap()
{
  display->lap();
}

void KhronoPanel::setCountdown()
{
  SetDialog input(this, "set_dialog");
  input.show();
  if( input.result()!=SetDialog::OK ) return;
  int d=input.getDays();
  int h=input.getHours();
  int m=input.getMins();
  int s=input.getSecs();
  m+=s/60; s=s%60;
  h+=m/60; m=m%60;
  d+=h/24; h=h%24;
  if( (!QTime(h, m, s).isValid())||(d>99) ) {
    char *msg=new char[128];
    if( d>99 ) sprintf(msg, "Invalid date: %d %02d:%02d:%02d (days>99)", d, h, m, s);
    else sprintf(msg, "Invalid date: %d %02d:%02d:%02d", d, h, m, s);
    QMessageBox::critical( this, "Error", msg );
    delete msg;
    return;
  }
  display->getTimeDisplay()->getTimer()->setCountdown(d, h ,m, s);
  saveCountdown(d, h, m, s);
}

void KhronoPanel::helpPopup()
{
  helpMenu->popup(mapToGlobal(QPoint(aboutButton->x(), aboutButton->y()+display->height())));
}

void KhronoPanel::help()
{
/*  QString html=KApplication::kde_htmldir();
  int pathDepth=0;
  int pos=0;
  while( pos!=-1 ) {
    pathDepth++;
    pos=html.find('/', pos+1);
  }
  QString toRootPath("");
  for(int i=0; i<pathDepth; i++) toRootPath+="/..";
  KApplication::getKApplication()->invokeHTMLHelp(toRootPath+"/.."+KApplication::localkdedir()+"/share/doc/HTML/en/"+APP_NAME+"/"+APP_NAME+".html", "");
*/
  KApplication::kApplication()->invokeHTMLHelp(QString(APP_NAME)+"/index.html", "");
}

void KhronoPanel::about()
{
//  char *msg=new char[512];
//  sprintf(msg, "%s\nVersion: %s\nProgramming: %s\nCopyright %s\nLicense: %s",
//          DESCRIPTION, VERSION, AUTHOR, COPYRIGHT, LICENSE);
//  QMessageBox::information( this, "About Khrono", msg, 1);
//  delete msg;
  KAboutApplication().show();
}

void KhronoPanel::setTheme()
{
  ThemeDialog input( colors, alarmFile, alarmRepeat, this, "theme_dialog");
  input.show();
  if( input.result()!=ThemeDialog::OK ) return;
  QString cols=input.getChosenTheme();
  if( colors!=0 ) delete colors;
  colors=new QString(cols);
  if( colors!=0 ) {
    saveTheme(colors);
    emit themeChanged(ThemeDialog::getThemeValue(colors));
  }
  alarmRepeat=input.getChosenRepeat();
  if( alarmFile!=0 ) delete alarmFile;
  const char *file=input.getChosenAlarm();
  if( (file!=0)&&(!QString(file).isNull()) ) {
    alarmFile=new QString(file);
    saveAlarm(alarmFile, alarmRepeat);
  }
}

void KhronoPanel::resizeEvent(QResizeEvent* _evt)
{
  saveSize(_evt->size());
}

void KhronoPanel::closeEvent(QCloseEvent* _evt)
{
  if( config_!=0 ) config_->sync();
  _evt->accept();
}

void KhronoPanel::alarm()
{
  for(int i=0; i<alarmRepeat; i++)
    KAudioPlayer::play((const char*)(*alarmFile));
/*
  Arts::SimpleSoundServer server(Arts::Reference("global:Arts_SimpleSoundServer"));
  if(server.isNull()) {
    fprintf(stderr, "Failed contacting audio server\n");
    return;
  }
  if( !server.play((const char*)(*alarmFile)) )
    fprintf(stderr, (const char*)(QString("Failed playing ")+(*alarmFile)+"\n"));
  else {
    for(int i=1; i<alarmRepeat; i++) server.play((const char*)(*alarmFile));
  }
*/
}

void KhronoPanel::saveTheme(QString* name)
{
  if( config_==0 ) return;
  config_->writeEntry(QString("theme.colors"), *name);
}

void KhronoPanel::saveAlarm(QString *file, int repeat)
{
  if( config_==0 ) return;
  config_->writeEntry(QString("theme.alarm.file"), *file);  
  config_->writeEntry(QString("theme.alarm.repeat"), repeat);  
}

void KhronoPanel::saveSize(const QSize& size)
{
  if( config_==0 ) return;
  config_->writeEntry(QString("size"), size);
}

void KhronoPanel::saveMode(int mode)
{
  if( config_==0 ) return;
  config_->writeEntry(QString("mode"), mode);
}

void KhronoPanel::saveCountdown(int days, int hours, int mins, int secs)
{
  if( config_==0 ) return;
  config_->writeEntry(QString("countdown.days"), days);
  config_->writeEntry(QString("countdown.hours"), hours);
  config_->writeEntry(QString("countdown.minutes"), mins);
  config_->writeEntry(QString("countdown.seconds"), secs);
}


// ************************************************************ //

SetDialog::SetDialog( QWidget *parent, const char *name )
    : QDialog( parent, name, TRUE )
{
  btDays=new QLineEdit( this, "days_input" );
  btDays->setMaximumSize(60, 20);
  lbDays=new QLabel( "Days", this, "days_label" );
  btHours=new QLineEdit( this, "hours_input" );
  btHours->setMaximumSize(60, 20);
  lbHours=new QLabel( "Hours", this, "hours_label" );
  btMins=new QLineEdit( this, "mins_input" );
  btMins->setMaximumSize(60, 20);
  lbMins=new QLabel( "Minutes", this, "mins_label" );
  btSecs=new QLineEdit( this, "secs_input" );
  btSecs->setMaximumSize(60, 20);
  lbSecs=new QLabel( "Seconds", this, "secs_label" );
  pnOk=new QWidget( this, "ok_panel" );
  btOk=new QPushButton( "Ok", pnOk, "ok_button" );
  btOk->setMaximumSize( 60, 20 );
  btCancel=new QPushButton( "Cancel", pnOk, "cancel_button" );
  btCancel->setMaximumSize( 60, 20 );
  loOk=new QHBoxLayout( pnOk );
  loOk->addWidget( btOk );
  loOk->addWidget( btCancel );

  lo=new QGridLayout( this, 3, 4 );
  lo->addColSpacing(0, 60);
  lo->addColSpacing(1, 60);
  lo->addColSpacing(2, 60);
  lo->addColSpacing(3, 60);
  lo->addRowSpacing(0, 25);
  lo->addRowSpacing(1, 25);
  lo->addRowSpacing(2, 30);
  lo->addWidget( lbDays, 0, 0 );
  lo->addWidget( btDays, 1, 0 );
  lo->addWidget( lbHours, 0, 1 );
  lo->addWidget( btHours, 1, 1 );
  lo->addWidget( lbMins, 0, 2);
  lo->addWidget( btMins, 1, 2 );
  lo->addWidget( lbSecs, 0, 3 );
  lo->addWidget( btSecs, 1, 3 );
  lo->addMultiCellWidget( pnOk, 2, 2, 0, 2 );
  lo->activate();
  setMaximumHeight(50);
  setMinimumHeight(50);
  setCaption("Countdown");

  QObject::connect( btOk, SIGNAL(clicked()), this, SLOT(okClicked()) );
  QObject::connect( btCancel, SIGNAL(clicked()), this, SLOT(cancelClicked()) );
}

void SetDialog::okClicked()
{
  done(OK);
}

void SetDialog::cancelClicked()
{
  done(CANCEL);
}

int SetDialog::getDays()
{
  return QString(btDays->text()).toInt();
}

int SetDialog::getHours()
{
  return QString(btHours->text()).toInt();
}

int SetDialog::getMins()
{
  return QString(btMins->text()).toInt();
}

int SetDialog::getSecs()
{
  return QString(btSecs->text()).toInt();
}

// ************************************************************ //

const QString& ThemeDialog::GREEN=QString("Green");
const QString& ThemeDialog::ORANGE=QString("Orange");
const QString& ThemeDialog::CLASSIC=QString("Classic");
const QString& ThemeDialog::RETRO_ORANGE=QString("RetroOrange");
const QString& ThemeDialog::RETRO_GREEN=QString("RetroGreen");
const QString& ThemeDialog::DEFAULT=GREEN;
QColorGroup** ThemeDialog::themes=0;
/*QColorGroup* ThemeDialog::themes[5]= {
  new QColorGroup(*(KhronoDisplay::defaultTheme)),
  new QColorGroup(QColor(240, 185, 115), black, black, QColor(240, 185, 115).dark(200), black, black, black),
  new QColorGroup(QColor(70, 70, 70), QColor(160, 178, 146), QColor(160, 178, 146), QColor(111, 137, 137), QColor(160, 178, 146), QColor(160, 178, 146), QColor(160, 178, 146)),
  new QColorGroup(QColor(100, 70, 10), QColor(255, 140, 39), QColor(255, 140, 39), QColor(150, 120, 20), QColor(255, 140, 39), QColor(255, 140, 39), QColor(255, 140, 39)),
  new QColorGroup(QColor(10, 100, 70), QColor(39, 255, 140), QColor(39, 255, 140), QColor(28, 168, 130), QColor(39, 255, 140), QColor(39, 255, 140), QColor(39, 255, 140))
};*/

ThemeDialog::ThemeDialog( QString* _theme, QString* _alarmFile, int _repeat, QWidget *parent, const char *name )
    : QDialog( parent, name, TRUE )
{
  if( (_theme==0)||(_theme->length()==0) ) colors=new QString(DEFAULT);
  colors=new QString(*_theme);

  btOk=new QPushButton( "Ok", this, "ok_button" );
  btOk->setMaximumHeight( 25 );
  btOk->setMinimumHeight( 25 );
  btCancel=new QPushButton( "Cancel", this, "cancel_button" );
  btCancel->setMaximumHeight( 25 );
  btCancel->setMinimumHeight( 25 );

  bg=new QButtonGroup(this, "theme_buttongroup");
  bg->setTitle("Themes");
  bt1=new QRadioButton(GREEN, bg, "themes_bt1");
  bt1->setMaximumHeight(15);
  bt2=new QRadioButton(ORANGE, bg, "themes_bt2");
  bt2->setMaximumHeight(15);
  bt3=new QRadioButton(CLASSIC, bg, "themes_bt3");
  bt3->setMaximumHeight(15);
  bt4=new QRadioButton(RETRO_ORANGE, bg, "themes_bt4");
  bt4->setMaximumHeight(15);
  bt5=new QRadioButton(RETRO_GREEN, bg, "themes_bt5");
  bt5->setMaximumHeight(15);
  if( !colors->compare(GREEN) ) bt1->setChecked(TRUE); else
  if( !colors->compare(ORANGE) ) bt2->setChecked(TRUE); else
  if( !colors->compare(CLASSIC) ) bt3->setChecked(TRUE); else
  if( !colors->compare(RETRO_ORANGE) ) bt4->setChecked(TRUE); else
  if( !colors->compare(RETRO_GREEN) ) bt5->setChecked(TRUE); else
  bt1->setChecked(TRUE);
  bg->setMinimumSize( 220, 100 );
  bt1->move(10, 20);
  bt2->move(10, 35);
  bt3->move(10, 50);
  bt4->move(10, 65);
  bt5->move(10, 80);
  
  testLCD=new QLCDNumber(5, this, "test_lcd");
  testLCD->setSegmentStyle(QLCDNumber::Flat);
  testLCD->setPalette(QPalette(*(getThemeValue(colors)), *(getThemeValue(colors)), *(getThemeValue(colors))));
  testLCD->display("88:88");
  testLCD->setMaximumSize(60, 40);
  testLCD->setMinimumSize(60, 40);
  
  gbAlarm=new QGroupBox("Alarm", this, "alarm_group");
  tfAlarm=new AlarmLineEdit(gbAlarm, "alarm_edit");
  tfAlarm->setMaximumSize(120,25);
  tfAlarm->setMinimumSize(120,25);
  if( (_alarmFile!=0)&&(!_alarmFile->isNull()) )
    tfAlarm->setText( _alarmFile->data() );
  btAlarm=new QPushButton("Browse", gbAlarm, "alarm_button");
  btAlarm->setMaximumSize(50,25);
  btAlarm->setMinimumSize(50,25);
  lbAlarm=new QLabel("Repeat: ", gbAlarm, "alarm_label");
  lbAlarm->setMaximumSize(40,25);
  lbAlarm->setMinimumSize(40,25);
  sbAlarm=new QSpinBox(1, 10, 1, gbAlarm, "alarm_spinbox");
  sbAlarm->setMaximumSize(35,25);
  sbAlarm->setMinimumSize(35,25);
  sbAlarm->setValue(_repeat);
  gbAlarm->setMinimumSize( 280, 50 );
  tfAlarm->move(10, 20);
  btAlarm->move(135, 20); // 10+120+5
  lbAlarm->move(190, 20); // 135+50+5
  sbAlarm->move(235, 20); // 190+40+5
  // -> // 235+35+10=280

  loTheme=new QHBoxLayout();
  loAlarm=new QHBoxLayout();
  loOk=new QHBoxLayout();
  lo=new QVBoxLayout( this );
  lo->addSpacing(10);
  lo->addLayout( loTheme );
  lo->addSpacing(10);
  lo->addLayout( loAlarm );
  lo->addSpacing(10);
  lo->addLayout( loOk );
  lo->addSpacing(10);
  loOk->addSpacing( 10 );
  loOk->addWidget( btOk );
  loOk->addSpacing( 5 );
  loOk->addWidget( btCancel );
  loOk->addSpacing( 10 );
  loAlarm->addSpacing( 10 );
  loAlarm->addWidget( gbAlarm );
  loAlarm->addSpacing( 10 );
  loTheme->addSpacing( 10 );
  loTheme->addWidget( bg );
  loTheme->addWidget( testLCD );
  loTheme->addSpacing( 10 );
  lo->activate();
  
  setMaximumSize(minimumSize());
  setCaption("Themes");
  QObject::connect( btOk, SIGNAL(clicked()), this, SLOT(okClicked()) );
  QObject::connect( btCancel, SIGNAL(clicked()), this, SLOT(cancelClicked()) );
  QObject::connect( bg, SIGNAL(clicked(int)), this, SLOT(themeChanged(int)) );
  QObject::connect( btAlarm, SIGNAL(clicked()), this, SLOT(alarmClicked()) );
}

QColorGroup* ThemeDialog::getThemeValue(QString* name)
{
  if( themes==0 ) initThemes();
  if( !name->compare(GREEN) ) return themes[0];
  if( !name->compare(ORANGE) ) return themes[1];
  if( !name->compare(CLASSIC) ) return themes[2];
  if( !name->compare(RETRO_ORANGE) ) return themes[3];
  if( !name->compare(RETRO_GREEN) ) return themes[4];
  return themes[0];
}

const QString& ThemeDialog::getThemeName(int id)
{
  switch(id) {
  case 0: return GREEN;
  case 1: return ORANGE;
  case 2: return CLASSIC;
  case 3: return RETRO_ORANGE;
  case 4: return RETRO_GREEN;
  default: return DEFAULT;
  }
}

void ThemeDialog::initThemes()
{
  if( themes!=0 ) return;
  themes=new (QColorGroup*)[5];
  {
    themes[0]=new QColorGroup(*(KhronoDisplay::defaultTheme));
  }
  {
    QColor back(black);
    QColor fore(240, 185, 115);
    QColor inactive(fore.dark());
    themes[1]=new QColorGroup(fore, back, back, inactive, back, back, back);
  }
  {
    QColor back(160, 178, 146);
    QColor fore(70, 70, 70);
    QColor inactive(111, 137, 137);
    themes[2]=new QColorGroup(fore, back, back, inactive, back, back, back);
  }
  {
    QColor back(255, 140, 39);
    QColor fore(100, 70, 10);
    QColor inactive(150, 120, 20);
    themes[3]=new QColorGroup(fore, back, back, inactive, back, back, back);
  }
  {
    QColor back(39, 255, 140);
    QColor fore(10, 100, 70);
    QColor inactive(28, 168, 130);
    themes[4]=new QColorGroup(fore, back, back, inactive, back, back, back);
  }
}

void ThemeDialog::okClicked()
{
  done(OK);
}

void ThemeDialog::cancelClicked()
{
  done(CANCEL);
}

void ThemeDialog::alarmClicked()
{
  const char* alarmFile=tfAlarm->text();
  QString path;
  if( (alarmFile!=0)&&(strlen(alarmFile)>0) ) {
    path=QFileInfo(alarmFile).dir().absPath();
  } else {
    path=QDir::current().absPath();
  }
  QString file=QFileDialog::getOpenFileName((const char*)path, "*.wav", this, "alarm_dialog");
  if( !file.isNull() ) tfAlarm->setText(file);
}

void ThemeDialog::themeChanged(int id)
{
  if( (colors!=0)&&(!colors->compare(getThemeName(id))) ) return;
  QString _colors=getThemeName(id);
  if( colors!=0 ) delete colors;
  colors=new QString(_colors);
  testLCD->setPalette(QPalette(*getThemeValue(colors), *getThemeValue(colors), *getThemeValue(colors)));
}

const QString& ThemeDialog::getChosenTheme()
{
  return *colors;
}

const char* ThemeDialog::getChosenAlarm()
{
  return tfAlarm->text();
}

int ThemeDialog::getChosenRepeat()
{
  return atoi(sbAlarm->text());
}
