/***************************************************************************
                          kpl.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPL_H
#define KPL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dcopobject.h>
#include <kmainwindow.h>
#include "kpldoc.h"

class QStrList;
class KURL;
class KAccel;
class KAction;
class KListAction;
class KToggleAction;
class KRecentFilesAction;
class KplDoc;
class KplView;
class ItemDlg;
class ArrayItem;
class FunItem;

/**
  * Kpl application.
  * Provides the kpl top level main window with menubar, toolbar and statusbar.
  * @author Werner Stille
  */

class KplApp : public KMainWindow, public DCOPObject
{
  K_DCOP
  Q_OBJECT

public:
  /** Constructor. Creates a kpl window. */
  KplApp(QWidget* parent = 0, const char* name = 0);
  /** Destructor. */
  ~KplApp();
	/**
	 * Returns a pointer to the current document connected to the KMainWindow
	 * instance and is used by the View class to access the document object's
	 * methods.
   * @return pointer to the current document.
   */	
	KplDoc* getDocument() const; 	

k_dcop:
  /** Creates a new plot in the current view. */
  void newPlot();
	/**
	 * Opens a plot file.
   * @param url URL of the file.
   * @return true when when succeeded.
	 */
  bool openPlot(const QString& url);
	/**
	 * Opens a data file.
   * @param url URL of the file.
   * @return true when when succeeded.
	 */
  bool openData(const QString& url);
	/** Performs plot. */
  void doPlot();
	/**
	 * Saves a plot file.
   * @param url URL of the file.
   * @return true when when succeeded.
	 */
  bool savePlot(const QString& url);
  /**
   * Generates PostScript file.
   * @param url URL of the file.
   * @param landscape true for landscape orientation.
   * @return true when when succeeded.
   */
  bool savePS(const QString& url, int landscape);
  /**
   * Prints plot.
   * @param printerName name of the printer.
   * @param fileName name of the output file (empty for output to printer).
   * @param landscape true for landscape orientation.
   * @param numCopies number of copies.
   * @return true when when succeeded.
   */
  bool printPlot(const QString& printerName, const QString & fileName,
                 int landscape, int numCopies);
  /**
   * Sets autoplot option.
   * @param setting true to plot automatically after changes.
   */
  void setAutoplotOption(int setting);
  /**
   * Sets path saving option.
   * @param setting true to save absolute paths.
   */
  void setPathOption(int setting);
  /**
   * Sets calculate PS bounding path option.
   * @param setting true to calculate automatically the box.
   */
  void setBoundingBoxOption(int setting);
  /**
   * Sets page size.
   * @param w width in cm.
   * @param h height in cm.
   */
  void setPageFormat(double w, double h);
  /**
   * Sets default colors.
   * @param colFrame color for frame and labels.
   * @param colGrid color for ticks and grid lines.
   * @param colData color for lines and markers.
   */
  void setColors(const QString& colFrame, const QString& colGrid,
                 const QString& colData);
  /**
   * Sets format for numeric output.
   * @param pres printf() presentation character.
   * @param prec precision.
   */
  void setOutputFormat(const QString& pres, int prec);
  /**
   * Sets Autoplot parameters.
   * @param x0 left page margin in centimeters.
   * @param w width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param h height in centimeters.
   * @param gridmode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels, 3 for grid, 4 for grid with labels.
   * @param sx x axis text.
   * @param sy y axis text.
   * @param sh heading text.
   * @param pathHeading true for heading = path.
   * @param ix x column index.
   * @param iy y column index.
   * @param ie y error column index.
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   * @param errbars true to plot error bars.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param autoNorm true for automatic normalization.
   * @param autoFit true for automatic parameter fit.
   */
  void setAutoplot(double x0, double w, double y0, double h,
                   int gridmode, const QString& sx, const QString& sy,
                   const QString& sh, int pathHeading, int ix, int iy,
                   int ie, int logx, int logy, int errbars, int symb,
                   int autoNorm, int autoFit);
  /**
   * Sets Autofit parameters.
   * @param path absolute path of the library for the fit function.
   * @param name fit function name.
   * @param nonLin true for nonlinear parameter fits.
   * @param errCol true for data weighing by error column, false for weighing
   * by error model function.
   * @param errModPath absolute path of the library for the error model
   * function.
   * @param errModName error model function name.
   * @param itMax maximum number of iterations (nonlinear fits only).
   * @param tol tolerance level (nonlinear fits only).
   * @param symb function plot: 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param showDlg true to display parameter fit dialog.
   * @param savePar true to save resulting parameter files.
   * @param follow true to follow parameter sets in fit series.
   */
  void setAutofit(const QString& path, const QString& name,
                  int nonLin, int errCol,
                  const QString& errModPath, const QString& errModName,
                  int itMax, double tol, int symb, const QString& color,
                  int showDlg, int savePar, int follow);
  /**
   * Loads Autofit function parameters from file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  bool loadAutofitPar(const QString& url);
  /**
   * Sets Autofit function parameter value.
   * @param iPar parameter index.
   * @param value parameter value.
   * @return true when when succeeded.
   */
  bool setAutofitPar(int iPar, double value);
  /**
   * Enables/disables (Auto)fit function parameter.
   * @param iPar parameter index.
   * @param fit true enables parameter fitting.
   * @return true when when succeeded.
   */
  bool enableFitPar(int iPar, int fit);
  /**
   * Loads error model function parameters from file.
   * @param url URL of the file.
   * @param iArr array list index.
   * @return true when when succeeded.
   */
  bool loadErrModPar(const QString& url, int iArr);
  /**
   * Sets error model function parameter value.
   * @param iPar parameter index.
   * @param value parameter value.
   * @param iArr array list index.
   * @return true when when succeeded.
   */
  bool setErrModPar(int iPar, double value, int iArr);
  /**
   * Deletes plot item.
   * @param iItem item index.
   * @return true when when succeeded.
   */
  bool deleteItem(int iItem);
  /**
   * Moves plot item.
   * @param is source item index.
   * @param id destination item index.
   * @return true when when succeeded.
   */
  bool moveItem(int is, int id);
  /**
   * Appends new frame item corresponding to arguments.
   * @param x0 left page margin in centimeters.
   * @param w width in centimeters.
   * @param y0 bottom page margin in centimeters.
   * @param h height in centimeters.
   * @param xmin x value of left data window margin in user units.
   * @param xmax x value of right data window margin in user units.
   * @param ymin y value of bottom data window margin in user units.
   * @param ymax y value of top window data margin in user units.
   * @param logx true for logarithmic x axis.
   * @param logy true for logarithmic y axis.
   * @param ndigx number of digits for x axis.
   * @param ndigy number of digits for y axis.
   * @param colFrame color for frame and labels.
   * @param colGrid color for grids and tics.
   * @param xtic distance between major tics of x axis.
   * @param ytic distance between major tics of y axis.
   * @param mticx number of intervals between major tics of x axis.
   * @param mticy number of intervals between major tics of y axis.
   * @param gridmode 0 for frame only, 1 for axes with tics,
   * 2 for axes with tics and labels, 3 for grid, 4 for grid with labels.
   * @param iex logarithm of normalization factor for x axis.
   * @param iey logarithm of normalization factor for y axis.
   * @param sx x axis text.
   * @param sy y axis text.
   * @param sh heading text.
   * @param relSize relative size of text, markers and lines.
   * @param active true for visible item.
   */
	void newFrameItem(double x0, double w, double y0, double h,
                    double xmin, double xmax, double ymin, double ymax,
                    int logx, int logy, int ndigx, int ndigy,
                    const QString& colFrame, const QString& colGrid,
                    double xtic, double ytic, int mticx, int mticy,
                    int gridmode, int iex, int iey,
                    const QString& sx, const QString& sy, const QString& sh,
                    double relSize, int active);
  /** Appends new frame item corresponding to autoplot settings. */
  void newAutoFrameItem();
  /**
   * Appends new array item.
   * @param path absolute path of the data file.
   * @param ix x column index.
   * @param iy y column index.
   * @param ie y error column index.
   * @param istart start row index.
   * @param n number of data points.
   * @param errbars true to plot error bars.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for data plot.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
	bool newArrayItem(const QString& path, int ix, int iy, int ie, int istart,
                    int n, int errbars, int symb, const QString& color,
                    double fx, double fy, int active);
  /**
   * Appends new function item.
   * @param path absolute path of the function library.
   * @param name function name.
   * @param xmin minimum argument value.
   * @param xmax maximum argument value.
   * @param dx argument stepsize.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
	bool newFunItem(const QString& path, const QString& name,
                  double xmin, double xmax, double dx,
                  int symb, const QString& color,
                  double fx, double fy, int active);
  /**
   * Appends new parametric function item.
   * @param pathx absolute path of the x function library.
   * @param namex x function name.
   * @param pathy absolute path of the y function library.
   * @param namey y function name.
   * @param tmin minimum argument value.
   * @param tmax maximum argument value.
   * @param dt argument stepsize.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param color color for function plot.
   * @param fx normalization factor for x values.
   * @param fy normalization factor for y values.
   * @param active true for visible item.
   * @return true when when succeeded.
   */
	bool newParFunItem(const QString& pathx, const QString& namex,
	                   const QString& pathy, const QString& namey,
                     double tmin, double tmax, double dt,
                     int symb, const QString& color,
                     double fx, double fy, int active);
  /**
   * Appends new legend item.
   * @param symb 0 for circles, < 0 for different markers,
   * >0 for different line types.
   * @param colSymb color for marker or line.
   * @param x x position in cm.
   * @param y y position in cm.
   * @param text legend text.
   * @param colText color for text.
   * @param xoff x offset for text in cm.
   * @param yoff y offset for text in cm.
   * @param relSize relative size of text.
   * @param active true for visible item.
   */
	void newLegendItem(int symb, const QString& colSymb, double x, double y,
                     const QString& text, const QString& colText,
                     double xoff, double yoff, double relSize,
                     int active);
  /**
   * Scales automatically all child items of a frame item.
   * @param iItem item index.
   * @param autoNorm true for automatic normalization.
   * @return true when when succeeded.
   */
  bool autoScaleFrame(int iItem, int autoNorm);
  /**
   * Loads function parameter file.
   * @param url URL of the file.
   * @param iItem item index.
   * @param yFun true for y function parameter, false for x function parameter
   *  (only for parametric functions).
   * @return true when when succeeded.
   */
  bool loadPar(const QString& url, int iItem, int yFun);
  /**
   * Sets function parameter value.
   * @param iItem item index.
   * @param iPar parameter index.
   * @param value parameter value.
   * @param yFun true for y function parameter, false for x function parameter
   *  (only for parametric functions).
   * @return true when when succeeded.
   */
  bool setPar(int iItem, int iPar, double value, int yFun);
  /**
   * Sets array row range.
   * @param iItem item index.
   * @param istart start row index.
   * @param n number of data points.
   * @return true when when succeeded.
   */
  bool setArrayRange(int iItem, int iStart, int n);
  /**
   * Adds array and function items to item lists for (multidimensional)
   * parameter fitting.
   * @param erase erases lists first when true.
   * @param iArr array item index.
   * @param iFun function item index.
   * @param errCol true for data weighing by error column, false for weighing
   * by error model function.
   * @param errModPath absolute path of the library for the error model
   * function.
   * @param errModName error model function name.
   * @return true when when succeeded.
   */
  bool addFitItems(int erase, int iArr, int iFun, int errCol,
                   const QString& errModPath, const QString& errModName);
  /**
   * Performs parameter fit.
   * @param nonLin true for nonlinear parameter fits.
   * @param savePar true to save resulting parameter files.
   * @param follow true to follow parameter sets in fit series.
   */
  bool fit(int nonLin, int savePar, int follow);
  /**
   * Returns chi-square of last fit.
   * @return chi-square.
   */
  double chiSquare();
  /**
   * Saves parameters and their errors resulting from a fit to a file.
   * @param url URL of the file.
   * @return true when when succeeded.
   */
  bool saveFitPar(const QString& url);
  /**
   * Exports function value table to file.
   * @param url URL of the file.
   * @param iItem item index.
   * @return true when when succeeded.
   */
  bool exportValues(const QString& url, int iItem);

protected:
  /** Initializes menu and toolbar. */
  void initActions();
  /** Calculates precision for statusbar. */
  void calcStatusBar();
  /** Inserts statusbar items. */
  void initStatusBar();
  /** Creates and initialize new statusbar. */
  void newStatusBar();
	/**
   * initKeyAccel creates the keyboard accelerator items for the available slots and changes the menu accelerators.
   * @see KAccel
   */
	void initKeyAccel();
	/**
   * Initializes the document object of the main window that is connected to the view in initView().
   * @see initView();
   */
	void initDocument();
  /**
   * Creates the centerwidget of the KMainWindow instance and sets it as the view.
   */
  void initView();
  /**
   * Saves the options.
   * @param saveAll save everything when true.
   */
  void saveOptions(bool saveAll);
  /** Reads the options on init. */
  void readOptions();
  /**
   * Opens an existing URL or local file for reading.
   * @param type one of the following: UNKNOWN, DATAFILE, PLOTFILE.
   */
  void openRead(KplDoc::FileTypes type);
  /**
   * Saves plot file using current or new name.
   * @param current true for save under current name.
   * @return true when file is saved.
   */
  bool save(bool current);
  /**
   * Saves plot file.
   * @param url URL of the file.
   * @return true when file is saved.
   */
  bool saveAs(const KURL &url);
  /**
   * Generates PostScript file.
   * @param url URL of the file.
   * @param landscape != 0 for landscape orientation.
   * @return true when file is saved.
   */
  bool savePSURL(const KURL& url, bool landscape);
  /**
   * Warns at unsaved changes.
   * @return true when ready to close file.
   */
  virtual bool queryClose();
  /** Reimplemented for message box on last window exit. */
  virtual bool queryExit();
  /**
   * Saves the window properties for each open window during session end
   * to the session config file, including saving the currently
   * opened file by a temporary filename provided by KApplication.
   * @see KMainWindow#saveProperties
   */
  virtual void saveProperties(KConfig*);
  /**
   * Reads the session config file and restores the application's state
   * including the last opened files and documents by reading the
   * temporary files saved by saveProperties().
   * @see KMainWindow#readProperties
   */
  virtual void readProperties(KConfig*);
  KAction* fileSave;
  KAction* fileSaveAs;
  KAction* fileClose;
  KAction* filePrint;
  KAction* filePlot;
  KAction* editUndo;
  KAction* editRedo;
  KAction* viewReload;
  KListAction* filePS;
  KToggleAction* showToolbar;
  KToggleAction* showStatusbar;
  KToggleAction* autoplot;
  KToggleAction* showSource;
  KToggleAction* boundingBox;
  KToggleAction* saveAbsPath;
  KToggleAction* warning;
  KToggleAction* saveAtEnd;
  KRecentFilesAction* recentFiles;
  KplView* view;
  KplDoc* doc;
  ItemDlg* itemDlg;
  KConfig* config;
  KAccel* key_accel;
  QTimer* timer;
  KToolBar::BarPosition toolBarPos;
  QList<ArrayItem> arr;
  QList<FunItem> fun;
  enum StatusId{StatusMsg = 1, StatusPos, StatusVal};
  bool plot;
  int tRefresh, xPrec, yPrec;
  double fz;

  friend KplView;

public slots:
	/**
	 * Opens a file.
   * @param url URL of the file.
	 */
  void openDocumentURL(const KURL& url);

protected slots:
  /** Creates a new plot in the current view. */
  void slotFileNew();
  /** Opens an existing plot file. */
  void slotFileOpenPlot();
  /** Opens an existing data file. */
  void slotFileOpenData();
  /** Saves the current plot file. */
  void slotFileSave();
  /** Saves the current plot as a file with different name. */
  void slotFileSaveAs();
  /** Closes the current file. */
  void slotFileClose();
  /** Plots the current file. */
  void slotFilePlot();
  /** Plots the current file conditionally. */
  void slotPlotCond();
  /**
   * Generates PostScript file.
   * @param landscape != 0 for landscape orientation.
   */
  void slotFilePS(int landscape);
  /** Prints the current file. */
  void slotFilePrint();
  /** Opens a new kpl window. */
  void slotFileNewWindow();
  /** Closes the current window. */
  void slotFileCloseWindow();
  /** Exits the application. */
  void slotFileQuit();
  /** Restores previous item from backup. */
  void slotEditUndo();
  /** Restores next item from backup. */
  void slotEditRedo();
  /** Edits plot items. */
  void slotEditItems();
  /** Increases magnification. */
  void slotViewZoomIn();
  /** Decreases magnification. */
  void slotViewZoomOut();
  /** Sets magnification. */
  void slotViewZoom();
  /** Watches file. */
  void slotWatchFile();
  /** Reloads file. */
  void slotViewReload();
  /** Reloads file periodically. */
  void slotViewAutoReload();
  /** Toggles the visibility of the toolbar. */
  void slotOptionsToolBar();
  /** Toggles the visibility of the statusbar. */
  void slotOptionsStatusBar();
  /** Toggles the autoplot option. */
  void slotOptionsAutoPlot();
  /** Toggles the show function source option. */
  void slotOptionsShowSource();
  /** Displays the key bindings dialog. */
  void slotOptionsKeys();
  /** Displays the toolbars configure dialog. */
  void slotEditToolbars();
  /** Displays the basic settings dialog. */
  void slotOptionsBasic();
  /** Displays the autoplot settings dialog. */
  void slotOptionsAuto();
  /** Displays the autofit settings dialog. */
  void slotOptionsAutoFit();
  /** Saves options. */
  void slotOptionsSave();
  /**
   * Changes the status message.
   * @param text pointer to new message text.
   */
  void slotStatusMsg(const QString& text);
  /**
   * Changes the status message of the whole statusbar temporarily.
   * @param text pointer to temporary message text.
   */
  void slotStatusHelpMsg(const QString& text);
  /**
   * Opens files from URL list.
   * @param list pointer to list with URLs.
   */
  void slotOpenList(QStrList* list);
  /** Updates titlebar, menus, plot, and item list.
   * @param updPlot true when plot should be uddated.
   * @param updList true when item list should be uddated.
   */
  void slotUpdate(bool updPlot, bool updList);
  /** Updates titlebar and menus. */
  void updWin();
  /**
   * Updates mouse position on statusbar.
   * @param x horizontal position of the mouse cursor.
   * @param y vertical position of the mouse cursor.
   */
  void updStatus(int x, int y);
  /** Clears mouse position information on statusbar. */
  void clearStatus();
  /** Shrinks window. */
  void shrink();
  /** Shows main popup menu. */
  void showPopupMenu();
};

#endif // KPL_H
