#ifndef _DB_INTERFACE_H_
#define _DB_INTERFACE_H_

// QT includes
#include <qlist.h>
#include <qstring.h>
#include <qdatetime.h>

// Local includes
#include "databaseentry.h"
#include "database.h"
#include "databaseresource.h"

/** This class provides an interface for all programs to the addressbook database in
 * KDE. The actual database is set up in a series of keys. For examply. When you store
 * DB entry, it is inserted into the list (according to the insertionPolicy) like this:<br>
 *<PRE>
 *       key #             element
 *
 *       100               LastName
 *       101               FirstName
 *       102               LastName
 *       103               Company
 *       104               Title
 *       ...               ...
 *</PRE>
 * This method allows 99 fields per entry. The entries are always starting at a key that 
 * is a multiple of 100. 
 *
 * If for some reason you need to delete an entry, only the key that is a multiple of 
 * 100 is deleted. This makes things faster for DB access.
 *
 * Field 0 of the databse will store the largest key. This can be obtained by calling 
 * largestKey(). Therefore, if you want to traverse through the list, you would do 
 * something like:
 *<PRE>
 * int key = 100;
 * while (key <= DBInterface->largestKey())
 *  {
 *    if (DBInterface->exists(key))
 *        {
 *          DBEntry* entry = fetchEntry(key);
 *        }
 *    key += 100;
 *  }
 *</PRE>
 * Storing the largestKey at key 0 in a database allows a program to support multiple
 * databases without having to know how many entries are in each DB. A user could
 * therefore have a 'business.database' and a 'personal.db' openable by any 
 * KDE address book application.
 *
 * @author Mike Pilone
 */

class DBInterface
{
  
public:


  /** Constructor
   */
  DBInterface();

  /** Destructor
   */
  ~DBInterface();

  /** Fetches the entry at key 'key'. Remember that key must be a multiple of 100. The 
   * entry returned will completely filled in from the database. It is possible that this
   * function will return NULL if 'key' doesn't exist
   */
  DBEntry* fetchEntry( int key );

  /** Fetches all the entries from the database and returns them in a QList. If the database
   * is empty, an empty QList will be returned.
   *
   * WARNING: Due to the fact that this method needs to read the entire database it could 
   * be very slow. It is recommended that you only use this function on small databases or 
   * at least alert the user it may take a few minutes
   */
  QList<DBEntry>* fetchAllEntries();

  /** Fetches the entries where keyOffset is equal to patternA. If the field has two options,
   * like phone numbers or custom fields, patternB will be used as well. A QList will always
   * be returned, but it could be empty
   */
  QList<DBEntry>* fetchEntries(int keyOffset, QString patternA, QString patternB = "");

  /** Returns how the databse is open. Options are:
   * <PRE>
   * dbOpenReadWrite = 0
   * dbOpenRead = 1
   * dbNotOpen = 2;
   * </PRE>
   */ 
  dbOpenType openDBType();

  /** Returns the largest key. It will return 0 if the databse is empty or not open
   */
  int largestKey() { return fLargestKey; };

  /** Checks to see if an entry exists at 'key'. Returns TRUE if it does, FALSE otherwise
   */
  bool exists( int key );

  /** Opens the database 'pathWithFile' using option 'flag'. Options for flag are:
   *<PRE>
   * dbReadWrite = 0
   * dbRead = 1
   * dbNotOpen = 2
   * </PRE>
   * If you choose dbNotOpen, the method will return FALSE. If the DB could be opened
   * according to 'flag', TRUE will be returned, otherwise FALSE will be returned
   */
  bool openDB(QString pathWithFile, dbOpenType flag = dbOpenReadWrite);

  /** Closes the database. Always call this so largestKey will be written to the databse.
   *
   * If the databse is open when this class is deleted, this will be called automatically.
   */
  bool closeDB();

  /** Stores entry into the databse. How it does it depends on both the insertion and storage
   * policies. Returns TRUE if the storage goes ok, FALSE otherwise.
   *
   * see also: setInsertionPolicy(), setStoragePolicy()
   */
  bool storeEntry(DBEntry* entry); 

  /** Returns the string stored at 'key'. This allows you to quickly grab information
   * from the database without loading a full entry. If 'key' doesn't exist, an empty
   * QString is returned
   */
  QString fetchString(int key);

  /** Returns the int stored at 'key'. This allows you to quickly grab information
   * from the database without loading a full entry. If 'key' doesn't exist, an -1
   * is returned
   */
  int fetchInt(int key);

  /** Deletes the entry at 'key'. Remember that 'key' must be a multiple of 100. Returns TRUE
   * if item was found and deleted, FALSE otherwise
   */
  bool deleteEntry(int key);

  /** Sets the insertion policy. Choices are:
   *
   * dbInsertReplace : Replaces the information at the key
   * dbInsertNoReplace : Will return an error if you attempt to store at a currently existing key
   */
  insertionPolicy setInsertionPolicy(insertionPolicy policy);

  /** Sets the storage policy. Choiuces are:
   *<PRE>
   * dbStorageAppend : Appends the new entry to the end of the list, after largestKey.
   * dbStorageInsert : Inserts the new entry at the first available key
   *</PRE>
   */
 storagePolicy setStoragePolicy(storagePolicy policy);

 /** Sets the largest key to 'key'. This would be needed if for instance you needed
   * to delete the last key. You would need to be sure you set the largest key to
   * the key before the now deleted key
   */
  void setLargestKey(int key) { fLargestKey = key; }; 
 
protected:
 int getAvailableKey();

 

  bool storeString(QString string, int keyNum, insertionPolicy policy = dbInsertReplace);

private:
  /** Stores the insertion policy
   */
  insertionPolicy fInsertionPolicy;

  /** Stores the storage policy
   */
  storagePolicy fStoragePolicy;

  /** The actual databse class. This is one level above the DB
   */
  Database* fDatabase;

  int fLargestKey;
  
};

#endif
