   /*

    $Id: kedit.cpp,v 1.110 2000/04/08 21:42:41 waba Exp $

    Copyright (C) 1997 Bernd Johannes Wuebben
                       wuebben@math.cornell.edu

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include <qdragobject.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtabdialog.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qcolor.h>

#include <kapp.h>
#include <kcolordlg.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ksavefile.h>
#include <kstdaccel.h>
#include <kspell.h>
#include <ksconfig.h>
#include <keditcl.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <kedittoolbar.h>
#include <kkeydialog.h>

#include "kedit.h"
#include "mail.h"
#include "optiondialog.h"
#include "urldlg.h"
#include "version.h"
#include "kspeech.h"
#include "kwidgetprobe.h"

#include <kaction.h>
#include <kstdaction.h>

QList<TopLevel> TopLevel::windowList;

int default_open =  TopLevel::OPEN_READWRITE;

TopLevel::TopLevel (QWidget *, const char *name)
    : KTMainWindow ( name)
{
  mOptionDialog = 0;
  kspell = 0;
  newWindow = false;

  windowList.setAutoDelete( FALSE );
  windowList.append( this );

  statusbar_timer = new QTimer(this);
  connect(statusbar_timer, SIGNAL(timeout()),this,SLOT(timer_slot()));

  connect(kapp,SIGNAL(kdisplayPaletteChanged()),this,SLOT(set_colors()));

  setupStatusBar();
  setupActions();

  readSettings();

  setupEditWidget();
  set_colors();

  setAcceptDrops(true);

  resize(editor_width,editor_height);
  setFileCaption();
}


TopLevel::~TopLevel ()
{
  windowList.remove( this );
  delete mOptionDialog;
}

QString TopLevel::name()
{
  if (location.isEmpty())
     return "[New Document]";
  return location;
}


void TopLevel::setupEditWidget()
{
  eframe = new KEdit (this, "eframe");

  connect(eframe, SIGNAL(CursorPositionChanged()),this,
	  SLOT(statusbar_slot()));
  connect(eframe, SIGNAL(toggle_overwrite_signal()),this,
	  SLOT(toggle_overwrite()));
  connect(eframe, SIGNAL(gotUrlDrop(QDropEvent*)), this,
	  SLOT(urlDrop_slot(QDropEvent*)));

  setView(eframe,FALSE);
  eframe->setMinimumSize(200, 100);

  if( mOptionState.misc.wrapMode == SMiscState::fixedColumnWrap )
  {
    eframe->setWordWrap(QMultiLineEdit::FixedColumnWidth);
    eframe->setWrapColumnOrWidth(mOptionState.misc.wrapColumn);
  }
  else if( mOptionState.misc.wrapMode == SMiscState::dynamicWrap )
  {
    eframe->setWordWrap(QMultiLineEdit::WidgetWidth);
  }
  else
  {
    eframe->setWordWrap(QMultiLineEdit::NoWrap);
  }
  eframe->setFont( mOptionState.font.font );
  eframe->setModified(false);

  setSensitivity();

  eframe->setFocus();

  /*
  right_mouse_button = new QPopupMenu;

  right_mouse_button->insertItem (i18n("Open..."),
				  this, 	SLOT(file_open()));
  right_mouse_button->insertItem (i18n("Save"),
				  this, 	SLOT(file_save()));
  right_mouse_button->insertItem (i18n("Save as..."),
				  this, SLOT(file_save_as()));
  right_mouse_button->insertSeparator(-1);
  right_mouse_button->insertItem(i18n("Copy"),
				 this, SLOT(copy()));
  right_mouse_button->insertItem(i18n("Paste"),
				 this, SLOT(paste()));
  right_mouse_button->insertItem(i18n("Cut"),
				 this, SLOT(cut()));
  right_mouse_button->insertItem(i18n("Select All"),
				 this, SLOT(select_all()));
  eframe->installRBPopup(right_mouse_button);
  */
}


void TopLevel::setupActions()
{

    // setup File menu
    KStdAction::openNew(this, SLOT(file_new()), actionCollection());
    KStdAction::open(this, SLOT(file_open()), actionCollection());
    recent = KStdAction::openRecent(this, SLOT(openRecent(const KURL&)),
                                          actionCollection());
    KStdAction::save(this, SLOT(file_save()), actionCollection());
    KStdAction::saveAs(this, SLOT(file_save_as()), actionCollection());
    KStdAction::close(this, SLOT(file_close()), actionCollection());
    KStdAction::print(this, SLOT(print()), actionCollection());
    KStdAction::mail(this, SLOT(mail()), actionCollection());
    KStdAction::quit(this, SLOT(close()), actionCollection());

    // setup edit menu
    KStdAction::undo(this, SLOT(undo()), actionCollection());
    KStdAction::redo(this, SLOT(redo()), actionCollection());
    KStdAction::cut(this, SLOT(cut()), actionCollection());
    KStdAction::copy(this, SLOT(copy()), actionCollection());
    KStdAction::paste(this, SLOT(paste()), actionCollection());
    KStdAction::selectAll(this, SLOT(select_all()), actionCollection());
    KStdAction::find(this, SLOT(search()), actionCollection());
    KStdAction::findNext(this, SLOT(search_again()), actionCollection());
    KStdAction::replace(this, SLOT(replace()), actionCollection());

    (void)new KAction(i18n("&Insert File"), 0, this, SLOT(file_insert()),
                      actionCollection(), "insert_file");
    (void)new KAction(i18n("I&nsert Date"), 0, this, SLOT(insertDate()),
                      actionCollection(), "insert_date");
    (void)new KAction(i18n("Cl&ean Spaces"), 0, this, SLOT(clean_space()),
                      actionCollection(), "clean_spaces");

    // setup Tools menu
    KStdAction::spelling(this, SLOT(spellcheck()), actionCollection());
    (void)new KAction(i18n("&Read All"), 0, this, SLOT(readText()),
                      actionCollection(), "read_text");
    (void)new KAction(i18n("&Read Selection"), 0, this, SLOT(readSelection()),
                      actionCollection(), "read_selection");
    (void)new KAction(i18n("&Stop Reading"), 0, this, SLOT(stopReading()),
                      actionCollection(), "stop_reading");
    (void)new KAction(i18n("Start &Talker"), 0, this, SLOT(startTalker()),
                      actionCollection(), "start_talker");
    (void)new KAction(i18n("Stop Talker"), 0, this, SLOT(stopTalker()),
                      actionCollection(), "stop_talker");

    // setup Go menu
    KStdAction::gotoLine(this, SLOT(gotoLine()), actionCollection());

    // setup Settings menu
    KStdAction::showToolbar(this, SLOT(toggleToolBar()), actionCollection());
    KStdAction::showStatusbar(this, SLOT(toggleStatusBar()), actionCollection());

    KStdAction::saveOptions(this, SLOT(save_options()), actionCollection());
    KStdAction::preferences(this, SLOT(customize()), actionCollection());

    KStdAction::keyBindings(this, SLOT(editKeys()), actionCollection());
    KStdAction::configureToolbars(this, SLOT(editToolbars()), actionCollection());

    createGUI();
}

void TopLevel::setupStatusBar(){

  KStatusBar *statusbar = new KStatusBar( this );
  statusbar->insertItem("Line:000000 Col: 000", ID_LINE_COLUMN);
  statusbar->insertItem("XXX", ID_INS_OVR);
  statusbar->insertItem("", ID_GENERAL);

  statusbar->changeItem("Line: 1 Col: 1", ID_LINE_COLUMN);
  statusbar->changeItem("INS", ID_INS_OVR);
  statusbar->changeItem("", ID_GENERAL);

  //statusbar->setInsertOrder(KStatusBar::RightToLeft);
  //statusbar->setAlignment(ID_INS_OVR,AlignCenter);

  //    statusbar->setInsertOrder(KStatusBar::LeftToRight);
  //    statusbar->setBorderWidth(1);

  setStatusBar( statusbar );

}




void TopLevel::saveProperties(KConfig* config)
{
  if(location.isEmpty() && !eframe->isModified())
    return;

  config->writeEntry("filename",name());
  config->writeEntry("modified",eframe->isModified());

  if(eframe->isModified())
  {
    QString tmplocation = kapp->tempSaveName(name());
    saveFile(tmplocation);
  }
}


void TopLevel::readProperties(KConfig* config){

    QString filename = config->readEntry("filename","");
    int modified = config->readNumEntry("modified",0);

    if(!filename.isEmpty() && modified){
        bool ok;

        QString fn = kapp->checkRecoverFile(filename,ok);

	if(ok){
	  openFile(fn, OPEN_READWRITE);
	  location = filename;
	  eframe->setModified();
	  setFileCaption();
	}
    }
    else{

      if(!filename.isEmpty()){
	openFile(filename, OPEN_READWRITE);
        location = filename;
        eframe->setModified(false);
        setFileCaption();
      }
    }
}


void TopLevel::undo()
{
  eframe->undo();
}


void TopLevel::redo()
{
  eframe->redo();
}


void TopLevel::copy()
{
  eframe->copyText();
}


void TopLevel::select_all()
{
  eframe->selectAll();
}




void TopLevel::insertDate(){

  int line, column;

  QString string;
  QDate dt = QDate::currentDate();
  string = KGlobal::locale()->formatDate(dt);

  eframe->getCursorPosition(&line,&column);
  eframe->insertAt(string,line,column);
  eframe->setModified(TRUE);

  statusbar_slot();
}

void TopLevel::paste(){

 eframe->paste();
 eframe->setModified(TRUE);

 statusbar_slot();
}

void TopLevel::cut(){

 eframe->cut();
 eframe->setModified(TRUE);
 statusbar_slot();

}

void TopLevel::file_new()
{
  if (!location.isEmpty() || eframe->isModified())
  {
     TopLevel *t = new TopLevel ();
     t->show();
     t->file_new();
     return;
  }
  eframe->clear();
  eframe->setModified(true);
  location = QString::null;
  setFileCaption();
  statusbar_slot();
}

void TopLevel::clean_space()
{
   if (!eframe);
   eframe->cleanWhiteSpace();
}

void TopLevel::readText()
{
  if (!eframe) return;

  statusBar()->changeItem("Reading:  Started.", ID_GENERAL);

  KSpeech *synth = KSpeech::getSynth();
  synth->read( eframe->text() );
}

void TopLevel::readSelection()
{
  if (!eframe) return;

  statusBar()->changeItem("Reading selection:  Started.", ID_GENERAL);

  KSpeech *synth = KSpeech::getSynth();
  synth->read( eframe->markedText() );
}

void TopLevel::stopReading()
{
  if (!eframe) return;

  statusBar()->changeItem("Reading:  Stopped.", ID_GENERAL);

  KSpeech *synth = KSpeech::getSynth();
  synth->stop();
}

void TopLevel::startTalker()
{
  probe = new KWidgetProbe( this );
}

void TopLevel::stopTalker()
{
  delete probe;
  probe = 0;
}

void TopLevel::spellcheck()
{
  if (!eframe) return;

  if (kspell) return; // In progress

  statusBar()->changeItem("Spellcheck:  Started.", ID_GENERAL);

  kspell = new KSpell(this, i18n("Spellcheck"), this,
	SLOT( spell_started(KSpell *)), &mOptionState.spell.config);

  connect (kspell, SIGNAL ( death()),
          this, SLOT ( spell_finished( )));

  connect (kspell, SIGNAL (progress (unsigned int)),
          this, SLOT (spell_progress (unsigned int)));
  connect (kspell, SIGNAL (misspelling (QString, QStringList *, unsigned)),
          eframe, SLOT (misspelling (QString, QStringList *, unsigned)));
  connect (kspell, SIGNAL (corrected (QString, QString, unsigned)),
          eframe, SLOT (corrected (QString, QString, unsigned)));
  connect (kspell, SIGNAL (done(const char *)),
		 this, SLOT (spell_done(const char *)));
}

void TopLevel::spell_started( KSpell *)
{
   eframe->spellcheck_start();
   kspell->setProgressResolution(2);
   kspell->check(eframe->text());
}

void TopLevel::spell_progress (unsigned int percent)
{
  QString s;
  s = i18n("Spellcheck:  %1% complete").arg(percent);

  statusBar()->changeItem (s, ID_GENERAL);
}

void TopLevel::spell_done(const char *newtext)
{
  eframe->spellcheck_stop();
  if (kspell->dlgResult() == 0)
  {
     eframe->setText( newtext);
     statusBar()->changeItem (i18n("Spellcheck:  Aborted"), ID_GENERAL);
  }
  else
  {
     statusBar()->changeItem (i18n("Spellcheck:  Complete"), ID_GENERAL);
  }
  kspell->cleanUp();
}

void TopLevel::spell_finished( )
{
  KSpell::spellStatus status = kspell->status();
  delete kspell;
  kspell = 0;
  if (status == KSpell::Error)
  {
     KMessageBox::sorry(this, i18n("ISpell could not be started.\n"
     "Please make sure you have ISpell properly configured and in your PATH."));
  }
  else if (status == KSpell::Crashed)
  {
     eframe->spellcheck_stop();
     statusBar()->changeItem (i18n("Spellcheck:  Crashed"), ID_GENERAL);
     KMessageBox::sorry(this, i18n("ISpell seems to have crashed."));
  }
}

void TopLevel::file_open( void )
{
  while( 1 )
  {
    KURL url = KFileDialog::getOpenURL( QString::null, "*", this );
    if( url.isEmpty() )
    {
      return;
    }

    TopLevel *toplevel;
    if( location.isEmpty() == false || eframe->isModified() == true )
    {
      toplevel = new TopLevel();
      if( toplevel == 0 )
      {
	return;
      }
    }
    else
    {
      toplevel = this;
    }

    QString tmpfile;
    KIO::NetAccess::download( url, tmpfile );
    int result = toplevel->openFile( tmpfile, 0 );
    KIO::NetAccess::removeTempFile( tmpfile );

    if( result == KEDIT_OK )
    {
      if( toplevel != this ) { toplevel->show(); }
      toplevel->location = url.path();
      toplevel->setFileCaption();
      recent->addURL( url );
      toplevel->eframe->setModified(false);
      toplevel->setGeneralStatusField(i18n("Done"));
      toplevel->statusbar_slot();
      break;
    }
    else if( result == KEDIT_RETRY )
    {
      if( toplevel != this ) { delete toplevel; }
    }
    else
    {
      if( toplevel != this ) { delete toplevel; }
      break;
    }
  }

}

void TopLevel::file_insert()
{
/*
#warning for now...
  bool tryAgain = false;
  do {
     QString fname = KFileDialog::getOpenFileName(QString::null, "*");
     if (fname.isEmpty())
        return; // Cancel

     loading_slot();
     int result = openFile(fname, OPEN_INSERT);
     switch (result)
     {
     case KEDIT_OK :	
       {
         setGeneralStatusField(i18n("Done"));
         eframe->setModified(false);
         add_recent_file(fname);
         break;
       }

     case KEDIT_RETRY :
       {
         tryAgain = true;
         break;
       }
     case KEDIT_OS_ERROR :
       {
         break;
       }
     default:
       break;
     }
  } while (!tryAgain);
  statusbar_slot();
*/
}

bool TopLevel::queryExit()
{
  writeSettings();
  return true;
}

bool TopLevel::queryClose()
{
  int result;
  if( !eframe->isModified() )
  {
     return true;
  }

  QString msg = i18n(""
        "This document has been modified.\n"
        "Would you like to save it?" );
  switch( KMessageBox::warningYesNoCancel( this, msg ) )
  {
     case KMessageBox::Yes: // Save, then exit
	  result = saveFile(location);
	  if( result == KEDIT_USER_CANCEL )
	  {
	    return false;
	  }
	  if( result != KEDIT_OK)
	  {
	    msg = i18n(""
	      "Could not save the file.\n"
	      "Exit anyways?");
	    switch( KMessageBox::warningYesNo( this, msg ) )
	    {
	      case KMessageBox::Yes:
	        return true;

	      case KMessageBox::No:
              default:
                return false;
	    }
	  }
          return true;

     case KMessageBox::No: // Don't save but exit.
          return true;

     case KMessageBox::Cancel: // Don't save and don't exit.
     default:
	  break; // Don't exit...
  }
  return false;
}



void TopLevel::openRecent(const KURL& url)
{
  if (!url.isEmpty() || eframe->isModified())
  {
     TopLevel *t = new TopLevel ();
     t->show();
     t->openRecent(url);
     return;
  }
  openURL( url, OPEN_READWRITE );
}

void TopLevel::file_close()
{
  if( eframe->isModified() )
  {
    QString msg = i18n("This document has been modified.\n"
                       "Would you like to save it?" );
    switch( KMessageBox::warningYesNoCancel( this, msg ) )
    {
      case KMessageBox::Yes: // Save, then close
        file_save();
        if (eframe->isModified())
           return; // Error during save.
      break;

      case KMessageBox::No: // Don't save but close.
      break;

      case KMessageBox::Cancel: // Don't save and don't close.
	return;
      break;
    }
  }
  eframe->clear();
  eframe->setModified(true);
  location = QString::null;
  setFileCaption();
  statusbar_slot();
}


void TopLevel::file_save()
{
  if (location.isEmpty())
  {
     file_save_as();
     return;
  }
  if ( eframe->isModified() )
  {
      KURL u( location );
      if ( !u.isMalformed() && u.protocol() != "file" )
      {
	
	  url_location = location;
	  saveURL( u );
	  statusbar_slot();
	  return;
      }

      int result = KEDIT_OK;

      result =  saveFile(location); // error messages are handled by saveFile

      if ( result == KEDIT_OK ){
	  QString string;
	  string = i18n("Wrote: %1").arg(location);
	  setGeneralStatusField(string);
      }
  }
  else
  {
      setGeneralStatusField(i18n("No changes need to be saved"));
  }
}

void TopLevel::setGeneralStatusField(const QString &text){

    statusbar_timer->stop();

    statusBar()->changeItem(text,ID_GENERAL);
    statusbar_timer->start(10000,TRUE); // single shot

}


void TopLevel::file_save_as()
{
  KURL u = KFileDialog::getSaveURL(location, "*");
  if (u.isEmpty())
     return;

  int result = saveURL(u); // error messages are handled by saveFile

  if ( result == KEDIT_OK )
    {
      location = u.url();
      setFileCaption();
      QString string = i18n("Saved as: %1").arg(location);
      setGeneralStatusField(string);
    }
}



void TopLevel::mail()
{
  Mail *maildlg = new Mail(this,"maildialog");
  if( maildlg->exec() == QDialog::Rejected )
  {
    delete maildlg;
    return;
  }

  kapp->processEvents();
  kapp->flushX();

  QString cmd;
  cmd = cmd.sprintf(mOptionState.misc.mailCommand.ascii(),
    maildlg->getSubject().ascii(),maildlg->getRecipient().ascii());

  delete maildlg;

  FILE* mailpipe = popen(cmd.ascii(),"w");
  if( mailpipe == NULL )
  {
    QString msg = i18n(""
      "Could not pipe the contents"
      "of this document into:\n %1").arg(cmd);
    KMessageBox::sorry( this, msg );
    return;
  }

  QTextStream t(mailpipe,IO_WriteOnly );
  int line_count = eframe->numLines();

  for(int i = 0 ; i < line_count ; i++){
    t << eframe->textLine(i) << '\n';
  }
  pclose(mailpipe);
}

/*
void TopLevel::fancyprint(){

  QPrinter prt;
  char buf[200];
  if ( prt.setup(0) ) {

    int y =10;
    QPainter p;
    p.begin( &prt );
    p.setFont( eframe->font() );
    QFontMetrics fm = p.fontMetrics();

    int numlines = eframe->numLines();
    for(int i = 0; i< numlines; i++){
      y += fm.ascent();
      QString line;
      line = eframe->textLine(i);
      line.replace( QRegExp("\t"), "        " );
      strncpy(buf,line.local8Bit(),160);
      for (int j = 0 ; j <150; j++){
	if (!isprint(buf[j]))
	    buf[j] = ' ';
      }
      buf[line.length()] = '\0';
      p.drawText( 10, y, buf );
      y += fm.descent();
    }

    p.end();
  }
  return ;
}
*/

void TopLevel::helpselected(){

  kapp->invokeHTMLHelp( "" , "" );

}

void TopLevel::search(){

      eframe->search();
      statusbar_slot();
}

void TopLevel::replace(){

      eframe->replace();
      statusbar_slot();
}


void TopLevel::customize( void )
{
  if( mOptionDialog == 0 )
  {
    mOptionDialog = new COptionDialog( topLevelWidget(), 0, false );
    if( mOptionDialog == 0 ) { return; }
    connect( mOptionDialog, SIGNAL(fontChoice(const SFontState &)),
	     this, SLOT(setFontOption(const SFontState &)) );
    connect( mOptionDialog, SIGNAL(colorChoice(const SColorState &)),
	     this, SLOT(setColorOption(const SColorState &)) );
    connect( mOptionDialog, SIGNAL(spellChoice(const SSpellState &)),
	     this, SLOT(setSpellOption(const SSpellState &)) );
    connect( mOptionDialog, SIGNAL(miscChoice(const SMiscState &)),
	     this, SLOT(setMiscOption(const SMiscState &)) );
  }
  if( mOptionDialog->isVisible() == false )
  {
    mOptionDialog->setState( mOptionState );
  }

  mOptionDialog->show();
}

void TopLevel::editKeys()
{
  KKeyDialog::configureKeys(actionCollection(), xmlFile());
}

void TopLevel::editToolbars()
{
  KEditToolbar dlg(actionCollection());

  if (dlg.exec())
    createGUI();
}

void TopLevel::setFontOption( const SFontState &font )
{
  mOptionState.font = font;
  eframe->setFont( mOptionState.font.font );
}


void TopLevel::setColorOption( const SColorState &color )
{
  mOptionState.color = color;
  set_colors();
}


void TopLevel::setSpellOption( const SSpellState &spell )
{
  mOptionState.spell = spell;
}


void TopLevel::setMiscOption( const SMiscState &misc )
{
  mOptionState.misc = misc;
  if( mOptionState.misc.wrapMode == SMiscState::fixedColumnWrap )
  {
    eframe->setWordWrap(QMultiLineEdit::FixedColumnWidth);
    eframe->setWrapColumnOrWidth(mOptionState.misc.wrapColumn);
  }
  else if( mOptionState.misc.wrapMode == SMiscState::dynamicWrap )
  {
    eframe->setWordWrap(QMultiLineEdit::WidgetWidth);
  }
  else
  {
    eframe->setWordWrap(QMultiLineEdit::NoWrap);
  }

  //eframe->saveBackupCopy(mOptionState.misc.backupCheck);
}



void TopLevel::toggleToolBar()
{
  if(toolBar()->isVisible())
    toolBar()->hide();
  else
    toolBar()->show();
}	


void TopLevel::toggleStatusBar()
{
  if (statusBar()->isVisible())
    statusBar()->hide();
  else
    statusBar()->show();
}


void TopLevel::search_again()
{
  eframe->repeatSearch();
  statusbar_slot();
}


void TopLevel::setFileCaption(){

  setCaption(name());
}


void TopLevel::gotoLine() {
	eframe->doGotoLine();
}

void TopLevel::statusbar_slot(){

  QString linenumber;

  linenumber = i18n("Line: %1 Col: %2")
		     .arg(eframe->currentLine() + 1)
		     .arg(eframe->currentColumn() +1);

  statusBar()->changeItem(linenumber,ID_LINE_COLUMN);
}



void TopLevel::print()
{
  QString command;
  QString com;
  QString file;

  int result;

  if( eframe->isModified() )
  {
    QString msg = i18n(""
      "The current document has been modified.\n"
      "Would you like to save the changes before\n"
      "printing this document?");
    switch( KMessageBox::warningYesNoCancel( this, msg ) )
    {
      case KMessageBox::Yes: // Save, then print
 	result = saveFile(location);
	if( result == KEDIT_USER_CANCEL )
	{
	  return;
	}
	if( result != KEDIT_OK)
	{
	  msg = i18n(""
	    "Could not save the file.\n"
	    "Print anyways?");
	  switch( KMessageBox::warningYesNo( this, msg ) )
	  {
	    case KMessageBox::Yes:
	    break;

	    case KMessageBox::No:
	      return;
	    break;
	  }
	}
      break;

      case KMessageBox::No: // Don't save but print.
      break;

      case KMessageBox::Cancel: // Don't save and don't print.
	return;
      break;
    }
  }

  PrintDialog *printDialog = new PrintDialog(this,"print",true);
  printDialog->setWidgets(pi);

  if(printDialog->exec() == QDialog::Rejected )
  {
    delete printDialog;
    return;
  }

  pi = printDialog->getCommand();
  delete printDialog;

  if( location.isEmpty() )
  {
    //
    // we go through all of this so that we can print an "Untitled" document
    // quickly without going through the hassle of saving it. This will
    // however result in a temporary filename and your printout will
    // usually show that temp name, such as /tmp/00432aaa
    // for a non "untitled" document we don't want that to happen so
    // we asked the user to save before we print the document.
    //
    // TODO find a smarter solution for the above!
    //

    QString tmpname = tmpnam(NULL);
    QFile file(tmpname);
    file.open(IO_WriteOnly);


    if(pi.selection)
    {
      if(file.writeBlock(eframe->markedText().ascii(),
			 eframe->markedText().length()) == -1)
      {
	result = KEDIT_OS_ERROR;
      }
      else
      {
	result = KEDIT_OK;
      }
    }
    else
    {
      if(file.writeBlock(eframe->text().ascii(),
			 eframe->text().length()) == -1)
      {
	result = KEDIT_OS_ERROR;
      }
      else
      {
	result = KEDIT_OK;
      }	
    }

    file.close();
    // TODO error handling

    if (pi.raw)
    {
      command = "lpr";
    }	
    else
    {
      command = pi.command;
    }

    com = QString("%1 %2 ; rm %3 &").arg(command).arg(tmpname).arg(tmpname);

    system(com.ascii());
    QString string;
    if(pi.selection)
      string = i18n("Printing: %1 Untitled (Selection)").arg(command);
    else
      string = i18n("Printing: %1 Untitled").arg(command);
    setGeneralStatusField(string);
  }
  else // document name != Untiteled
  {
    QString tmpname = tmpnam(NULL);
    if( pi.selection )
    {
      // print only the selection
      QFile file(tmpname);
      file.open(IO_WriteOnly);


      if(file.writeBlock(eframe->markedText().ascii(),
			 eframe->markedText().length()) == -1)
      {
	result = KEDIT_OS_ERROR;
      }
      file.close();
      // TODO error handling
    }

    if(pi.raw)
    {
      command = "lpr";
    }	
    else
    {
      command = pi.command;
    }

    if(!pi.selection) // print the whole file
    {
      com = QString ("%s '%s' &").arg(command).arg(location);
      system(com.ascii());
      QString string;	
      string = i18n("Printing: %1").arg(com);
      setGeneralStatusField(string);
    }
    else // print only the selection
    {
      com = QString ("%1 %2 ; rm %3 &").arg(command).arg(tmpname).arg(tmpname);
      system(com.ascii());
      QString string = i18n("Printing: %1 %2 (Selection)")
	.arg(command).arg(location);
      setGeneralStatusField(string);
    }
    //kdebug(0, 0, com.ascii());
  }
}



void TopLevel::setSensitivity (){

}

void TopLevel::save_options(){

  writeSettings();

}

int TopLevel::saveURL( const KURL& _url )
{
    if ( _url.isMalformed() )
    {
        KMessageBox::sorry(this, i18n("Malformed URL"));
        return KEDIT_RETRY;
    }

    // Just a usual file ?
    if ( _url.isLocalFile() )
    {
        return saveFile( _url.url() );
    }

    QString tmpFile;
    tmpFile = QString("/tmp/kedit%1").arg(time( 0L ));
    eframe->setModified( true );
    saveFile( tmpFile );

    if (KIO::NetAccess::upload( tmpFile, _url ) == false)
    {
      KMessageBox::error(this, "Could not save remote file"); 
      return KEDIT_RETRY;
    }

    return true;
}

int TopLevel::openFile( const QString& _url, int _mode )
{
  location=_url;
  setFileCaption();
  KURL *u = new KURL( _url );
  if ( u->isMalformed() )
  {
     KMessageBox::sorry(this, i18n("This is not a valid filename.\n"));
     return KEDIT_RETRY;
  }
  if ( !u->isLocalFile() )
  {
     KMessageBox::sorry(this, i18n("This is not a local file.\n"));
     return KEDIT_RETRY;
  }

  QFileInfo info(u->path());

  if(!info.exists())
  {
     KMessageBox::sorry(this, i18n("The specified file does not exist"));
     return KEDIT_RETRY;
  }

  if(info.isDir())
  {
     KMessageBox::sorry(this, i18n("You have specified a directory"));
     return KEDIT_RETRY;
  }

  QFile file(u->path());

  if(!file.open(IO_ReadOnly))
  {
     KMessageBox::sorry(this, i18n("You do not have read permission to this file."));
     return KEDIT_RETRY;
  }

  QTextStream stream(&file);
  if ((_mode & OPEN_INSERT) == 0)
  {
     eframe->clear();
  }
  eframe->insertText( &stream );
  eframe->setModified(false);
  return KEDIT_OK;

}

int TopLevel::saveFile( const QString& _url )
{
  KURL *u = new KURL( _url );
  if ( u->isMalformed() )
  {
     KMessageBox::sorry(this, i18n("This is not a valid filename.\n"));
     return KEDIT_RETRY;
  }
  if ( !u->isLocalFile() )
  {
     KMessageBox::sorry(this, i18n("This is not a local file.\n"));
     return KEDIT_RETRY;
  }

  QFileInfo info(u->path());

  if(info.isDir())
  {
     KMessageBox::sorry(this, i18n("You have specified a directory"));
     return KEDIT_RETRY;
  }

  KSaveFile file(u->path());

  if(file.status() != 0)
  {
     KMessageBox::sorry(this, i18n("Unable to write to file."));
     return KEDIT_RETRY;
  }

  eframe->saveText( file.textStream() );
  if (!file.close())
  if(file.status() != 0)
  {
     KMessageBox::sorry(this, i18n("Could not save file."));
     return KEDIT_RETRY;
  }
  eframe->setModified(false);
  return KEDIT_OK;
}


void TopLevel::openURL( const KURL& _url, int _mode )
{
    qDebug("TopLevel::openURL: %s", _url.url().ascii());
    QString netFile = _url.url();
    if ( _url.isMalformed() )
    {
        QString string;
        string = i18n( "Malformed URL\n%1").arg(netFile);

        KMessageBox::sorry(this, string);
        return;
    }

    QString target;
    if (KIO::NetAccess::download(_url, target) == false)
    {
      // this needs to be handled within KIO sooooon
      KMessageBox::error(this, i18n("Cannot download file!"));
      return;
    }

    int result = openFile(target, 0);
    if (result == KEDIT_OK)
    {
        location = netFile;
        setFileCaption();
        recent->addURL(_url);
        eframe->setModified(false);
        setGeneralStatusField(i18n("Done"));
    }
}

void TopLevel::urlDrop_slot(QDropEvent* e) {

  dropEvent(e);
}

void TopLevel::dragEnterEvent(QDragEnterEvent* e)
{
  e->accept(QUriDrag::canDecode(e));
}

void TopLevel::dropEvent(QDropEvent* e)
{

    QStrList list;

    // This should never happen, but anyway...
    if(!QUriDrag::decode(e, list))
        return;

    char *s;

    for ( s = list.first(); s != 0L; s = list.next() )
    {
	// Load the first file in this window
	if ( s == list.getFirst() && !eframe->isModified() )
	{
            openURL( KURL(s), OPEN_READWRITE );
	}
	else
	{
	    setGeneralStatusField(i18n("New Window"));
	    TopLevel *t = new TopLevel ();
	    t->show ();
	    setGeneralStatusField(i18n("New Window Created"));
		t->openURL( KURL(s), OPEN_READWRITE );
	    setGeneralStatusField(i18n("Load Command Done"));
	}
    }
}

void TopLevel::timer_slot(){

  statusBar()->changeItem("",ID_GENERAL);

}



void TopLevel::set_colors(){


  QPalette mypalette = (eframe->palette()).copy();

  QColorGroup cgrp = mypalette.normal();
  QColorGroup ncgrp( mOptionState.color.textFg, cgrp.background(),
    cgrp.light(),cgrp.dark(),cgrp.mid(), mOptionState.color.textFg,
    mOptionState.color.textBg );

  mypalette.setNormal(ncgrp);
  mypalette.setDisabled(ncgrp);
  mypalette.setActive(ncgrp);

  eframe->setPalette(mypalette);
  eframe->setBackgroundColor(mOptionState.color.textBg);

}


void TopLevel::readSettings( void )
{
  //
  // Default settings. The values in mOptionState is defined in
  // optionstate.h
  //
  editor_width = 550;
  editor_height = 400;
  show_statusbar = TRUE;
  show_toolbar = TRUE;
  pi.command = "enscript -2rG";
  pi.raw = 1;
  pi.selection = 0;

  QString str;
  config = kapp->config();

  config->setGroup( "Text Font" );
  mOptionState.font.font = config->readFontEntry("KEditFont",
    &mOptionState.font.font);

  recent->loadEntries( config );

  config->setGroup("General Options");
  url_location = config->readEntry( "default_url",
    "ftp://localhost/welcome.msg" );

  mOptionState.misc.mailCommand = config->readEntry("MailCmd",
    mOptionState.misc.mailCommand );

  str = config->readEntry("Width");
  if( str.isNull() == false )
    editor_width = str.toInt();

  str = config->readEntry("Height");
  if( str.isNull() == false )
    editor_height = str.toInt();

  str = config->readEntry( "StatusBar" );
  if( str.isNull() == false && str.find( "off" ) == 0 )
  {
    show_statusbar = FALSE;
    enableStatusBar( KStatusBar::Hide );
  }
  else
  {
    show_statusbar = TRUE;
    enableStatusBar( KStatusBar::Show );
  }

  str = config->readEntry( "ToolBar" );
  if( str.isNull() == false && str.find( "off" ) == 0 )
  {
    show_toolbar = FALSE;
    enableToolBar( KToolBar::Hide );
  }
  else
  {
    show_toolbar = TRUE;
    enableToolBar( KToolBar::Show );
  }

  str = config->readEntry("WrapMode");
  if( str.isNull() == false )
    mOptionState.misc.wrapMode = str.toInt();

  str = config->readEntry("WrapColumn");
  if( str.isNull() == false )
    mOptionState.misc.wrapColumn = str.toInt();

  str = config->readEntry("BackupCopies");
  if( str.isNull() == false )
    mOptionState.misc.backupCheck = (bool) str.toInt();

  str = config->readEntry( "ForeColor" );
  if( str.isNull() == false )
    mOptionState.color.textFg.setNamedColor( str );

  str = config->readEntry( "BackColor" );
  if( str.isNull() == false )
    mOptionState.color.textBg.setNamedColor( str );

  ///////////////////////////////////////////////////

  config->setGroup("Printing");
  str = config->readEntry("PrntCmd1");
  if ( str.isNull() == false )
    pi.command = str;

  str = config->readEntry("PrintSelection");
  if ( str.isNull() == false )
    pi.selection = str.toInt();

  str = config->readEntry("PrintRaw");
  if ( str.isNull() == false )
    pi.raw = str.toInt();

}

void TopLevel::writeSettings( void )
{
  config = kapp->config();

  config->setGroup( "Text Font" );
  config->writeEntry("KEditFont", mOptionState.font.font );

  recent->saveEntries( config );

  config->setGroup("General Options");
  config->writeEntry("default_url", url_location);
  config->writeEntry("Width", QString().setNum(width()) );
  config->writeEntry("Heigh", QString().setNum(height()) );

  config->writeEntry( "StatusBar", show_statusbar ? "on" : "off" );
  config->writeEntry( "ToolBar", show_toolbar ? "on" : "off" );
  config->writeEntry("MailCmd", mOptionState.misc.mailCommand );

  QString string;
  string.setNum( mOptionState.misc.wrapMode );
  config->writeEntry("WrapMode", string);

  string.setNum( mOptionState.misc.wrapColumn );
  config->writeEntry("WrapColumn", string);

  string="";
  string.setNum( (int) mOptionState.misc.backupCheck );
  config->writeEntry("BackupCopies", string);

  QColor &fg = mOptionState.color.textFg;
  string.sprintf("#%02x%02x%02x", fg.red(), fg.green(), fg.blue() );
  config->writeEntry( "ForeColor", string );

  QColor &bg = mOptionState.color.textBg;
  string.sprintf("#%02x%02x%02x", bg.red(),bg.green(), bg.blue());
  config->writeEntry( "BackColor", string );
	
  ////////////////////////////////////////////////////

  config->setGroup("Printing");

  config->writeEntry("PrntCmd1", pi.command);

  string.setNum( pi.selection );
  config->writeEntry("PrintSelection", string);

  string.setNum( pi.raw );
  config->writeEntry("PrintRaw", string);

  config->sync();
}



void TopLevel::toggle_overwrite(){

  if(eframe->isOverwriteMode()){
    statusBar()->changeItem("OVR",ID_INS_OVR);
  }
  else{
    statusBar()->changeItem("INS",ID_INS_OVR);
  }

}

static const char *description=I18N_NOOP("A KDE Text Editor");

static const KCmdLineOptions options[] =
{
	{ "+file", I18N_NOOP("File or URL to Open"), 0 },
	{ 0, 0, 0}
};

int main (int argc, char **argv)
{
	bool have_top_window = false;

	KAboutData aboutData( "ktalkedit", I18N_NOOP("KTalkEdit"),
		KEDITVERSION, description, KAboutData::License_GPL,
		"(c) 1997, Bernd Johannes Wuebben");
	aboutData.addAuthor("Bernd Johannes Wuebben",0, "wuebben@math.cornell.edu");
	KCmdLineArgs::init( argc, argv, &aboutData );
	KCmdLineArgs::addCmdLineOptions( options );

	KApplication a;
	//CT KIO::Job::initStatic();

	if ( a.isRestored() )
	{
		int n = 1;

		while (KTMainWindow::canBeRestored(n))
		{
			TopLevel *tl = new TopLevel();
			tl->restore(n);
			n++;
			have_top_window = true;
		}
	}
	else
	{
		have_top_window = false;
		KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
	
		for(int i = 0; i < args->count(); i++)
		{
			TopLevel *t = new TopLevel;
			t->show ();
			have_top_window = true;

            KURL url = args->url(i);
			if( url.isLocalFile() )
			{
				// a normal file, let's see whether it exists. If it doesn't
				// exist try to create and open it
				QFileInfo info(url.path());
				if ( !info.exists())
				{
					QFile file(url.path());
	
					if(!file.open( IO_ReadWrite))
					{
						QString string = i18n("Can not create:\n%1").arg(url.path());
						KMessageBox::sorry(0, string);
					}
					file.close();
				}
				// Ok! f contains the filename, open it
				t->openFile(url.path(), default_open);	
			}
            else
				t->openURL( url, default_open );
		}
		args->clear();
	}
	
	if(!have_top_window)
	{
		TopLevel *t = new TopLevel ();
		t->show ();
	}

	return a.exec ();
}


#include "kedit.moc"

