/***************************************************************************
                          knd_pcap.cpp  -  description                              
                             -------------------                                         
    begin                : Thu Mar 25 14:26:46 GMT 1999
                                           
    copyright            : (C) 1999 by Mike Richardson                         
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include	<stdlib.h>
#include	<unistd.h>
#include	<fcntl.h>

#include	"knd_pcap.h"

#define		__FAVOR_BSD
#include	<net/if.h>
#include	<netinet/ip.h>
#include	<netinet/if_ether.h>
#include	<netinet/tcp.h>
#include	<netinet/udp.h>
#include	<netdb.h>

#define	STRERR	strerror(errno)

typedef	struct	ether_header	ETH_HDR	;
typedef	struct	ip		IP	;
typedef	struct	ether_arp	ETH_ARP	;
typedef	struct	tcphdr		TCP_HDR	;
typedef	struct	udphdr		UDP_HDR	;

static	PktInfo	*pktFree ;
static	int	cntFree	 ;
#define	MAXFREE	(MAXPACK*3)

/*  getPktInfo	: Allocate a PktInfo object				*/
/*  (returns)	: PktInfo *	: Allocated object			*/

PktInfo	*getPktInfo ()
{
	PktInfo	*pkt	;

	if (pktFree != NULL)
	{	pkt	= pktFree	;
		pktFree	= pktFree->next ;
		cntFree-=1 ;
	}
	else	pkt	= new PktInfo	;

	return	pkt	;
}

/*  freePktInfo	: Free a PktInfo object					*/
/*  pkt		: PktDisp *	: Object to free			*/
/*  (returns)	: void		:					*/

void	freePktInfo
	(	PktInfo	*pkt
	)
{
	assert(pkt != NULL) ;

	if (cntFree < MAXFREE)
	{	pkt->next = pktFree ;
		pktFree   = pkt	    ;
		cntFree  += 1	    ;
	}
	else	delete	pkt	    ;
}

/*  PCapInfo								*/
/*  Structure used to pass information into the PCAP callback routine.	*/
/*  Contains the communications socket and the PCAP structure pointer.	*/

typedef	struct
{	int	sock	;
	pcap_t	*pcap	;
}	PCapInfo	;

/*  pcapCallback: Monitor process/thread callback function		*/
/*  user	: u_char *	: User data				*/
/*  h		: pcap_pkthdr *	: Packet header information		*/
/*  p		: u_char *	: Packet data				*/
/*  (returns)	: void		:					*/

void	pcapCallback
	(	u_char		*user,
		const	struct	pcap_pkthdr *h,
		const	u_char	*p
	)
{
	PktData	pktdata	;

	pktdata.pktlen	= h->len	;
	pktdata.caplen	= h->caplen	;
	pktdata.tval	= h->ts		;
	pktdata.accept	= ((PCapInfo *)user)->pcap->md.stat.ps_recv   ;
	pktdata.dropped	= ((PCapInfo *)user)->pcap->md.stat.ps_drop   ;
	pktdata.missed	= ((PCapInfo *)user)->pcap->md.stat.ps_ifdrop ;

	memcpy	(pktdata.data, p, h->caplen < sizeof(pktdata.data) ? h->caplen : sizeof(pktdata.data)) ;
	write	(((PCapInfo *)user)->sock, &pktdata, sizeof(pktdata)) ;
}

/*  pcapGo	: Start routine for capture process or thread		*/
/*  arg		: void *	: Argument (actually KNDPCap *)		*/
/*  (returns)	: void		:					*/

void	*pcapGo
	(	void	*arg
	)
{
	/* Get a pointer at the monitor object, and thence the packet	*/
	/* capture structure and the communications socket.		*/
	PCapInfo pcinfo	  ;
	KNDPCap	 *monitor = (KNDPCap *)arg  ;

	pcinfo.sock	  = monitor->wrSock () ;
	pcinfo.pcap	  = monitor->getPcap() ;

	/* Close all file descriptors other than those associated with	*/
	/* the packet capture and the communications socket. This	*/
	/* ensures that if the parent crashes then we will terminate.	*/
	for (int fd = 0 ; fd < 64 ; fd += 1)
		if ((fd != pcinfo.sock) && (fd != pcinfo.pcap->fd))
			close (fd) ;

	/* Now run the packet capture loop. Note that we should never	*/
	/* return from the loop; if the parent sniffer is stopped then	*/
	/* this process/thread will simply be killed.			*/
	pcap_loop (pcinfo.pcap, -1, pcapCallback, (u_char *)&pcinfo) ;
	return	  NULL ;
}

static	void	JoinStr
	(	char	*buff,
		char	*str1,
		char	*str2
	)
{
	strcpy	(buff, str1) ;
	strcat	(buff, str2) ;
}

/*  pkt_ip	: Decode in internet (IP) packet			*/
/*  pktdata	: PktData *	: Captured packet			*/
/*  pktinfo	: PktInfo *	: Results structure			*/
/*  _p		: char *	: Packet data				*/
/*  (returns)	: void		:					*/

static	void	pkt_ip
	(	PktData	*pktdata,
		PktInfo	*pktinfo,
		char	*_p
	)
{
	IP	*ip	= (IP *)_p ;

	int	off	;
	int	hlen	;
	int	dlen	;
	char	*data	;

	TCP_HDR	*tcp	;
	UDP_HDR	*udp	;

	hlen	= ip->ip_hl * 4      ;	/* IP header length		*/
	dlen	= ntohs (ip->ip_len) ;	/* Total datagram length ...	*/
	dlen   -= hlen		     ;	/* ... hence IP data length	*/
	off	= ntohs (ip->ip_off) ;	/* Fragment offset and flags	*/
	data	= (char *)ip + hlen  ;	/* Start of data		*/

	/* If the low-order 13 bits of the fragment offset are all	*/
	/* zero then this is the first fragmment, so we have all the	*/
	/* information needed.						*/
	if ((off & 0x1fff) == 0)
	{
		pktinfo->pktlen	 = pktdata->pktlen ;
		pktinfo->datlen	 = dlen		   ;

		sprintf	(pktinfo->size, "%4d,%4d",
					pktinfo->pktlen,
					pktinfo->datlen) ;

		switch (ip->ip_p)
		{
			case IPPROTO_TCP  :
				tcp	= (TCP_HDR *)data ;

				strcpy	(pktinfo->proto, "tcp") ;
				JoinStr	(pktinfo->srce,
						FindHost    ((char *)&ip->ip_src), 
						FindService (ntohs (tcp->th_sport))) ;
				assert	(strlen(pktinfo->srce) < sizeof(pktinfo->srce)) ;
				JoinStr	(pktinfo->dest,
						FindHost    ((char *)&ip->ip_dst),
						FindService (ntohs (tcp->th_dport))) ;
				assert	(strlen(pktinfo->dest) < sizeof(pktinfo->dest)) ;
				break	;

			case IPPROTO_UDP  :
				udp	= (UDP_HDR *)data ;

				strcpy	(pktinfo->proto, "udp") ;
				JoinStr	(pktinfo->srce,
						FindHost    ((char *)&ip->ip_src), 
						FindService (ntohs (udp->uh_sport))) ;
				assert	(strlen(pktinfo->srce) < sizeof(pktinfo->srce)) ;
				JoinStr	(pktinfo->dest,
						FindHost    ((char *)&ip->ip_dst),
						FindService (ntohs (udp->uh_dport))) ;
				assert	(strlen(pktinfo->dest) < sizeof(pktinfo->dest)) ;
				break	;

			case IPPROTO_ICMP :
				strcpy	(pktinfo->proto, "icmp") ;
				break	;

			default :
				sprintf	(pktinfo->proto, "proto %d", ip->ip_p) ;
				break	;
		}
	}

	/* If either the fragments-follow bit 0x2000 is set, or the	*/
	/* flagment offset is non-zero then generate some information	*/
	/* about the flagment.						*/
	/* data). Make do with the IP address.				*/
	if ((off & 0x3fff) != 0)
	{
		sprintf	(pktinfo->info, "%d=%d@%d%s",
				ntohs (ip->ip_id),
				dlen,
				(off & 0x1fff) * 8,
				(off & IP_MF ) ? "+" : "") ;
		assert	(strlen(pktinfo->info) < sizeof(pktinfo->info)) ;
	}

	/* If the flagment offset is non-zero then we do not have the	*/
	/* next-level protocol header (ie., the header at the start of	*/
	/* the IP data) so make do with the addresses.			*/
	if ((off & 0x1fff) != 0)
	{
		strcpy	(pktinfo->proto, "frag") ;

		strcpy	(pktinfo->srce, FindHost ((char *)&ip->ip_src)) ;
		assert	(strlen(pktinfo->srce) < sizeof(pktinfo->srce)) ;
		strcpy	(pktinfo->dest, FindHost ((char *)&ip->ip_dst)) ;
		assert	(strlen(pktinfo->dest) < sizeof(pktinfo->dest)) ;
	}
}

/*  pkt_arp	: Decode in (R)ARP packet				*/
/*  pktdata	: PktData *	: Captured packet			*/
/*  pktinfo	: PktInfo *	: Results structure			*/
/*  _p		: char *	: Packet data				*/
/*  (returns)	: void		:					*/

static	void	pkt_arp
	(	PktData	*pktdata,
		PktInfo	*pktinfo,
		char	*_p
	)
{
#define	REVARP_REQUEST	(3)
#define	REVARP_REPLY	(4)

	ETH_ARP	*earp	= (ETH_ARP *)_p ;
//	ETH_HDR	*ehdr	= (ETH_HDR *)(&pktdata.data[0]) ;
	u_short	pro	;
	u_short	hrd	;
	u_short	op	;

	strcpy	(pktinfo->proto, "(r)arp") ;

	/* Extract:							*/
	/*	pro	- Protocol requesting resolution		*/
	/*	hrd	- Hardware type					*/
	/*	op	- Operation code				*/
	pro	= (u_short)ntohs(*(u_short *)&earp->arp_pro) ;
	hrd	= (u_short)ntohs(*(u_short *)&earp->arp_hrd) ;
	op	= (u_short)ntohs(*(u_short *)&earp->arp_op ) ;

	/* Check that we know about the requesting protocol, and that	*/
	/* the hardware and protocol address lengths are as expected	*/
	/* for an ethernet.						*/
	if ( ((pro != ETHERTYPE_IP) && (pro != ETHERTYPE_TRAIL)) ||
	     (earp->arp_hln != sizeof(earp->arp_sha)) ||
	     (earp->arp_pln != sizeof(earp->arp_spa)) )
	{
		return	;
	}

	switch (op)
	{
		case ARPOP_REQUEST  :
			/* ARP Request: put the requestor in the source	*/
			/* field and the reuested target in the		*/
			/* destination field.				*/
			strcpy	(pktinfo->proto, "arp who-has") ;
			strcpy	(pktinfo->srce, FindHost ((char *)&earp->arp_spa)) ; 
			strcpy	(pktinfo->dest, FindHost ((char *)&earp->arp_tpa)) ;
			break	;

		case ARPOP_REPLY    :
			/* ARP Reply: put the sender in the source	*/
			/* field.					*/
			strcpy	(pktinfo->proto, "arp is-at") ;
			strcpy	(pktinfo->srce, FindHost ((char *)&earp->arp_spa)) ; 
			break	;

		case REVARP_REQUEST :
			strcpy	(pktinfo->proto, "rarp who-is") ;
			break	;

		case REVARP_REPLY   :
			strcpy	(pktinfo->proto, "rarp reply") ;
			break	;

		default	:
			sprintf	(pktinfo->proto, "(r)arp %d", op) ;
			break	;
	}
}

/*  SetTime	: Set time as string					*/
/*  pktdata	: PktData *	: Capture packet data			*/
/*  pktinfo	: PktInfo *	: Packet information structure		*/
/*  (returns)	: void		:					*/

static	void	SetTime
	(	PktData	*pktdata,
		PktInfo	*pktinfo
	)
{
	int	hours	;
	int	mins	;
	int	secs	;

	/* Slave the last decoded second. If stuff is coming thick and	*/
	/* fast, this might save a bit. Note that when using threads	*/
	/* this is the main thread, so there is no problem with static	*/
	/* data.							*/
	static	int	lasts	  ;
	static	char	lastt[20] ;

	if (pktdata->tval.tv_usec != lasts)
	{
		lasts	= pktdata->tval.tv_sec	;
		secs	= pktdata->tval.tv_sec	;
		secs   %= 3600  *   24 	  	;
		hours	= secs  / 3600		;
		secs   -= hours * 3600		;
		mins	= secs	/   60		;
		secs   -= mins  *   60  	;

		sprintf	(lastt,  "%02d:%02d:%02d",  hours, mins, secs) ;
	}

	sprintf	(pktinfo->time, "%s.%06ld", lastt, pktdata->tval.tv_usec) ;
}

/*  PktDecode	: Packet decode main routine				*/
/*  pktdata	: PktData *	: Captured packet			*/
/*  pktinfo	: PktInfo *	: Results structure			*/
/*  options	: int		: Decode options			*/
/*  (returns)	: void		:					*/

void	PktDecode
	(	PktData	*pktdata,
		PktInfo	*pktinfo,
		int	options
	)
{
	ETH_HDR	*ehdr	  = (ETH_HDR *)(&pktdata->data[0]) ;
	char	*data	  = (char    *)(&pktdata->data[sizeof(ETH_HDR)]) ;

	/* Initialise the packet data structure to all empty with zero	*/
	/* lengths. These will be filled in by the various protocol	*/
	/* decoders.							*/
	pktinfo->tval	  = pktdata->tval    ;
	pktinfo->accept	  = pktdata->accept  ;
	pktinfo->dropped  = pktdata->dropped ;
	pktinfo->missed	  = pktdata->missed  ;

	SetTime	(pktdata, pktinfo) ;

	pktinfo->proto[0] = 0 ;
	pktinfo->srce [0] = 0 ;
	pktinfo->dest [0] = 0 ;
	pktinfo->info [0] = 0 ;
	pktinfo->size [0] = 0 ;
	pktinfo->pktlen	  = 0 ;
	pktinfo->datlen	  = 0 ;

	/* This is the top-level decode for an ethernet, based on the	*/
	/* protocol type in the ethernet frame header.			*/
	switch (ntohs (ehdr->ether_type))
	{
		case ETHERTYPE_IP 	:
			pkt_ip	(pktdata, pktinfo, data) ;
			break	;

		case ETHERTYPE_ARP	:
		case ETHERTYPE_REVARP	:
			pkt_arp	(pktdata, pktinfo, data) ;
			break	;

		default	:
			sprintf	(pktinfo->proto, "et %d", ntohs(ehdr->ether_type)) ;
			break	;
	}
}

/*  KNDPCap								*/
/*  KNDPCap	: Constructor for monitor module			*/
/*  graphic	: KNDGraphic *	: Graphic display module		*/
/*  packets	: KNDPackets *	: Packet display module			*/
/*  multi	: KNDMulti *	: Multiple graph object			*/
/*  slot	: int		: Monitor slot number			*/
/*  (returns)	: KNDPCap	:					*/

KNDPCap::KNDPCap
	(	KNDGraphic	*graphic,
		KNDPacket	*packets,
		KNDMulti	*multi,
		int		slot
	)
	:
	graphic	(graphic),
	packets	(packets),
	multi	(multi),
	slot	(slot)
{
	pd	 = NULL	 ;
	snaplen	 = 68	 ;
	sockp[0] = -1	 ;
	sockp[1] = -1	 ;
	ksock	 = NULL	 ;
	logfd	 = NULL	 ;
	logfile	 = NULL	 ;
	options	 = 0xffff;
	execing	 = false ;

#ifdef	_PTHREAD
	thread	 = 0	 ;
#else	/* _PTHREAD */
	pid	 = -1	 ;
	signal	(SIGCHLD, SIG_IGN) ;
#endif	/* _PTHREAD */
}

/*  KNDPCap								*/
/*  ~KNDPCap	: Destructor for monitor module				*/
/*  (returns)	:		:					*/

KNDPCap::~KNDPCap ()
{
	clear	() ;
}

/*  KNDPCap	:							*/
/*  execute	: Start or stop execution of the monitor		*/
/*  _execing	: bool		: TRUE to start				*/
/*  _logfd	: FILE *	: Log file or NULL			*/
/*  _binary	: bool		: Select binary logging			*/
/*  (returns)	: bool		: Monitor executing			*/

bool	KNDPCap::execute
	(	bool	_execing,
		FILE	*_logfd,
		bool	_binary
	)
{
	/* If we are asked to start but ar not already doing so then	*/
	/* create a new process or thread to handle the monitor.	*/
	if (_execing && !execing)
	{
		if (ksock == NULL)
			return	false ;

#ifdef	_PTHREAD
		if (pthread_create (&thread, NULL, threadStart, (void *)this) != 0)
		{
			Error	("Ksnuffle thread error", STRERR) ;
			return	false ;
		}
#else	/* _PTHREAD */
		if	((pid = fork ()) < 0)
		{	Error	("Ksnuffle fork error", STRERR) ;
			return	false ;
		}
		else if (pid == 0)
		{	pcapGo	((void *)this) ;
			exit	(1) ;
		}
#endif	/* _PTHREAD */

		logfd	= _logfd  ;
		binary	= _binary ;
		execing	= true	  ;

		return	true ;
	}

	/* If we are asked to stop and are currently executing the	*/
	/* monitor then simplay cancel the process/thread.		*/
	if (!_execing && execing)
	{
#ifdef	_PTHREAD
		if (thread != 0) pthread_cancel (thread) ;
		thread	= 0	;
#else	/* _PTHREAD */
		if (pid    >= 0) kill (pid, SIGKILL) ;
		pid	= -1	;
#endif	/* _PTHREAD */
		execing	= false ;
		logfile	= NULL	;

		return	false	;
	}

	return	execing	;
}

/*  KNDPCap	:							*/
/*  clear	: Clear up current monitor				*/
/*  (returns)	: void		:					*/

void	KNDPCap::clear ()
{
	/* If there is an associated processthread running then cancel	*/
	/* it before remving any data or other information.		*/
#ifdef	_PTHREAD
	if (thread != 0)
	{	pthread_cancel (thread) ;
		thread	= 0 ;
	}
#else	/* _PTHREAD */
	if (pid    >= 0)
	{	kill	(pid, SIGKILL) ;
		pid	= -1 ;
	}
#endif	/* _PTHREAD */

	/* If there is a monitor open then close it. Note that there	*/
	/* is currently no way to free the compiled code. Also, since	*/
	/* we have opened the monitor in the parent process/thread, we	*/
	/* close it here.						*/
	if (pd != NULL)
	{	pcap_close    (pd) ;
		pd = NULL ;
	}

	/* There should always be a KDE socket at this point, which we	*/
	/* delete, but make sure just in case.				*/
	if (ksock != NULL)
	{	delete	ksock	   ;
		ksock	 = NULL	   ;
	}

	/* Close the socket pair that was used for communicating	*/
	/* between the associated process/thread and the main process.	*/
	if (sockp[0] >= 0)
	{	::close	(sockp[0]) ;
		::close	(sockp[1]) ;
		sockp[0] = -1	   ;
		sockp[1] = -1	   ;
	}
}

/*  KNDPCap	:							*/
/*  setProg	: Set the monitoring program				*/
/*  iface	: const char *	: Interface (device) name		*/
/*  prog	: const char *	: Monitor program			*/
/*  (returns)	: bool		: Success				*/

bool	KNDPCap::setProg
	(	const	char	*iface,
		const	char	*prog
	)
{
	char	ebuf	[PCAP_ERRBUF_SIZE] ;
	BFPI32	locnet	;
	BFPI32	netmask	;

	/* Clean up anything left over from a previous run of this	*/
	/* particular monitor.						*/
	clear	() ;

	/* First stage is to open the interface, which we do in		*/
	/* promiscuous mode ....					*/
	if ((pd = pcap_open_live ((char *)iface, snaplen, 1, 1000, ebuf)) == NULL)
	{	Error	("Ksnuffle error",   ebuf) ;
		return	false	;
	}

	/* Second stage is to determin the local network address and	*/
	/* the network mask for the interface ....			*/
	if (pcap_lookupnet ((char *)iface, &locnet, &netmask, ebuf))
		Error	("Ksnuffle warning", ebuf, QMessageBox::Information) ;
	

	/* There is no call to free program space, although the pcal	*/
	/* header file does define one. So, we just clear the program	*/
	/* structure each time,						*/
	memset	(&program, 0, sizeof(BPFPROG)) ;

	/* We can now attempt to compile the monitor program supplied	*/
	/* by the configuration object. If this is OK then we can	*/
	/* apply it to the opened device.				*/
	if (pcap_compile (pd, &program, (char *)prog, 0, netmask) < 0)
	{	Error	("Ksnuffle error",   pcap_geterr (pd)) ;
		clear	()	;
		return	false	;
	}

	if (pcap_setfilter (pd, &program) < 0)
	{	Error	("Ksnuffle error",  pcap_geterr (pd)) ;
		clear	()	;
		return	false	;
	}

	/* Since the real monitor runs in a processthread, create a	*/
	/* socket pair by which it can send results back to us. The	*/
	/* pair is in the Unix doamin, as this is supposed to be a bit	*/
	/* more efficient than the InterNet domain, but who knows?	*/
	if (socketpair (AF_UNIX, SOCK_STREAM, 0, sockp) < 0)
	{	Error	("Ksnuffle socketpair error", STRERR) ;
		clear	()	;
		return	false	;
	}

	if (fcntl (sockp[1], F_SETFL, O_NONBLOCK) < 0)
	{	Error	("Ksnuffle socket nonblock error", STRERR) ;
		clear	()	;
		return	false	;
	}

	/* Create a KDE socket which we use to wait for data coming up	*/
	/* the socket to us. It is attached to one end of the socket	*/
	/* pair, and reading is enabled. The read event is connected	*/
	/* to a slot on this object.					*/
	ksock	= new KSocket (sockp[1]) ;
	ksock->enableRead (true) ;
	connect	(ksock, SIGNAL(readEvent(KSocket *)), this, SLOT(sockRead(KSocket *))) ;

	return	true	;
}

/*  KNDPCap	:							*/
/*  setOptions	: Set options flags					*/
/*  _options	: Options flags						*/
/*  (returns)	: void							*/

void	KNDPCap::setOptions
	(	int	_options
	)
{
	options	= _options ;
}


/*  KNDPCap								*/
/*  sockRead	: Socket read slot					*/
/*  sock	: KSocket *	: Socket				*/
/*  (returns)	: void		:					*/

void	KNDPCap::sockRead
	(	KSocket	*sock
	)
{
	PktData	pktdata	;
	PktInfo	*pktinfo;
	long	timenow	;
	int	got	;

	while ((got = read (sock->socket(), &pktdata, sizeof(pktdata))) == sizeof(pktdata))
	{
		pktinfo	  = getPktInfo () ;
		pktinfo->inuse = false	  ;

		PktDecode (&pktdata, pktinfo, options) ;

		/* If the log file is open then log the packet. This	*/
		/* is either a binary dump of the data we got from the	*/
		/* monitor processthread, or a text version suitably	*/
		/* formatted.						*/
		if (logfd != NULL)
			if (binary)
				fwrite	(&pktdata, sizeof(pktdata), 1, logfd) ;
			else	fprintf	(logfd, "%s: %4d: %s %s -> %s : %s\n",
						pktinfo->time,
						pktinfo->pktlen,
						pktinfo->proto,
						pktinfo->srce,
						pktinfo->dest,
						pktinfo->info) ;

		/* Before passing the packet to the display objects,	*/
		/* ensure that we have an up-to-date view of the time,	*/
		/* in case a timer tick as got behind a packet.		*/
		timenow	     = upToTick (pktinfo->tval.tv_sec) ;
		pktinfo->ago = timenow - pktinfo->tval.tv_sec  ;

		/* Worst case is the packet appears to have arrived in	*/
		/* the future. This should never happen!		*/
		if (pktinfo->ago < 0)
		{	fprintf	(stderr, "KSnuffle: packet arrived in future by %d\n",
					 -pktinfo->ago) ;
			freePktInfo (pktinfo) ;
			continue ;
		}

		packets->addPacket (pktinfo) ;
		graphic->addPacket (pktinfo) ;
		multi  ->addPacket (pktinfo, slot) ;

		if (!pktinfo->inuse) freePktInfo (pktinfo) ;
	}

	if (got < 0)
		if (errno != EAGAIN)
			Error	("Ksnuffle socket read error", STRERR) ;
		else	return	;

	clear	() ;
}

