/*
 *   kscan - a scanning program
 *   Copyright (C) 1998 Ivan Shvedunov
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <qfiledlg.h>
#include <qstring.h>

#define __PREVIEW_CPP__
#include "preview.h"

Preview::Preview(QWidget *parent = NULL, const char *name = NULL):
	QWidget(parent,name)
{
	brightness = 0;
	contrast = 0;
	type = COLOR;
	moving = MOVE_NONE;
	gamma = 100;
	s_img = new ScanImage(type);
	type = s_img->getType(); //s_img knows better
	//s_img->setArea(0,0,SANE_FIX(150.),SANE_FIX(100.));
	SANE_Fixed tlx,tly,brx,bry;
	s_img->getArea(tlx,tly,brx,bry);
	//FIXME: getArea should be emulated if there's no scanner
	printf("%g, %g, %g, %g\n",
	       SANE_UNFIX(tlx),SANE_UNFIX(tly),
	       SANE_UNFIX(brx),SANE_UNFIX(bry));
	xmms = SANE_UNFIX(brx);
	ymms = SANE_UNFIX(bry);
	int h = KApplication::desktop()->height()/2;
	int w = (int)((double)h*SANE_UNFIX(brx)/SANE_UNFIX(bry));
	setFixedSize(w,h);
	px = new QPixmap(w,h);
	px->fill(QColor(150,150,150));
	zoomed = FALSE;
	acquired = FALSE;
	//zoomOut();
	selected = new QRect;
	setCursor(crossCursor);
	cr1 = 0;
	startTimer(100);
	setMouseTracking(TRUE);
	if(s_img->picturePresent()) {
		acquired = TRUE;
		QSize sz = s_img->image()->size();
		setFixedSize(sz);
		px->resize(sz);
		s_img->adjust_and_convert(gamma,brightness,contrast);
		px->convertFromImage(*s_img->image());
	}
}

void Preview::zoomOut()
{
	if(!s_img->scannerPresent()) {
		noScanner();
		return;
	}
	double sch = (double)KApplication::desktop()->height();
	s_img->setResolution((int)(sch*.5*mms_per_inch/ymms));
	s_img->setArea(0,0,SANE_FIX(xmms),SANE_FIX(ymms));
	acquire();
	if(zoomed&&!selected->isEmpty()) {
		int xpx = (int)(x_offs*(double)width()/xmms);
		int ypx = (int)(y_offs*(double)height()/ymms);
		double xcoef = xmms_cur/xmms;
		double ycoef = ymms_cur/ymms;
		selected->setRect(xpx+(int)((double)selected->left()*xcoef),
				  ypx+(int)((double)selected->top()*ycoef),
				  (int)((double)selected->width()*xcoef),
				  (int)((double)selected->height()*ycoef));
	}
	zoomed = FALSE;
	s_img->save();
}

void Preview::zoomIn()
{
	if(!s_img->scannerPresent()) {
		noScanner();
		return;
	}
	if(selected->isEmpty()) return;
	int sw = selected->width(), sh = selected->height();
	bool expHor = TRUE;
	//First, decide what should we expand to maximum
	if(sh*width()/selected->width()>height()) expHor = FALSE;
	int res = expHor?(int)(width()/xInches()):
		(int)(height()/yInches());
	if(res>max_dpi) {
		QMessageBox::warning(NULL,"Warning",
				     "Cannot zoom :\nresolution too high");
		return;
	}
	s_img->setResolution(res);
	if(!zoomed) x_offs = y_offs = 0;
	x_offs += xtomms(selected->left());
	y_offs += ytomms(selected->top());
	if(expHor) {
		xmms_cur = xtomms(sw);
		ymms_cur = xmms_cur*ymms/xmms;
	} else {
		ymms_cur = ytomms(sh);
		xmms_cur = ymms_cur*xmms/ymms;
	}
	printf("** AREA : %g, %g, %g, %g\n",x_offs,y_offs,xmms_cur,ymms_cur);
	s_img->setArea(SANE_FIX(x_offs),SANE_FIX(y_offs),
			SANE_FIX(x_offs+xmms_cur),
			SANE_FIX(y_offs+ymms_cur));
	zoomed = TRUE;
	acquire();
	if(expHor) selected->setRect(0,0,width(),sh*width()/sw);
	else selected->setRect(0,0,sw*height()/sh,height());
	selected->normalize();
	*selected = selected->intersect(rect());
}

void Preview::acquire() //Acquire the preview image
{
	if(!s_img->scannerPresent()) {
		noScanner();
		return;
	}
	s_img->setBrightness(0);
	s_img->setContrast(0);
	s_img->acquire();
	acquired = TRUE;
	QSize sz = s_img->image()->size();
	setFixedSize(sz);
	px->resize(sz);
	emit sizeChanged();
	redraw();
}

double Preview::xtomms(double x)
{
	return (zoomed?xmms_cur:xmms)*x/(double)width();
}

double Preview::ytomms(double y)
{
	return (zoomed?ymms_cur:ymms)*y/(double)height();
}

void Preview::save(int dpi)
{
	if(!s_img->scannerPresent()) {
		noScanner();
		return;
	}
	//s_img->image()->save("Preview.bmp","BMP");
	if(selected->isEmpty()) {
		QMessageBox::warning(NULL,"Warning","Nothing to scan!");
		return;
	}
	QString fname = QFileDialog::getSaveFileName(NULL,"*.bmp");
	if(fname.isNull()) return;
	ScanImage im2(type,FALSE);
	double x1 = xtomms(selected->left());
	double x2 = xtomms(selected->right());
	double y1 = ytomms(selected->top());
	double y2 = ytomms(selected->bottom());
	if(zoomed) x1 += x_offs, x2 += x_offs, y1 += y_offs, y2 += y_offs;
	printf("RESOLUTION: %d; (%g,%g) - (%g,%g)\n",dpi,x1,y1,x2,y2);
	printf("inches : %g, %g; -> %g, %g\n",xInches(),yInches(),
	       (x2-x1)/mms_per_inch,(y2-y1)/mms_per_inch);
	printf("pixels : %d, %d\n",(int)(xInches()*(double)dpi),
	       (int)(yInches()*(double)dpi));
	im2.setArea(SANE_FIX(x1),SANE_FIX(y1),SANE_FIX(x2),SANE_FIX(y2));
	im2.setResolution(dpi);
	im2.setBrightness(brightness);
	im2.setContrast(contrast);
	printf("getResolution() = %d\n",im2.getResolution());
	im2.acquire();
	im2.adjust_and_convert(gamma);
	im2.image()->save((const char *)fname,"BMP");
	printf("We got : %d dpi\n",
	       (int)((double)im2.image()->width()/xInches()));
}

void Preview::paintEvent(QPaintEvent *)
{
	//bitBlt(this,0,0,px,0,0,-1,-1,CopyROP);
	QPainter p(this);
	p.drawPixmap(0,0,*px);
	drawAreaBorder(&p);
}

void Preview::timerEvent(QTimerEvent *)
{
	if(moving!=MOVE_NONE) return;
	cr1++;
	QPainter p(this);
	drawAreaBorder(&p);
}

void Preview::mousePressEvent(QMouseEvent *ev)
{
	if(ev->button()!=LeftButton) return;
	if(moving==MOVE_NONE) {
		QPainter p(this);
		drawAreaBorder(&p,TRUE);
		int x = lx = ev->x(),y = ly = ev->y();
		moving = classifyPoint(x,y);
		if(moving==MOVE_NONE) { //Create new area
			selected->setCoords(x,y,x,y);
			moving = MOVE_BOTTOM_RIGHT;
		}
		drawAreaBorder(&p);
	} else printf("MouseReleaseEvent() missed\n");
}

void Preview::mouseReleaseEvent(QMouseEvent *ev)
{
	if(ev->button()!=LeftButton) return;
	if(moving!=MOVE_NONE) {
		QPainter p(this);
		drawAreaBorder(&p,TRUE);
		moving = MOVE_NONE;
		*selected = selected->normalize();
		if(selected->width()<MIN_AREA_WIDTH||
		   selected->height()<MIN_AREA_HEIGHT) {
			selected->setWidth(0);
			selected->setHeight(0);
			emit noRect();
			return;
		}
		*selected = selected->intersect(rect());
		drawAreaBorder(&p);
		emit newRect();
	} else printf("mousePressEvent() missed\n");
}

void Preview::mouseMoveEvent(QMouseEvent *ev)
{
	static cursor_type ps = HREN;
	int x = ev->x(),y = ev->y();
	switch(moving!=MOVE_NONE?moving:classifyPoint(x,y)) {
	case MOVE_NONE:
		if(ps!=CROSS) {
			setCursor(crossCursor);
			ps = CROSS;
		}
		break;
	case MOVE_LEFT:
	case MOVE_RIGHT:
		if(ps!=HSIZE) {
			setCursor(sizeHorCursor);
			ps = HSIZE;
		}
		break;
	case MOVE_TOP:
	case MOVE_BOTTOM:
		if(ps!=VSIZE) {
			setCursor(sizeVerCursor);
			ps = VSIZE;
		}
		break;
	case MOVE_TOP_LEFT:
	case MOVE_BOTTOM_RIGHT:
		if(ps!=FDIAG) {
			setCursor(sizeFDiagCursor);
			ps = FDIAG;
		}
		break;
	case MOVE_TOP_RIGHT:
	case MOVE_BOTTOM_LEFT:
		if(ps!=BDIAG) {
			setCursor(sizeBDiagCursor);
			ps = BDIAG;
		}
		break;
	case MOVE_WHOLE:
		if(ps!=ALL) {
			setCursor(sizeAllCursor);
			ps = ALL;
		}
	}
	//At ButtonRelease : normalize + clip
	if(moving!=MOVE_NONE) {
		QPainter p(this);
		drawAreaBorder(&p,TRUE);
		switch(moving) {
		case MOVE_NONE: //Just to make compiler happy
		case MOVE_TOP_LEFT:
			selected->setLeft(x);
		case MOVE_TOP:
			selected->setTop(y);
			break;
		case MOVE_TOP_RIGHT:
			selected->setTop(y);
		case MOVE_RIGHT:
			selected->setRight(x);
			break;
		case MOVE_BOTTOM_LEFT:
			selected->setBottom(y);
		case MOVE_LEFT:
			selected->setLeft(x);
			break;
		case MOVE_BOTTOM_RIGHT:
			selected->setRight(x);
		case MOVE_BOTTOM:
			selected->setBottom(y);
			break;
		case MOVE_WHOLE:
			selected->moveBy(x-lx,y-ly);
		}
		drawAreaBorder(&p);
		lx = x;
		ly = y;
	}
}

void Preview::redraw()
{
	if(!acquired) return;
	s_img->adjust_and_convert(gamma,brightness,contrast);
	px->convertFromImage(*s_img->image());
	repaint(FALSE);
}

void Preview::drawHAreaBorder(QPainter &p,int x1,int x2,int y,int r = FALSE)
{
	if(moving!=MOVE_NONE) cr2 = 0;
	int inc = 1;
	if(x2<x1) inc = -1;
	if(!r) {
		if(cr2&4) p.setPen(black);
		else p.setPen(white);
	} else if(!acquired) p.setPen(QPen(QColor(150,150,150)));
	for(;;) {
		if(rect().contains(QPoint(x1,y))) {
			if(r&&acquired) p.setPen(QPen(QColor(s_img->image()->
							     pixel(x1,y))));
		}
		p.drawPoint(x1,y);
		if(!r) {
			cr2++;
			cr2 &= 7;
			if(!(cr2&3)) {
				if(cr2&4) p.setPen(black);
				else p.setPen(white);
			}
		}
		if(x1==x2) break;
		x1 += inc;
	}
}

void Preview::drawVAreaBorder(QPainter &p,int x,int y1,int y2,int r = FALSE)
{
	if(moving!=MOVE_NONE) cr2 = 0;
	int inc = 1;
	if(y2<y1) inc = -1;
	if(!r) {
		if(cr2&4) p.setPen(black);
		else p.setPen(white);
	} else if(!acquired) p.setPen(QPen(QColor(150,150,150)));
	for(;;) {
		if(rect().contains(QPoint(x,y1))) {
			if(r&&acquired) p.setPen(QPen(QColor(s_img->image()->
							     pixel(x,y1))));
			p.drawPoint(x,y1);
		}
		if(!r) {
			cr2++;
			cr2 &= 7;
			if(!(cr2&3)) {
				if(cr2&4) p.setPen(black);
				else p.setPen(white);
			}
		}
		if(y1==y2) break;
		y1 += inc;
	}
}

void Preview::drawAreaBorder(QPainter *p,int r = FALSE)
{
	if(selected->isNull()) return;
	cr2 = cr1;
	int xinc = 1;
	if(selected->right()<selected->left()) xinc = -1;
	int yinc = 1;
	if(selected->bottom()<selected->top()) yinc = -1;
	if(selected->width())
		drawHAreaBorder(*p,selected->left(),
				selected->right(),selected->top(),r);
	if(selected->height()) {
		drawVAreaBorder(*p,selected->right(),
				selected->top()+yinc,selected->bottom(),r);
		if(selected->width()) {
			drawHAreaBorder(*p,selected->right()-xinc,
					selected->left(),
					selected->bottom(),r);
			drawVAreaBorder(*p,selected->left(),
					selected->bottom()-yinc,
					selected->top()+yinc,r);
		}
	}
}

preview_state Preview::classifyPoint(int x,int y)
{
	if(selected->isEmpty()) return MOVE_NONE;
	QRect a = selected->normalize();
	int top = 0,left = 0,right = 0,bottom = 0;
	int lx = a.left()-x,rx = x-a.right();
	int ty = a.top()-y,by = y-a.bottom();
	if(a.width()>delta*2+2)
		lx = abs(lx), rx = abs(rx);
	if(a.height()>delta*2+2)
		ty = abs(ty), by = abs(by);
	if(lx>=0&&lx<=delta) left++;
	if(rx>=0&&rx<=delta) right++;
	if(ty>=0&&ty<=delta) top++;
	if(by>=0&&by<=delta) bottom++;
	if(y>=a.top()&&y<=a.bottom()) {
		if(left) {
			if(top) return MOVE_TOP_LEFT;
			if(bottom) return MOVE_BOTTOM_LEFT;
			return MOVE_LEFT;
		}
		if(right) {
			if(top) return MOVE_TOP_RIGHT;
			if(bottom) return MOVE_BOTTOM_RIGHT;
			return MOVE_RIGHT;
		}
	}
	if(x>=a.left()&&x<=a.right()) {
		if(top) return MOVE_TOP;
		if(bottom) return MOVE_BOTTOM;
		if(selected->contains(QPoint(x,y))) return MOVE_WHOLE;
	}
	return MOVE_NONE;
}

double Preview::xInches()
{
	return (zoomed?xmms_cur:xmms)*(double)selected->width()/
		((double)width()*mms_per_inch);
}

double Preview::yInches()
{
	return (zoomed?ymms_cur:ymms)*(double)selected->height()/
		((double)height()*mms_per_inch);
}

int Preview::calculateDpi(int xw)
{
	return (int)((double)xw/xInches());
}

#include "preview.moc"
