// -*- C++ -*-

//
//  konline
//
//  Copyright (C) 1998 Christoph Neerfeld
//  email:  Christoph.Neerfeld@home.ivm.de or chris@kde.org
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

extern "C" {
#include <unistd.h>
#include <time.h>

};

#include <qchkbox.h>
#include <qcombo.h>
#include <qframe.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qmenubar.h>
#include <qpushbt.h>
#include <qkeycode.h>
#include <qmsgbox.h>
#include <qobjcoll.h>
#include <qfileinf.h>
#include <qdir.h>
#include <qcursor.h>
#include <qsocknot.h>
#include <qtimer.h>

#include <kprocess.h>
#include <kmsgbox.h>
#include <ktoolbar.h>
#include <kiconloader.h>

#include "konline.h"
#include "konline.moc"
#include "confdialog.h"
#include "mainlist.h"
#include "server.h"
#include "users.h"
#include "messages.h"

#define global_pix_loader KApplication::getKApplication()->getIconLoader()

extern Server *serv;
MainRowTable *watchlist;

Konline::Konline(const char* name = 0) 
  : KTopLevelWidget( name )
{
  initMetaObject();

  details = 0;

  keep_timer = new QTimer(this);
  connect( keep_timer, SIGNAL(timeout()), this, SLOT(sendKeepAlive()) );

  for( int i = 0; i < MAX_RECV_WIN; i++ )
    recv_msg[i] = 0;

  setCaption(KApplication::getKApplication()->getCaption());
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup("konline");

  popup_menu = new QPopupMenu;
  popup_menu->insertItem( klocale->translate("Send message"), this, SLOT(sendMsg()) );
  //popup_menu->insertItem( klocale->translate("Send email"), this, SLOT(sendEmail()) );
  popup_menu->insertItem( klocale->translate("Show details"), this, SLOT(showDetails()) );
  popup_menu->insertItem( klocale->translate("Remove"), this, SLOT(removeList()) );

  menubar = new KMenuBar( this, "MenuBar_1" );
  menubar->setGeometry( 0, 0, 620, 28 );
  menubar->setFrameStyle( 34 );
  menubar->setLineWidth( 2 );
  
  f_main = new QFrame( this, "Frame_1" );
  f_main->move( 0, 28 );
  f_main->setMinimumSize( 520, 370 );
  
  lb_watchlist = new MainRowTable( f_main, "ListBox_1" );
  lb_watchlist->setMinimumSize( 504, 60 );
  lb_watchlist->setFrameStyle( 51 );
  lb_watchlist->setLineWidth( 2 );
  lb_watchlist->setAutoDelete(TRUE);
  watchlist = lb_watchlist;

  h_list = new KHeader( f_main, "Header_1", 4, KHeader::Resizable | KHeader::Buttons );
  h_list->setMinimumSize( 504, 20 );
  h_list->setHeaderLabel( 0, klocale->translate("A"), AlignLeft | AlignVCenter );
  h_list->setHeaderLabel( 1, klocale->translate("Name"), AlignLeft | AlignVCenter );
  h_list->setHeaderLabel( 2, klocale->translate("First name"), AlignLeft | AlignVCenter );
  h_list->setHeaderLabel( 3, klocale->translate("Nickname"), AlignLeft | AlignVCenter );

  connect( h_list, SIGNAL(sizeChanged(int,int)),
           lb_watchlist, SLOT(setColumnWidth(int,int)) );
  connect( lb_watchlist, SIGNAL(hSliderMoved(int)),
           h_list, SLOT(setOrigin(int)) );
  connect( lb_watchlist, SIGNAL(rightButtonClicked()), this, SLOT(popupMenu()) );

  top2bottom = new QGridLayout( f_main, 2, 1 );
  top2bottom->addWidget( h_list, 0, 0, AlignCenter );
  top2bottom->addWidget( lb_watchlist, 1, 0, AlignCenter );
  top2bottom->setRowStretch( 1, 1 );
  top2bottom->activate();

  file = new QPopupMenu;
  file->insertItem(klocale->translate("&Register user"), this, SLOT(registerUser()));
  file->insertItem(klocale->translate("&Add user to watchlist"), this, SLOT(addWatchList()));
  file->setId( 1, 1);
  file->insertItem(klocale->translate("&Connect"), this, SLOT(serverConnect()));
  file->setId( 2, 0);
  file->insertItem(klocale->translate("&Disconnect"), this, SLOT(serverDisconnect()));
  file->setId( 3, 2);
  file->insertItem(klocale->translate("&Quit"), qApp, SLOT(quit()));
  file->setItemEnabled(1, FALSE);
  file->setItemEnabled(2, FALSE);

  QPopupMenu *conf_menu = new QPopupMenu;
  conf_menu->insertItem(klocale->translate("&Configure"), this, SLOT(configureKOnline()) );

  QString about = (QString ) "Konline " + VERSION + "\n(C) " +
    klocale->translate("by") + " Christoph Neerfeld\n" + 
    "chris@kde.org";
  QPopupMenu *help = KApplication::getKApplication()->getHelpMenu(FALSE, about );

  menubar->insertItem( klocale->translate("&File"), file );
  menubar->insertItem( klocale->translate("&Options"), conf_menu );
  menubar->insertSeparator();
  menubar->insertItem( klocale->translate("&Help"), help );  

  lb_watchlist->setFocus();

  toolbar = new KToolBar(this);
  toolbar->insertButton(global_pix_loader->loadIcon("connected.xpm"), 0, SIGNAL(clicked()),
			this, SLOT(serverConnect()), TRUE, klocale->translate("Connect"));
  toolbar->insertButton(global_pix_loader->loadIcon("disconnected.xpm"), 1, SIGNAL(clicked()),
			this, SLOT(serverDisconnect()), TRUE, klocale->translate("Disconnect"));
  toolbar->insertButton(global_pix_loader->loadIcon("reload.xpm"), 1, SIGNAL(clicked()),
			this, SLOT(getMsg()), TRUE, klocale->translate("Get Messages"));
  toolbar->insertButton(global_pix_loader->loadIcon("useradd.xpm"), 1, SIGNAL(clicked()),
			this, SLOT(addWatchList()), TRUE,
			klocale->translate("Add user to watchlist"));
  addToolBar(toolbar);
  setMenu(menubar);
  setView(f_main, FALSE);

  toolbar->setItemEnabled(1, FALSE);
  // resize window
  f_main->resize( 200, 300 );
  setMinimumSize(200, 250);
  int width, height;
  config->setGroup("konline");
  width = config->readNumEntry("Width");
  height = config->readNumEntry("Height");
  if( width < minimumSize().width() )
    width = minimumSize().width();
  if( height < minimumSize().height() )
    height = minimumSize().height(); 
  //  move( config->readNumEntry("XPos", 0), config->readNumEntry("YPos", 0) );

  h_list->setHeaderSize( 0, config->readNumEntry("aSize", 24 ) );
  h_list->setHeaderSize( 1, config->readNumEntry("nameSize", 80 ) );
  h_list->setHeaderSize( 2, config->readNumEntry("fnameSize", 80 ) );
  h_list->setHeaderSize( 3, config->readNumEntry("nnameSize", 60 ) );

  if( config->readBoolEntry("SavePasswd", FALSE ) )
    passwd = config->readEntry("Passwd");
  else
    passwd = "";

  notifier = 0;
  connected = config->readNumEntry("Connected", 0);
  if(connected)
    serverConnect();

  resize(width, height);
}

Konline::~Konline()
{
  saveProperties( (KConfig *) 0 );
}

void Konline::saveProperties(KConfig*)
{
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup("konline");
  config->writeEntry("Width", width());
  config->writeEntry("Height", height());
  config->writeEntry("XPos", x());
  config->writeEntry("YPos", y());
  config->writeEntry("aSize", h_list->getHeaderSize(0) );
  config->writeEntry("nameSize", h_list->getHeaderSize(1) );
  config->writeEntry("fnameSize", h_list->getHeaderSize(2) );
  config->writeEntry("nnameSize", h_list->getHeaderSize(3) );
  config->writeEntry("Connected", connected );
  if( config->readBoolEntry("SavePasswd", FALSE) )
    config->writeEntry("Passwd", passwd);
  else
    config->writeEntry("Passwd", "");
  if( connected )
    serverDisconnect();
  config->sync();
}

void Konline::configureKOnline()
{
  conf_diag = new ConfDialog(0);
  if( conf_diag->exec() )
    {
      if(connected)
	{
	  KConfig *config = KApplication::getKApplication()->getConfig();
	  config->setGroup("konline");
	  QString ip = config->readEntry("Server");
	  QString port = config->readEntry("Port");
	  QString a_id = config->readEntry("AId");
	  conf_diag->writeConfig();
	  config->setGroup("konline");
	  if( ip != config->readEntry("Server")
	      || port != config->readEntry("Port")
	      || a_id != config->readEntry("AId") )
	    {
	      short rcode;
	      rcode = serv->serverDisconnect();
	      if( rcode == RE_OK )
		{
		  setConnected(FALSE);
		}
	      serverConnect();
	    }
	  else
	    serverConnect();                         // this is a reconnect
	}
      else
	conf_diag->writeConfig();
    }
  delete conf_diag;
  conf_diag = 0;
}

void Konline::registerUser()
{
  register_user = new RegisterUser(0, passwd);
  if( register_user->exec() )
    {
      if(connected)
	{
	  KConfig *config = KApplication::getKApplication()->getConfig();
	  config->setGroup("konline");
	  QString a_id = config->readEntry("AId");
	  register_user->writeConfig();
	  config->setGroup("konline");
	  if( ((QString) a_id) != config->readEntry("AId") )
	    {
	      short rcode;
	      serv->setAccount( (const char *) passwd, a_id.toULong() );
	      rcode = serv->serverDisconnect();
	      if( rcode == RE_OK )
		{
		  setConnected(FALSE);
		}
	      passwd = register_user->getPasswd();
	      serverConnect();
	    }
	  else
	    {
	      passwd = register_user->getPasswd();
	      serverConnect();                         // this is a reconnect
	    }
	}
      else
	{
	  register_user->writeConfig();
	  passwd = register_user->getPasswd();
	}
    }
  delete register_user;
  register_user = 0;
}

void Konline::addWatchList()
{
  add_watch_list = new AddWatchList(0);
  add_watch_list->exec();
  if(connected)
    {
      serverConnect(); // this is a reconnect
    }
  delete add_watch_list;
  add_watch_list = 0;
}

void Konline::serverConnect()
{
  PasswdDialog dialog(0, 0);
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup("konline");
  QString ip = config->readEntry("Server");
  QString port = config->readEntry("Port");
  QString a_id = config->readEntry("AId","0");
  unsigned short port_nr = port.toUShort();
  unsigned long id = a_id.toULong();
  if( serv->init( ip, port_nr, (const char *) passwd, id) )
    {
      QMessageBox::critical( 0, klocale->translate("Konline"),
			     klocale->translate("I can't find the server.\n
Please check IP and PORT settings.") );
      setConnected(FALSE);
      return;
    }
  short rcode;
  struct users rec;
  User drec;
  char act;
  while( (rcode = serv->serverConnect(&rec, &act)) != RE_OK && rcode != RE_NOTFOUND )
    {
      switch( rcode ) {
      case RE_PASSWD:
	config->setGroup("konline");
	if( config->readBoolEntry("SavePasswd", FALSE) )
	  QMessageBox::critical( 0, klocale->translate("Konline"),
				 klocale->translate("The given password is wrong.") );
	dialog.restart();
	if( dialog.exec() )
	  {
	    passwd = dialog.getPasswd();
	    serv->setAccount((const char *) passwd, id);
	  }
	else
	  {
	    setConnected(FALSE);
	    return;
	  }
	break;
      case RE_ACCESS_DENIED:
	QMessageBox::critical( 0, klocale->translate("Konline"),
			       klocale->translate("The server has denied your access.") );
	setConnected(FALSE);
	return;
      default:
	QMessageBox::critical( 0, klocale->translate("Konline"),
			       klocale->translate("I can't connect to the server.") );
	setConnected(FALSE);
	return;
      };
    }
  lb_watchlist->setUpdatesEnabled(FALSE);
  lb_watchlist->clear();
  if( rcode == RE_OK )
    {
      do {
	drec.name = rec.name;
	drec.fname = rec.fname;
	drec.nname = rec.nname;
	drec.email = rec.email;
	drec.id = rec.user_id;
	lb_watchlist->insertItem( &drec, (bool) act ); 
      } while( serv->nextWUserId(&rec, &act) == RE_OK );
    }
  lb_watchlist->setUpdatesEnabled(TRUE);
  lb_watchlist->repaint(TRUE);
  setConnected(TRUE);
}

void Konline::serverDisconnect()
{
  KConfig *config = KApplication::getKApplication()->getConfig();
  config->setGroup("konline");
  QString ip = config->readEntry("Server");
  QString port = config->readEntry("Port");
  QString a_id = config->readEntry("AId","0");
  short rcode;
  rcode = serv->serverDisconnect();
  if( rcode != RE_OK )
    {
      if( rcode == RE_PASSWD )
	debug(klocale->translate("Konline::serverDisconnect: Wrong passwd."));
      else
	debug(klocale->translate("Konline:serverDisconnect: Error while disconnecting."));
    }
  setConnected(FALSE);
}

void Konline::dataReceived()
{
  int rcode;
  char inbuf[1025];
  inbuf[0] = 0;
  unsigned long id;
  struct users rec;
  QString name;
  int win_id = -1;
  int i;
  for( i = 0; i < MAX_RECV_WIN; i++ )
    {
      if( recv_msg[i] == 0 )
	{
	  win_id = i;
	  break;
	}
    }
  if( win_id < 0 )
    rcode = serv->incomingData(&id, 0, &rec);
  else
    rcode = serv->incomingData(&id, inbuf, &rec);
  switch(rcode) {
  case USER_ONLINE:
    lb_watchlist->setActive(id, TRUE);
    break;
  case USER_OFFLINE:
    lb_watchlist->setActive(id, FALSE);
    break;
  case SERVER_SHUTDOWN:
    QMessageBox::critical( 0, klocale->translate("Konline"),
			   klocale->translate("Server was shut down.\nGoing offline.") );
    if( notifier )
      {
	delete notifier;
	notifier = 0;
      }
    toolbar->setItemEnabled(0, TRUE);
    toolbar->setItemEnabled(1, FALSE);
    lb_watchlist->setUpdatesEnabled(TRUE);
    lb_watchlist->clear();
    lb_watchlist->setUpdatesEnabled(FALSE);
    connected = 0;
    break;
  case STILL_ALIVE:
    // handled by Server object
    break;
  case INCOMING_MESSAGE:
    if( win_id >= 0 )
      {
	recv_msg[win_id] = new RecvMsg(win_id);
	connect( recv_msg[win_id], SIGNAL(close(int)), this, SLOT(closeMsgWin(int)) );
	connect( recv_msg[win_id], SIGNAL(reply(unsigned long, QString, QString, QString)), this, 
		 SLOT(sendMsg(unsigned long, QString, QString, QString)) );
	name = (QString) rec.fname;
	name.detach();
	name += (QString) ", " + rec.name + " (" + rec.nname + ")";
	recv_msg[win_id]->setRecv((const char *) name);
	recv_msg[win_id]->setMsg(inbuf);
	QString date = rec.date;
	QString formated = date.mid(6,2) + '.' + date.mid(4,2) + '.' + date.left(4) 
	  + ' ' + date.mid(8,2) + ':' + date.mid(10,2) + ' ' + date.right(3);
	recv_msg[win_id]->setDate((const char *) formated);
	recv_msg[win_id]->setInfo(&rec);
	recv_msg[win_id]->show();
      }
    break;
  };
}

void Konline::sendMsg()
{
  char date[16];
  time_t ltime;
  send_msg = new SendMsg();
  MainRow *recv;
  recv = (MainRow *) lb_watchlist->getRow(lb_watchlist->currentItem());
  QString name = recv->fname;
  name.detach();
  name += (QString) ", " + recv->name + " (" + recv->nname + ")";
  send_msg->setText(name);
  QString info, temp;
  while(1)
    {
      if( send_msg->exec() )
	{
	  if( send_msg->text().length() > MAX_QUEUED_MSG_LEN )
	    {
	      if( send_msg->text().length() > MAX_MSG_LEN )
		{
		  info = (QString) klocale->translate("Your message is ");
		  temp.setNum( send_msg->text().length() - MAX_MSG_LEN );
		  info += temp;
		  info += (QString) klocale->translate(" characters to long.\n");
		  info += (QString) klocale->translate("Please edit your message.\n");
		  QMessageBox::warning(0, klocale->translate("Konline"), info );
		  continue;
		}
	      info = (QString) klocale->translate("Your message is ");
	      temp.setNum( send_msg->text().length() - MAX_QUEUED_MSG_LEN );
	      info += temp;
	      info += (QString) klocale->translate(" characters to long.\n");
	      info += (QString) klocale->translate("Note that your message will be truncated,\n") +
		klocale->translate("if it has to be queued by the server.\n") +
		klocale->translate("Do you want to edit your message ?");
	      if( QMessageBox::information(0, klocale->translate("Konline"), info,
					   klocale->translate("Send"),
					   klocale->translate("Edit") ) == 1 )
		continue;
	    }

	  ltime = time(0);
	  strftime( date, 16, "%Y%m%d%H%M", localtime(&ltime) );
	  sprintf(date,"%s%+.2li", date, (long) timezone/(60*60)*(-1));
	  int rcode = serv->sendMessage( lb_watchlist->accId(lb_watchlist->currentRow()),
					 (const char *) send_msg->text(), date);
	  if( rcode == RE_MSG_QUEUED )
	    {
	      QMessageBox::information(0, klocale->translate("Konline"),
				       klocale->translate("Message queued for delivery.") );
	    }
	  if( rcode == RE_MSG_DELIVERED )
	    {
	      QMessageBox::information(0, klocale->translate("Konline"),
				       klocale->translate("Message delivered.") );
	    }
	  break;
	}
      else
	{
	  break;
	}
    }
  delete send_msg;
  send_msg = 0;
}

void Konline::sendMsg( unsigned long id, QString n, QString fn, QString nn )
{
  char date[16];
  time_t ltime;
  send_msg = new SendMsg();
  QString name = fn;
  name.detach();
  name += (QString) ", " + n + " (" + nn + ")";
  QString info, temp;
  send_msg->setText(name);
  while(1)
    {
      if( send_msg->exec() )
	{
	  if( send_msg->text().length() > MAX_QUEUED_MSG_LEN )
	    {
	      if( send_msg->text().length() > MAX_MSG_LEN )
		{
		  info = (QString) klocale->translate("Your message is ");
		  temp.setNum( send_msg->text().length() - MAX_MSG_LEN );
		  info += temp;
		  info += (QString) klocale->translate(" characters to long.\n");
		  info += (QString) klocale->translate("Please edit your message.\n");
		  QMessageBox::warning(0, klocale->translate("Konline"), info );
		  continue;
		}
	      info = (QString) klocale->translate("Your message is ");
	      temp.setNum( send_msg->text().length() - MAX_QUEUED_MSG_LEN );
	      info += temp;
	      info += (QString) klocale->translate(" characters to long.\n");
	      info += (QString) klocale->translate("Note that your message will be truncated,\n") +
		klocale->translate("if it has to be queued by the server.\n") +
		klocale->translate("Do you want to edit your message ?");
	      if( QMessageBox::information(0, klocale->translate("Konline"), info,
					   klocale->translate("Send"),
					   klocale->translate("Edit") ) == 1 )
		continue;
	    }

	  ltime = time(0);
	  strftime( date, 16, "%Y%m%d%H%M", localtime(&ltime) );
	  sprintf(date,"%s%+.2li", date, (long) timezone/(60*60)*(-1));

	  int rcode = serv->sendMessage( id, (const char *) send_msg->text(), date);
	  if( rcode == RE_MSG_QUEUED )
	    {
	      QMessageBox::information(0, klocale->translate("Konline"),
				       klocale->translate("Message queued for delivery.") );
	    }
	  if( rcode == RE_MSG_DELIVERED )
	    {
	      QMessageBox::information(0, klocale->translate("Konline"),
				       klocale->translate("Message delivered.") );
	    }
	  break;
	}
      else
	{
	  break;
	}
    }
  delete send_msg;
  send_msg = 0;
}

/*
void Konline::sendEmail()
{
  MainRow *recv;
  recv = (MainRow *) lb_watchlist->getRow(lb_watchlist->currentItem());
  QString name = recv->fname;
  name.detach();
  
}
*/

void Konline::showDetails()
{
  MainRow *user;
  user = (MainRow *) lb_watchlist->getRow(lb_watchlist->currentItem());
  if( !details )
    {
      details = new DetailsDialog;
      connect( details, SIGNAL(close()), this, SLOT(hideDetails()) );
    }
  details->setInfo(user->id,
		   user->name,
		   user->fname,
		   user->nname,
		   user->email);
  details->show();
}

void Konline::hideDetails()
{
  if( details )
    {
      delete details;
      details = 0;
    }
}

void Konline::removeList()
{
  if( QMessageBox::warning(0, klocale->translate("Remove user from watchlist"),
			   klocale->translate("Are you sure you want to remove
this user from your watchlist ?"),
			   klocale->translate("Ok"),
			   klocale->translate("Cancel"), 0, 0, 1) != 0 )
    {
      return;
    }
  if( serv->removeWatchList( lb_watchlist->accId(lb_watchlist->currentItem()) ) != RE_OK )
    {
      QMessageBox::critical( 0, klocale->translate("Konline"),
			     klocale->translate("I can't remove this user from your watchlist.") );
    }
  else
    {
      lb_watchlist->setUpdatesEnabled(FALSE);
      lb_watchlist->setAutoDelete(FALSE);
      int count = lb_watchlist->count();
      MainRow *array[count];
      int i, row;
      row = lb_watchlist->currentItem();
      for( i = 0; i < count; i++ )
	{
	  array[i] = (MainRow *) lb_watchlist->getRow(i);
	}
      lb_watchlist->clear();
      for( i = 0; i < count; i++ )
	{
	  if( i == row )
	    delete array[i];
	  else
	    lb_watchlist->insertItem( array[i] );
	}
      lb_watchlist->setUpdatesEnabled(TRUE);
      lb_watchlist->setAutoDelete(TRUE);
      lb_watchlist->repaint(TRUE);
    }
}

void Konline::closeMsgWin( int win_id )
{
  int i;
  if( recv_msg[win_id] )
    {
      recv_msg[win_id]->hide();
      delete recv_msg[win_id];
      recv_msg[win_id] = 0;
    }
  for( i = 0; i < MAX_RECV_WIN; i++ ) // test if there are still message windows
    {
      if( recv_msg[i] != 0 )
	return;
    }
  getMsg();  // no message windows, so check for new messages
}

void Konline::getMsg()
{
  serv->getMsg();
}

void Konline::popupMenu()
{
  if( lb_watchlist->currentRow() == -1 )
    return;  
  popup_menu->popup(QCursor::pos());
}

void Konline::sendKeepAlive()
{
  if( connected )
    {
      //debug("sending keep alive");
      serv->sendKeepAlive();
    }
}

void Konline::setConnected( bool status )
{
  if( status )
    {
      if( !notifier )
	{
	  notifier = new QSocketNotifier(serv->listenSocket(), QSocketNotifier::Read);
	  QObject::connect( notifier, SIGNAL(activated(int)),
			    this, SLOT(dataReceived()) );
	}
      toolbar->setItemEnabled(0, FALSE);
      toolbar->setItemEnabled(1, TRUE);
      file->setItemEnabled(0, FALSE);
      file->setItemEnabled(1, TRUE);
      file->setItemEnabled(2, TRUE);
      keep_timer->start(1000*KEEP_ALIVE_TIME);
      connected = 1;
    }
  else
    {
      if( notifier )
	{
	  delete notifier;
	  notifier = 0;
	}
      toolbar->setItemEnabled(0, TRUE);
      toolbar->setItemEnabled(1, FALSE);
      file->setItemEnabled(0, TRUE);
      file->setItemEnabled(1, FALSE);
      file->setItemEnabled(2, FALSE);
      lb_watchlist->setUpdatesEnabled(TRUE);
      lb_watchlist->clear();
      lb_watchlist->setUpdatesEnabled(FALSE);
      keep_timer->stop();
      connected = 0;
    }
}



