/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <stdlib.h>
#include <time.h>
#include <string.h>

#include <qprogressdialog.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qsize.h>
#include <qfont.h>
#include <qpushbutton.h>
#include <qtooltip.h>
#include <qstring.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "maildlg.h"
#include "filters.h"
#include "global.h"



MailDlg *maildlg = 0;



MailClient::MailClient() : DwSmtpClient()
{
	connected = false;
	srand(time(0));
}


MailClient::~MailClient()
{
	if (connected)
		disconnect();
}


bool MailClient::SendDataMult(const DwString& aStr)
{
	return SendDataMult(aStr.data(), aStr.length());
}


bool MailClient::SendDataMult(const char *aBuf, int aBufLen)
{
	return PSend(aBuf, aBufLen) == aBufLen;
}


void MailClient::createBoundary()
{
	int i;

	strcpy(boundary, "0_=_vwxy");
	for (i=8; i < 39; i++)
		boundary[i] = 65 + (int)(26.0 * rand() / (RAND_MAX + 1.0));
	boundary[39] = 0;
}


bool MailClient::connect(const char *server, int port)
{
	if (Open(server, port))
		if (Helo() == 250)
			connected = true;
		else
			Close();

	return connected;
}


bool MailClient::sendHeader(const char *from,  const char *to, const char *subj)
{
	DwString msg;

	createBoundary();

	if (Mail(from) != 250)
		return false;

	if (Rcpt(to) != 250)
		return false;

	if (Data() != 354)
		return false;

	msg.clear();

	if (*subj) {
		msg.append("Subject: ");
		msg.append(subj);
		msg.append("\r\012");
	}

	msg.append("X-Mailer: KMLOFax ");
	msg.append(VERSION);
	msg.append("\r\012MIME-Version: 1.0\r\012");
	msg.append("Content-Type: multipart/mixed;\r\012 boundary=\"");
	msg.append(boundary);
	msg.append("\"\r\012\r\012");

	return SendDataMult(msg);
}


bool MailClient::sendBody(const char *body)
{
	DwString msg;
	char buff[1000];
	int i = 0, j = 0;

	msg.assign("--");
	msg.append(boundary);
	msg.append("\r\012Content-Type: text/plain; charset=iso-8859-1\r\012");
	msg.append("Content-Transfer-Encoding: 8bit\r\012\r\012");

	if (!SendDataMult(msg))
		return false;

	while (body[i]) {
		buff[j++] = body[i++];
		if ((!body[i]) || (buff[j-1] == '\n') || (j > 996)) {
			if (buff[j-1] == '\n')
				j--;
			if ((j == 1) && (buff[0] == '.'))
				buff[j++] = '.';
			buff[j++] = '\r';
			buff[j++] = '\012';
			buff[j] = 0;
			if (!SendDataMult(buff, j))
				return false;
			j = 0;
		}
	}

	return true;
}


bool MailClient::sendFile(FILE *stream)
{
	DwString msg;
	char buff[258];
	int i;
	QProgressDialog *prog;

	fseek(stream, 0, SEEK_END);
	
	msg.assign("--");
	msg.append(boundary);
	msg.append("\r\012Content-Type: application/postscript; name=\"facsimile.ps\"\r\012");
	msg.append("Content-Transfer-Encoding: 7bit\r\012");
	msg.append("Content-Disposition: attachment; filename=\"facsimile.ps\"\r\012\r\012");

	if (!SendDataMult(msg)) {
		return false;
	}

	prog = new QProgressDialog(i18n("Sending mail..."), i18n("Cancel"), (int)ftell(stream), 0, "mailprogress", true);
	CHECK_PTR(prog);
	prog->setCaption(i18n("Sending Mail"));

	rewind(stream);
	prog->setProgress(0);
	
	while (fgets(buff, 256, stream)) {
		if (prog->wasCancelled()) {
			delete prog;
			return false;
		}

		i = strlen(buff);
		if (i &&  buff[i-1] == '\n')
			i--;
		buff[i++] = '\r';
		buff[i++] = '\012';
		buff[i] = 0;
		if (!SendDataMult(buff, i)) {
			delete prog;
			return false;
		}

		prog->setProgress((int)ftell(stream));
	}

	delete prog;

	msg.assign("\r\012--");
	msg.append(boundary);
	msg.append("--\r\012");

	return SendData(msg) == 250;
}


void MailClient::disconnect()
{
	Quit();
	Close();
}



MailDlg::MailDlg(QWidget *parent, const char *name) : QDialog (parent, name)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QGroupBox *gb;
	QGridLayout *grid;
	QLabel *label;
	QPushButton *ok, *cancel;
	QSize size;
	QFont font;

	config = kapp->getConfig();
	
	gotToolTips = false;

	vbox = new QVBoxLayout(this, 8);

	gb = new QGroupBox(this);

	grid = new QGridLayout(gb, 2, 2, 8);

	label = new QLabel(i18n("&To:"), gb);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0);

	header_to = new QLineEdit(gb);
	label->setBuddy(header_to);
	font = header_to->font();
	font.setFamily("courier");
	header_to->setFont(font);
	header_to->setMinimumSize(header_to->sizeHint());
	grid->addWidget(header_to, 0, 1);

	label = new QLabel(i18n("&Subject:"), gb);
	label->setMinimumSize(label->sizeHint());

	grid->addWidget(label, 1, 0);

	header_subject = new QLineEdit(gb);
	label->setBuddy(header_subject);
	header_subject->setFont(font);
	header_subject->setMinimumSize(header_subject->sizeHint());
	grid->addWidget(header_subject, 1, 1);

	grid->setColStretch(1, 1);

	grid->activate();

	vbox->addWidget(gb, 0);

	body = new QMultiLineEdit(this);
	body->setFont(font);
	
	vbox->addWidget(body, 1);

	hbox = new QHBoxLayout(8);
	vbox->addLayout(hbox);

	ok = new QPushButton(i18n("OK"), this);
	ok->setDefault(true);

	cancel = new QPushButton(i18n("Cancel"), this);

	size = ok->sizeHint();
	size = size.expandedTo(cancel->sizeHint());

	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addStretch(1);
	hbox->addWidget(ok);
	hbox->addWidget(cancel);

	vbox->activate();

	connect(ok, SIGNAL(clicked()), SLOT(okClicked()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	setCaption(i18n("KMLOFax Mail"));
}


MailDlg::~MailDlg()
{
}


void MailDlg::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(header_to, i18n("Email address of the receiver"));
		QToolTip::add(header_subject, i18n("Subject of the email"));
		QToolTip::add(body, i18n("Body of the email"));
	}
	else {
		QToolTip::remove(header_to);
		QToolTip::remove(header_subject);
		QToolTip::remove(body);
	}
}	


void MailDlg::mailFax(const QString& name, const QString& alias)
{
	QString s;

	fname = name;
	salias = alias;

	if (alias.isEmpty())
		s = i18n("Facsimile");
	else
		s.sprintf(i18n("Facsimile from %s"), (const char *)salias);
	header_subject->setText(s);
	body->setText(i18n("This is a facsimile file forwarded with KMLOFax."));

	if (isVisible())
		raise();
	else
		show();
}


void MailDlg::okClicked()
{
	MailClient mclient;
	QString from, server, name, org, email, reply;
	int port;
	MLO2PSFilter filter;
	FILE *f;
	
	config->setGroup(ID_PREF_GROUP_MAIL);
	server = config->readEntry(ID_PREF_MAIL_SERVER, PREF_MAIL_SERVER);
	port = config->readNumEntry(ID_PREF_MAIL_PORT, PREF_MAIL_PORT);
	name = config->readEntry(ID_PREF_MAIL_NAME, PREF_MAIL_NAME);
	org = config->readEntry(ID_PREF_MAIL_ORG, PREF_MAIL_ORG);
	email = config->readEntry(ID_PREF_MAIL_EMAIL, PREF_MAIL_EMAIL);
	reply = config->readEntry(ID_PREF_MAIL_REPLY, PREF_MAIL_REPLY);
	
	if (server.isEmpty()) {
		KMsgBox::message(0, i18n("Mail Error"), i18n("No SMTP server specified."), KMsgBox::EXCLAMATION);
		return;
	}

	if (email.isEmpty()) {
		KMsgBox::message(0, i18n("Mail Error"), i18n("No sender specified."), KMsgBox::EXCLAMATION);
		return;
	}

	if (!*(header_to->text())) {
		KMsgBox::message(0, i18n("Mail Error"), i18n("No receiver specified."), KMsgBox::EXCLAMATION);
		return;
	}

	if (!mclient.connect(server, port)) {
		KMsgBox::message(0, i18n("SMTP Error"), i18n("Cannot connect to server."), KMsgBox::EXCLAMATION);
		return;
	}

	from = name;

	if (name.isEmpty())
		from = email;
	else {
		from += " <";
		from += email;
		from += ">";
	}

	if (!mclient.sendHeader(from.data(), header_to->text(), header_subject->text())) {
		KMsgBox::message(0, i18n("SMTP Error"), i18n("Cannot send mail header."), KMsgBox::EXCLAMATION);
		mclient.disconnect();
		return;
	}

	if (!mclient.sendBody(body->text().data())) {
		KMsgBox::message(0, i18n("SMTP Error"), i18n("Cannot send mail body."), KMsgBox::EXCLAMATION);
		mclient.disconnect();
		return;
	}

	if (!(f = tmpfile())) {
		KMsgBox::message(0, i18n("File Error"), i18n("Cannot create temporary file."), KMsgBox::EXCLAMATION);
		mclient.disconnect();
		return;
	}
	
	filter.setFile(fname);
	
	if (!filter.convertFile(f, salias)) {
		fclose(f);
		mclient.disconnect();
		return;
	}

	if (!mclient.sendFile(f)) {
		KMsgBox::message(0, i18n("SMTP Error"), i18n("Cannot send file."), KMsgBox::EXCLAMATION);
	}

	mclient.disconnect();

	hide();
}
