/*
    KMLOFax
    
    A utility to process facsimile received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 1999-2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <unistd.h>
#include <stdlib.h>

#include <qglobal.h>

#include <kapp.h>
#include <kmsgbox.h>

#include "downloaddlg.h"
#include "global.h"



DownloadDlg::DownloadDlg(QWidget *parent, const char *name, bool modal, WFlags f) : QProgressDialog(parent, name, modal, f)
{
	config = kapp->getConfig();
	
	modem = new Modem(this, "modem");
	CHECK_PTR(modem);
	init();

	setMinimumDuration(0);

	connect(modem, SIGNAL(gotLine(const char *)), SLOT(fetchModemLine(const char *)));
	connect(modem, SIGNAL(gotXBlock(const uchar *, int)), SLOT(fetchXModemBlock(const uchar *, int)));
	connect(modem, SIGNAL(xmodemDone(bool)), SLOT(fetchXModemDone(bool)));
	connect(modem, SIGNAL(timeout()), SLOT(fetchTimeout()));
	connect(this, SIGNAL(cancelled()), SLOT(cancel()));

	setCaption(i18n("Download"));
}


DownloadDlg::~DownloadDlg()
{
	reset();
}


bool DownloadDlg::startDownload()
{
	static int speeds[] = { 19200, 38400, 57600, 115200, 230400 };
	QString dev;
	int speed;

	if (status)
		return false;

	config->setGroup(ID_PREF_GROUP_GENERAL);
	load_new = config->readBoolEntry(ID_PREF_GEN_LOADNEW, PREF_GEN_LOADNEW);
	delete_fax = config->readBoolEntry(ID_PREF_GEN_DELETEFAX, PREF_GEN_DELETEFAX);

	config->setGroup(ID_PREF_GROUP_MODEM);
	dev = config->readEntry(ID_PREF_MOD_DEVICE, PREF_MOD_DEVICE);
	speed = config->readNumEntry(ID_PREF_MOD_SPEED, PREF_MOD_SPEED);
	if ((speed < 0) || (speed > 4))
		speed = PREF_MOD_SPEED;
	
	modem->setDevice(dev);
	modem->setSpeed(speeds[speed]);
	modem->setData(8);
	modem->setParity('N');
	modem->setStop(1);

	if (!modem->open()) {
		KMsgBox::message(0, i18n("Modem Error"), i18n("Cannot open modem device."), KMsgBox::EXCLAMATION);
		return false;
	}
	if (!modem->dsrOn()) {
		KMsgBox::message(0, i18n("Modem Error"), i18n("Modem is off."), KMsgBox::EXCLAMATION);
		modem->close();
		return false;
	}
	if (!modem->ctsOn()) {
		KMsgBox::message(0, i18n("Modem Error"), i18n("Modem is busy."), KMsgBox::EXCLAMATION);
		modem->close();
		return false;
	}

	modem->writeLine("");
	usleep(250000);
	modem->flush();
	modem->writeLine("ATZ");

	status = 1;

	return true;
}


void DownloadDlg::cancel()
{
	QProgressDialog::cancel();

	if (status == 3)
		modem->abortXModem();
	else
		reset();
}


void DownloadDlg::fetchModemLine(const char *line)
{
	char buff[81], *p;
	fitem_t *fitem;
	int year, month, day, hour, minute, size;

	switch (status) {
		case 1:					/* "ATZ" sent	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				if (load_new)
					modem->writeLine("AT$JDIR=,FAX,N");
				else
					modem->writeLine("AT$JDIR=,FAX");
				status++;
			}
			break;
		case 2:					/* "AT$JDIR=,FAX[,N]" sent	*/
			if (strlen(line) > 59) {
				usleep(250000);
				strncpy(buff, line, 80);
				buff[80] = 0;
				if (sscanf(&buff[41], "%d.%d.%d, %d:%d, %d", &day, &month, &year, &hour, &minute, &size) == 6) {
					fitem = (fitem_t *)malloc(sizeof(fitem_t));
					CHECK_PTR(fitem);
					fitem->next = fitems;
					fitems = fitem;
					if ((p = strchr(buff, ',')))
						*p = 0;
					strncpy(fitems->name, buff, 18);
					fitems->name[18] = 0;
					sprintf(fitems->file, "%04d%02d%02d%02d%02d00.FAX", year, month, day, hour, minute);
					if (!strncmp(fitems->name, fitems->file, 12)) {
						if ((fitems->name[12] >= '0') && (fitems->name[12] <= '9'))
							fitems->file[12] = fitems->name[12];
						if ((fitems->name[13] >= '0') && (fitems->name[13] <= '9'))
							fitems->file[13] = fitems->name[13];
					}
					fitems->size = size;
					nfiles++;
				}
			} else if (!strcmp(line, "OK")) {
				usleep(250000);
				count = 0;
				status++;
				downloadFile();
			}
			break;
		case 3:					/* XModem confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);

				if (delete_fax) {
					sprintf(buff, "AT$JDEL=\"%s\"", fitems->name);
					modem->writeLine(buff);
					status++;
				} else {
					setProgress(fitems->size + 1);
					fitem = fitems;
					fitems = fitems->next;
					free(fitem);
					downloadFile();
				}
			}
			break;
		case 4:					/* Delete confirmed	*/
			if (!strcmp(line, "OK")) {
				usleep(250000);
				setProgress(fitems->size + 1);
				fitem = fitems;
				fitems = fitems->next;
				free(fitem);
				status = 3;
				downloadFile();
			}
	}
}


void DownloadDlg::fetchXModemBlock(const uchar *block, int size)
{
	int diff;

	if (file.isOpen() && ((diff = fitems->size - progress()) > 0)) {
		if (size > diff)
			size = diff;
		file.writeBlock((const char *)block, size);
		setProgress(progress () + size);
	}
}


void DownloadDlg::fetchXModemDone(bool success)
{
	QString s;

	if (file.isOpen()) {
		file.close();
		if (success)
			emit gotFile(fitems->file);
		else {
			emit message(s.sprintf("%s: %s", i18n("Error"), i18n("Download failed.")));

			reset();
		}
		setLabelText(i18n("Waiting for confirmation..."));
	}
}


void DownloadDlg::fetchTimeout()
{
	QString s;
	
	KMsgBox::message(0, i18n("Modem Error"), i18n("Modem response timeout."), KMsgBox::EXCLAMATION);

	emit message(s.sprintf("%s: %s", i18n("Error"), i18n("Modem response timeout.")));

	reset();
}


void DownloadDlg::init()
{
	load_new = false;
	delete_fax = false;
	
	status = 0;

	fitems = 0;
	nfiles = 0;
	count = 0;
}


void DownloadDlg::reset()
{
	fitem_t *p;

	QProgressDialog::reset();

	modem->close();

	while (fitems) {
		p = fitems;
		fitems = fitems->next;
		free(p);
	}

	if (file.isOpen())
		file.close();

	init();

	emit done();
}


void DownloadDlg::downloadFile()
{
	QString s;

	if (fitems) {
		count++;
		file.setName(expandPath(fitems->file));
		if (!file.open(IO_WriteOnly)) {
			KMsgBox::message(0, i18n("File Error"), i18n("Cannot create file."), KMsgBox::EXCLAMATION);
			emit message(s.sprintf("%s: %s!", i18n("Error"), i18n("Download failed")));
			reset();

			return;
		}
		modem->writeLine(s.sprintf("AT$JDNL=\"%s\"", fitems->name));
		setLabelText(s.sprintf(i18n("Receiving file %d of %d..."), count, nfiles));
		setTotalSteps(fitems->size + 1);
		setProgress(0);
		modem->receiveXModem(true);
	} else {
		if (nfiles)
			emit message(s.sprintf(i18n("Received %d file(s)."), nfiles));
		else
			emit message(i18n("No facsimile in modem."));

		reset();
	}
}
