#include "mem_zone.h"
#include "memdata.h"

#include <qmsgbox.h>
#include <qcolor.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qdatetime.h>

#include "memdata.h"

MemZone::MemZone( MemData *m, QWidget *parent, const char *name)
	: QWidget(parent, name), memdata( m )
{
	painter = new QPainter() ;
	draw_buffer = new QPixmap( size() ) ; 

	page_offset = 0 ;
	setPagesOnRow( DEFAULT_PAGES_ON_ROW ) ;

	memcolors[ MEM_NONE ] = backgroundColor() ;

	connect( m, SIGNAL(dataChanged()),
			this, SLOT(redrawBuffer()) ) ;
}

MemZone::~MemZone()
{
	#ifdef DEBUG
	printf("Deleting MemZone\n");
	#endif
}

/* initialize static members */ 
const int MemZone::MIN_PAGES_ON_ROW = 10 ;
const int MemZone::MAX_PAGES_ON_ROW = 400 ;
const int MemZone::DEFAULT_PAGES_ON_ROW = 124 ;

QColor MemZone::memcolors[] = { QColor(white), QColor(255,128,0),
		QColor(128,128,0), QColor(255,0,0), QColor(blue),
			QColor(white) } ;

QColor MemZone::getColor( int pos )
{
	if ( pos>=0 && pos<number_of_colors )
		return memcolors[pos] ;
	else return QColor(black) ; /* TODO fix this up */
}

void MemZone::setColor( QColor c, int pos )
{
	if ( pos>=0 && pos<number_of_colors )
		memcolors[pos]=c ;

}

const char * MemZone::getMemoryType( int type )
{
	switch( type )
	{
		case MEM_FREE:		return i18n("Free memory") ;
		case MEM_RESERVED:	return i18n("Reserved memory") ;
		case MEM_USED:		return i18n("Used memory") ;
		case MEM_BUFFER:	return i18n("Buffer memory") ;
		case MEM_CACHED:	return i18n("Cache memory") ;
		case MEM_NONE:		return i18n("Nonexistent memory") ;
		default:
			return i18n("No such memory") ;
	}
}

void MemZone::redrawBuffer()
{
	#ifdef DEBUG
	//debug("MemZone::redrawBuffer()" ) ;
	//QTime time ;
	//time.start() ;
	#endif


	if( draw_buffer==0 || draw_buffer->isNull() )
		return ;

	char t ;
	int pages = memdata->numberOfPages() ;

	/* watch out, this pointer could be 0 ! */
 	const char *page_pointer = memdata->getPagePointer() ;
	if( pages==0 )
		return ;

	/* start drawing here */
	painter->begin( draw_buffer ) ;

	int offset = lineOffset() ;

	/* TODO: don't update entire buffer */
	for( int j=offset; j<pagesInColumn(); j++ )
	for( int i=0; i<pagesOnRow(); i++ )
	{
		int pos = i+pagesOnRow()*j ;
		if ( pos >= pages )
			t = MEM_NONE ;
		else	t = page_pointer[pos] ;
		//printf("page_pointer was %d\n", page_pointer[pos]);

		painter->setPen( memcolors[t] ) ;
		painter->fillRect( i*page_width,
				j*page_height, page_width,
				page_height, memcolors[t] ) ;
 	}

	painter->end() ; 

	paintEvent( 0 ) ;

	#ifdef DEBUG
	//printf("time elapsed: %d\n", time.elapsed() ) ; 
	#endif
}

void MemZone::redraw()
{
	update() ;
}

void MemZone::paintEvent( QPaintEvent * )
{
	#ifdef DEBUG
	//printf("MemZone: paintEvent\n" ) ;
	#endif

	int offset = lineOffset()*page_height ;
	int w = pagesOnRow()*page_width;
	int h = pagesInColumn()*page_height ;
	bitBlt( this, 0, 0, draw_buffer, 0, offset, w, h ) ;
}


int MemZone::lineOffset()
{
	return int(pageOffset()/pagesOnRow()) ;
}

int MemZone::linesVisible()
{
	printf("linesVisible() : page_height %f, he() %d\n",
			page_height, he() ) ;
	return (int)(he()/page_height) ;
}

/* inline int MemZone::pageOffset() ; */ 

void MemZone::setPageOffset( int  p )
{
	if ( page_offset != p && p>=0 && p<numberOfPages() )
	{
		#ifdef DEBUG
		//printf("MemZone: settting page_offset to %d\n", p ) ;
		#endif
		page_offset = p ;
		emit pageOffsetChanged(p) ;
		emit scrollOffsetChanged( pageToScroll(p) ) ;
	}

	update() ;
}

int MemZone::scrollOffset()
{
	return pageToScroll( page_offset ) ;
}

void MemZone::setScrollOffset( int p )
{
	setPageOffset( scrollToPage( p ) ) ; 

	#ifdef DEBUG
	//printf("MemZone::setScrollOffset(%d)->%d\n", p,scrollToPage(p) ) ;
  	#endif
}

int MemZone::scrollToPage( int s )
{
	return s ;

/*	int visible = linesVisible()*pagesOnRow() ;
	printf("visible: %d\n", visible ) ;

	int scroll_end = numberOfPages() - visible ;

	return (int)( s*scroll_end / numberOfPages() ) ; 
*/
}

int MemZone::pageToScroll( int p )
{
	return p ;
/*	int visible = linesVisible()*pagesOnRow() ;
	int scroll_end = numberOfPages() - visible ;

	int ret = ( p*numberOfPages()/scroll_end ) ; 

	return ret<numberOfPages()?ret:numberOfPages() ;
*/
}

void MemZone::addToPageOffset( int  p )
{
	if( page_offset+p>=0 && page_offset+p<memdata->numberOfPages() ) 
		page_offset+=p ;

	emit pageOffsetChanged(page_offset) ;  

	update() ;
}

/* inline int MemZone::pagesInColumn() ; */

void MemZone::setPagesInColumn()
{
	pages_in_column = 1+(memdata->numberOfPages()/pagesOnRow()) ; 
}

/* inline int MemZone::pagesOnRow() ; */

void MemZone::setPagesOnRow( int i )
{
	if ( i>=MIN_PAGES_ON_ROW && i< MAX_PAGES_ON_ROW )
	{
		pages_on_row = i ;
		setPagesInColumn() ;

		resizeEvent( 0 ) ;
	}
}

void MemZone::mouseMoveEvent( QMouseEvent *m )
{
	if( mouse_pressed )
	{
		#ifdef DEBUG
		//printf("MemZone: mouseMoveEvent\n" ); 
		#endif

		int diff = mouse_position.y() - m->pos().y() ;
		diff *= pagesOnRow()/page_height ;

		setPageOffset( last_page_offset + diff ) ;		
	}
}
 
void MemZone::mouseReleaseEvent( QMouseEvent *m ) 
{
	mouse_pressed = false ;
}

void MemZone::leaveEvent( QEvent *e ) 
{
	mouse_pressed = false ;
}

void MemZone::mousePressEvent( QMouseEvent *m ) 
{
	if( m->button() & LeftButton!=0 )
	{
		/* scroll widget */
		mouse_pressed=true ;
		mouse_position = m->pos() ;
		last_page_offset = pageOffset() ;
	}
	else	emit notifyMousePressEvent( m ) ;
}
 
void MemZone::resizeEvent( QResizeEvent *r )
{
	#ifdef DEBUG
	//printf("MemZone::resizeEvent()\n" ) ;
	#endif

	int width, height ;

	QSize s = size() ;
	width = s.width() ;

	page_width = width/pages_on_row ;
	page_height = page_width ;
	height = page_height * pages_in_column ;

	draw_buffer->resize( width, height );
	draw_buffer->fill( memcolors[MEM_NONE] ) ;

 	redrawBuffer() ;
}


int MemZone::wi()
{
	return this->width() ;
}

int MemZone::he()
{
	return this->height() ;
}


