/****************************************************************
**
** Implementation of LPanel class
**
****************************************************************/

extern "C" {
    #include <unistd.h>
}

#include <iostream.h>
#include <fstream.h>

#include "lpanel.h"

#include <kprocess.h>


const int LPanel::MIN_WIDTH  = 10;
const int LPanel::MIN_HEIGHT = 10;

// Constructor
LPanel::LPanel(int n,QWidget *parent, const char *name )
        : QWidget( parent, name )
{
    numPoints = 0;
    numGraphs = n;
    toolTip = "";
    description = "";
    name = "";
    yScale = 1.0;
    xScale = 1;  // FIXME: this is not yet correctly implemented!
    // redrawInterval = 2 * QSECOND; // in milliseconds
    doShift = false;
    
    myWidth = myHeight = 0;
    
    setDoubleClickCmd("");
    setLowWM(0); setHighWM(0);
    setLowBoundCmd("");
    setHighBoundCmd("");
    lowWMTriggerEnabled = highWMTriggerEnabled = true;
    
    incremental = true;
    filled = true;
    
    setMinimumSize(MIN_WIDTH,MIN_HEIGHT);
    
    graphColors = new QColor[numGraphs];
    setBackgroundColor(QColor(gray));
    
    setYLines(-1); // DEFAULT: Do not draw scale lines
    scaleColor = QColor(black);
 
    data = new FArray[numGraphs];
    for (int i = 0;i < numGraphs;i++) {
        data[i] = FArray(width());
    }
}


// Destructor 
LPanel::~LPanel() 
{
    delete[] data;
    delete[] graphColors;
}


void LPanel::setToolTip(const char *s) 
{
    toolTip = s;
    QToolTip::add(this,toolTip);
}


// PaintEvent: Called after resize and expose events
void LPanel::paintEvent(QPaintEvent *)
{
    fullPaint();
}


// Slot normally called by timer Events
void LPanel::wakeUp()
{
    resample();         // Gain new data
    if (!isVisible()) 
        return;
        
    if (rescale()) {;   // Get new y scaling
        fullPaint();    // draw panel
    } 
    else  {
        diffPaint();
    }
       
    intervalCmds();     // Check Interval Violations and trigger Cmds
}


// Do the hard part I: draw panel using actual geometry, colors, scale and data
// preparing steps
int LPanel::rescale()
{
    float yMax;
    
    // Compute max. y value
    yMax = -1;
    for (int i = 0;i < numGraphs;i++) {
        for (int j = 0;j < numPoints;j++) {
            if (data[i][j] > yMax)  
                 yMax = data[i][j];
        }
    }
   
    if (yLines > 0.0) {
        float oldYScale = yScale;
        yScale = int((yMax + getYLines()) / getYLines()) * getYLines();
        
        return (yScale != oldYScale); 
    }
    // else: yScale remains unchanged
    
    return false; // No need to repaint
}


void LPanel::intervalCmds()
{
    if (numPoints > 0) {
        // Compute max. value
        float max;
        max = data[0][numPoints - 1];
        for (int i = 1;i < numGraphs;i++) {
            if (data[i][numPoints - 1] > max) {
                max = data[i][numPoints - 1];
            }
        }

        // Falling below lower water mark
        if (max < lowWM && lowWM > 0) {
            if (lowWMTriggerEnabled) {
                callCmd(getLowBoundCmd());
            }
            lowWMTriggerEnabled = false;
            highWMTriggerEnabled = true;
        }
        else if (max > highWM && highWM > 0) {
            if (highWMTriggerEnabled) {
                callCmd(getHighBoundCmd());
            }
            lowWMTriggerEnabled = true;
            highWMTriggerEnabled = false;
        }         
    }
}


// Do the hard part II: draw panel using actual geometry, colors, scale and data
// Draw the complete Panel
void LPanel::fullPaint()
{
    QPixmap  pix(this->size() );  
      
    int x1,y1,x2,y2;
    QPainter p, panelPainter(this);
    int w = width(), h = height();
    
    pix.fill(this,0,0);
    p.begin(&pix);

    p.translate(0,height());
    p.scale(1,-1.0 / yScale);
   
    for (int i = 0;i < numGraphs;i++) {
        QPen pen(graphColors[i]);
        p.setPen(pen);
        
        if (filled) {          // always incremental
            for (int j = 1;j < numPoints;j++) {
                if (i == 0) {
                    p.drawLine(j, 0, j, int(h * data[i][j] + 0.5) ); 
                }
                else {
                    p.drawLine(j, int(h * data[i-1][j] + 0.5), 
                               j, int(h * data[i][j]   + 0.5));
                }
            }
        }
        else {
            x1 = 0;
            y1 = int(h * data[i][0]);
        
            p.moveTo(x1,y1);
            for (int j = 1;j < numPoints;j++) {
                x2 = j * xScale;
                y2 = int(h * data[i][j]);
                p.lineTo(x2,y2);
                x1 = x2;
                y1 = y2;
            }
        }
    }
    
    if (yLines > 0) { // Draw scale lines!
        p.setPen(QPen(scaleColor));
        
        for (float i = yLines;i < yScale;i+= yLines) {
            p.drawLine(0, int(i * h) , w, int(i * h));
       }
    }
    
    p.end();                          // Finish pixmap painting
    panelPainter.drawPixmap(0,0,pix); // Copy pixmap to LPanel widget
}


// Do paint the last column
void LPanel::diffPaint() 
{
    int x = numPoints - 1;
    int h = height();
    int w = width();
    
    // shift left by one if necessary
    if (x == (w - 1)) {
        if (doShift) {
            bitBlt(this,0,0,this,1,0);
        } 
        doShift = true;
    }
    
    QPainter p(this);

    p.setPen(QPen(backgroundColor()));
    p.drawLine(x,0,x, h - 1);
    
    // Draw LAST row
    p.translate(0, h);
    p.scale(1,-1.0 / yScale);
        
   for (int i = 0;i < numGraphs;i++) {
        QPen pen(graphColors[i]);
        p.setPen(pen);
        if (filled) {      // Always incremental
            if (i == 0) {
                p.drawLine(x, 0, x, int(h * data[i][x] + 0.5) ); 
            }
            else {
                p.drawLine(x, int(h * data[i-1][x] + 0.5), 
                           x, int(h * data[i][x]   + 0.5));
            }
        }
        else {  // Not filled
            int myX = (x == 0) ? 0 : x - 1;
            p.drawLine(myX, int(h * data[i][myX]), x, int(h * data[i][x]));
        }
    }
        
    if (yLines > 0) { // Draw scale lines!
        p.setPen(QPen(scaleColor));
        
        for (float i = yLines;i < yScale;i+= yLines) {
            p.drawPoint(x, int(h * i));
        }
    }
 }

// Slot called by resize events
// Reallocate internal data structures 
// No need to deal with changes in height
void LPanel::resizeEvent(QResizeEvent *re)
{
    this->setSize(re->size().width(),re->size().height());
}


void LPanel::setSize(int new_w, int new_h)
{
    int old_w = myWidth;
    
    if (new_w > old_w) {
        doShift = false;
    }
    
    if (new_w < numPoints) {
        lshift(numPoints - new_w);
        numPoints = new_w;
    }
    
    if (new_w != old_w) {
        for (int i = 0;i < numGraphs;i++) {
            data[i].resize(new_w);
        }
    }
    
    myWidth = new_w;
    myHeight = new_h;    
}


// left shift data by <num> points
void LPanel::lshift(int num)
{
    for (int i = 0;i < numGraphs;i++) {
        for (int j = 0;j < (numPoints - num);j++) {
            data[i][j] = data[i][j+num];
        }
    }
    numPoints -= num;
}


// Prepare for Insertion of next Column, i.e. shift left by one if
// necessary
void LPanel::nextColumn()
{
    int w = myWidth;
    if (numPoints > w) {
        cerr << "LPanel::nextColumn(): numPoints > myWidth" << endl;
        exit(-1);
    }
    else if (numPoints == w) {
        lshift(1);
    }
}


void LPanel::resample() {
    FArray row(getNumGraphs());
    
    if (getNewData(&row)) {
        append(row);
    }
}


// This is the very function for derived classes to append new data!
void LPanel::append(FArray &r)
{
    if (width() == 0)
        return;
    
    nextColumn();
    
    float sum = 0.0;
    for (int i = 0;i < numGraphs;i++) {
        data[i][numPoints] = sum + r[i];
        if (incremental) {
            sum += r[i];
        }
    }
    numPoints++;
}


// This signal is triggered upon double click
void LPanel::mouseDoubleClickEvent(QMouseEvent *)
{
    callCmd(getDoubleClickCmd());
}


// Call a shell cmd asynchronously
void LPanel::callCmd(const QString s)
{
    if (s.isNull() || s.isEmpty() || (s.stripWhiteSpace()).length() == 0)
        return;
        
    int pid;
    if ((pid = fork()) < 0) {
        cerr << "Cannot fork in LPanel::callCmd()" << endl;
        return;
    }
    else if (pid > 0) {
        return;
    }
    else {
       execlp("/bin/sh", "sh", "-c", s.data(), 0);
    }
}


void LPanel::mousePressEvent(QMouseEvent *e)
{
    if (e->button() == RightButton) {
        emit rightMouseButtonPressed(mapToGlobal(e->pos()));
    }

}

