/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: StdVid.cpp,v 1.3 1999/01/14 17:33:06 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>

#include "StdVid.h"

#define RANGE_INCREMENT 1
#define RANGE_LOWER_LIMIT 3
#define RANGE_UPPER_LIMIT 15

StdVid::StdVid( 
  int sysc1, 
  int sysc2, 
  int sysc3,
  int sysc4,
  double grMin, 
  double grMax ) : Sensor(sysc1, sysc2, sysc3, sysc4, grMin, grMax)
{
  SensorValue val;
  double scaleFactor;
  
  // the vid sensor is a special voltage sensors which has no
  // name (according to lm_sensors). We provide local storage
  // and defaults for its name. It also has no limits or alarms
  // that it can trigger. Documentation indicates that it is just
  // a more accurate Vcore reading. This class provides emulated 
  // limits for this sensor.
  if(Sensor::getSettings( val, sensorName, scaleFactor ))
  {
    setName( sensorName );
    lowAlarm=val.min;
    highAlarm=val.max;
  }
  else
  {
    lowAlarm = grMin;
    highAlarm = grMax;
  }	
  if (sensorName.length()==0)
  {
    setName( "VID" );
  }	
}

StdVid::~StdVid()
{
}

void StdVid::name( QString &name )
{	
  name = sensorName;
}

bool StdVid::setName( QString &name )
{
  sensorName = name;
  return true;
}	


bool StdVid::save()
{
  SensorValue val;
  
  if( read( val ) )
  {
    return Sensor::saveSettings( val, sensorName, 0 );
  }
  
  return false;
}

bool StdVid::read( SensorValue &val )
{
  if (!Sensor::getVidVolt( val ))
  {
    val.min = val.max = val.value = 0;
    return false;
  }
  else
  {
    val.value = (val.value/100);
    val.min = lowAlarm;
    val.max = highAlarm;
  }
  
  return true;
}

bool StdVid::write( SensorValue &val )
{
  lowAlarm = val.min;
  highAlarm = val.max;
  
  return true;
}

bool StdVid::displayDecimals()
{
  return true;
}

bool StdVid::displayUnits()
{
  return true;
}

void StdVid::units( QString &unitStr )
{
  unitStr="V";
}

bool StdVid::increaseGraphRange()
{
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max += RANGE_INCREMENT;
  
  if (max > RANGE_UPPER_LIMIT)
  {
    max = RANGE_UPPER_LIMIT;
  }
  
  return Sensor::setGraphRange( min ,max );
}

bool StdVid::decreaseGraphRange()
{
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max -= RANGE_INCREMENT;
  
  if (max < RANGE_LOWER_LIMIT)
  {
    max = RANGE_LOWER_LIMIT;
  }
  
  return Sensor::setGraphRange( min ,max );	
}

bool StdVid::scalable()
{
  return false;
}

bool StdVid::isAlarm()
{
  SensorValue val;
  
  read( val );
  val.min = fabs(val.min);
  val.max = fabs(val.max);
  val.value = fabs(val.value);
  
  return ( (val.value < val.min) || (val.value > val.max) );
}

bool StdVid::lowAlarmSettable()
{
  return false;
}

bool StdVid::highAlarmSettable()
{
  return false;
}
