/*
 *            klm: A lm_sensors front end for the KDE project
 *
 * $Id: MinMaxTemp.cpp,v 1.2 1999/02/13 11:45:59 humphrey Exp $
 *
 *            Copyright (C) 1998 Brendon Humphrey
 *                   brendy@swipnet.se
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>

#include "MinMaxTemp.h"

#define RANGE_INCREMENT 5
#define RANGE_LOWER_LIMIT 10
#define RANGE_UPPER_LIMIT 100

#define OFFSET_INCREMENT 0.25
#define OFFSET_LOWER_LIMIT -20
#define OFFSET_UPPER_LIMIT 20

MinMaxTemp::MinMaxTemp( 	
  int sysc1, 
  int sysc2, 
  int sysc3,
  int sysc4,
  double grMin, 
  double grMax,
  char *defaultName ) : Sensor(sysc1, sysc2, sysc3, sysc4, grMin, grMax)
{
  SensorValue	val;
  double 		scaleFactor;
  
  // attempt to read the sensor config from the application config
  // and write the config to the lm_sensors module. 
  // must be root for the write operation to succeed. If it fails
  // the klm will read existing data from the sensors.
  
  if(Sensor::getSettings( val, sensorName, scaleFactor ))
  {
    // scale factor is ignored - not supported in hardware		
    write( val );
    setName( sensorName );
  }
  else
  {
    setName( defaultName );	
  }
}

MinMaxTemp::~MinMaxTemp()
{
}

bool MinMaxTemp::save()
{
  QString		sensorName;
  SensorValue	val;
  
  name( sensorName );
  if( read( val ) )
  {
    return Sensor::saveSettings( val, sensorName, 0 );
  }
  
  return false;
}

void MinMaxTemp::name( QString &name )
{
  name = sensorName;	
}

bool MinMaxTemp::setName( QString &name )
{
  sensorName = name;	
  return true;
}

bool MinMaxTemp::read( SensorValue &val )
{
  double offset;
  
  offset=Sensor::getOffset();
  if( Sensor::getTemp( val ) )
  {
    val.value = (val.value/10) + offset;
    val.min = (val.min/10)  + offset;
    val.max = (val.max/10) + offset;	
  }
  else
  {
    val.value = val.min = val.max = 0.0;
    return false;
  }	
  
  return true;
}

bool MinMaxTemp::write( SensorValue &val )
{
  double offset;
  
  offset=Sensor::getOffset();
  
  val.value = (val.value * 10) - offset;
  val.min = (val.min * 10) - offset;
  val.max = (val.max * 10) - offset;;
  
  return Sensor::writeTemp( val );
}

bool MinMaxTemp::isTemp()
{
  return true;
}


bool MinMaxTemp::scalable()
{
  return false;
}

bool MinMaxTemp::displayDecimals()
{
  return true;
}

bool MinMaxTemp::displayUnits()
{
  return true;
}

void MinMaxTemp::units( QString &unitStr )
{
  unitStr="C";
}

bool MinMaxTemp::increaseGraphRange()
{
  SensorValue	val;
  double 		min;
  double 		max;
  
  Sensor::graphRange( min, max );
  
  max += RANGE_INCREMENT;
  
  if (max > RANGE_UPPER_LIMIT)
  {
    max = RANGE_UPPER_LIMIT;
  }
  
  if (!read( val ))
  {
    return false;
  }
  
  if ( fabs(val.value) > max )
  {
    max = fabs(val.value) + (RANGE_INCREMENT * 2);
  }	
  
  return Sensor::setGraphRange( min ,max );
}

bool MinMaxTemp::decreaseGraphRange()
{
  SensorValue val;
  double min;
  double max;
  
  Sensor::graphRange( min, max );
  
  max -= RANGE_INCREMENT;
  
  if (max < RANGE_LOWER_LIMIT)
  {
    max = RANGE_LOWER_LIMIT;
  }
  
  if (!read( val ))
  {
    return false;
  }
  
  if ( fabs(val.value) > max )
  {
    max = fabs(val.value) + (RANGE_INCREMENT * 2);
  }	
  
  return Sensor::setGraphRange( min ,max );	
}

bool MinMaxTemp::increaseOffset()
{
  double offset;
  
  offset = Sensor::getOffset();
  
  offset += OFFSET_INCREMENT;
  
  if ( offset <= OFFSET_UPPER_LIMIT )
  {
    return Sensor::setOffset( offset );
  }
  
  return false;
}

bool MinMaxTemp::decreaseOffset()
{
  SensorValue val;
  double offset;
  
  offset = Sensor::getOffset();
  if (!read( val ))
  {
    return false;
  }
  
  offset	-= OFFSET_INCREMENT;
  
  if ( (val.value + offset) < 0 )
  {
    return false;
  }
  
  if ( offset >= OFFSET_LOWER_LIMIT )
  {
    return Sensor::setOffset( offset );
  }
  
  return false;
}


bool MinMaxTemp::isAlarm()
{
  SensorValue val;
  
  read( val );
  val.min = fabs(val.min);
  val.max = fabs(val.max);
  val.value = fabs(val.value);
  
  return ( val.value > val.max ) || ( val.value < val.min );
}
