/*
 * $Id: klavg.cc,v 1.4 1999/12/19 07:52:17 lord Exp $
 *
 * KLAVG Simple applet showing load average graph on KDE toolbar.
 *
 * Vadim Zaliva <lord@crocodile.org>
 * http://www.crocodile.org/
 * 
 * Copyright (C) 1999 Vadim Zaliva
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#include <kconfig.h> 
#include <kstddirs.h> // for KConfig
#include <kapp.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kwm.h>
#include <klocale.h>

#include "klavg.h"

#define LAVG_FILE "/proc/loadavg"

#ifndef i18n
# define i18n(x) x
# else
# warning i18n macro not defined - source will be non-localizeable
#endif


KLavg::KLavg() : KApplet(NULL, "KLavg")
{
    graph = new Graph(this);
    popup = new QPopupMenu();
    
    popup->insertItem(i18n("&Refresh" ), this, SLOT(tick  ()));
    popup->insertItem(i18n("&Settings"), this, SLOT(config()));
    popup->insertItem(i18n("&Exec"    ), this, SLOT(click ()));
    popup->insertItem(i18n("&About"   ), this, SLOT(about ()));
    popup->insertSeparator();
    popup->insertItem(i18n("&Quit"    ), this, SLOT(quit  ()));

    connect(&timer, SIGNAL(timeout()),
            this, SLOT(tick()) );

    connect(graph, SIGNAL(rightMouseButtonPressed()),
            this, SLOT(popupMenu()) );

    connect(graph, SIGNAL(leftMouseButtonPressed()),
            this, SLOT(click()) );

#ifdef ENABLE_DYNAMIC_TIP
    tip = new DynamicTip(this,"Load average graph");
#else
    QToolTip::add(this,"Load average graph");
#endif 
    
    loadCfg();
    newCfg(cfg);
    tick();
}

KLavg::~KLavg()
{
    timer.stop();
}

void KLavg::loadCfg()
{
    KConfig *kconfig = kapp->config();

    kconfig->setGroup("General");
    cfg.refresh_rate = kconfig->readNumEntry("RefreshRate",1     );
    cfg.cmd          = kconfig->readEntry("Command"       ,"ktop");
    
    kconfig->setGroup("Colours");
    cfg.bg_colour    = kconfig->readColorEntry ("Background",&black );
    cfg.fg_colour    = kconfig->readColorEntry ("Foreground",&yellow);
    cfg.grid_colour  = kconfig->readColorEntry ("Grid"      ,&red   );
}

void KLavg::saveCfg()
{
    KConfig *kconfig = kapp->config();

    kconfig->setGroup("General");
    kconfig->writeEntry("RefreshRate"   , cfg.refresh_rate  );
    kconfig->writeEntry("Command"       , cfg.cmd           );
    
    kconfig->setGroup("Colours");
    kconfig->writeEntry("Background", cfg.bg_colour     );
    kconfig->writeEntry("Foreground", cfg.fg_colour     );
    kconfig->writeEntry("Grid"      , cfg.grid_colour   );
    
    kconfig->sync();
}

void KLavg::newCfg(KlavgPrefs &newcfg)
{
    cfg=newcfg;
    graph->setColours(cfg.bg_colour,  
                      cfg.fg_colour,  
                      cfg.grid_colour
    );
    process.clearArguments();
    process << cfg.cmd;
    timer.changeInterval(cfg.refresh_rate * 1000);
    saveCfg();
}

// Update GUI
void KLavg::tick()
{
#ifdef ENABLE_DYNAMIC_TIP
    char s[80];
    float c;
    graph->addPoint(c=getLAVG());
    sprintf(s,"Load average: %.2f",c);
    tip->setText(s);
#else
    graph->addPoint(getLAVG());
#endif
}

// User left clicked on applet.
void KLavg::click()
{
    process.start(KProcess::DontCare);
}

// Show about dialogue
void KLavg::about()
{
    KMessageBox::about(NULL,
                       i18n("klavg v" KLAVGVERSION "\n\n"
                            "KDE load average monitor\n\n"
                            "Vadim Zaliva <lord@crocodile.org>\n"
                            "http://www.crocodile.org/"
                       ),
                       i18n("About klavg")
    );
}

// show preferences dialogue.
void KLavg::config()
{
    prefsdlg prefs(this, i18n("KLAVG settings"), cfg);
    if(prefs.exec()==QDialog::Accepted)
    {
        KlavgPrefs *n=prefs.getData();
        if(n)
        {
            newCfg(*n);
            delete n;
        }
    }
}

// exit
void KLavg::quit()
{
    kapp->quit();
    exit(0);
}

void KLavg::popupMenu()
{
    popup->exec(mapToGlobal(graph->pos()));
}

void KLavg::setupGeometry( Orientation orientation, int width, int height )
{
    //fprintf(stderr,"SETUP %d %d\n",width, height);
    KApplet::setupGeometry( orientation, width, height );
}

void KLavg::resizeEvent(QResizeEvent *e)
{
    //fprintf(stderr,"RESIZE %d %d\n",e->size().width(), e->size().height());
    graph->setGeometry(0,0, width(), height());
}

float KLavg::getLAVG()
{
    float res;

#ifdef HAVE_GETLOADAVG
    double currentLoad;
    if(getloadavg(&currentLoad, 1) < 0)
        return 0.0;
    res = currentLoad;
#else
    FILE *f=fopen(LAVG_FILE,"r");
    if(!f)
        return 0.0;
    
    fscanf(f,"%f",&res);
    fclose(f);
#endif    
    //fprintf(stderr,"LAVG=%f\n",res); 
    return res;
}

#include "klavg.moc"
