/***********************************************************************
**
**   map.h
**
**   This file is part of KFLog
**
************************************************************************
**
**   Copyright (c):  1999 by Heiner Lamprecht ( heiner@kijumfo.de )
**   begin        :  Mon Jun  7 1999
**
**
**   This program is free software; you can redistribute it and/or
**   modify it under the terms of the GNU General Public License as
**   published by the Free Software Foundation; either version 2 of
**   the License, or (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY; without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
**   GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public
**   License along with this program; if not, write to the Free
**   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
***********************************************************************/

#ifndef MAP_H
#define MAP_H

// include the app-specific header
#include <kflog.h>
#include <flight.h>

// include the qt-header
#include <qregion.h>
#include <qwidget.h>

class MapView;

/**
  * This class provides basic functions for displaying the map.
  * @author Heiner Lamprecht
  * @version 0.5
  */
class Map : public QWidget
{
  Q_OBJECT

 public:
  /**
    * The constructor creates a new Map-object, read the configdata for the
    * standard parallels, the center of the map, the scale used the last
    * time. It also creates the icon used as a cursor in the map, and
    * connects some tool-bar-button for controling the map.
    */
  Map(QWidget* parent = 0, KFLogApp* main = 0, KFLogView* view = 0);
  /**
    * Destroys the Map-object and saves the current scale and center of the
    * map, so the map will appear the same, when KFLog is started next.
    */
  ~Map();
  /**
    * Increases the scale of the map and redraws it.
    * @see #scaleSub
    * @see #redrawMap
    * @see #scale
    */
//  void scaleAdd();
  /**
    * Decreases the scale of the map and redraws it.
    * @see #scaleAdd
    * @see #redrawMap
    * @see #scale
    */
//  void scaleSub();
  /**
    * Creates a new Flight-object and invoces loading the flight.
    * Returns the number of flights in the list.
    */
  void loadFlight(MapContents* mContents, QFile* flightDataFile);
  void loadFlight(MapContents* mContents, QString flightDataFileName);
  /** */
  void loadWaypoint(MapContents* mContents, QString flightDataFileName);

  /** Hier zunchst nur zum Testen, mal sehen, wie schnell das geht. */
  int* xPos;
  int* yPos;

 public slots:
  void slotConfigureMap();
  /**
    * Forces increasing the scale.
    * @see #scaleAdd
    * @see #scale
    */
  void slotScaleAdd();
  /**
    * Forces decreasing the scale.
    * @see #scaleSub
    * @see #scale
    */
  void slotScaleSub();
  /**
    * Opens a dialog-window to configure the type of mapobjects displayed
    * in the map.
    */
//  void slotConfigureMap();
  /**
    * Cancels configuring the map.
    */
//  void slotCancel();
  /**
    * Forces drawing the map.
    * @see redrawMap
    */
//  void slotDrawMap();
  /**
    * Displays the flight in the map.
    */
  void slotDrawFlight();
  /**
    * Opens a dialog and asks for the flight to be removed from the list.
    * If there is only one flight in the lists, this flight will be removed
    * without any quastion.
    */
  void slotDeleteFlight();
  /** */
  void slotRedrawMap();
  /** */
  void slotShowLayer();
  /**
    * Enables or disables drawing.
    */
  void slotEnableMap(bool paint);

 protected:
  /**
    * Moves the center of the map to point given in the event, if the
    * middle-button is pressed.
    */
  virtual void mousePressEvent(QMouseEvent* event);
  /**
    * Displays the coordinates of the mousepointer in the status-bar.
    */
  virtual void mouseMoveEvent(QMouseEvent* event);
  /**
    * Redefinition of the paintEvent.
    */
  virtual void paintEvent(QPaintEvent* event);
  /**
    * Redefinition of the resizeEvent.
    */
  virtual void resizeEvent(QResizeEvent* event);

 private:
  /**
    * Redraws the map.
    */
  void redrawMap();
  /**
    * Draws the map. The type of mapobjects to be drawn is controlled
    * via slotConfigureMap.
    * @see #slotConfigureMap
    */
  void drawMap();
  /**
    * Draws the flight into the map.
    */
  void drawFlight();
  /**
    * Draws the grid tino the map.
    */
  void drawGrid();
  /**
    * Pointer to the parent window.
    */
  QWidget* parentWidget;
  /**
    * Pointer to the current KFLog-instance.
    */
  KFLogApp* main;
  /**
    * Coordinates of the current mouse-position, needed
    * to show in the status bar.
    */
  QPoint _current;
//  QPoint _start;
//  QPoint _stop;
  /**
    * This pixmap is used to store the currently displayed map.
    * This painter is about the same size as the map-widget, but only
    * used for internal buffering the map. Whenever the widget is about
    * to be drawn, this buffer is used to get the content.
    */
  QPixmap pixBuffer;
  /**
    * Contains cities and villages.
    */
  QPixmap pixCity;
  /**
    * Contains the grid, needed as the basis.
    */
  QPixmap pixGrid;
  /**
    * Contains the obstacles and spots.
    */
  QPixmap pixTopo;
  /**
    * The layer containing the waypoints.
    */
  QPixmap pixWaypoints;
  /**
    * The layer containing the glider-sites.
    */
  QPixmap pixGlider;
  /**
    * Contains the airports, airfields, ...
    */
  QPixmap pixAirport;
  /**
    * Contains the additinal sites, like ballon, ulf, ...
    */
  QPixmap pixAddSites;
  /**
    * Contains the airspaces, radio-navigation facilities as well as the
    * compulsory reporting points.
    */
  QPixmap pixNav;
  /**
    * Contains the flights.
    */
  QPixmap pixFlight;
  /**
    * Contains the highways and roads.
    */
  QPixmap pixRoad;
  /**
    * Contains the railways, stations and aerial-railways.
    */
  QPixmap pixRail;
  /**
    * Contains all lakes, rivers, coastlines, locks and dams.
    */
  QPixmap pixHydro;
  /**
    * Contains the airspace-structure.
    */
  QPixmap pixAirspace;
  /**
    * Contains the landmarks.
    */
  QPixmap pixLand;
  /**
    * Contains the outlandingfields.
    */
  QPixmap pixOut;
  /**
    * Pointer to the current main window.
    */
  KFLogApp *app;
  /**
    * The minimum scale. Here the height of 1 point is about 6360.0m.
    */
  static const int minScale = 1000;
  /**
    * The maximum scale. Here the height of 1 point is about 12.72m.
    */
  static const int maxScale = 750000;
  /**
    * The mapCenter is the position displayed in the center of the map.
    * It is used in two different ways:
    * 1.: Determine the area shown in the map-widget
    * 2.: Calculating the difference in latitude between a point in the
    * map and the center.
    *
    * The latitude of the center of the map.
    */
  long mapCenterLat;
  /**
    * The longitude of the center of the map.
    */
  long mapCenterLon;
  /**
    * Enables drawing of the map.
    * Used to guarantee, that the map will only be drawn, if the widget
    * is already build. Otherwise, the map would be drawn twice when
    * opening a window!
    */
  bool isEnabled;
  /**
    * flightList contains the loaded flights.
    */
  QList<Flight> flightList;
  bool displayFlights;

  bool showAddSites;
  bool showAirport;
  bool showAirspace;
  bool showCity;
  bool showFlight;
  bool showGlider;
  bool showHydro;
  bool showLand;
  bool showNav;
  bool showOut;
  bool showRail;
  bool showRoad;
  bool showTopo;
  bool showWaypoint;

  /** */
  unsigned int posNum;
  QList<QRegion>* airspaceRegList;
  QList<QRegion>* cityRegList;

  KFLogView* view;
};

#endif
