/*
	searchdlg.cpp - part of KDiskCat the KDE Disk Catalog.

	Copyright (c) 1998,1999 Balzs Ternyi <terenyi@freemail.c3.hu>

	KDiskCat is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
			
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
					
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <kapp.h>
#include <kmsgbox.h>
#include <kquickhelp.h>
#include <qregexp.h>
#include "searchdlg.h"

#include <iostream.h>

SearchDlg::SearchDlg(QWidget* parent,const char* name,QStrList* categories,QString current)
	:QDialog(parent,name,true)
{
	QDateTime dt;
	QTime time;
	
	init();
	initQuickHelp();

	current_category=current;

   setCaption(i18n("Search"));
   pb_search->setDefault(true);

   dt.setDate(QDate::currentDate());
   time.setHMS(0,0,0);
   dt.setTime(time);
  	ke_date_from->setData(&dt);
  	time.setHMS(23,59,59,999);
   dt.setTime(time);
   ke_date_to->setData(&dt);

   lbox_categories->insertStrList(categories);
   lbox_categories->setCurrentItem(0);

   loadSettings();

   slotCbxTextSearch();
	slotCbxSizeFrom();
	slotCbxSizeTo();
	slotCbxDateFrom();
	slotCbxDateTo();

	if (cbx_text_search->isChecked())
	{
		sle_reg_expr->setFocus();
	}

   connect(pb_search,SIGNAL(clicked()),this,SLOT(slotOk()));
   connect(pb_cancel,SIGNAL(clicked()),this,SLOT(reject()));
   connect(cbx_text_search,SIGNAL(clicked()),this,SLOT(slotCbxTextSearch()));
   connect(cbx_size_from,SIGNAL(clicked()),this,SLOT(slotCbxSizeFrom()));
   connect(cbx_size_to,SIGNAL(clicked()),this,SLOT(slotCbxSizeTo()));
   connect(cbx_date_from,SIGNAL(clicked()),this,SLOT(slotCbxDateFrom()));
   connect(cbx_date_to,SIGNAL(clicked()),this,SLOT(slotCbxDateTo()));
   connect(pb_select_all,SIGNAL(clicked()),this,SLOT(slotSelectAll()));
   connect(pb_select_none,SIGNAL(clicked()),this,SLOT(slotSelectNone()));
   connect(pb_select_current,SIGNAL(clicked()),this,SLOT(slotSelectCurrent()));
}

SearchDlg::~SearchDlg()
{
}

void SearchDlg::slotSelectAll()
{
   lbox_categories->setCurrentItem(0);
   for (int i=0;i < (int)lbox_categories->count();i++)
   	lbox_categories->setSelected(i,true);
}

void SearchDlg::slotSelectNone()
{
   lbox_categories->setCurrentItem(0);
   lbox_categories->clearSelection();
}

void SearchDlg::slotSelectCurrent()
{
	int i;
	
   for (i=0;i < (int)lbox_categories->count() && lbox_categories->text(i) != current_category;i++)
   {
   }
   if (lbox_categories->text(i) == current_category)
   {
	   lbox_categories->setCurrentItem(0);
   	lbox_categories->clearSelection();
	  	lbox_categories->setSelected(i,true);
	}
}

bool SearchDlg::checkFilter()
{
   QString msg;
   QRegExp* regexp;
   bool ret=true;
   bool is_any_selected=false;
   bool b;
   int i;
   uint si_from=0;
   uint si_to=0;
   QString s;
   QDateTime dt_from,dt_to;
   
   if (!cbx_text_search->isChecked() && !cbx_size_from->isChecked() &&
		 !cbx_size_to->isChecked() && !cbx_date_from->isChecked() &&
		 !cbx_date_to->isChecked())
   {
      msg=i18n("- At least one of the conditions must be checked!\n");
      ret=false;
   }

   if (cbx_text_search->isChecked() && !cbx_match_filename->isChecked() && !cbx_match_descriptions->isChecked())
   {
      msg+=i18n("- In case of text search you should enable at least one of the Match filename\n");
      msg+=i18n("  or Match descriptions checkboxes!\n");
      ret=false;
   }

   if (cbx_text_search->isChecked())
   {
	   regexp=new QRegExp(sle_reg_expr->currentText(),cbx_case_sens->isChecked(),rb_wildcard->isChecked());

	   if (!regexp->isValid())
   	{
      	msg+=i18n("- Invalid text search expression!\n");
	      ret=false;
   	}
	   delete regexp;
   }

   // Size condition checks
   if (cbx_size_from->isChecked())
   {
   	s=sle_size_from->text();
   	si_from=s.toUInt(&b);
   	if (!b)
   	{
   		msg+=i18n("- The size from field contains no valid number!\n");
   		ret=false;
   	}
   }

   if (cbx_size_to->isChecked())
   {
   	s=sle_size_to->text();
   	si_to=s.toUInt(&b);
   	if (!b)
   	{
   		msg+=i18n("- The size to field contains no valid number!\n");
   		ret=false;
   	}
   }

   if (cbx_size_from->isChecked() && cbx_size_to->isChecked() && si_to<si_from)
   {
  		msg+=i18n("- The size to field can't be smaller than size from!\n");
  		ret=false;   	
   }
	
   // Datetime condition checks
   if (cbx_date_from->isChecked())
   {
   	if (!ke_date_from->getData(&dt_from))
   	{
   		msg+=i18n("- The datetime from field contains no valid data!\n");
   		ret=false;
   	}
   }

   if (cbx_date_to->isChecked())
   {
   	if (!ke_date_to->getData(&dt_to))
   	{
   		msg+=i18n("- The datetime to field contains no valid data!\n");
   		ret=false;
   	}
   }

   if (cbx_date_from->isChecked() && cbx_date_to->isChecked() && dt_to<dt_from)
   {
  		msg+=i18n("- The date to field can't contain earlier date than the date from field!\n");
  		ret=false;   	
   }

	// Category check
   for (i=0; i < (int)lbox_categories->count() && !is_any_selected; i++)
   {
   	is_any_selected=lbox_categories->isSelected(i);
   }

   if (!is_any_selected)
   {
      msg+=i18n("- There is no category selected!\n");
      ret=false;
   }

   if (!ret)
   {
   	msg=i18n("The specified searching expression is not valid because of the followings:\n\n")+msg;
      KMsgBox::message(this,i18n("Attention!"),(const char *) msg);
   }

   return ret;
}

void SearchDlg::slotOk()
{
   if (checkFilter())
   {
      accept();
      sle_reg_expr->insertItem(sle_reg_expr->currentText(),0);
      /* This truncates the list to 10 elements.
      	Calling setMaxCount only one time doesn't work. */
		sle_reg_expr->setMaxCount(9);
      saveSettings();
   }
}

void SearchDlg::slotCbxTextSearch()
{
   if (cbx_text_search->isChecked())
   {
      cbx_match_filename->setEnabled(true);
      cbx_match_descriptions->setEnabled(true);
      rb_regexp->setEnabled(true);
      sle_reg_expr->setEnabled(true);
      lb_reg_expr->setEnabled(true);
      rb_wildcard->setEnabled(true);
      cbx_case_sens->setEnabled(true);
   }
   else
   {
      cbx_match_filename->setEnabled(false);
      cbx_match_descriptions->setEnabled(false);
      rb_regexp->setEnabled(false);
      sle_reg_expr->setEnabled(false);
      lb_reg_expr->setEnabled(false);
      rb_wildcard->setEnabled(false);
      cbx_case_sens->setEnabled(false);
   }
}

void SearchDlg::slotCbxSizeFrom()
{
	if (cbx_size_from->isChecked())
	{
		sle_size_from->setEnabled(true);
	}
	else
	{
		sle_size_from->setEnabled(false);
	}
}

void SearchDlg::slotCbxSizeTo()
{
	if (cbx_size_to->isChecked())
	{
		sle_size_to->setEnabled(true);
	}
	else
	{
		sle_size_to->setEnabled(false);
	}
}

void SearchDlg::slotCbxDateFrom()
{
	if (cbx_date_from->isChecked())
	{
		ke_date_from->setEnabled(true);
	}
	else
	{
		ke_date_from->setEnabled(false);
	}
}

void SearchDlg::slotCbxDateTo()
{
	if (cbx_date_to->isChecked())
	{
		ke_date_to->setEnabled(true);
	}
	else
	{
		ke_date_to->setEnabled(false);
	}
}

QStrList SearchDlg::getSelectedCategories()
{
	QStrList ret;
	int i;
	
	for (i=0; i < (int)lbox_categories->count(); i++)
	{
		if (lbox_categories->isSelected(i))
		{
			ret.append(lbox_categories->text(i));
		}
	}
	
	return ret;
}

void SearchDlg::saveSettings()
{
	KConfig* config;
	QStrList strs;
	int i;
	
	config=kapp->getConfig();
	
   config->setGroup("SearchDlg");

	for (i=0;i < sle_reg_expr->count();i++)
	{
		strs.append(sle_reg_expr->text(i));
	}
   config->writeEntry("ExprList",strs,'|');

  	strs.clear();
   for (i=0;i < (int)lbox_categories->count();i++)
   {
   	if (lbox_categories->isSelected(i))
   	{
   		strs.append(lbox_categories->text(i));
   	}
   }
   config->writeEntry("SelectedCats",strs,'|');

   config->writeEntry("CaseSens",cbx_case_sens->isChecked());
   config->writeEntry("DateFrom",cbx_date_from->isChecked());
   config->writeEntry("DateTo",cbx_date_to->isChecked());
   config->writeEntry("MatchDesc",cbx_match_descriptions->isChecked());
   config->writeEntry("MatchFilename",cbx_match_filename->isChecked());
   config->writeEntry("SizeFrom",cbx_size_from->isChecked());
   config->writeEntry("SizeTo",cbx_size_to->isChecked());
   config->writeEntry("TextSearch",cbx_text_search->isChecked());
   config->writeEntry("RegExpMode",rb_regexp->isChecked());
   config->writeEntry("WildcardMode",rb_wildcard->isChecked());

   config->writeEntry("DateFromField",ke_date_from->text());
   config->writeEntry("DateToField",ke_date_to->text());
   config->writeEntry("SizeFromField",sle_size_from->text());
   config->writeEntry("SizeToField",sle_size_to->text());
}

void SearchDlg::loadSettings()
{
	KConfig* config;
	QStrList strs;
	int i;
	
	config=kapp->getConfig();
	
   config->setGroup("SearchDlg");

   config->readListEntry("ExprList",strs,'|');
	sle_reg_expr->insertStrList(&strs);

   config->readListEntry("SelectedCats",strs,'|');
   for (i=0;i < (int)lbox_categories->count();i++)
   {
   	if (strs.find((const char*)lbox_categories->text(i)) != -1)
   	{
   		lbox_categories->setSelected(i,true);
   	}
   }

   cbx_case_sens->setChecked(config->readBoolEntry("CaseSens",false));
   cbx_date_from->setChecked(config->readBoolEntry("DateFrom",false));
	cbx_date_to->setChecked(config->readBoolEntry("DateTo",false));
   cbx_match_descriptions->setChecked(config->readBoolEntry("MatchDesc",false));
   cbx_match_filename->setChecked(config->readBoolEntry("MatchFilename",true));
   cbx_size_from->setChecked(config->readBoolEntry("SizeFrom",false));
   cbx_size_to->setChecked(config->readBoolEntry("SizeTo",false));
   cbx_text_search->setChecked(config->readBoolEntry("TextSearch",true));
   rb_regexp->setChecked(config->readBoolEntry("RegExpMode",false));
   rb_wildcard->setChecked(config->readBoolEntry("WildcardMode",true));

   if (cbx_date_from->isChecked())
   {
   	ke_date_from->setText(config->readEntry("DateFromField",ke_date_from->text()));
   }
   if (cbx_date_to->isChecked())
   {
   	ke_date_to->setText(config->readEntry("DateToField",ke_date_to->text()));
   }
   if (cbx_size_from->isChecked())
   {
   	sle_size_from->setText(config->readEntry("SizeFromField",sle_size_from->text()));
   }
   if (cbx_size_to->isChecked())
   {
   	sle_size_to->setText(config->readEntry("SizeToField",sle_size_to->text()));
   }
}

void SearchDlg::getCriteria(CatCriteria* crit)
{
	QString s;
	
  	crit->categories=getSelectedCategories();
  	
	if (cbx_text_search->isChecked())
	{
   	crit->rexp=sle_reg_expr->currentText();
   	crit->rexp.setCaseSensitive(cbx_case_sens->isChecked());
   	crit->rexp.setWildcard(rb_wildcard->isChecked());
	  	crit->filename_on=cbx_match_filename->isChecked();
  		crit->description_on=cbx_match_descriptions->isChecked();
 	}
 	else
 	{
 		crit->filename_on=false;
 		crit->description_on=false;
 	}
	
	crit->size_from_on=cbx_size_from->isChecked();
	crit->size_to_on=cbx_size_to->isChecked();
	if (crit->size_from_on)
	{
  		s=sle_size_from->text();
  		crit->size_from=s.toUInt();
  	}
  	if (crit->size_to_on)
  	{
  		s=sle_size_to->text();
  		crit->size_to=s.toUInt();
  	}
 	
 	crit->date_from_on=cbx_date_from->isChecked();
  	crit->date_to_on=cbx_date_to->isChecked();
  	if (crit->date_from_on)
  	{
  		ke_date_from->getData(&crit->date_from);
  	}
  	if (crit->date_to_on)
  	{
  		ke_date_to->getData(&crit->date_to);
  	} 		
}

void SearchDlg::init()
{
	int width;

	// left side
	pb_select_all=new QPushButton(this,"pb_select_all");
	pb_select_all->setGeometry( 10, 10, 40, 20 );
	pb_select_all->setText( i18n("&A"));
	pb_select_all->setAutoRepeat( FALSE );
	pb_select_all->setAutoResize( FALSE );

	pb_select_none=new QPushButton(this,"pb_select_none");
	pb_select_none->setGeometry( 60, 10, 40, 20 );
	pb_select_none->setText( i18n("&N"));
	pb_select_none->setAutoRepeat( FALSE );
	pb_select_none->setAutoResize( FALSE );
	
	pb_select_current=new QPushButton(this,"pb_select_current");
	pb_select_current->setGeometry(110, 10, 40, 20 );
	pb_select_current->setText( i18n("&C"));
	pb_select_current->setAutoRepeat( FALSE );
	pb_select_current->setAutoResize( FALSE );
	
	lbox_categories=new QListBox(this,"lbox_categories");
	lbox_categories->setGeometry( pb_select_all->x(), 35,
				      pb_select_current->x()+pb_select_current->width()-pb_select_all->x(), 250 );
	lbox_categories->setMultiSelection(true);

	// Search, Cancel
	pb_search = new QPushButton( this, "pb_search" );
	pb_search->setGeometry( lbox_categories->x(), 295, lbox_categories->width(), 35 );
	pb_search->setText( i18n("Search") );
	pb_search->setAutoRepeat( FALSE );
	pb_search->setAutoResize( FALSE );

	pb_cancel = new QPushButton( this, "pb_cancel" );
	pb_cancel->setGeometry( pb_search->x(), 335, pb_search->width(), 35 );
	pb_cancel->setText( i18n("Cancel") );
	pb_cancel->setAutoRepeat( FALSE );
	pb_cancel->setAutoResize( FALSE );

        // Text condition
	fr_reg_expr = new QFrame( this, "fr_reg_expr" );
	fr_reg_expr->setGeometry( 160, 10, 495, 190 );
	fr_reg_expr->setFrameStyle( 34 );

	cbx_text_search = new QCheckBox( fr_reg_expr, "cbx_text_search" );
	cbx_text_search->setGeometry( 10, 5, fr_reg_expr->width()-20, 30 );
	cbx_text_search->setText( i18n("&Text condition") );
	cbx_text_search->setAutoRepeat( FALSE );
	cbx_text_search->setAutoResize( FALSE );

	rb_wildcard = new QRadioButton( fr_reg_expr, "rb_wildcard" );
	rb_wildcard->setGeometry( 30, 30, fr_reg_expr->width()-40, 30 );
	rb_wildcard->setText( i18n("Search using &wildcards") );
	rb_wildcard->setAutoRepeat( FALSE );
	rb_wildcard->setAutoResize( FALSE );

	rb_regexp = new QRadioButton( fr_reg_expr, "rb_regexp" );
	rb_regexp->setGeometry( 30, 55, fr_reg_expr->width()-40, 30 );
	rb_regexp->setText( i18n("Search using &regular expression") );
	rb_regexp->setAutoRepeat( FALSE );
	rb_regexp->setAutoResize( FALSE );

	bg_search_mode = new QButtonGroup( fr_reg_expr, "bg_search_mode" );
	bg_search_mode->setAlignment( 1 );
	bg_search_mode->insert( rb_regexp );
	bg_search_mode->insert( rb_wildcard );
	bg_search_mode->hide();

	lb_reg_expr = new QLabel( fr_reg_expr, "lb_reg_expr" );
	lb_reg_expr->setText( i18n("Expression:") );
	width = lb_reg_expr->fontMetrics().width(lb_reg_expr->text());
	lb_reg_expr->setGeometry( 30, 90, width+5, 30 );
	lb_reg_expr->setAlignment( 289 );
	lb_reg_expr->setMargin( -1 );

	sle_reg_expr=new QComboBox( true, fr_reg_expr, "sle_reg_expr");
	sle_reg_expr->setGeometry( 40+width, 90, fr_reg_expr->width()-(40+width)-10, 30);
	sle_reg_expr->setInsertionPolicy(QComboBox::NoInsertion);
	sle_reg_expr->setMaxCount(9);
	sle_reg_expr->setAutoCompletion(true);

	cbx_case_sens = new QCheckBox( fr_reg_expr, "cbx_case_sens" );
	cbx_case_sens->setGeometry( 30, 125, fr_reg_expr->width()-40, 30 );
	cbx_case_sens->setText( i18n("Case &sensitive") );
	cbx_case_sens->setAutoRepeat( FALSE );
	cbx_case_sens->setAutoResize( FALSE );

	cbx_match_filename = new QCheckBox( fr_reg_expr, "cbx_match_filenames" );
	cbx_match_filename->setGeometry( 30, 150, (fr_reg_expr->width()-40)/2-10, 30 );
	cbx_match_filename->setText( i18n("Match &filenames") );
	cbx_match_filename->setAutoRepeat( FALSE );
	cbx_match_filename->setAutoResize( FALSE );

	cbx_match_descriptions = new QCheckBox( fr_reg_expr, "cbx_match_descriptions" );
	cbx_match_descriptions->setGeometry( 50+cbx_match_filename->width(), 150,
					     (fr_reg_expr->width()-40)/2-10, 30 );
	cbx_match_descriptions->setText( i18n("Match &descriptions") );
	cbx_match_descriptions->setAutoRepeat( FALSE );
	cbx_match_descriptions->setAutoResize( FALSE );


	// Size condition
	fr_size = new QFrame( this, "fr_size" );
	fr_size->setGeometry( 160, 210, 495, 75 );
	fr_size->setFrameStyle( 34 );

	lb_size_condition = new QLabel( fr_size, "lb_size_condition" );
	lb_size_condition->setGeometry( 10, 5, fr_size->width()-20, 30 );
	lb_size_condition->setText( i18n("Size condition") );

	cbx_size_from = new QCheckBox( fr_size, "cbx_size_from" );
	cbx_size_from->setAutoResize( TRUE );
	cbx_size_from->move( 10, 35 );
	cbx_size_from->setText( i18n("From:") );
	cbx_size_from->resize( cbx_size_from->width(), 30 );

	sle_size_from = new KRestrictedLine( fr_size, "sle_size_from" );
	sle_size_from->setValidChars("0123456789");
	sle_size_from->setGeometry( 20+cbx_size_from->width(), 35, 
				    fr_size->width()/2-(25+cbx_size_from->width()), 30 );
	sle_size_from->setMaxLength( 20 );
	sle_size_from->setEchoMode( QLineEdit::Normal );
	sle_size_from->setFrame( TRUE );

	cbx_size_to = new QCheckBox( fr_size, "cbx_size_to" );
	cbx_size_to->setAutoResize( TRUE );
	cbx_size_to->move( fr_size->width()/2+5, 35 );
	cbx_size_to->setText( i18n("To:") );
	cbx_size_to->resize( cbx_size_to->width(), 30 );

	sle_size_to = new KRestrictedLine( fr_size, "sle_size_to" );
	sle_size_to->setValidChars("0123456789");
	sle_size_to->setGeometry( fr_size->width()/2+15+cbx_size_to->width(), 35,
				  fr_size->width()/2-25-cbx_size_to->width(), 30 );
	sle_size_to->setMaxLength( 20 );
	sle_size_to->setEchoMode( QLineEdit::Normal );
	sle_size_to->setFrame( TRUE );


	// Datetime condition
	fr_date_search = new QFrame( this, "fr_date_search" );
	fr_date_search->setGeometry( 160, 295, 495, 75 );
	fr_date_search->setFrameStyle( 34 );

	lb_date_condition = new QLabel( fr_date_search, "lb_date_condition" );
	lb_date_condition->setGeometry( 10, 5, fr_date_search->width()-20, 30 );
	lb_date_condition->setText( i18n("Datetime condition") );

	cbx_date_from = new QCheckBox( fr_date_search, "cbx_date_from" );
	cbx_date_from->setAutoResize( TRUE );
	cbx_date_from->move( 10, 35 );
	cbx_date_from->setText( i18n("From:") );
	cbx_date_from->resize( cbx_date_from->width(), 30 );
	
	ke_date_from = new KEditMask( fr_date_search, "ke_date_from" );
	ked_from=new KEditMaskDefinitionDateTime(ke_date_from,"ked_from",i18n("yyyy-mm-dd hh:MM:ss"));
	ke_date_from->setMaskDefinition(ked_from);
	ke_date_from->setGeometry( 20+cbx_date_from->width(), 35, 
				    fr_date_search->width()/2-(25+cbx_date_from->width()), 30 );

	cbx_date_to = new QCheckBox( fr_date_search, "cbx_date_to" );
	cbx_date_to->setAutoResize( TRUE );
	cbx_date_to->move( fr_date_search->width()/2+5, 35 );
	cbx_date_to->setText( i18n("To:") );
	cbx_date_to->resize( cbx_date_to->width(), 30 );

	ke_date_to = new KEditMask( fr_date_search, "ke_date_to" );
	ked_to=new KEditMaskDefinitionDateTime(ke_date_to,"ked_to", i18n("yyyy-mm-dd hh:MM:ss"));
	ke_date_to->setMaskDefinition(ked_to);
	ke_date_to->setGeometry( fr_date_search->width()/2+15+cbx_date_to->width(), 35,
				 fr_date_search->width()/2-25-cbx_date_to->width(), 30 );

	setTabOrder(sle_reg_expr,cbx_case_sens);
	setTabOrder(cbx_case_sens,cbx_match_filename);
	setTabOrder(cbx_match_filename,cbx_match_descriptions);
	setTabOrder(cbx_match_descriptions,cbx_size_from);
	setTabOrder(cbx_size_from,sle_size_from);
	setTabOrder(sle_size_from,cbx_size_to);
	setTabOrder(cbx_size_to,sle_size_to);
	setTabOrder(sle_size_to,cbx_date_from);
	setTabOrder(cbx_date_from,ke_date_from);
	setTabOrder(ke_date_from,cbx_date_to);
	setTabOrder(cbx_date_to,ke_date_to);
	setTabOrder(ke_date_to,pb_select_all);
	setTabOrder(pb_select_all,pb_select_none);
	setTabOrder(pb_select_none,pb_select_current);
	setTabOrder(pb_select_current,lbox_categories);
	setTabOrder(lbox_categories,pb_search);
	setTabOrder(pb_search,pb_cancel);
	setTabOrder(pb_cancel,cbx_text_search);
	setTabOrder(cbx_text_search,rb_wildcard);
	setTabOrder(rb_wildcard,rb_regexp);
	setTabOrder(rb_regexp,sle_reg_expr);
				
	resize( 665, 380 );
}

void SearchDlg::initQuickHelp()
{
	KQuickHelp::add(pb_select_all,i18n("Use this to select all categories."));
	KQuickHelp::add(pb_select_current,i18n("Use this to select the current category."));
	KQuickHelp::add(pb_select_none,i18n("Use this to unselect all."));
	KQuickHelp::add(lbox_categories,i18n("<brown><b>Selecting the categories<black></b>\n\n"
													 "This list contains all the categories from the database.\n"
													 "The search will be done only in the selected categories.\n\n"
													 "You can also use the buttons above this list to select\n"
													 "the appropriate categories."));
	KQuickHelp::add(cbx_text_search,i18n("If checked text search will be done with the given expression\n"
													 "in the selected categories."));
	KQuickHelp::add(rb_wildcard,
	KQuickHelp::add(rb_regexp,i18n("<brown><b>Text search mode</b><black>\n\n"
											 "The text search mode tells the search engine\n"
											 "how to interpret the given expression.\n\n"
											 "<b>Wildcard mode:</b>\n"
											 "To compose an expression you can use the followings:\n\n"
											 "<b>*</b> matches any sequence of characters\n"
											 "<b>?</b> matches any single character\n"
											 "<b>[]</b> matches a defined set of characters, e.g. [a-zA-Z0-9]\n"
											 "<b>^x</b> matches everything except x\n\n"
											 "<b>Regular expression mode:</b>\n"
											 "To compose an expression you can use the followings:\n\n"
											 "<b>c</b> matches the character 'c'\n"
											 "<b>.</b> matches any character\n"
                                  "<b>^</b> matches start of input (except [^x] which matches NOT [x])\n"
                                  "<b>$</b> matches end of input\n"
                                  "<b>[]</b> matches a set of characters, for example [a-z0-9_]\n"
                                  "<b>a*</b> matches a sequence of zero or more a's\n"
                                  "<b>a+</b> matches a sequence of one or more a's\n"
                                  "<b>a?</b> matches an optional a\n\n"
                                  "For more info on regular expressions see the man page of\n"
                                  "the <i>grep</i> command."
											 )));
	KQuickHelp::add(lb_reg_expr,
	KQuickHelp::add(sle_reg_expr,i18n("<brown><b>Text search expression</b><black>\n\n"
											 "The entered expression tells the search engine what you are looking for. To match\n"
											 "the required filenames or descriptions compose a pattern with wildcards or type a\n"
											 "regular expression.\n\n"
											 "Example wildcard pattern:\n"
											 "<i+><b>a*.txt</b> matches every text file begining with 'a'<i->\n\n"
											 "Example regular expression:\n"
											 "<i+><b>^kde.*gz$</b> matches every file begining with 'kde' ending with 'gz'<i->\n\n"
											 "You must specify the search mode by checking either <i>Search using wildcards</i>\n"
											 "or <i>Search using regular expression</i>."
											 )));
	KQuickHelp::add(cbx_case_sens,i18n("If checked the given expression\n"
												  "will be matched case sensitive."));
	KQuickHelp::add(cbx_match_filename,i18n("If checked filenames will\n"
														 "be searched."));
	KQuickHelp::add(cbx_match_descriptions,i18n("If checked descriptions\n"
															  "will be searched."));
	KQuickHelp::add(lb_size_condition,
	KQuickHelp::add(cbx_size_from,
	KQuickHelp::add(sle_size_from,
	KQuickHelp::add(cbx_size_to,
	KQuickHelp::add(sle_size_to,i18n("<brown><b>Size condition</b><black>\n\n"
												"You can set up a size condition by defining an interval.\n"
												"Switch on the <i>from</i> and <i>to</i> fields and type the lower and\n"
												"upper limits in bytes.\n\n"
												"Using only the <i>from</i> field means more or equal, switching\n"
												"on only the <i>to</i> field means less or equal.\n\n"
												"You can combine the text, size and datetime conditions\n"
												"to create a complex searching term."
											 ))))));
	KQuickHelp::add(lb_date_condition,
	KQuickHelp::add(cbx_date_from,
	KQuickHelp::add(ke_date_from,
	KQuickHelp::add(cbx_date_to,
	KQuickHelp::add(ke_date_to,i18n("<brown><b>Datetime condition</b><black>\n\n"
											  "Using this two fields allows searching for files last\n"
											  "modified within a specified time period. You can use\n"
											  "the two fields independently to specify open periods.\n\n"
											  "You can combine the text, size and datetime conditions\n"
											  "to create a complex searching term."
											 ))))));
}
