/*
 * KBatMon - Copyright 1999 Russ Steffen <rsteffen@ia.net>
 *
 * Released under the terms of the GNU General Public License
 *
 *
 * $Id: apmInterface.cc,v 1.9 1999/01/15 06:08:47 russ Exp $
 *
 * $Log: apmInterface.cc,v $
 * Revision 1.9  1999/01/15 06:08:47  russ
 * Added file headers.
 *
 *
 */
 
#include "apmInterface.h"

#include <kapp.h>
#include <kprocess.h>

const int SCAN_INTERVAL = 1000; /* msec */

APMInterface::APMInterface() : haveAPM(false), haveDevAccess(false),
    timer(0), apm_dev(0), apm_dev_fd(-1)
{
    int rc;

    rc = apm_exists();

    if( rc == 0 ) {
        haveAPM = true;
    } else {
        haveAPM = false;
        return;
    }

    /* We have an APM BIOS, so query it for
     the initial information */

    rc = apm_read( &status );

    if( rc != 0 ) {
        /* Probably no /proc support */
        haveAPM = false;
        return;
    }

    timer = new QTimer();
    connect( timer, SIGNAL(timeout()), this, SLOT(timerEvent()) );
    timer->start(SCAN_INTERVAL);
    
    rc = apm_open();

    if( rc < 0 ) {
        /* No access to the apm device, we
         are read only */
        haveDevAccess = false;
    } else {
        haveDevAccess = true;
        apm_dev_fd = rc;
    }

}

APMInterface::~APMInterface()
{
    if( haveDevAccess ) {
        apm_close( apm_dev_fd );
    }
}

bool
APMInterface::haveAPMBIOS(void)
{
    return haveAPM;
}

// Slots

void APMInterface::apmEvent()
{
    if( haveAPM ) {
        int rc;

        rc = apm_read( &status );
        if( rc == 1 ) {
            /* APM disappeared? */
        }

        if( haveDevAccess ) {
            /* Process APM event message */
        }

        emit update();

    }
}

void APMInterface::timerEvent()
{
    if( haveAPM ) {
        int rc;

        rc = apm_read( &status );
        if( rc == 1 ) {
            /* APM disappeared? */
        }

        emit update();
    }
}
    
void APMInterface::suspend()
{
#ifndef NO_APM_PROXY
    KProcess proxyProc;
    
    QString apmProxyPath;

    apmProxyPath = kapp->kde_bindir() + "/apm_proxy";

    proxyProc.clearArguments();
    proxyProc << apmProxyPath;
    proxyProc << "suspend";

    proxyProc.start( KProcess::DontCare, KProcess::NoCommunication );

#else
    if(haveDevAccess) {
        apm_suspend( apm_dev_fd );
    }
#endif

}

void APMInterface::standby()
{
#ifndef NO_APM_PROXY
    KProcess proxyProc;
    
    QString apmProxyPath;

    apmProxyPath = kapp->kde_bindir() + "/apm_proxy";

    proxyProc.clearArguments();
    proxyProc << apmProxyPath;
    proxyProc << "standby";

    proxyProc.start( KProcess::DontCare, KProcess::NoCommunication );

#else
    if(haveDevAccess) {
        apm_standby( apm_dev_fd );
    }
#endif

}

QString APMInterface::getBatteryStatusString( void )
{
    QString statusString;


    if( status.battery_flags & 0x80 ) {
        statusString = i18n("No System Battery");
        return statusString;
    }

    switch( status.battery_status )
    {
    case 0x00:
        statusString = i18n("Battery Charge: High");
        break;
    case 0x01:
        statusString = i18n("Battery Charge: Low");
        break;
    case 0x02:
        statusString = i18n("Battery Charge: Critical");
        break;
    case 0x03:
        statusString = i18n("Battery is Charging");
        break;
    default:
        statusString = "Unknown";
    }
    
    return statusString;

}

QString APMInterface::getPowerStatusString( void )
{
    QString statusString;

    switch( status.ac_line_status )
    {
    case 0x00 :
        statusString = i18n("AC Power Offline");
        break;
    case 0x01 :
        statusString = i18n("AC Power On-Line");
        break;
    case 0x02 :
        statusString = i18n("On backup Power");
        break;
    case 0xff :
    default:
        statusString = i18n("Unknown");
        break;
    }

    return statusString;
}

/* Data Access members */

QString APMInterface::getAPMDriverVersion( void )
{
    QString version;

    if( haveAPMBIOS() ) {
        version = status.driver_version;
    } else {
        version = "Not Present";
    }

    return version;
}

QString APMInterface::getAPMBIOSVersion( void )
{
    QString version;

    if( haveAPMBIOS() ) {
        version.sprintf( "%d.%d", status.apm_version_major,
                         status.apm_version_minor );
    } else {
        version = "Not Present";
    }

    return version;
}

bool APMInterface::haveBattery(void)
{
    if( status.battery_flags & 0x80 )
        return false;
    else
        return true;
}

int APMInterface::getAPMFlags( void )
{
    return status.apm_flags;
}

int APMInterface::getBatteryFlags( void )
{
    return status.battery_flags;
}

int APMInterface::getBatteryStatus( void )
{
    return status.battery_status;
}

int APMInterface::getBatteryPercentage( void )
{
    return status.battery_percentage;
}

int APMInterface::getBatteryTime( void )
{
    return status.battery_time;
}

bool APMInterface::getACLineStatus( void )
{
    if( status.ac_line_status & 0x01 ) {
        return true;
    } else {
        return false;
    }
}

