// This file is part of KLyX, The KDE Document Processor
//
// Copyright (C) 1995 Matthias Ettrich
// Copyright (C) 1997-1999 KLyX Team

#include "InsertCitationReferenceDialog.h"
#include "filetools.h"
#include "insetbib.h"
#include "LString.h"
#include "LyXView.h"

#include <kapp.h>
#include <kbuttonbox.h>

#include <qcombo.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qstrlist.h>

#include <algorithm>


extern void UpdateInset( Inset *inset, bool mark_dirty = true );



InsertCitationReferenceDialog::InsertCitationReferenceDialog( LyXView *view, QWidget *parent=0, 
							      const char *name=0, WFlags f=0 )
    : QDialog( parent, name, true, f ),
      _view( view )
{
    setCaption( i18n( "Insert Citation" ) );
    QGridLayout *layout = new QGridLayout( this, 4, 3, 10 );
  
    QLabel* dbLA = new QLabel( i18n( "&Database:" ), this );
    dbLA->setAlignment( AlignVCenter | AlignRight );
    dbLA->setMinimumSize( dbLA->sizeHint() );
    layout->addWidget( dbLA, 0, 0, AlignRight );
    dbCB = new QComboBox( true, this );
    dbCB->setFixedHeight( dbCB->sizeHint().height() );
    dbCB->setMinimumWidth( dbCB->sizeHint().width() );
    layout->addMultiCellWidget( dbCB, 0, 0, 1, 2, AlignCenter );
    dbLA->setBuddy( dbCB );
    connect( dbCB, SIGNAL( activated( const char * ) ), SLOT( newDatabase( const char * ) ) );

    QLabel* keyLA = new QLabel( i18n( "&Key:" ), this );
    keyLA->setAlignment( AlignVCenter | AlignRight );
    keyLA->setMinimumSize( keyLA->sizeHint() );
    layout->addWidget( keyLA, 1, 0, AlignRight );
    keyCB = new QComboBox( true, this );
    keyCB->setFixedHeight( keyCB->sizeHint().height() );
    keyCB->setMinimumWidth( keyCB->sizeHint().width() );
    layout->addMultiCellWidget( keyCB, 1, 1, 1, 2, AlignCenter );
    keyLA->setBuddy( keyCB );
    
    QLabel* remarkLA = new QLabel( i18n( "&Remark:" ), this );
    remarkLA->setAlignment( AlignVCenter | AlignRight );
    remarkLA->setMinimumSize( remarkLA->sizeHint() );
    layout->addWidget( remarkLA, 2, 0, AlignRight );
    remarkED = new QLineEdit( this );
    remarkED->adjustSize();
    remarkED->setFixedHeight( remarkED->sizeHint().height() );
    remarkED->setMinimumWidth( remarkED->sizeHint().width() );
    layout->addMultiCellWidget( remarkED, 2, 2, 1, 2, AlignCenter );
    remarkLA->setBuddy( remarkED );

    KButtonBox *bbox = new KButtonBox( this );
    bbox->addStretch( 1 );
    QPushButton* okPB = bbox->addButton( i18n( "OK" ) );
    okPB->setMinimumSize( okPB->sizeHint() );
    okPB->setDefault( true );
    connect( okPB, SIGNAL( clicked() ), SLOT( okClicked() ) );
    bbox->addStretch( 1 );
    QPushButton* cancelPB = bbox->addButton( i18n( "Cancel" ) );
    cancelPB->setMinimumSize( cancelPB->sizeHint() );
    connect( cancelPB, SIGNAL( clicked() ), SLOT( cancelClicked() ) );
    bbox->addStretch( 1 );
    bbox->layout();
    layout->addMultiCellWidget( bbox, 3, 3, 0, 2 );

    layout->addColSpacing( 1, keyCB->sizeHint().width() );
    layout->addColSpacing( 2, keyCB->sizeHint().width() );
    layout->activate();
    resize( sizeHint() );
}



void InsertCitationReferenceDialog::cancelClicked()
{
    keys.erase( keys.begin(), keys.end() );
    reject();
}



void InsertCitationReferenceDialog::ensureBibItem( const char* item )
{
    bool found = false;
    map< string, set< string > >::iterator db = keys.begin();
    unsigned int db_idx = 0;
    while( db !=  keys.end() && ! found ) {
	set< string >::iterator key = db->second.begin();
	unsigned int key_idx = 0;
	while( key !=  db->second.end() && ! found ) {
	    if( ! key->compare( item ) ) {
		dbCB->setCurrentItem( db_idx );
		keyCB->setCurrentItem( key_idx );
		found = true;
	    }
	    key++;
	    key_idx++;
	}
	db++;
	db_idx++;
    }
    if( ! found ) {
	dbCB->setCurrentItem( 0 );
	keyCB->insertItem( item );
    }
    return;
}



void InsertCitationReferenceDialog::newDatabase( const char *db )
{
    unsigned int i = 0;
    keyCB->clear();
    set< string >::iterator iter = keys[ db ].begin();
    while( iter !=  keys[ db ].end() ) {
	keyCB->insertItem( iter->c_str(), i++ );
	iter++;
    }
}



void InsertCitationReferenceDialog::okClicked()
{
    keys.erase( keys.begin(), keys.end() );
    if( !_view->currentBuffer()->isReadonly() ) {
	_inset->setContents( keyCB->currentText() );
	_inset->setOptions( remarkED->text() );
	// shouldn't mark the buffer dirty unless something was actually altered
	UpdateInset( _inset );
    }
    accept();
}



void InsertCitationReferenceDialog::setRemark( const char *remark )
{
    remarkED->setText( remark );
}



void InsertCitationReferenceDialog::updateBibItems()
{
    string bibDb;
    set< string > list;
    if( ! _view->currentView()->available() )
	return;
    LyXParagraph *par = _view->currentBuffer()->paragraph;
    while( par ) {
	int pos = 0;
	Inset *inset = par->ReturnNextInsetPointer( pos );
	while( inset ) {
	    switch( inset->LyxCode() ) {
	    case Inset::BIBTEX_CODE:
		// more BibTeX databases
		bibDb += dynamic_cast< InsetBibtex * >( inset )->getContents().c_str() + string( "," );
		break;
	    case Inset::BIBITEM_CODE:
	    case Inset::CITATION_CODE:
		// add key from document
		list.insert( dynamic_cast< InsetCommand * >( inset )->getContents().c_str() );
		break;
	    default:
		break;
	    }
	    pos++;
	    inset = par->ReturnNextInsetPointer( pos );
	}
	par = par->next;
    }
    // save internal keys
    keys.insert( make_pair( i18n( "Internal keys" ), list ) );
    if( ! bibDb.empty() ) {
	// Get keys from the bibtex databases (based on InsetBibtex::getKeys from LyX-1.0.2 )
	// This is a quick and dirty adaption to KLyX :-(

	// We need to create absolute path names for bibliographies. First look for bib-file in same
	// directory as document, then in all directories listed in environment variable  BIBINPUTS
	LString bibfiles( bibDb.c_str() ), linebuf, tmp;
	bibfiles.split( tmp, ',' );
	while( ! tmp.empty() ) {
	    string db( tmp.c_str() );
	    list.erase( list.begin(), list.end() );
	    if( IsFileReadable( MakeAbsPath( tmp, _view->currentBuffer()->filepath) + ".bib" ) )
		tmp = MakeAbsPath( tmp, _view->currentBuffer()->filepath ) + ".bib";
	    else {
		tmp = FileOpenSearch( getEnvPath( "BIBINPUTS" ), tmp, "bib" );
		if( tmp.empty() )
		    tmp = FileOpenSearch( getEnvPath( "BIBINPUT" ), tmp, "bib" );
	    }
	    // If we didn't find a matching file name just fail silently
	    if( ! tmp.empty() ) {
		// This is a _very_ simple parser for Bibtex database files. All it does is to look for
		// lines starting in @ and not being @preamble and @string entries. It does NOT do any
		// syntax checking!
		FilePtr file( tmp, FilePtr::read );
		char c;
		// On some systems where feof() is a macro, the () after file is needed (JMarc)
		while( ! feof( file() ) ) {
		    c = fgetc(file);
		    // At end of each line check if line begins with '@'
		    if( c == '\n' ) {
			if ( linebuf.prefixIs( "@" ) ) {
			    linebuf.subst( '{', '(' );
			    linebuf.split( tmp, '(' );
			    tmp.lowercase();
			    if ( ! tmp.prefixIs( "@string" ) && ! tmp.prefixIs( "@preamble" ) ) {
				linebuf.split( tmp, ',' );
				if( ! tmp.empty() )
				    list.insert( tmp.strip().c_str() );
			    }
			}
			linebuf.clean();
		    } else {
			linebuf += c;
		    }
		}
		// store keys
		keys.insert( make_pair( db, list ) );
	    }
	    // Get next file name
	    bibfiles.split( tmp, ',' );
	}
    }
    // stable_sort( ++( keys.begin() ), keys.end() );
    dbCB->clear();
    map< string, set< string > >::iterator iter = keys.begin();
    unsigned int i = 0;
    while( iter !=  keys.end() ) {
	dbCB->insertItem( iter->first.c_str(), i++ );
	iter++;
    }
    dbCB->setCurrentItem( 0 );
    newDatabase( keys.begin()->first.c_str() );
}



// Local Variables:
// mode: C++
// c-file-style: "Stroustrup"
// End:
