/* Copyright (C) 1999 Claus Wilke <claus.wilke@gmx.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <qwidget.h>
#include <qtextstream.h>
#include <qdom.h>
#include <klocale.h>

#include "entrytypes.h"
#include "kbibliodoc.h"

//#include "entrytypes.cpp"


BibEntryFieldType::BibEntryFieldType()
{
    mType = NOTHING;
    mName = "";
    mDescription = "";
    mEditable = false;
}


BibEntryFieldType::BibEntryFieldType( const DataType &type,
				      const QString &name,
				      const QString &description, bool editable )
{
    mType = type;
    mName = name;
    mDescription = description;
    mEditable = editable;
}

BibEntryFieldType::BibEntryFieldType( const BibEntryFieldType &ef )
{
    mType = ef.mType;
    mName = ef.mName;
    mDescription = ef.mDescription;
    mEditable = ef.mEditable;
}


BibEntryFieldType::~BibEntryFieldType()
{
}

bool
BibEntryFieldType::operator==( const BibEntryFieldType &ef2 ) const
{
    if ( mName == ef2.mName ) return true; // we distinguish BibEntryFieldTypes
    // only by their name
    else return false;
}

bool
BibEntryFieldType::operator!=( const BibEntryFieldType &ef2 ) const
{
    if ( mName != ef2.mName ) return true; // we distinguish BibEntryFieldTypes
    // only by their name
    else return false;
}

BibEntryFieldType::DataType
BibEntryFieldType::dataType() const
{
    return mType;
}

QString
BibEntryFieldType::name() const
{
    return mName;
}

QString
BibEntryFieldType::description() const
{
    return mDescription;
}

bool
BibEntryFieldType::isEditable() const
{
    return mEditable;
}

void
BibEntryFieldType::setEditable( bool editable )
{
    mEditable = editable;
}



//==================================================
// class BibEntryType

BibEntryType::BibEntryType( const QStringList &fieldsList, const QString &name, const QString &description )
{
    mFieldsList = new QStringList( fieldsList );
    mName = name;
    mDescription = description;
}

// convenience method for easier creation of BibEntryTypes.
// The QString array has to end in "-1";

BibEntryType::BibEntryType( const QString *fields, const QString &name, const QString &description )
{
    mFieldsList = new QStringList();
    mName = name;
    mDescription = description;
    int i=0;
    QString tmpString = fields[0];

    do{
	mFieldsList->append( tmpString );
	i++;
	tmpString = fields[i];
    } while ( tmpString != "-1" );
}

BibEntryType::~BibEntryType()
{
    delete mFieldsList;
}

bool
BibEntryType::operator==( const BibEntryType &entryType ) const
{
    //let's assume equal names imply equal BibEntryTypes
    if ( mName == entryType.mName )
    return true;
    else return false;
}

bool
BibEntryType::operator!=( const BibEntryType &entryType ) const
{
    //let's assume equal names imply equal BibEntryTypes
    if ( mName != entryType.mName )
    return true;
    else return false;
}

QString
BibEntryType::name() const
{
    return mName;
}

QString
BibEntryType::description() const
{
    return mDescription;
}

QStringList
BibEntryType::fieldTypesNames() const
{
    return QStringList( *mFieldsList );
}

//============================================================
//
//               class KBiblioDoc

KBiblioDoc::KBiblioDoc()
{
    setupEntryFieldTypeDict();
    setupEntryTypeDict();
    setupCompletionsListDict();
    
    setEmptyDoc();
    mEdited = false;
}


KBiblioDoc::~KBiblioDoc()
{
    delete mEntryFieldTypeDict;
    delete mEntryTypeDict;
    delete mCompletionsListDict;
}


QDomDocument
KBiblioDoc::domDocument() const
{
    return mDomDocument;
}


QDomNode
KBiblioDoc::entryListNode() const
{
    return mDomDocument.firstChild();
}


QDomElement
KBiblioDoc::firstEntry() const
{
    return entryListNode().firstChild().toElement();
}


void
KBiblioDoc::setContent( const QString &text )
{
    mDomDocument.setContent( text );
    setEdited( true );
}


void
KBiblioDoc::setEmptyDoc()
{
   setContent( "<!DOCTYPE KBIBLIO_DOC ><KBIBLIO_DOC/>\n" );
}


void
KBiblioDoc::addEntry( QDomElement entry )
{
    mDomDocument.firstChild().appendChild( entry );
    setEdited( true );
}


void
KBiblioDoc::removeEntry( QDomElement entry )
{
    mDomDocument.firstChild().removeChild( entry );
    setEdited( true );
}


void
KBiblioDoc::replaceEntry( QDomElement oldEntry, QDomElement newEntry )
{
    mDomDocument.firstChild().replaceChild( newEntry, oldEntry );
    setEdited( true );
}


void
KBiblioDoc::save( QTextStream &t ) const
{
    mDomDocument.save( t );
}


bool
KBiblioDoc::edited() const
{
    return mEdited;
}


void
KBiblioDoc::setEdited( bool edited )
{
    mEdited = edited;
}


BibEntryFieldType*
KBiblioDoc::lookupEntryFieldType( const QString &name )
{
    return mEntryFieldTypeDict->find( name );
}


BibEntryType*
KBiblioDoc::lookupEntryType( const QString &name )
{
    return mEntryTypeDict->find( name );
}


QStringList* 
KBiblioDoc::lookupCompletionsList( const QString &fieldName )
{
    return mCompletionsListDict->find( fieldName );
}


QStringList
KBiblioDoc::entryTypesNames()
{
    return QStringList( mEntryTypesNames );
}


QStringList
KBiblioDoc::fieldTypesNames( const QString &entryName )
{
    BibEntryType *entryType = lookupEntryType( entryName );

    if ( entryType != 0 )
	return entryType->fieldTypesNames();
    else{
	return QStringList();
    }
}


QStringList
KBiblioDoc::allFieldTypesNames()
{
    return QStringList( mFieldTypesNames );
}


void
KBiblioDoc::setupEntryFieldTypeDict()
{
    mEntryFieldTypeDict = new QDict<BibEntryFieldType> ( 137, false );
    mEntryFieldTypeDict->setAutoDelete( true );

    int i;
    BibEntryFieldType *ft;
    for ( i=0; i<gNoFieldTypes; i++ ){
	ft = new BibEntryFieldType( gFieldTypes[i] );
	mFieldTypesNames.append( ft->name() );
	mEntryFieldTypeDict->insert( ft->name(), ft );
    }
    mFieldTypesNames.sort();
}


void
KBiblioDoc::setupEntryTypeDict()
{
    mEntryTypeDict = new QDict<BibEntryType> ( 137, false );
    mEntryTypeDict->setAutoDelete( true );

    BibEntryType *et;

    //article
    mEntryTypesNames.append( "ARTICLE" );
    et = new BibEntryType( gArticleType, "ARTICLE", i18n( "Article in Journal" ) );
    mEntryTypeDict->insert( et->name(), et );

    //book
    mEntryTypesNames.append( "BOOK" );
    et = new BibEntryType( gBookType, "BOOK", i18n( "Book" ) );
    mEntryTypeDict->insert( et->name(), et );

    //inProceedings
    mEntryTypesNames.append( "INPROCEEDINGS" );
    et = new BibEntryType( gInProceedingsType, "INPROCEEDINGS", i18n( "Article in Conference Proceedings" ) );
    mEntryTypeDict->insert( et->name(), et );

    //Proceedings
    mEntryTypesNames.append( "PROCEEDINGS" );
    et = new BibEntryType( gProceedingsType, "PROCEEDINGS", i18n( "Conference Proceedings" ) );
    mEntryTypeDict->insert( et->name(), et );

    //inCollection
    mEntryTypesNames.append( "INCOLLECTION" );
    et = new BibEntryType( gInCollectionType, "INCOLLECTION", i18n( "Article in Collection" ) );
    mEntryTypeDict->insert( et->name(), et );

    //Collection
    mEntryTypesNames.append( "COLLECTION" );
    et = new BibEntryType( gCollectionType, "COLLECTION", i18n( "Collection of Articles" ) );
    mEntryTypeDict->insert( et->name(), et );
    
    //Unpublished
    mEntryTypesNames.append( "UNPUBLISHED" );
    et = new BibEntryType( gUnpublishedType, "UNPUBLISHED", i18n( "Unpublished" ) );
    mEntryTypeDict->insert( et->name(), et );
    
    //Misc
    mEntryTypesNames.append( "MISC" );
    et = new BibEntryType( gMiscType, "MISC", i18n( "Miscellaneous" ) );
    mEntryTypeDict->insert( et->name(), et );
    
}


void 
KBiblioDoc::setupCompletionsListDict()
{
    mCompletionsListDict = new QDict<QStringList> ( 137, false );
    mCompletionsListDict->setAutoDelete( true );
    
    QStringList *cl = new QStringList();
    QString tmpStr;
    
    int i = 0;
    tmpStr = gJournalCompletions[0];
    while ( tmpStr != "-1" ){
	cl->append( tmpStr );
	i += 1;
	tmpStr = gJournalCompletions[i];
    }
    mCompletionsListDict->insert( "JOURNAL", cl );
    
    cl = new QStringList();
    i = 0;
    tmpStr = gPublisherCompletions[0];
    while ( tmpStr != "-1" ){
	cl->append( tmpStr );
	i += 1;
	tmpStr = gPublisherCompletions[i];
    }
    mCompletionsListDict->insert( "PUBLISHER", cl );

    cl = new QStringList();
    i = 0;
    tmpStr = gAddressCompletions[0];
    while ( tmpStr != "-1" ){
	cl->append( tmpStr );
	i += 1;
	tmpStr = gAddressCompletions[i];
    }
    mCompletionsListDict->insert( "ADDRESS", cl );

    cl = new QStringList();
    i = 0;
    tmpStr = gMonthCompletions[0];
    while ( tmpStr != "-1" ){
	cl->append( tmpStr );
	i += 1;
	tmpStr = gMonthCompletions[i];
    }
    mCompletionsListDict->insert( "MONTH", cl );
 
    
    
    
    
    
    

}

#include "kbibliodoc.moc"





