/* Copyright (C) 1999 Claus Wilke <claus.wilke@gmx.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/


#include <qdom.h>
#include <qstring.h>
#include <qregexp.h>
#include <stdio.h>

#include "kbiblioshell.h"
#include "entrymanipulator.h"

EntryManipulator::EntryManipulator()
{
    mShell = 0;
    mDocument = 0;
    mEntryName = QString::null;
}


EntryManipulator::EntryManipulator( QDomElement entry, KBiblioShell *shell )
{
    mShell = shell;
    mDocument = shell->document();
    mEntry = entry;
    mEntryName = mEntry.nodeName().upper();
}


EntryManipulator::EntryManipulator( QDomElement entry, KBiblioDoc *doc )
{
    mShell = 0;
    mDocument = doc;
    mEntry = entry;
    mEntryName = mEntry.nodeName().upper();
}


EntryManipulator::EntryManipulator( const EntryManipulator &manip )
{
    mShell = manip.mShell;
    mDocument = manip.mDocument;
    mEntry = manip.mEntry;
    mEntryName = manip.mEntryName;
}


EntryManipulator::~EntryManipulator()
{
}


EntryManipulator&
EntryManipulator::operator=( const EntryManipulator &manip )
{
    mShell = manip.mShell;
    mDocument = manip.mDocument;
    mEntry = manip.mEntry;
    mEntryName = manip.mEntryName;

    return *this;
}


BibEntryFieldType::DataType
EntryManipulator::fieldDataType( const QString &fieldName ) const
{
    BibEntryFieldType *entryFieldType = mDocument->lookupEntryFieldType( fieldName );
    if ( entryFieldType != 0 )
	return entryFieldType->dataType();
    else return BibEntryFieldType::NOTHING;
}


QString
EntryManipulator::name() const
{
    return mEntryName;
}


QString
EntryManipulator::label() const
{
    return mEntry.attribute( "label" );
}


QString
EntryManipulator::fieldFlatText( const QString &fieldName, int length ) const
{
    QDomElement field = mEntry.namedItem( fieldName ).toElement();

    QString returnString = "";

    if ( !field.isNull() )
	switch ( fieldDataType( fieldName ) ){
	case BibEntryFieldType::STRING:
	    returnString = stringFieldText( field );
	    break;
	case BibEntryFieldType::TEXT:
	    returnString = textFieldText( field );
	    break;
	default:
	    returnString = "not Supported";
	}
    returnString.replace( QRegExp("&amp;"), "&" );
    returnString.replace( QRegExp("&lt;"), "<" );
    returnString.replace( QRegExp("&gt;"), ">" );

    returnString.simplifyWhiteSpace();
    if ( length >= 0 )
	returnString.truncate( length );

    return returnString;
}


QString
EntryManipulator::fieldRichText( const QString &fieldName ) const
{
    QDomElement field = mEntry.namedItem( fieldName ).toElement();

    QString returnString = "";

    if ( !field.isNull() )
	switch ( fieldDataType( fieldName ) ){
	case BibEntryFieldType::STRING:
	    returnString = stringFieldText( field );
	    break;
	case BibEntryFieldType::TEXT:
	    returnString = textFieldText( field );
	    break;
	default:
	    returnString = "not Supported";
	}

    return returnString;
}


QString
EntryManipulator::richDisplayText() const
{
    QString result = "";
    QStringList fields = mShell->viewableFields( mEntryName );
    QStringList::Iterator it = fields.begin();

    QString title = fieldRichText( "TITLE" );
    if ( title.isEmpty() ) title = QString( "(No Title)" );

    /*
    QString url = fieldRichText( "URL" );
    if ( !url.isEmpty() )
	title = QString( "<a href=\"%1\">%2</a>" ).arg( url ).arg( title );
    */
	
    QString author = fieldRichText( "AUTHOR" );
    if ( author.isEmpty() ) author = QString( "(No Author)" );

    result += QString( "<big><big><b>%1</b></big></big><br>").arg( title );

    result += QString( "<big><b>%1</b></big><br><br>").arg( author );

    QString descr, value;

    for ( ; it != fields.end(); ++it ){
	if ( ( *it != QString( "TITLE" ) ) && ( *it != QString( "AUTHOR" ) ) )
	    {
		if ( BibEntryFieldType::TEXT == fieldDataType( *it ) ){
		    descr = mShell->fieldDescription( *it );
		    value = fieldRichText( *it );
		    if ( !value.isEmpty() )
			result += QString( "<br><b>%1:</b><br>%2<br>").
				  arg( descr ).arg( value );
		}
		else {
		    descr = mShell->fieldDescription( *it );
		    value = fieldRichText( *it );
		    if ( !value.isEmpty() )
			result += QString( "<b>%1:</b> %2<br>").
				  arg( descr ).arg( value );
		}
	    }
    }
    return result;
}


bool
EntryManipulator::find( const QRegExp &regExp, const QString &fieldName )
{
    if ( fieldFlatText( fieldName ).find( regExp ) >= 0 )
	return true;
    else return false;
}


void
EntryManipulator::setValue( const QString &fieldName, const QString &value )
{
    QString v = value;

    v.replace( QRegExp("&"), "&amp;" );
    v.replace( QRegExp("<"), "&lt;" );
    v.replace( QRegExp(">"), "&gt;" );

    BibEntryFieldType *entryFieldType =
				       mDocument->lookupEntryFieldType( fieldName );

    if ( entryFieldType != 0 ){
	QDomElement field = mEntry.namedItem( fieldName ).toElement();
	
	if ( !v.isEmpty() ){
	    if ( field.isNull() ){
		field = mDocument->domDocument().createElement( fieldName );
		mEntry.appendChild( field );
	    }
	    switch ( entryFieldType->dataType() ){
	    case BibEntryFieldType::STRING:
		stringFieldSetValue( field, v );
		break;
	    case BibEntryFieldType::TEXT:
		textFieldSetValue( field, v );
		break;
	    default:
		break;
	    }
	}
	// empty value: remove Dom Element
	else {
	    if ( !field.isNull() )
		mEntry.removeChild( field );
	}
    }
}


void
EntryManipulator::setLabel( const QString &label )
{
    mEntry.setAttribute( "label", label );
}


QString
EntryManipulator::stringFieldText( QDomElement field ) const
{
    return field.firstChild().nodeValue();
}


QString
EntryManipulator::textFieldText( QDomElement field ) const
{
    return field.firstChild().nodeValue();
}


void
EntryManipulator::stringFieldSetValue( QDomElement field, const QString &value )
{
    QDomText text = field.firstChild().toText();
    if ( text.isNull() ){ // TODO: works only if either the first child is really a text node or there is no first child! should be fixed!
	text = mDocument->domDocument().createTextNode( value );
	field.appendChild( text );
    }
    else
	text.setNodeValue( value );
}


void
EntryManipulator::textFieldSetValue( QDomElement field, const QString &value )
{
    stringFieldSetValue( field, value );
}
