/***************************************************************************
                          knetstartdoc.cpp  -  A simple netconfig tool
                             -------------------
    begin                : Thu Sep 30 20:48:28 CDT 1999
    copyright            : (C) 1999 by Federico David Sacerdoti
    email                : fds@slinuxmachines.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/

// include files for Qt
#include <qdir.h>
#include <qfileinfo.h>
#include <qwidget.h>
#include <qtimer.h>

// include files for KDE
#include <kapp.h>
#include <kmsgbox.h>

// application specific includes
#include <knetstartdoc.h>
#include "knetstart.h"
#include "knetstartview.h"
#include <kprocess.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <iostream.h>
#include <stdlib.h>
#include "ipinput.h"
#include "netmap.h"
extern "C" {
	#include "net.h"
	//#include "pump.h"
}

QList<KnetstartView>* KnetstartDoc::viewList = 0L;

KnetstartDoc::KnetstartDoc(QWidget *parent, const char* name)
	:QObject(parent, name)
{
	if( !viewList )
		viewList = new QList<KnetstartView>;
	viewList->setAutoDelete(true);
	
	// Setup attributes for ping-based network discovery
	ping = new KProcess;
	pingTimer = new QTimer(this);
	
	connect( ping, SIGNAL(processExited(KProcess*)), this, SLOT(slotPingCleaner(KProcess*)));
	connect( pingTimer, SIGNAL(timeout()), this, SLOT(slotPingDead()));
}

KnetstartDoc::~KnetstartDoc()
{
}

void KnetstartDoc::readRecent (KConfig *config)
{
	config->setGroup("Network");
	dhcp = config->readBoolEntry("DHCP", false);
  ip = config->readEntry("IP", "");
  gateway = config->readEntry("Gateway", "");
  ns = config->readEntry("NameServer", "");
}

void KnetstartDoc::addView(KnetstartView* m_pView)
{
	viewList->append(m_pView);
}

void KnetstartDoc::removeView(KnetstartView* m_pView)
{
	viewList->remove(m_pView);
}
const QString& KnetstartDoc::getPathName() const
{
	return m_path;
}

void KnetstartDoc::slotUpdateAllViews(KnetstartView* pSender)
{
	KnetstartView* w;
	if(viewList)
	{
		for( w = viewList->first(); w; w = viewList->next() )
		{ if( w != pSender)
				w->repaint();
		}
	}

}

/*
void KnetstartDoc::pathName( const char* path_name)
{
	m_path=path_name;
}
void KnetstartDoc::title( const char* title)
{
	m_title=title;
}

const QString& KnetstartDoc::getTitle() const
{
	return m_title;
}


bool KnetstartDoc::saveModified()
{

		return true;

}

void KnetstartDoc::closeDocument()
{
	deleteContents();
}

bool KnetstartDoc::newDocument()
{
	
	/////////////////////////////////////////////////
	// TODO: Add your document initialization code here
	/////////////////////////////////////////////////
	b_modified=false;
	m_path=QDir::homeDirPath();
	m_title="Untitled";
	return true;
}

bool KnetstartDoc::openDocument(const char* filename, const char* format)
{
	QFileInfo fileInfo(filename);
	m_title=fileInfo.fileName();
	m_path=fileInfo.absFilePath();
	//const char *tFormat = format ? format : "r+";
	
	/////////////////////////////////////////////////
	// TODO: Add your document opening code here
	
	//FILE* fin = fopen(filename, tFormat);
	//if (fin) {
	
	//} else perror("Knetstart got stuck on an open");
	/////////////////////////////////////////////////
	
	b_modified=false;
	return true;
}
*/

void KnetstartDoc::slotDhcp (bool toggleState)
{
	dhcp = toggleState;
	emit dhcpToggled (toggleState);
}

/* FIXME. I would like to put up some progress meter during this operation. Especially
if the DHCP server cannot be reached, the user needs to be 'anesthetized'. Otherwise
the person is liable to click the button 50 times and get really mad when the computer
waits for the DHCP timeout * 50.

Also, I tried to use pump v0.7.3 code, but it did not work at all for me. It always
returned "Operation Failed" and I had trouble tracing the problem with gdb. So I have
reverted to version 0.6.7, which is simpler, but seems to work fine.
*/
bool KnetstartDoc::pumpUpTheDhcp () {
	struct intfInfo result;
	//struct pumpNetIntf result;
	
	// Run a DHCP client (Pump)
	if ( startPump(&result) != 0) {
		viewList->current()->nmap->lostGateway();
		viewList->current()->nmap->lostNs();
		emit statusMsg(i18n("Cannot contact the DHCP server. Is there really one on your network?"));
		return false;
	}
	
	ip = inet_ntoa(result.ip);
	gateway = inet_ntoa(result.gateway);
	ns = inet_ntoa(result.dnsServers[0]);
	
	//Write an eth0 config with DHCP set, for the next time system boots.
	commitEthConfig(ip);
	
	//Pump was sucessful, start network discovery
	viewList->current()->nmap->findGateway();
	node = "Gateway";
	emit statusMsg(i18n(QString("Looking for Gateway (" + gateway + ") on network...")));
	pingNode(gateway);
	
	return true;
}

void KnetstartDoc::activateNet()
{
	KProcess *network = new KProcess;
	//Check to see if script exists
	QFile networkScript("/etc/rc.d/init.d/network");
	if (networkScript.exists() == false) {
		emit statusMsg(i18n("Cannot find network script. Is this a RedHat Linux system?"));
		return;
	}
	//network.clearArguments();
	*network <<  "/etc/rc.d/init.d/network" << "restart";
	
	connect(network, SIGNAL(processExited(KProcess*)), this, SLOT(slotNetworkCleaner(KProcess*)));
		
	if (network->start() == false) {
		emit statusMsg(i18n("Cannot run network script. Try running it manually."));
	}
	
	return;
}

void KnetstartDoc::pingNode( const QString & nodeIP )
{
	ping->kill();
	ping->clearArguments();
	*ping << "/bin/ping" << "-c 2" << nodeIP;
	
	if (ping->start() == false) {
		emit statusMsg(i18n("Cannot ping network node " + nodeIP ));
	}	
	//If ping has not exited after a while, the node is unreachable
	pingTimer->start (5000,TRUE);
	return;
}

void KnetstartDoc::slotPingDead ()
{
	if (ping->isRunning() == true) ping->kill();	
}

bool KnetstartDoc::slotNetworkCleaner(KProcess * network)
{
	
	if (network->normalExit() == true) {
		if (network->exitStatus() == 0) {
			//Eth setup successful. Start network discovery
			viewList->current()->nmap->findGateway();
			node = "Gateway";
			emit statusMsg(i18n("Looking for Gateway on network..."));
			pingNode(gateway);
		}
	}
	else emit statusMsg(i18n("The network restart script had problems. Try running it manually."));
	
	delete network;
	return true;
}

bool KnetstartDoc::slotPingCleaner(KProcess * p)
{
	if (p->normalExit() == true) {
		if (p->exitStatus() == 0) {
			if (node == "Gateway") {
				//Gateway reachable. Find nameserver
				viewList->current()->nmap->foundGateway();
				viewList->current()->nmap->findNs();
				node = "Nameserver";
				emit statusMsg(i18n(QString("Looking for Nameserver ("+ns+") on network...")));
				pingNode(ns);
			}
			else if (node == "Nameserver") {
				//Nameserver reachable. Machine has connected successfully
				viewList->current()->nmap->foundNs();
				emit statusMsg(i18n("Connected to network. Go test your new connection!") );
			}
		}
	}
	// Ping failed
	else {
		emit statusMsg(i18n("Cannot reach " + node + ". Check your settings again."));
		if (node == "Gateway") {
			viewList->current()->nmap->lostGateway();
			viewList->current()->gateway->bad();
		}
		if (node == "Nameserver") {
			viewList->current()->nmap->lostNs();
			viewList->current()->ns->bad();
		}
	}
	
	return true;
}


bool KnetstartDoc::lookFor(const char* line, const char* keyword)
{
  const char *p = line;

  // skip leading space and tabs
  while(*p == ' ' || *p == '\t')
    p++;
  // compare pattern with keyword
  if(strncmp(keyword, p, sizeof(keyword)) != 0)
    return false;	

  return true;
}

bool KnetstartDoc::installConfig(const char *installMe, const char *original)
{
	//KnetstartApp* theApp=(KnetstartApp*)parent();
	// /tmp/knetstartNet.txt -> /etc/sysconfig/network
	
	// Save old config in data directory and put in recent file list
	
	//QString recentConfig = KApplication::localkdedir();
	//recentConfig += "/share/apps/knetstart/recentConfig";
	//cerr << "I will save the original config file as: " << recentConfig << endl;
	
	//if (rename(original, recentConfig) == -1) return false;
	//theApp->addRecentFile(recentConfig);
	
	if (rename(installMe, original) == -1) return false;
	
	return true;
}

bool KnetstartDoc::commitEthConfig(const QString & i)
{	
	ip = i;
	
	int old_umask = umask(0022);
	char
		//*tmpEth = "/tmp/knetstartEth.txt",
		*ethConfig = "/etc/sysconfig/network-scripts/ifcfg-eth0";
	FILE *fout;
	
	// Open the ethernet config file in the network-scripts folder.
	fout = fopen(ethConfig, "w");		
	if (fout) {
		//Generate the whole file. The subnet mask will be calculated automatically by /bin/ipcalc
		fprintf(fout, "DEVICE=eth0\n");
		if (dhcp == false)
			fprintf(fout, "IPADDR=%s\n", (const char*) ip);
		else
			fprintf(fout, "BOOTPROTO=dhcp\n");
		fprintf(fout, "ONBOOT=yes\n");
	 	fclose(fout);
	   	
	} else emit statusMsg(i18n("Cannot open /etc/sysconfig/network-scripts/ifcfg-eth0. Is this a RedHat Linux system with an Ethernet card?"));
	 	
	// restore umask
 	umask(old_umask);
	return true;	
}


bool KnetstartDoc::commitNetConfig(const QString & g, const QString & n)
{
		/* FIXME these are obviously fake, but I dont want to run pump -i eth0 --status
		  if the server is not configured correctly - there will be another 5 min
		  TCP timeout wait. */
	
	// We are pretty sure that these are 'valid' addresses
	gateway = g;
	ns = n;
	
	char line[100];
	int old_umask = umask(0022);
	char
		*tmpNet = "/tmp/knetstartNet.txt",
		*tmpNs = "/tmp/knetstartNs.txt",
		*network = "/etc/sysconfig/network",
		*resolv = "/etc/resolv.conf";		
	FILE *fout, *fin;
	
	// Put the gateway in the 'network' file
	fout = fopen(tmpNet, "w");		
	if (fout) {
		//Open Network file for Redhat 5.1 - 6.0 Linux
		fin = fopen(network, "r");
		if (fin) {
			fprintf(fout, "NETWORKING=yes\n");
	   	//Go through the old file and pick out the useful stuff
	   	while(fgets(line, sizeof(line), fin)) {
	   		if (lookFor(line, "NETWORKING")) continue;	    		
	    	else if (lookFor(line, "GATEWAY")) continue;	
	   		else if (lookFor(line, "GATEWAYDEV")) continue;
	   		//Copy the rest of file unchanged
	   		else fputs(line, fout);
	   		// in case LF is missing at EOF
        if(line[strlen(line)-1] != '\n')
          putc('\n', fout);
	   	}
	   	fprintf(fout, "GATEWAY=%s\n", (const char*) gateway);
	   	fprintf(fout, "GATEWAYDEV=%s\n", NETDEVICE);
	   	
	   	fclose(fin);
	   	fclose(fout);
	   	
	   	if ( installConfig(tmpNet, network) == false) return false;
		
		} else emit statusMsg(i18n("Cannot find network config file. Is this a RedHat Linux system?"));		
	} else emit statusMsg(i18n("Cannot open temp file, where is /tmp ?"));
	
	// Put nameserver in /etc/resolv.conf file
	fout = fopen(tmpNs, "w");
	if (fout) {
		//Open resolver file for all RedHat Linux systems
		fin = fopen(resolv, "r");
		if	(fin) {
			while (fgets(line, sizeof(line), fin)) {
				//Dont use any old nameservers
				if (lookFor(line, "nameserver")) continue;
				//Copy the search domain lines
	   		else fputs(line, fout);
	   		// in case LF is missing at EOF
        if(line[strlen(line)-1] != '\n')
          putc('\n', fout);
      }
      fprintf(fout, "nameserver %s\n", (const char*) ns);
    	fclose(fin);
    	fclose(fout);
    	
    	if ( installConfig(tmpNs, resolv) == false ) return false;
    	
    } else emit statusMsg(i18n("Cannot find nameserver config file. Is this a RedHat Linux system?"));
  } else emit statusMsg(i18n("Cannot open temp file, where is /tmp ?"));

 	// restore umask
 	umask(old_umask);
 	return true;
}


/*
bool KnetstartDoc::saveDocument(const char* filename, const char* format)
{

	/////////////////////////////////////////////////
	// TODO: Add your document saving code here
	/////////////////////////////////////////////////

	b_modified=false;
	return true;
}

void KnetstartDoc::deleteContents()
{
	/////////////////////////////////////////////////
	// TODO: Add implementation to delete the document contents
	/////////////////////////////////////////////////

}

*/

































































































































