// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/modules/MSequential.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:02:16 $
// $State: Exp $
// **************************************************************

#define MODULE_NAME "sequential"

#include "ModuleMacros.h"
#include "minmaxabs.h"
#include "NullSound.h"

BEGIN_MODULE_DEFINITION(Sequential);
    Slot  *sslot_input1;
    Slot  *sslot_input2;
END_MODULE_DEFINITION(Sequential);

class SSSequentialOutput : public Signal
{
public:
    SSSequentialOutput(MSequential *m) 
	: Signal(SOUND_TYPE, "output", "Input1 and input2 after each other", m) {};
    PreparedSignal *prepareSignal(Metainfo *, const Parameterset *);
};

class PSSSequentialOutput : public PreparedSoundSignal
{
    friend class SSSequentialOutput;
    PreparedSoundSignal *pssig_input1;
    PreparedSoundSignal *pssig_input2;
    long cut_from_in1, cut_to_in1;
    long cut_from_in2, cut_to_in2;
    long glue_point;
public:
    PSSSequentialOutput(PreparedSoundSignal *,PreparedSoundSignal *,
			long, long, long, long, long);
    ~PSSSequentialOutput() { delete pssig_input2; delete pssig_input1; };
    SoundPortion getSoundPortion(long start_time, long number_of_samples);
private:
    void computeSamples(Number *, long start_time, long number_of_samples) const;
};


// ---------------------------------------------------------------------------
//                             Implementation
// ---------------------------------------------------------------------------

#include <values.h>

#include "SharedSoundStore.h"


inline PSSSequentialOutput::PSSSequentialOutput(
    PreparedSoundSignal *di,
    PreparedSoundSignal *i2,
    long cfd, long ctd, long cf2, long ct2, long gl) :
    
	pssig_input1(di),
	pssig_input2(i2),
	cut_from_in1(cfd),
	cut_to_in1(ctd),
	cut_from_in2(cf2),
	cut_to_in2(ct2),
	glue_point(gl)
{
}


MSequential::MSequential(string)
{
    addConnector(sslot_input1 = new Slot(SOUND_TYPE, "input1", "First sound to play", this, 5));
    addConnector(sslot_input2 = new Slot(SOUND_TYPE, "input2", "Second sound to play", this, 6));
    addConnector(new SSSequentialOutput(this));
}


PreparedSignal *SSSequentialOutput::prepareSignal(Metainfo *mi,
						  const Parameterset *parset)
{
    Seconds sampling_interval = getSamplingInterval(getModule(), mi, parset);

    Metainfo mi_in1 = *mi;
    mi_in1.setDuration(); // I need the duration to glue together
    mi_in1.setCutFrom();  // Optimization here very important!
    mi_in1.setCutTo();    // Optimization here very important!
    
    PreparedSoundSignal *input1 
	= getPreparedSoundSignal(this, ((MSequential *)getModule())->sslot_input1, &mi_in1, parset);
    
    if (!mi_in1.containsDuration()) // Didn't get any duration
    {
	getModule()->reportError("No duration",
				 "The sound signal on input1 has no duration information. "
				 "The sequential module needs to know how long the first "
				 "sound lasts to glue both sounds together. Use the module "
				 "set-duration to give a duration to a sound. ");
	delete input1;
	return NullSound::getPreparedNullSignal(mi, parset);
    }
    
    long glue_point = (long)(mi_in1.getDuration() / sampling_interval);
    
    Metainfo mi_in2 = *mi;
    mi_in2.setCutFrom();  // Optimization here very important!
    mi_in2.setCutTo();    // Optimization here very important!

    PreparedSoundSignal *input2 
	= getPreparedSoundSignal(this, ((MSequential *)getModule())->sslot_input2, &mi_in2, parset);
    
    // duration
    
    Metainfo asked_for = *mi;
    *mi = mi_in1; // take all meta information from the dominant input

    // I save the cutting ranges of the two inputs for later optimizations

    long cut_from_in1 = mi_in1.containsCutFrom() ? mi_in1.getCutFrom() : MINLONG; 
    long cut_to_in1   = mi_in1.containsCutTo()   ? mi_in1.getCutTo()   : MAXLONG;

    long cut_from_in2 = mi_in2.containsCutFrom() ? mi_in2.getCutFrom() : MINLONG;
    long cut_to_in2   = mi_in2.containsCutTo()   ? mi_in2.getCutTo()   : MAXLONG;

    // except for metainfos cut from, cut to and duration:

    if (asked_for.containsCutFrom()) {
	if (mi_in1.containsCutFrom() && mi_in2.containsCutFrom())
	    mi->setCutFrom(min(cut_from_in1, glue_point + cut_from_in2));
	else mi->clearCutFrom();
    }
    if (asked_for.containsCutTo()) {
	if (mi_in1.containsCutTo() && mi_in2.containsCutTo())
	    mi->setCutTo(max(cut_to_in1, glue_point + cut_to_in2));
	else mi->clearCutTo();
    }
    if (asked_for.containsDuration()) {
	if (mi_in2.containsDuration()) mi->setDuration(mi_in1.getDuration() 
						      + mi_in2.getDuration());
	else mi->clearDuration();
    }
    
    
    return new PSSSequentialOutput(input1, input2,
				   cut_from_in1, cut_to_in1,
				   cut_from_in2, cut_to_in2,
				   glue_point);
}


SoundPortion PSSSequentialOutput::getSoundPortion(long start_time, long nsamples)
{
    long end_time = start_time + nsamples;

    // I pay respect to the cutting ranges of the inputs. 
    // I don't handle the case the interval outside BOTH cutting ranges. I this case
    // the interval also is outside MY cutting range. This case is assumed to be handled
    // by the module one level higher.

    // If only one input is to be calculated, because the interval is outside
    // the cutting range of the other, no samples will be copied!

    if (start_time >= cut_to_in1 || end_time <= cut_from_in1)
	return pssig_input2->getSoundPortion(start_time - glue_point, nsamples);

    if (start_time-glue_point >= cut_to_in2 || end_time-glue_point <= cut_from_in2)
	return pssig_input1->getSoundPortion(start_time, nsamples);

    Number *output_buffer = new Number[nsamples];
    computeSamples(output_buffer, start_time, nsamples);
    return SoundPortion(output_buffer, new SharedSoundStore(output_buffer));
}	


void PSSSequentialOutput::computeSamples(Number *output,long start_time, long nsamples) const
{
    SoundPortion sp_in1 = pssig_input1->getSoundPortion(start_time, nsamples);
    SoundPortion sp_in2 = pssig_input2->getSoundPortion(start_time - glue_point, nsamples); 
    const Number *in1   = sp_in1.getSamples();
    const Number *in2   = sp_in2.getSamples();
    while (nsamples--) *output++ = *in1++ + *in2++;
}

