// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/modules/MRealOutput.cc,v $
// $Revision: 1.3 $
// $Date: 1999/05/15 22:48:47 $
// $State: Exp $
// **************************************************************

#define MODULE_NAME "real-output"

#include <linux/soundcard.h>
#include <values.h>

#include "ModuleMacros.h"
#include "SamplingConfig.h"
#include "toString.h"

BEGIN_MODULE_DEFINITION(RealOutput);
  Slot *sslot_input;
  PreparedSoundSignal *pssig_input;
  long  end_time;
  int   audio; // file descriptor number for /dev/audio
  long  audio_block_size;
  char *audio_buffer;
  long  bytes_in_audio_buffer;
  const SamplingConfig *sampling_config;
public:
  bool isExecutable() const { return true; };
  void prepareForExecution(const SamplingConfig *);
  bool executeBlock(long, long);
  bool mustWait();
  void finishExecution();
private:
  void writeDWord(long);
  void writeWord(short);
END_MODULE_DEFINITION(RealOutput);


// ---------------------------------------------------------------------------
//                            Implementation
// ---------------------------------------------------------------------------

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <iostream.h> // debugging!

#include "language.h" // TODO: is this the right place to include the file?

#define AUDIO_DEVICE_PATH "/dev/audio"

MRealOutput::MRealOutput(string) :
    pssig_input(0),
    audio(-1),
    audio_buffer(0)
{
    addConnector(sslot_input = new Slot(SOUND_TYPE, "input", "Sound signal to play", this, 1)); 
}


void MRealOutput::prepareForExecution(const SamplingConfig *sc)
{
    sampling_config = sc;

    pssig_input = 0;
    
    // open audio device
    audio = open(AUDIO_DEVICE_PATH, O_WRONLY, 0);
    if (audio == -1) {
	reportError(l_MRealOutput_1tt, l_MRealOutput_1tx);
	return;
    }

    // Set buffer size of audio device
    long fragsize = sampling_config->fragmentsize;
    audio_block_size = 1 << fragsize;
    if (0 > ioctl(audio, SNDCTL_DSP_SETFRAGMENT, &fragsize))
    {
	close(audio);
	audio = -1;
	string errormsrg = 
	    "I cannot set the block size to " 
	    + ::toString(audio_block_size) 
	    + " bytes. Please configure another fragment size.";
	reportError("Problem with audio device", errormsrg.c_str());
	return;
    }

    // Set sample size
    unsigned long sample_size = sampling_config->samplesize;
    ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &sample_size);
    if (sample_size != sampling_config->samplesize) {
	string errmsg = string(l_MRealOutput_4txa)
	    + ::toString(16L) + l_MRealOutput_4txb;
	reportError(l_MRealOutput_4tt, errmsg.c_str());
	close(audio);
	audio = -1;
	return;
    }
    
    // Set STEREO/MONO
    long stereo = 0; // TODO: stereo detection
    ioctl(audio, SNDCTL_DSP_STEREO, &stereo);
    
    // Set sampling rate
    long sr = sampling_config->samplingrate;
    if (-1 == ioctl(audio, SNDCTL_DSP_SPEED, &sr)) {
	string errormsg = string(l_MRealOutput_5txa) + ::toString(sr) + l_MRealOutput_5txb;
	reportError(l_MRealOutput_5tt, errormsg.c_str());
	close(audio);
	audio = -1;
	return;
    }

    audio_buffer = new char[audio_block_size];
    bytes_in_audio_buffer = 0;
    
    // Now I prepare my sound source.
    
    Parameterset parset;
    parset.setSamplingInterval(1.0 / Number(sampling_config->samplingrate));
    Metainfo mi;
    mi.setCutTo(); // means I wan't to know about this parameter.
    // TODO: auch CutFrom() beruecksichtigen.
    pssig_input = getPreparedSoundSignal(0, sslot_input, &mi, &parset);
    end_time = mi.containsCutTo() ? mi.getCutTo() : MAXLONG; // <values.h>
}


bool MRealOutput::executeBlock(long start_time, long nsamples) 
{
    if (audio == -1) return false;
    
    // Don't play more than until end_time, even if nsamples would be more.
    if (end_time < start_time + nsamples) nsamples = end_time - start_time;
    if (nsamples <= 0) return false;
    
    // Is this the last block?
    bool should_continue = start_time + nsamples < end_time;
    
    // Compute the sound
    SoundPortion sp = pssig_input->getSoundPortion(start_time, nsamples);
    const Number *soundsource = sp.getSamples();
    
    int bytes_per_sample = (sampling_config->samplesize == 8 ? 1 : 2);

    // If nsamples is more than one buffer, player buffer after buffer.
    while (nsamples) {
	long samples_to_copy = 
	    min(nsamples, (audio_block_size - bytes_in_audio_buffer) / bytes_per_sample);
	nsamples -= samples_to_copy;
	if (bytes_per_sample == 2) {
	    short *output = (short *)(&audio_buffer[bytes_in_audio_buffer]);
	    bytes_in_audio_buffer += samples_to_copy * 2;
	    while (samples_to_copy--) *output++ = (short)(*soundsource++ * 32767);
	}
	else {
	    unsigned char *output = (unsigned char *)(&audio_buffer[bytes_in_audio_buffer]);
	    bytes_in_audio_buffer += samples_to_copy;
	    while (samples_to_copy--) *output++ = (unsigned char)(128 + *soundsource++ * 128);
	}
	
	// write out to audio device if the buffer is full, or if it's the last block
	if (bytes_in_audio_buffer == audio_block_size || !should_continue)
	{
	    if (write(audio, audio_buffer, bytes_in_audio_buffer) != bytes_in_audio_buffer)
	    {
	      string errmsg = string(l_MRealOutput_6txa)
		+ ::toString(bytes_in_audio_buffer) + l_MRealOutput_6txb;
	      reportError(l_MRealOutput_6tt, errmsg.c_str());
	      close(audio);
	      audio = -1;
	      return false;
	    }
	    bytes_in_audio_buffer = 0;
	}
    }

    return should_continue;
}


void MRealOutput::finishExecution()
{
    if (pssig_input) {
	delete pssig_input;
	pssig_input = 0;
    }
    if (audio_buffer) {
	delete audio_buffer;
	audio_buffer = 0;
    }
    if (audio != -1) {
	close(audio);
	audio = -1;
    }
}

bool MRealOutput::mustWait()
{
    // Controll write-ahead. 
    audio_buf_info info;
    ioctl(audio, SNDCTL_DSP_GETOSPACE, &info);
    return ((unsigned)(info.fragstotal - info.fragments) > sampling_config->writeahead);
}
