// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/editor/MacroModuleLook.cc,v $
// $Revision: 1.4 $
// $Date: 1999/05/16 19:14:53 $
// $State: Exp $
// **************************************************************

#include <unistd.h> // getlogin(), access()
#include <stdio.h>  // remove()
#include <math.h>   // floor()

#define INCLUDE_MENUITEM_DEF // HACK!!

#include <iostream.h> // debugging!
#include <fstream.h>
#include <qpixmap.h>
#include <qkeycode.h>
#include <kmsgbox.h>
#include <qtimer.h>
#include <qmenudta.h>
#include <qfiledlg.h>

#include "Slot.h"
#include "Signal.h"
#include "DirScan.h"
#include "toString.h"
#include "Dirs.h"
#include "minmaxabs.h"

#include "LookInfoDialog.h"
#include "MacroModuleLook.h"
#include "MacroModuleEditor.h"
#include "MusicdrawTopLevel.h"
#include "ModuleGenerator.h"
#include "ModuleLookGenerator.h"
#include "ModuleErrorReport.h"
#include "InputModuleFile.h"
#include "OutputModuleFile.h"
#include "Wirebox.h"
#include "MLGPlugin.h"
#include "LookInfo.h"
#include "LookInfoTypes.h"
#include "ESNothing.h"
#include "ESCarryingForInsertion.h"
#include "ESWireing.h"
#include "rendering.h"
#include "gridspacing.h"
#include "rectutils.h"
#include "language.h" // Remove this sooner or later
#include "split.h"

#include "copy.xpm"
#include "cut.xpm"
#include "paste.xpm"
#include "undo.xpm"
#include "redo.xpm"
#include "rewire.xpm"
#include "play.xpm"
#include "pause.xpm"
#include "stop.xpm"
    
MacroModuleLook::MacroModuleLook(MusicdrawTopLevel *toplevel, MacroModule *macromodule, bool editor)
    : ModuleLook(macromodule, 0),
      toplevel(toplevel),
      mme(0),
      macromodule(macromodule),
      wire_menu(0),
      toolbar(0),
      statusbar(0),
      macro_module_file_dialog(new QFileDialog("", "*.md", 0,"", TRUE)),
      undo_level(0),
      highest_undo_file(-1), // no undo file existing
      playing(false),
      mouse_pressed_over(0)
{
    if (!editor) initialize(0);
    modified = false; // initialize() sets modified to true, as ModuleLooks are added
}


void MacroModuleLook::initialize(MacroModuleEditor *mme)
{
    this->mme = mme;
    module_looks.setAutoDelete(true);
    module_looks.setAutoDelete(true); // induces memory leakage, avoids bug in multiple inheritance.
    wire_looks.setAutoDelete(true);
    annotations.setAutoDelete(true);
    scanPlugins();
    
    wirebox = new Wirebox(canvasGridSize());
    if (mme) {
	setMouseTracking(TRUE); // want to get mouse events    
	initializeEditMenu();
	initializeMacroModuleMenu();
	initializeModulesMenu();
	initializeExecuteMenu();
    }

    editor_state = new ESNothing(this);
    editor_state->enter();

    readSamplingConfig();

    syncNewModules();
    refreshLayoutImage();
    doGeneralUpdate();
}


MacroModuleLook::~MacroModuleLook()
{
    if (wire_menu) delete wire_menu;
    editor_state->leave();
    delete editor_state;
    cleanupUndoFiles(0);       // remove undo files in /tmp
    delete wirebox;
}

void MacroModuleLook::setMouseTracking(bool enable)
{
    mme->setMouseTracking(enable);
}


void MacroModuleLook::readSamplingConfig()
{
    KConfig *config = kapp->getConfig();
    config->setGroup("Sampling");
    sampling_config.samplingrate 
	= config->readUnsignedLongNumEntry("Samplingrate", sampling_config.samplingrate);

    sampling_config.samplesize 
	= config->readUnsignedNumEntry("Samplesize", sampling_config.samplesize);
    if (sampling_config.samplesize != 8 && sampling_config.samplesize != 16) {
	fprintf(stderr, "Unknown sampling size %u. Using default of 16 bits/sample.\n",
		sampling_config.samplesize);
	sampling_config.samplesize = 16;
    }

    sampling_config.fragmentsize 
	= config->readUnsignedNumEntry("Fragmentsize", sampling_config.fragmentsize);
    
    sampling_config.writeahead
	= config->readUnsignedNumEntry("Writeahead", sampling_config.writeahead);

    sampling_config.show_seconds
	= config->readBoolEntry("ShowSeconds", sampling_config.show_seconds);

    sampling_config.frags_per_display_refresh = config
	->readUnsignedNumEntry("FragsPerDisplayRefresh", sampling_config.frags_per_display_refresh);
}

void MacroModuleLook::writeSamplingConfig()
{
    KConfig *config = kapp->getConfig();
    config->setGroup("Sampling");
    config->writeEntry("Samplingrate", sampling_config.samplingrate);
    config->writeEntry("Samplesize", sampling_config.samplesize);
    config->writeEntry("Fragmentsize", sampling_config.fragmentsize);
    config->writeEntry("Writeahead", sampling_config.writeahead);
    config->writeEntry("ShowSeconds", sampling_config.show_seconds);
    config->writeEntry("FragsPerDisplayRefresh", sampling_config.frags_per_display_refresh);
    config->sync();
}

void MacroModuleLook::addMenus(KMenuBar *menubar)
{
    menubar->insertItem("&Edit",     &edit_menu);
    menubar->insertItem("&Look",     &layout_menu);
    menubar->insertItem("&Modules",  &modules_menu);
    menubar->insertItem("E&xecute",  &execute_menu);
}


void MacroModuleLook::addToolbarButtons(KToolBar *tb)
{
    toolbar = tb;
    #define TOOLBARBUTTON(func,text)								\
    toolbar->insertButton(									\
	QPixmap((const char **)func ## _xpm), toolbarid_ ## func,				\
	SIGNAL(clicked()), mme, SLOT(func()), true, text);					\
    toolbar->addConnection(toolbarid_ ## func, SIGNAL(clicked()), mme, SLOT(doGeneralUpdate()));

    TOOLBARBUTTON(copy,   "Copy selected modules into clipboard");
    TOOLBARBUTTON(cut,    "Cut selected modules into clipboard");
    TOOLBARBUTTON(paste,  "Insert clipboard contents");
    TOOLBARBUTTON(undo,   "Undo last operation");
    TOOLBARBUTTON(redo,   "Redo undone operation");
    TOOLBARBUTTON(rewire, "Nicely relayout all wires");
    TOOLBARBUTTON(play,   "Start playing");
    TOOLBARBUTTON(pause,  "Suspend playing for a while");
    TOOLBARBUTTON(stop,   "Abort playing");
}


void MacroModuleLook::addStatusbarItems(KStatusBar *sb)
{
    int id = 1;
    statusbar = sb;
    statusbar->insertItem("00:00:00,00",      statusbarid_time       = id++);
    statusbar->insertItem(l_PLAYING,          statusbarid_pause      = id++);
    statusbar->insertItem("Anyway last item", statusbarid_modulename = id++);
    statusbar->insertItem("",                 statusbarid_modified   = id++);
    showPlayingTime();
    showPlayingStatus();
    showModuleName("");
    showModified();
}


void MacroModuleLook::removeToolbarButtons()
{
    toolbar->removeItem(toolbarid_copy);
    toolbar->removeItem(toolbarid_cut);
    toolbar->removeItem(toolbarid_paste);
    toolbar->removeItem(toolbarid_undo);
    toolbar->removeItem(toolbarid_redo);
    toolbar->removeItem(toolbarid_rewire);
    toolbar->removeItem(toolbarid_play);
    toolbar->removeItem(toolbarid_pause);
    toolbar->removeItem(toolbarid_stop);
}

// ===========================================================================
//                     creating and maintaining menues
// ===========================================================================

void MacroModuleLook::initializeEditMenu()
{
    menuid_undo  = edit_menu.insertItem
	(QPixmap((const char **)undo_xpm), l_UNDO,  mme, SLOT(undo()), CTRL + Key_Z);
    menuid_redo  = edit_menu.insertItem
	(QPixmap((const char **)redo_xpm), l_REDO,  mme, SLOT(redo()));
    edit_menu.insertSeparator();
    menuid_copy  = edit_menu.insertItem
	(QPixmap((const char **)copy_xpm), l_COPY,  mme, SLOT(copy()), CTRL + Key_C);
    menuid_cut   = edit_menu.insertItem
	(QPixmap((const char **)cut_xpm),  l_CUT,   mme, SLOT(cut()), CTRL + Key_X);
    menuid_paste = edit_menu.insertItem
	(QPixmap((const char **)paste_xpm), l_PASTE, mme, SLOT(paste()), CTRL + Key_V);
    menuid_select_all = edit_menu.insertItem
	(l_SELECT_ALL, mme, SLOT(selectAllModules()), CTRL + Key_A);
    menuid_save_sel = edit_menu.insertItem
	(l_SAVE_SEL_TO_FILE, mme, SLOT(saveSelectionToFile()));

    edit_menu.insertSeparator();
    menuid_flip_horiz = edit_menu.insertItem
	(l_FLIP_HORIZ, mme, SLOT(flipHorizontally()));
    menuid_flip_vert  = edit_menu.insertItem
	(l_FLIP_VERT, mme, SLOT(flipVertically()));

    edit_menu.insertSeparator();
    menuid_rewire = edit_menu.insertItem(QPixmap((const char **)rewire_xpm), 
					 l_REWIRE, mme, SLOT(rewire()), CTRL + Key_R);

    edit_menu.insertSeparator();
    menuid_look_dialog = edit_menu.insertItem("&Module's Properties", mme, SLOT(lookDialog()));
    mme->connect(&edit_menu, SIGNAL(activated(int)), mme, SLOT(doGeneralUpdate(int)));
}


void MacroModuleLook::initializeMacroModuleMenu()
{
    menuid_show_rect  = layout_menu.insertItem
	("&Hilite Visible Modules", mme, SLOT(editLook()));
    menuid_add_to_layout = layout_menu.insertItem
	("Make &Visible", mme, SLOT(addToLayout()));
    menuid_remove_from_layout = layout_menu.insertItem
	("Make &Invisible", mme, SLOT(removeFromLayout()));

    mme->connect(&layout_menu, SIGNAL(activated(int)), mme, SLOT(doGeneralUpdate(int)));
}


void MacroModuleLook::initializeModulesMenu()
{
    int submenu_id = 10000;

    const DLList *module_list = ModuleLookGenerator::generatorList();
    if (module_list) 
    {
	DLListNode *node = module_list->getFirst();
	while (!module_list->isEndOfList(node)) {
	    ModuleLookGenerator *lmg = (ModuleLookGenerator *)node;
	    QPopupMenu *menu = &modules_menu;
	    const char *menupath = lmg->menuPath();
	    if (menupath) {
		string submenu_names[16]; // Up to 16 submenu levels
		int levels = split(string(menupath), submenu_names, 16, string("/"));
		for (int level=0; level < levels; level++) 
		{
		    // Try to find, if submenu already exists.
		    bool found=false;
		    for (unsigned int index=0; index < menu->count(); index++)
		    {
			int id = menu->idAt(index);
			if (submenu_names[level] == menu->text(id)) { // found!
			    menu = menu->findItem(id)->popup();
			    found = true;
			break;
			}
		    }
		    if (!found) { // Must create is
			QPopupMenu *new_menu = new QPopupMenu();
			mme->connect(new_menu, SIGNAL(activated(int)), mme, SLOT(modulesMenuEvent(int)));
			mme->connect(new_menu, SIGNAL(activated(int)), mme, SLOT(doGeneralUpdate(int)));
			menu->insertItem(submenu_names[level].c_str(), new_menu, submenu_id++, 0);
			menu = new_menu;
		    }
		} // Build or scan menu path
		menu->insertItem(lmg->getPixmap(), lmg->getName().c_str(), lmg->uniqueId());
	    }
	    node = node->getNext();
	}
	
	modules_menu.insertSeparator();
    }
    modules_menu.insertItem("&From File...", mme, SLOT(insertMacroModule()));
    
    mme->connect(&modules_menu, SIGNAL(activated(int)), mme, SLOT(modulesMenuEvent(int)));
    mme->connect(&modules_menu, SIGNAL(activated(int)), mme, SLOT(doGeneralUpdate(int)));
}


void MacroModuleLook::initializeExecuteMenu()
{
    menuid_play = execute_menu.insertItem
	(QPixmap((const char **)play_xpm), l_PLAY, mme, SLOT(play()));
    menuid_pause = execute_menu.insertItem
	(QPixmap((const char **)pause_xpm), l_PAUSE, mme, SLOT(pause()));
    menuid_stop = execute_menu.insertItem
	(QPixmap((const char **)stop_xpm), l_STOP, mme, SLOT(stop()));
    menuid_samplingconfig = execute_menu.insertItem
	("Configure..", mme, SLOT(samplingConfiguration()));

    mme->connect(&execute_menu, SIGNAL(activated(int)), mme, SLOT(doGeneralUpdate(int)));
}


// ===========================================================================
//                          Loading the whole stuff
// ===========================================================================

bool MacroModuleLook::insertFile(InputModuleFile& modulefile, const QPoint& /* offset */)
{
    string parameterlist = "";
    if (macromodule->loadFromFile(modulefile, parameterlist, true)) // true means "insert"
    {
	syncNewModules();
	return true;
    }
    else return false;
}


bool MacroModuleLook::reloadFromFile(InputModuleFile& modulefile)
{
    cancelCurrentUserAction();
    deleteModuleLooks();
    string parameterlist = "";
//    return loadFromFile(modulefile, false, parameterlist); // false: not only insert, but load also parameters
    return true;
}


void MacroModuleLook::syncNewModules()
{
    Module *m;
    Wire *w;
    while (0 != (m = macromodule->nextNewModule())) addModuleLook(m);
    while (0 != (w = macromodule->nextNewWire()))   addWireLook(w);
}


// ===========================================================================
//
//                                Saving
//
// ===========================================================================
bool MacroModuleLook::saveToFile(OutputModuleFile& modulefile, 
				 bool now_not_modified, bool selection_only)
{
    preparePreview();
    
    macromodule->saveToFile(modulefile, selection_only);

    if (modulefile.isGood()) {
	if (now_not_modified) {
	    clearModified();
	    showModified();
	}
	return true;
    }
    else return false; // TODO: explanation for Error.
}


void MacroModuleLook::preparePreview()
{
    QPixmap pixmap = getPixmap(); // from LayoutElement
    
    // Now I have to save the pixmap into a memory area.
    // The only problem: Qt can't do this :-(. So I have go
    // via tempfile.

    char tempfilename[128];
    sprintf(tempfilename, "/tmp/%d.preview.md", getpid());
    pixmap.save(tempfilename, "BMP");
    FILE *file = fopen(tempfilename, "r");
    if (file) {
	fseek(file, 0, SEEK_END);
	long size = ftell(file);
	fseek(file, 0, SEEK_SET);
	unsigned char *buffer = new unsigned char[size];
	long read_blocks = fread(buffer, size, 1, file);
        fclose(file);
	if (read_blocks == 1) macromodule->lookInfo().preview.setImage(buffer, size);
	else delete buffer;
    }
    remove(tempfilename);
}

// ===========================================================================
//
//                                Painting
//
// ===========================================================================

void MacroModuleLook::paintEvent(QPaintEvent *event)
{
    QRect update_rect = event->rect();
    QPixmap pixmap(update_rect.size());
    pixmap.fill(mme->backgroundColor());
    QPainter p;
    p.begin(&pixmap);
    p.translate(- update_rect.left(), - update_rect.top()); // Allows normal painting
    
    paintCanvas(p, update_rect);
    paintLayoutRect(p, update_rect);
    paintAuras(p, update_rect);
    paintGrid(p, update_rect);
    wirebox->paint(p, update_rect);
    paintModuleLooks(p, update_rect);
    paintAnnotations(p, update_rect);
    editor_state->paintEvent(p, update_rect);

    p.end(); // close painter
    bitBlt(mme, update_rect.topLeft(), &pixmap);
}


void MacroModuleLook::paintCanvas(QPainter& p, const QRect& update_rect) const
{
    ::paintCanvas(p, canvasRect().intersect(update_rect));
}


// This function makes no use of MacroModuleLook members!
void MacroModuleLook::paintGrid(QPainter& p, const QRect& update_rect) const
{
    short x      = roundUpToGrid(update_rect.left());
    short y      = roundUpToGrid(update_rect.top());
    short width  = min(update_rect.width(),  canvasSize().width()  - x);
    short height = min(update_rect.height(), canvasSize().height() - y);
    ::paintGrid(p, QRect(x, y, width, height));
}

short MacroModuleLook::roundUpToGrid(short v) const
{
    return ((v + GRID_SPACING - 1) / GRID_SPACING) * GRID_SPACING;
}

void MacroModuleLook::paintAuras(QPainter& p, const QRect& update_rect)
{
    if (doesEditLook()) {
	for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
	{
	    if (ml->getModule()->isVisible() && ml->rectOfAura().intersects(update_rect))
	    {
		ml->paintAura(p);
	    }
	}
    }
}


void MacroModuleLook::paintLayoutRect(QPainter& p, const QRect& update_rect) const
{
    QRect rect = layoutRect() * 2 * GRID_SPACING;
    if (enlargeRect(rect, LAYOUT_RECT_THICKNESS).intersects(update_rect) &&
	doesEditLook())
    {
	p.save();

	short x1i = rect.x() - 1,
	    x1o = x1i - LAYOUT_RECT_THICKNESS,
	    y1i = rect.y() - 1,
	    y1o = y1i - LAYOUT_RECT_THICKNESS,
	    x2i = rect.right() + 2,
	    x2o = x2i + LAYOUT_RECT_THICKNESS,
	    y2i = rect.bottom() + 2,
	    y2o = y2i + LAYOUT_RECT_THICKNESS;
	
	p.setBrush(QColor(158,170,175));
	p.drawRect(QRect(QPoint(x1i,y1i),QPoint(x2i,y2i)));

	if (macromodule->lookInfo().image_layout) {
	    p.save();
	    p.scale(2.0, 2.0);
	    p.drawPixmap(rect.x()/2, rect.y()/2, layout_image, 0, 0, 
			 rect.width()/2, rect.height()/2);
	    p.restore();
	}

	p.setPen(QColor(190,190,190));
	p.setBrush(QBrush(QColor(190, 190, 190), SolidPattern));
	p.drawRect(QRect(QPoint(x1o,y1o), QPoint(x1i,y2o))); // left
	p.drawRect(QRect(QPoint(x2i,y1o), QPoint(x2o,y2o))); // right
	p.drawRect(QRect(QPoint(x1o,y1o), QPoint(x2o,y1i))); // top   
	p.drawRect(QRect(QPoint(x1o,y2i), QPoint(x2o,y2o))); // bottom
	
	p.setPen(QColor(230,230,230)); // light shadows
	p.drawLine(x1o,y1o,x1o,y2o); // left
	p.drawLine(x2i,y1i,x2i,y2i); // right
	p.drawLine(x1o,y1o,x2o,y1o); // top
	p.drawLine(x1i,y2i,x2i,y2i); // bottom
	
	p.setPen(QColor( 30, 30, 30)); // dark shadows
	p.drawLine(x1i,y1i,x1i,y2i); // left
	p.drawLine(x2o,y1o,x2o,y2o); // right
	p.drawLine(x1i,y1i,x2i,y1i); // top
	p.drawLine(x1o,y2o,x2o,y2o); // bottom
	
	p.restore();
    }
}


void MacroModuleLook::paintModuleLooks(QPainter& p, const QRect& update_rect)
{
    for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
    {
	if (ml->intersects(update_rect)) {
	    p.save();
	    p.translate( ml->left(), ml->top());
	    ml->paintBorder(p);
	    ml->paint(p);
	    if (ml->getModule()->isSelected()) ml->paintSelection(p);
	    p.restore();
	}    
    }
}

void MacroModuleLook::paintAnnotations(QPainter& p, const QRect& update_rect)
{
    for (Annotation *anno = annotations.first(); anno ; anno = annotations.next())
	if (anno->intersects(update_rect)) anno->paint(p);
}


void MacroModuleLook::needsUpdate(const QRect& rect)
{
    if (rectangle_to_update.isEmpty()) rectangle_to_update = rect.normalize();
    else rectangle_to_update = rectangle_to_update.unite(rect.normalize());
}


void MacroModuleLook::needsUpdate(const ModuleLook *ml)
{
    if (ml->getModule()->isVisible()) needsUpdate(ml->rectOfAura());
    needsUpdate(ml->rectWithBorder()); // This in any case, even with Aura.
}


void MacroModuleLook::needsUpdate(const Annotation *anno)
{
    needsUpdate(anno->rect());
}


void MacroModuleLook::needsUpdateLayoutRect()
{
    needsUpdate(enlargeRect(layoutRect() * (2 * GRID_SPACING), LAYOUT_RECT_THICKNESS + 2));
}


void MacroModuleLook::doUpdate()
{
    if (!rectangle_to_update.isEmpty())
    {
	mme->repaint(rectangle_to_update, FALSE);
	rectangle_to_update = QRect();
    }
} 


// ===========================================================================
//
//                              Receiving events
//
// ===========================================================================

void MacroModuleLook::handleEvent(QEvent *e)
{
    goToEditorState(editor_state->handleEvent(e));
    doGeneralUpdate();
}

void MacroModuleLook::keyPressed(QKeyEvent *e)
{
    if (e->key() == Key_Escape) {
	cancelCurrentUserAction();
	doGeneralUpdate();
    }

    else handleEvent(e);
}


// ===========================================================================
//
//                        Carry out user commands
//
// ===========================================================================

void MacroModuleLook::goToEditorState(EditorState *new_state)
{
    if (new_state != editor_state)
    {
	editor_state->leave();
	delete editor_state;
	editor_state = new_state;
	editor_state->enter();
    }
}

void MacroModuleLook::popupWireMenu(ModuleLook *ml, Connector *connector)
{
    if (wire_menu) delete wire_menu;
    wire_menu = ml->createWireMenu(connector);
    if (!connector) mme->connect(wire_menu, SIGNAL(activated(int)), mme, SLOT(startWireingEvent(int)));
    else mme->connect(wire_menu, SIGNAL(activated(int)), mme, SLOT(finishWireingEvent(int)));
    wire_menu->popup(QCursor::pos());
    produced_wire_menu = ml;
}


void MacroModuleLook::startWireingEvent(int connector_id)
{
    Module *module = produced_wire_menu->getModule();
    Connector *connector = module->connectorNumber(connector_id);

    // If this is a slot that has been wired already, then
    // we first rip out the old wire.
    
    if (connector->isSlot()) deleteAllWiresConnectedTo(0, connector);

    // If this is a slot that has been declared as parameter, then
    // undo this, too.
    
    if (connector->isSlot() && ((Slot *)connector)->isParameter()) ((Slot *)connector)->disconnect();
    
    goToEditorState(new ESWireing(this, produced_wire_menu, connector));
    doGeneralUpdate();
}


void MacroModuleLook::finishWireingEvent(int connector_id)
{
    ModuleLook *ml_1 = ((ESWireing *)editor_state)->getModuleLook();
    Connector  *c_1  = ((ESWireing *)editor_state)->getConnector();
    ModuleLook *ml_2 = produced_wire_menu;
    Connector  *c_2  = ml_2->getModule()->connectorNumber(connector_id);
    
    if (macromodule->checkWire(c_1, c_2)) {                   // checkWire comes from MacroModule
	saveForUndo();                           // now is the very last moment for this :-)
	Wire *wire = macromodule->addWire(c_1, c_2, ml_1->getModule(), ml_2->getModule()); // addWire comes from MacroModule
	if (wire) addWireLook(ml_1, ml_2, wire); // else: slot was defined as parameter
    }
    else { // TODO: genaue Fehlermeldung
	KMsgBox::message(mme, l_WIRE_ERROR_CAPTION, l_WIRE_ERROR_TEXT, KMsgBox::EXCLAMATION);
    }

    goToEditorState(new ESNothing(this));
    doGeneralUpdate();
}


void MacroModuleLook::cancelCurrentUserAction()
{
    goToEditorState(new ESNothing(this));
}


void MacroModuleLook::selectModule(ModuleLook *ml)
{
    if (!ml->getModule()->isSelected()) {
	ml->getModule()->select(true);
	needsUpdate(ml);
    }
}

void MacroModuleLook::deselectModule(ModuleLook *ml)
{
    if (ml->getModule()->isSelected()) {
	ml->getModule()->select(false);
	needsUpdate(ml);
    }
}


void MacroModuleLook::toggleSelection(ModuleLook *ml)
{
    ml->getModule()->select(!ml->getModule()->isSelected());
    needsUpdate(ml);
}


/**
  * Deselects all layout module. This happens, when the user clicks anywhere
  * into the background. All red borders will disappear. No repaint() is done here.
  */
void MacroModuleLook::deselectAllModules()
{
    ModuleLook *ml;
    for (ml = module_looks.first(); ml != 0; ml = module_looks.next())
	deselectModule(ml);
}

void MacroModuleLook::rectangularSelection(const QRect& old_rect, const QRect& new_rect)
{
    for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
    {
	bool intersects_old = ml->intersects(old_rect.normalize());
	bool intersects_new = ml->intersects(new_rect.normalize());
	if (intersects_old && !intersects_new) deselectModule(ml);
	else if (intersects_new && !intersects_old) selectModule(ml);
    }
}

void MacroModuleLook::deleteModuleLooks(bool selection_only)
{
    ModuleLook *ml;
    for (ml = module_looks.first(); ml != 0;)
    {
	if (!selection_only || ml->getModule()->isSelected())
	{
	    wirebox->releaseRegion(ml->gridRect());
	    needsUpdate(ml);
	    deleteAllWiresConnectedTo(ml); // wires must be removed first!
	    macromodule->deleteModule(ml->getModule());
	    module_looks.remove(); // Remove current item from list. ml is deleted by Qt!
	    ml = module_looks.current();
	}
	else ml = module_looks.next(); // only to next if nothing is removed! See QList-Docu!
    }
}


// ===========================================================================
//
//                        Carry out user actions
//
// User actions are differ from user commands in that they represent a state.
// For example when the user insert a new module, a white frame will appear
// to indicate the position, where the module is going to be inserted. Mouse
// movement will move the white frame, but no further command is executed.
//
// ===========================================================================


bool MacroModuleLook::dragSelectedModules(const QPoint& grid_drag_vector) 
{
    ModuleLook *ml;

    // I will only move, if no module will be move outside
    bool inside = true;
    for (ml = module_looks.first(); inside && ml != 0; ml = module_looks.next())
	if (ml->getModule()->isSelected()) 
	    inside &= canvasGridRect().contains(QRect(ml->gridPos() + grid_drag_vector, ml->gridSize()));
    if (inside) {
	for (ml = module_looks.first(); ml != 0; ml = module_looks.next())
	    if (ml->getModule()->isSelected()) moveModuleLook(ml, ml->gridPos() + grid_drag_vector);
    } 
    return inside;
}




void MacroModuleLook::flipSelection(bool horizontally)
{
    // find the minimal enclosing rectangle.
    QRect mer;
    ModuleLook *ml;
    for (ml = module_looks.first(); ml != 0; ml = module_looks.next())
	if (ml->getModule()->isSelected()) mer = ml->gridRect().unite(mer);
    
    // flip within this rectangle
    if (!mer.isEmpty())
    {
	for (ml = module_looks.first(); ml != 0; ml = module_looks.next())
	{
	    if (ml->getModule()->isSelected()) {
		if (horizontally) 
		    moveModuleLook(ml, QPoint(mer.right() - (ml->gridLeft() - mer.left()) 
						 - ml->gridWidth() + 1, ml->gridTop()));
		else
		    moveModuleLook(ml, QPoint(ml->gridLeft(), mer.bottom()-(ml->gridTop()-mer.top())
						 - ml->gridHeight() + 1));
	    }
	}
    }
}


void MacroModuleLook::moveModuleLook(ModuleLook *ml, const QPoint& gridpos)
{
    if (ml->gridPos() != gridpos) {
	needsUpdate(ml);
	wirebox->releaseRegion(ml->gridRect());
	ml->gridMoveTo(gridpos);
	needsUpdate(ml);
	wirebox->lockRegion(ml->gridRect()); // Don't wire through the Modules.
	relayoutWiresConnecting(ml);
	setModified();
    }
}


// ===========================================================================
//                              Annotations
// ===========================================================================
void MacroModuleLook::addAnnotation(Annotation *anno)
{
    annotations.append(anno); // add as first: newer errors should be on top.
    needsUpdate(anno);
}


void MacroModuleLook::annotationBack(Annotation *anno)
{
    annotations.setAutoDelete(false);
    annotations.remove(anno);
    annotations.setAutoDelete(true);
    annotations.insert(0, anno);
    needsUpdate(anno);
}


void MacroModuleLook::removeAnnotation(Annotation *anno)
{
    needsUpdate(anno);
    annotations.remove(anno);  // auto delete is enabled
}


void MacroModuleLook::removeAllAnnotations()
{
    while (!annotations.isEmpty()) {
	Annotation *anno = annotations.first();
	needsUpdate(anno);
	annotations.remove(); // auto delete is enabled, annotation will be deleted by Qt!
    }
}


void MacroModuleLook::showErrorAnnotation(ModuleLook *le, ModuleErrorReport *error_report)
{
    // TODO: smart placement of annotations.
    Annotation *anno =
	new Annotation(error_report->getTitle(), error_report->getText(), Annotation::ERROR,
		       le->center() + QPoint(50,30), le->center());
    delete error_report;
    addAnnotation(anno);
}


Annotation *MacroModuleLook::findAnnotationAt(const QPoint& pos)
{
    // I must search backwards, because last drawn items should be found first.
    Annotation *anno;
    for (anno = annotations.last(); anno; anno = annotations.prev())
	if (anno->mouseClickHits(pos)) return anno;
    return 0;
}


// ===========================================================================
//
//                              statusbar
//
// ===========================================================================

void MacroModuleLook::showPlayingTime()
{
    if (statusbar) {
	if (playing) {
	    double playing_time = (double)sampling_time / (double)sampling_config.samplingrate;
	    long hours   = (long)floor(playing_time / 3600);
	    long minutes = (long)floor((playing_time - (double)hours*3600) / 60);
	    long seconds = (long)floor(((playing_time - (double)hours*3600 - (double)minutes*60)));
	    long cents   = (long)floor(((playing_time - floor(playing_time)) * 100));
	    
	    char string[20];
	    sprintf(string, "%02ld:%02ld:%02ld,%02ld", hours, minutes, seconds, cents);
	    statusbar->changeItem(string, statusbarid_time);
	}
	else statusbar->changeItem("", statusbarid_time);
    }
}


void MacroModuleLook::showPlayingStatus()
{
    if (statusbar) statusbar->changeItem(playing ? (paused ? l_PAUSED : l_PLAYING) : "", statusbarid_pause);
}


void MacroModuleLook::showModuleName(const char *modulename)
{
    if (statusbar) statusbar->changeItem(modulename, statusbarid_modulename);
}


void MacroModuleLook::showModified()
{
    if (statusbar) statusbar->changeItem(modified ? "modified" : "", statusbarid_modified);
}


// ===========================================================================
//
//                            helper functions
//
// ===========================================================================

string MacroModuleLook::undoFilename(int level) const
{
    char str[128];
    sprintf(str, "/tmp/%d.undolevel-%d.md", getpid(), level);
    return string(str);
}


void MacroModuleLook::saveForUndo()
{
/*    cleanupUndoFiles(undo_level); // Redo would make no sense anymore
    OutputModuleFile *modulefile = toplevel->getOutputFile(undoFilename(undo_level));
    if (modulefile) {
	saveToFile(*modulefile);
	delete modulefile;
	highest_undo_file = undo_level;
	undo_level ++;
    }        
*/}


void MacroModuleLook::cleanupUndoFiles(int from_level)
{
    for (int i = from_level; i <= highest_undo_file; i++)
	::remove(undoFilename(i).c_str()); // see man 3 remove. The "::" to avoids DLListNode::remove()!
}


string MacroModuleLook::clipboardFilename() const
{
    char str[128];
    sprintf(str, "/tmp/%d.clipboard.md", getuid());
    return string(str);
};


bool MacroModuleLook::clipboardFileExists() const
{
    return 0 == access(clipboardFilename().c_str(), R_OK); // <unistd.h>
}


void MacroModuleLook::createAndInsertModule(ModuleLookGenerator *mlg, const QPoint& pos)
{
    // Find Generator for creating the underlying Module.
    ModuleGenerator *mg = (ModuleGenerator *)ModuleGenerator::findGenerator(mlg->getName().c_str());
    if (mg) 
    {
	Module *module = mg->create();
	module->moveTo(pos.x(), pos.y());
	macromodule->addModule(module);
	ModuleLook *ml = mlg->create(module);
	addModuleLook(ml);
    }
    else fprintf(stderr, "Can't find Module of type %s in libm3modules.\n", mlg->getName().c_str());
}

/**
  * Looks for a ModuleLook, whose area contains a given Pixel.
  */
ModuleLook *MacroModuleLook::findModuleLookAt(const QPoint& pos)
{
    // I must search backwards, because last drawn items should be found first.
    ModuleLook *le;
    for (le = module_looks.last(); le != 0; le = module_looks.prev())
	if (le->rect().contains(pos)) return le;
    return 0;
}


QPoint MacroModuleLook::nearestGridPosition(const QPoint& pos, int align) const
{
    QPoint result = pos + QPoint((GRID_SPACING * align)/2, (GRID_SPACING * align)/2);

    if (result.x() < 0) result.rx() -= GRID_SPACING * align;
    if (result.y() < 0) result.ry() -= GRID_SPACING * align;
    return  (result / (GRID_SPACING * align));
}

QSize MacroModuleLook::canvasGridSize() const
{ 
    return QSize(macromodule->editorInfo().canvas_width, 
	       macromodule->editorInfo().canvas_height); 
}

QRect MacroModuleLook::canvasGridRect() const
{
    return QRect(QPoint(0,0), canvasGridSize());
}


QSize MacroModuleLook::canvasSize() const
{
    return canvasGridSize() * GRID_SPACING;
}

QRect MacroModuleLook::canvasRect() const
{
    return QRect(QPoint(0,0), canvasSize());
}


QRect MacroModuleLook::layoutRect() const
{
    return toQ(macromodule->lookInfo().layout_rect);
}


void MacroModuleLook::setLayoutRect(const QRect& rect)
{
    if (rect != layoutRect()) {
	needsUpdateLayoutRect();
	macromodule->lookInfo().layout_rect = fromQ(rect);
	needsUpdateLayoutRect();
    }
}

bool MacroModuleLook::doesEditLook() const
{
    return macromodule->editorInfo().edit_look;
}

void MacroModuleLook::addModuleLook(Module *module)
{
    ModuleLookGenerator *mlg = 
	(ModuleLookGenerator *)ModuleLookGenerator::findGenerator(module->getName().c_str());
    if (mlg) {
	ModuleLook *ml = mlg->create(module);
	addModuleLook(ml);
    }
    else fprintf(stderr, "No look available for module type %s.\n", module->getName().c_str());
}


void MacroModuleLook::addModuleLook(ModuleLook *ml)
{
    module_looks.append(ml);
    wirebox->lockRegion(ml->gridRect());
    needsUpdate(ml);
    setModified();
}


void MacroModuleLook::addWireLook(Wire *wire)
{
    // Find the two ModuleLooks this wire belongs to. 
    const Module *m1 = wire->getSignalModule();
    const Module *m2 = wire->getSlotModule();

    ModuleLook *ml1 = findModuleLook(wire->getSignalModule());
    if (!ml1) {
 	fprintf(stderr, "Internal bug: ModuleLook for Module %s wire signal %s not found.\n",
		m1->getName().c_str(), wire->getSignal()->getName().c_str());
    }
    ModuleLook *ml2 = findModuleLook(wire->getSlotModule());
    if (!ml2) {
 	fprintf(stderr, "Internal bug: ModuleLook for Module %s wire slot %s not found.\n",
		m2->getName().c_str(), wire->getSlot()->getName().c_str());
    }
    if (ml1 && ml2) addWireLook(ml1, ml2, wire);
}


ModuleLook *MacroModuleLook::findModuleLook(const Module *module)
{
    // I scan the list *backwards*, because the Wire surely connects
    // to relatively *new* Modules, and those are found at the end of
    // the list.

    ModuleLook *ml;
    for (ml = module_looks.last(); ml != 0; ml = module_looks.prev())
	if (ml->getModule() == module) return ml;
    return 0;
}


void MacroModuleLook::addWireLook(ModuleLook *wl_from, ModuleLook *wl_to, Wire *wire)
{
    WireLook *wl = new WireLook(wl_from, wl_to, wire);
    wire_looks.append(wl);
    needsUpdate(wl->nicelyWireIntoWirebox(wirebox));
    setModified();
}


void MacroModuleLook::relayoutWiresConnecting(ModuleLook *ml)
{
    WireLook *wl;
    for (wl = wire_looks.first(); wl != 0; wl = wire_looks.next())
    {
	if (!ml || wl->isConnectedTo(ml)) {
	    needsUpdate(wl->ripOutOfWirebox(wirebox));
	    needsUpdate(wl->nicelyWireIntoWirebox(wirebox));
	} 
    } 
}


void MacroModuleLook::deleteAllWiresConnectedTo(const ModuleLook *ml, const Connector *c)
{
    WireLook *wl;
    for (wl = wire_looks.first(); wl != 0;)
    {
	if ((ml && wl->isConnectedTo(ml))
	    || (c && wl->isConnectedTo(c)))
 	{
	    needsUpdate(wl->ripOutOfWirebox(wirebox));
	    macromodule->deleteWire(wl->getWire());
	    wire_looks.remove(); // remove current item. It's deleted by Qt. See Qt docu for QList.
	    wl = wire_looks.current();
	    setModified();
	}
	else wl = wire_looks.next();
   }
}

bool MacroModuleLook::somethingSelected()
{
    ModuleLook *ml;
    for (ml = module_looks.first(); ml != 0; ml = module_looks.next())
	if (ml->getModule()->isSelected()) return true;
    return false;
}

bool MacroModuleLook::preparedToDie()
{
    if (!modified) return true;
    else {
	int answer = KMsgBox::yesNo(mme, "Warning!", "You didn't save your darn changes yet.\n"
				    "Anyway they aren't worth a penny.\n"
				    "So are we going to expunge them?",
				    KMsgBox::EXCLAMATION, "Sure. Cool.", "Nooooarghl!");
	return (answer == 1);
    }
}

void MacroModuleLook::changeVisibility(bool visible)
{
    for (ModuleLook *ml = module_looks.first(); ml ; ml = module_looks.next())
    {
	if (ml->getModule()->isSelected() && ml->getModule()->isVisible() != visible)
	{
	    needsUpdate(ml);
	    ml->getModule()->setVisibility(visible);
	    needsUpdate(ml);
	}
    }
}

void MacroModuleLook::refreshLayoutImage()
{
    if (macromodule->lookInfo().image_layout && macromodule->lookInfo().image.imagesize)
	layout_image = toQ(macromodule->lookInfo().image);
    else layout_image = QPixmap();
}


// ===========================================================================
//
//                           public slots
//
// ===========================================================================

/**
  * An item from the modules menu has been selected. The menu-id corresponds to
  * the unique id of the ModuleLookGenerator.
  */
void MacroModuleLook::modulesMenuEvent(int module_id)
{
    ModuleLookGenerator *generator
	= (ModuleLookGenerator *)ModuleLookGenerator::findGenerator(module_id);
    if (generator) goToEditorState(new ESCarryingForInsertion(this, generator));
    // else: any other menu item in the Modules Menu.
}


void MacroModuleLook::insertMacroModule()
{
    macro_module_file_dialog->setCaption("Select Module to Insert...");
    macro_module_file_dialog->rereadDir();
    macro_module_file_dialog->show();
    if (macro_module_file_dialog->result())
    {
	QString filename = macro_module_file_dialog->selectedFile();
	if (filename.length() > 0) {
	    InputModuleFile modulefile(filename);
	    if (modulefile.isGood())
	    {
		MacroModule *new_module = new MacroModule(MacroModule::ET_FILEREF, "", &modulefile);
		if (modulefile.isGood()) {
		    MacroModuleLook *to_insert = new MacroModuleLook(0, new_module, false);
		    goToEditorState(new ESCarryingForInsertion(this, to_insert));
		}
		else {
		    KMsgBox::message(mme, "Can't insert Module:", modulefile.errorText().c_str(), 
				     KMsgBox::EXCLAMATION);
		    delete new_module;
		}
	    }
	    else KMsgBox::message(mme, "Can't insert Module:", modulefile.errorText().c_str(), 
				  KMsgBox::EXCLAMATION);
	} 
    } 
}

void MacroModuleLook::undo()
{
/*    // This is a little tricky: The current state may not yet be saved for undo. Therefore
    // I do this now, if neccessary.

    if (highest_undo_file < undo_level) { // no undo file exists for this level
	saveForUndo();
	undo_level --; // not interested in just saved undo file, of course!
    }

    if (undo_level > 0) { // For safety. Otherwise the menu item should be disabled anyway.
	undo_level --;
	InputModuleFile *modulefile = toplevel->getInputFile(undoFilename(undo_level));
	if (!modulefile) {
	    cerr << "Can't open undofile " << undoFilename(undo_level) << "!" << endl; // TODO: MBox
	}
	else {
	    reloadFromFile(*modulefile);
	    delete modulefile;
	}
    }
*/}


void MacroModuleLook::redo()
{
/*    if (undo_level < highest_undo_file) {
	undo_level ++;
	InputModuleFile *modulefile = toplevel->getInputFile(undoFilename(undo_level));
	if (!modulefile) {
	    cerr << "Can't open undofile " << undoFilename(undo_level) << "!" << endl; // TODO: MBox
	}
	else {
	    reloadFromFile(*modulefile);
	    delete modulefile;
	}
    }
*/}


void MacroModuleLook::copy()
{
    OutputModuleFile *modulefile = toplevel->getOutputFile(clipboardFilename().c_str());
    if (modulefile) {
	saveToFile(*modulefile, false, true); // stays modified, only selected modules
	delete modulefile;
    } 
}

void MacroModuleLook::cut()
{
    saveForUndo();
    copy();
    deleteModuleLooks(true); // true = only selected ModuleLooks
}


void MacroModuleLook::paste()
{
    saveForUndo();
    InputModuleFile *modulefile = toplevel->getInputFile(clipboardFilename().c_str());
    if (modulefile) {
	insertFile(*modulefile);
	delete modulefile;
    }
}

void MacroModuleLook::selectAllModules()
{
    ModuleLook *ml;
    for (ml = module_looks.first(); ml != 0; ml = module_looks.next()) selectModule(ml);
}


/**
  * The user has pressed the PLAY-Button or selected Play from a menu.
  */
void MacroModuleLook::play()
{
    if (playing && paused) {
	paused = false;
	playNextBlock();
    }
    else if (!playing) {
	removeAllAnnotations();
	playing = true;
	paused = false;
	sampling_time = 0;
	frags_to_next_display_refresh = 0;
	modules_to_play.clear();
	ModuleLook *ml;
	for (ml = module_looks.first(); ml != 0; ml = module_looks.next())
	{
	    Module *module = ml->getModule();
	    if (module->isExecutable()) {
		modules_to_play.append(module);
		module->prepareForExecution(&sampling_config);
	    }
	}
	ftime(&time_started_playing);
	if (!checkForModuleErrors()) 
	{
	    // I could start playing right now. But it's better to
	    // start playing after the event loop has been cleared.
	    // This avoids "flickering" of the sound on slow machines
	    // at the beginning.
	    QTimer::singleShot(0, mme, SLOT(playNextBlock()));
	}
	else stop();
    }
    showPlayingStatus();
    showPlayingTime();
}


void MacroModuleLook::playNextBlock()
{
    if (playing && !paused)
    {
	// I can only play, if no module has to wait (to write out its
	// buffer). I don't want to play ahead too much.

	for (Module *module = modules_to_play.first(); module != 0;)
	{
	    if  (module->mustWait()) {
		// I could give here a time to wait in ms. But this would
		// only work, if the resolution of the timer is really at
		// one ms. If it is one tenth of a second, I could loose
		// samples against realtime.
		QTimer::singleShot(1 /* ms */, mme, SLOT(playNextBlock()));
		return;
	    }
	    else module = modules_to_play.next();
	}

	int bytes_per_sample = (sampling_config.samplesize == 8 ? 1 : 2);
	long nsamples = (1<<sampling_config.fragmentsize) / bytes_per_sample;
	for (Module *module = modules_to_play.first(); module != 0;)
	{
	    if  (!module->executeBlock(sampling_time, nsamples)) {
		module->finishExecution();
		modules_to_play.remove();
		module = modules_to_play.current();
	    }
	    else module = modules_to_play.next();
	}
	sampling_time += nsamples;

	if (checkForModuleErrors()) {
	    stop(); // errors occured during playing
	    doGeneralUpdate(); // will otherwise not be done, because this is timer event.
	}
	    
	else if (modules_to_play.isEmpty()) {
	    playing = false;
	    showPlayingStatus();
	}

	else QTimer::singleShot(0, mme, SLOT(playNextBlock()));
    }
    if (sampling_config.show_seconds) showPlayingTime();

    // repaint display modules - not every time, but every once and then.
    if (frags_to_next_display_refresh == 0)
    {
	frags_to_next_display_refresh = 1 << sampling_config.frags_per_display_refresh;
	for (ModuleLook *ml = module_looks.first(); ml ; ml = module_looks.next())
	{
	    if (ml->needsRepaint()) 
	    {
		QPainter p;
		p.begin(widget());
		p.translate(ml->left(), ml->top());
		ml->paint(p);
		if (ml->getModule()->isSelected()) ml->paintSelection(p);
		// No need to paint border :-)
		p.end();
	    }
	}
    }
    frags_to_next_display_refresh --;
}


bool MacroModuleLook::checkForModuleErrors()
{
    bool error_found = false;
    ModuleLook *ml;
    for (ml = module_looks.first(); ml ; ml = module_looks.next())
    {
	ModuleErrorReport *error_report = ml->getModule()->getErrorReport();
	if (error_report) {
	    showErrorAnnotation(ml, error_report);
	    error_found = true;
	}
    }
    return error_found;
}


void MacroModuleLook::pause()
{
    paused = true; // TODO: Toggle paused and button.
    showPlayingStatus();
    showPlayingTime();
}


void MacroModuleLook::stop()
{
    if (playing) {
	for (Module *module = modules_to_play.first(); module != 0; module = modules_to_play.next())
	    module->finishExecution();
	playing = false;
	modules_to_play.clear(); // Should not be neccessary, but is cleaner style.
    }
    showPlayingStatus();
    showPlayingTime();
}


void MacroModuleLook::samplingConfiguration()
{
    if (!playing) {
	SamplingConfigDialog scd(&sampling_config);
	if (scd.exec() == QDialog::Accepted) writeSamplingConfig();
    }
}

void MacroModuleLook::doGeneralUpdate(int)
{
    if (!mme) return;

    // First update graphics, if neccessary
    
    doUpdate();

    bool something_selected = somethingSelected();
    bool clipboard_exists = clipboardFileExists();

    // update menu enabling

    layout_menu.setItemEnabled(menuid_add_to_layout, doesEditLook() && something_selected);
    layout_menu.setItemEnabled(menuid_remove_from_layout, doesEditLook() && something_selected);

    edit_menu.setItemChecked(menuid_show_rect, doesEditLook());
    edit_menu.setItemEnabled(menuid_undo, undo_level > 0);
    edit_menu.setItemEnabled(menuid_redo, undo_level < highest_undo_file);
    edit_menu.setItemEnabled(menuid_copy, something_selected);
    edit_menu.setItemEnabled(menuid_cut, something_selected);
    edit_menu.setItemEnabled(menuid_paste, clipboard_exists);
    edit_menu.setItemEnabled(menuid_save_sel, something_selected);
    edit_menu.setItemEnabled(menuid_select_all, macromodule->numberOfModules() > 0);
    edit_menu.setItemEnabled(menuid_flip_vert, something_selected);
    edit_menu.setItemEnabled(menuid_flip_horiz, something_selected);

    execute_menu.setItemEnabled(menuid_play,  macromodule->numberOfModules() > 0 && (!playing || paused));
    execute_menu.setItemEnabled(menuid_pause, playing && !paused);
    execute_menu.setItemEnabled(menuid_stop, playing);
    execute_menu.setItemEnabled(menuid_samplingconfig, !playing);

    // update toolbar enabling
    
    if (toolbar) { // Within MacroModuleLook::MacroModuleLook() toolbar is 0!
 	toolbar->setItemEnabled(toolbarid_copy, something_selected);
 	toolbar->setItemEnabled(toolbarid_cut, something_selected);
 	toolbar->setItemEnabled(toolbarid_paste, clipboard_exists);
	toolbar->setItemEnabled(toolbarid_undo, undo_level > 0);
	toolbar->setItemEnabled(toolbarid_redo, undo_level < highest_undo_file);
	toolbar->setItemEnabled(toolbarid_play, macromodule->numberOfModules() > 0 && (!playing || paused));
	toolbar->setItemEnabled(toolbarid_pause, playing && !paused);
	toolbar->setItemEnabled(toolbarid_stop, playing);
    }

    // update statusbar
    showModified();
}


void MacroModuleLook::saveSelectionToFile()
{
    OutputModuleFile *modulefile = toplevel->getOutputFile();
    if (modulefile) {
	saveToFile(*modulefile, false, true); // stays modified, only selected modules
	delete modulefile;
    }
}

void MacroModuleLook::rewire()
{
    relayoutWiresConnecting(0);
}


void MacroModuleLook::flipVertically()
{
    flipSelection(true);
}


void MacroModuleLook::flipHorizontally()
{
    flipSelection(false);
}


void MacroModuleLook::addToLayout()
{
    changeVisibility(true);
}


void MacroModuleLook::removeFromLayout()
{
    changeVisibility(false);
}


void MacroModuleLook::editLook()
{
    macromodule->editorInfo().edit_look = !layout_menu.isItemChecked(menuid_show_rect);
    layout_menu.setItemChecked(menuid_show_rect, doesEditLook());
    needsUpdate(mme->rect());
}

void MacroModuleLook::lookDialog()
{
    LookInfoDialog mld(this, macromodule->lookInfo());
    if(mld.exec() == QDialog::Accepted)
    {
	mld.getNewLook(macromodule->lookInfo());
	refreshLayoutImage();
	if (doesEditLook()) needsUpdateLayoutRect();
    }
}


void MacroModuleLook::scanPlugins() const
{
    static bool already_scanned = false;
    if (!already_scanned) {
	already_scanned = true; // MUST be done first to avoid infinite recursion!
 	DirScan dirscan(global_plugins_dir);
 	while (dirscan.toNext())
 	{
	    string menupath = dirscan.relativeDirPath();
	    string filename = dirscan.fullFilename();
	    MLGPlugin *mgpl = new MLGPlugin(filename, menupath);
	    if (!mgpl->isGood()) {
		mgpl->remove();
		printf("Skipped %s: not valid module file.\n", filename.c_str());
	    } 
	} 
    } 
} 


// Fullfill inheritance of ModuleLook

bool MacroModuleLook::isControl()
{
    for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
	if (moduleLookIsVisible(ml) && ml->isControl()) return true;
    return false;
}


bool MacroModuleLook::moduleLookIsVisible(ModuleLook *ml) const
{
    QRect must_be_contained_in_here = enlargeRect(layoutRect() * 2 * GRID_SPACING, 1);
    return (must_be_contained_in_here.contains(ml->rectOfAura())
	    && ml->getModule()->isVisible());
}


bool MacroModuleLook::mousePressed(const QPoint& pos)
{
    // Find the LaytoutElement, the mouse was pressed over.
    for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
    {
	if (moduleLookIsVisible(ml) && ml->isControl())
	{
	    if (QRect(positionInLayout(ml), ml->size()).contains(pos)) {
		mouse_pressed_over = ml;
		return ml->mousePressed(pos - positionInLayout(ml));
	    }
	}
    }
    return false; // no repaint neccessary, since mouseclick was not recognized.
}


bool MacroModuleLook::mouseMoved(const QPoint& pos)
{
    if (mouse_pressed_over) 
	return mouse_pressed_over->mouseMoved(pos - positionInLayout(mouse_pressed_over));
    else return false; // no repaint neccessary
}


bool MacroModuleLook::mouseReleased()
{
    if (mouse_pressed_over) {
	bool repaint_neccessary = mouse_pressed_over->mouseReleased();
	mouse_pressed_over = 0;
	return repaint_neccessary;
    }
    else return false; // no repaint neccessary
}

QPoint MacroModuleLook::positionInLayout(ModuleLook *ml) const
{
    return QPoint((ml->rectOfAura().left() - layoutRect().left() * 2 * GRID_SPACING) / 2, 
		  (ml->rectOfAura().top()  - layoutRect().top()  * 2 * GRID_SPACING) / 2);
}


// Inheritance from LayoutElement

QSize MacroModuleLook::gridSize() const 
{
    return layoutRect().size();
}

QRect MacroModuleLook::sizeWithBorder() const
{
    if (macromodule->lookInfo().threed_border) {
	if (macromodule->lookInfo().rounded_edges) 
	    return ::sizeWithRoundBorder(LayoutElement::size(), 50);
	else return ::sizeWithBorder(LayoutElement::size()); 
    }
    else return QRect(QPoint(0,0), LayoutElement::size());
}


bool MacroModuleLook::needsRepaint()
{
    // Important: The loops must go through ALL modules,
    // so that all know, that they will be repainted.
    bool needs_repaint = false;
    for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
	if (ml->needsRepaint()) needs_repaint = true;
    return needs_repaint;
}


void MacroModuleLook::paint(QPainter& p)
{
    paint(p, &macromodule->lookInfo());
}


void MacroModuleLook::paint(QPainter& p, const LookInfo *li)
{
    p.save();
    
    QRect rect(QPoint(0,0), LayoutElement::size());
    p.setBrush(toQ(li->backgroundcolor));
    p.setPen(NoPen);
    if (li->rounded_edges) p.drawRoundRect(rect, 50, 50);
    else p.drawRect(rect);
    if (li->text_layout) 
    {
	p.setPen(toQ(li->textcolor));
	p.drawText(rect,  AlignCenter | AlignVCenter, (li->layout_text).c_str());
    }
    if (li->image_layout && li->image.imagesize) {
	if (li == &macromodule->lookInfo())
	    	p.drawPixmap(0, 0, layout_image, 0, 0, LayoutElement::width(), LayoutElement::height());
	else {
	    QPixmap pixmap = toQ(li->image);
	    p.drawPixmap(0, 0, pixmap, 0, 0, LayoutElement::width(), LayoutElement::height());
	}
    }
    
    // Paint all visible Modules.
    for (ModuleLook *ml = module_looks.first(); ml != 0; ml = module_looks.next())
	if (moduleLookIsVisible(ml))
	{
	    QPoint pos_in_layout = positionInLayout(ml);
	    p.save();
	    p.translate(pos_in_layout.x(), pos_in_layout.y());
	    ml->paint(p); // , positionInLayout(ml), li->separators);
	    p.restore();
	}


    p.restore();
}

QPixmap MacroModuleLook::preview(const LookInfo *li, const QSize& size)
{
    QPixmap pixmap(size);
    QPainter p;
    p.begin(&pixmap);

    ::paintCanvas(p, QRect(QPoint(0,0), size));
    ::paintGrid(p, QRect(QPoint(0,0), size));

    p.translate((size.width() - LayoutElement::width()) / 2,
		(size.height() - LayoutElement::height()) / 2);
    paintBorder(p, li);
    paint(p, li);
    p.end();
    return pixmap;
}


void MacroModuleLook::paintBorder(QPainter& p) const
{
    paintBorder(p, &macromodule->lookInfo());
}


void MacroModuleLook::paintBorder(QPainter& p, const LookInfo *li) const
{
    if (li->threed_border) {
	if (li->rounded_edges) ::paintRoundBorder(p, LayoutElement::size(), 50);
	else ::paintBorder(p, LayoutElement::size()); 
    }
}


void MacroModuleLook::paintSelection(QPainter& p) const
{
    ::paintSelection(p, sizeWithBorder());
}


QWidget *MacroModuleLook::widget()
{
    if (mme) return mme;
    else {
	fprintf(stderr, "Internal Error: MacroModuleLook::widget() was called with mme == 0.\n");
	*(char *)0xdeafbeaf = 0;
	return 0;
    }
}
