// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/base/AVLTree.h,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:00:35 $
// $State: Exp $
// **************************************************************
// -*- c++ -*-

#ifndef AVLTree_h
#define AVLTree_h

// No duplicates will be stored in this AVL-Tree!

// This tree is not yet an AVL tree but yet a most simple binary search tree.
// But i hope you know the three great laws of optimization:
//
//           I.   Don't optimze yet.
//           II.  Don't even optimze yet.
//           III. Even now it's far too soon to optimize.

#define AVL_NIL ((AVLTreeNode *)0)


/**
  * @short Base class for objects stored in an AVLTree.
  */
class AVLTreeNode
{
  friend class AVLTree;

  enum left_or_right { left=0, right=1 };
  AVLTreeNode *father;
  AVLTreeNode *child[2];
private:
  bool isSuperRoot() const { return father == AVL_NIL; };
  bool isRoot() const { return father->isSuperRoot(); };
  bool isLeaf() const { return child[left] == AVL_NIL && child[right] == AVL_NIL; };
  bool isSonAs(left_or_right lor) const { return father->child[lor] == this; };
  left_or_right sonAs() const { return isSonAs(left) ? left : right; };
  inline AVLTreeNode *getExtreme(left_or_right);
  inline AVLTreeNode *traverse(left_or_right);
public:
  AVLTreeNode() { child[left] = child[right] = AVL_NIL; };
  inline void remove();
  virtual ~AVLTreeNode() { remove(); };
  inline void insert(AVLTreeNode *n);
  AVLTreeNode *getLargest() { return getExtreme(right); };
  AVLTreeNode *getSmallest() { return getExtreme(left); };
  AVLTreeNode *getNext() { return traverse(right); };
  AVLTreeNode *getPrevious() { return traverse(left); };
  inline AVLTreeNode *find(void *c);     // entry exactly equal to c
  inline AVLTreeNode *findNext(void *c); // entry just greater than c
protected:
  virtual int comparedTo(AVLTreeNode *) const { return 0; };
  virtual int comparedTo(void *) const { return 0; }; // 2 dummies for super_root
};


/**
  * @short Kind of balanced tree structure. Used to store cached portions of sound.
  */
class AVLTree
{
  friend class  AVLTreeNode;
  AVLTreeNode   super_root; // This avoids case decisions. child[left]=root.
  AVLTreeNode*& root() { return super_root.child[AVLTreeNode::left]; };
public:
  AVLTree() { root() = AVL_NIL; super_root.father = AVL_NIL; };
  bool isEmpty() { return root() == AVL_NIL; };
  void deleteMembers() { while (!isEmpty()) delete root(); };
  void insert(AVLTreeNode *n);
  AVLTreeNode *getSmallest()     { return root() ? root()->getSmallest() : AVL_NIL; };
  AVLTreeNode *getLargest()      { return root() ? root()->getLargest()  : AVL_NIL; };
  AVLTreeNode *find(void *c)     { return root() ? root()->find(c)       : AVL_NIL; };
  AVLTreeNode *findNext(void *c) { return root() ? root()->findNext(c)   : AVL_NIL; };
};


// inline functions for AVLTree -----------------------------------------

inline void AVLTree::insert(AVLTreeNode *n) 
{
  if (isEmpty()) { root() = n; n->father = &super_root; }
  else root()->insert(n);
}


// inline functions for AVLTreeNode -------------------------------------

inline void AVLTreeNode::insert(AVLTreeNode *n) 
{
  left_or_right lor;
  int comp = this->comparedTo(n);
  if (comp == 0)     return;
  else if (comp < 0) lor = right;
  else               lor = left;

  if (child[lor] != AVL_NIL) 
    child[lor]->insert(n);
  else { 
    child[lor] = n; 
    n->father = this; 
  }
}


inline void AVLTreeNode::remove() 
{ 
  // The destructor of AVLTreeNode calls remove(). But the super-root may not
  // be removed!

  if (isSuperRoot()) return;

  // The root case needs no special handling, because we have the
  // "super-root" in AVLTree.  For the root always holds sonAs() == left,
  // since the super-root uses child[left] as a pointer to the root.

  if (isLeaf()) father->child[sonAs()] = AVL_NIL; // leaf
  else if (child[left] == AVL_NIL) {              // half-leaf left
    child[right]->father = father;
    father->child[sonAs()] = child[right];     
  }
  else if (child[right] == AVL_NIL) {             // half-leaf right
    child[left]->father = father;
    father->child[sonAs()] = child[left];
  }
  else {                                       // inner node
    AVLTreeNode *replacement = child[left]->getLargest();
    replacement->remove(); // is at least a half-leaf -> no endless recursion.

    // Each node has three pointers and is pointed to by three pointers.
    // I must therefore adapt 6 pointers:

    replacement->child[left] = child[left];
    replacement->child[right] = child[right];
    replacement->father = father;

    if (child[left] != AVL_NIL)
      child[left]->father = replacement; // I removed one node left!
    child[right]->father = replacement;
    father->child[sonAs()] = replacement;
  }
}


inline AVLTreeNode *AVLTreeNode::find(void *searched_for)
{
  
  int comp = this->comparedTo(searched_for);
  if (comp<0) return child[right]!=AVL_NIL ? child[right]->find(searched_for):AVL_NIL;
  else if (comp == 0) return this;
  else return child[left ]!=AVL_NIL ? child[left ]->find(searched_for) : AVL_NIL;
}

 
inline AVLTreeNode *AVLTreeNode::findNext(void *searched_for)
{
  int comp =  this->comparedTo(searched_for);
  if (comp < 0) {
    if (child[right]) return child[right]->findNext(searched_for);
    else return traverse(right);
  }
  else if (comp == 0) return traverse(right);
  else {
    if (child[left]) return child[left]->findNext(searched_for);
    else return this;
  }
}


inline AVLTreeNode *AVLTreeNode::traverse(left_or_right lor)
{
  if (child[lor] != AVL_NIL) 
    return child[lor]->getExtreme(lor == left ? right : left);
  else {
    AVLTreeNode *n = this;
    while (!n->isRoot() && n->isSonAs(lor)) n = n->father;
    return n->isRoot() ? AVL_NIL : (n->father);
  }
}


inline AVLTreeNode *AVLTreeNode::getExtreme(left_or_right lor) 
{ 
  if (child[lor] != AVL_NIL) return child[lor]->getExtreme(lor);
  else return this;
};



#endif // AVLTree_h
