/*
  implements a node (n sources, m devices)
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */





#include <devices/nodeDevice.h>

#include <graph/daisyChain.h>            

static DaisyChain* daisyChain=new DaisyChain();
static int instance=0;



NodeDevice::NodeDevice() {
  sources=new Edges();
  listeners=new Edges();
  yafGraph=new YafGraph();

  instance++;

  pthread_mutex_init(&writeInMut,NULL);
  pthread_mutex_init(&changeMut,NULL);

  pthread_mutex_init(&writeOutMut,NULL);
  pthread_cond_init(&changeCond,NULL);
}


NodeDevice::~NodeDevice() {
  int i;
  int n;
  YafGraph* neighbourGraph;
  NodeDevice* nodeDevice;
  lockDeviceTree();
  yafGraph->getSources(this,sources);
  n=sources->getElements(); 
  for(i=0;i<n;i++) {
    nodeDevice=sources->getNodeDevice(i);
    neighbourGraph=nodeDevice->yafGraph;
    neighbourGraph->disconnect(nodeDevice,this);
    // now reload the neighbours listener cache
    neighbourGraph->getListeners(nodeDevice,nodeDevice->listeners);
  }
  // and now remove the source from our listeners
  yafGraph->getListeners(this,listeners);
  n=listeners->getElements();
  for(i=0;i<n;i++) {
    nodeDevice=listeners->getNodeDevice(i);
    neighbourGraph=nodeDevice->yafGraph;
    neighbourGraph->disconnect(this,nodeDevice);
    // now reload the neighbours source cache
    neighbourGraph->getSources(nodeDevice,nodeDevice->sources);
  }
  
  unlockDeviceTree();
  delete sources;
  delete listeners;
}


void* NodeDevice::getDaisyChain() {
  return daisyChain;
}




void NodeDevice::lockDeviceTree() {
  daisyChain->lockStreamProducer();
}


void NodeDevice::unlockDeviceTree() {
  daisyChain->unlockStreamProducer();
} 


void NodeDevice::addListener(NodeDevice* nodeDevice) {
  lockDeviceTree();
  if (nodeDevice == NULL) {
    cout << "Your NodeDevice->addListener(..) speaks to you:"<<endl;
    cout << "***Inserting a NULL pointer as Listener! I better exit***"<<endl;
    exit(0);
  }
  yafGraph->connect(this,nodeDevice);
  nodeDevice->addSource(this);
  yafGraph->getListeners(this,listeners);
  unlockDeviceTree();
}


void NodeDevice::removeListener(NodeDevice* nodeDevice) {
  lockDeviceTree();
  if (nodeDevice == NULL) {
    cout << "Your NodeDevice->removeListener(..) speaks to you:"<<endl;
    cout << "***Removing a NULL pointer as Listener! I better exit***"<<endl;
    exit(0);
  }  
  yafGraph->disconnect(this,nodeDevice);
  yafGraph->getListeners(this,listeners);
  nodeDevice->removeSource(this);
  unlockDeviceTree();
}


void NodeDevice::addSource(NodeDevice* source) {
  yafGraph->connect(source,this);
  yafGraph->getSources(this,sources);
}


void NodeDevice::removeSource(NodeDevice* source) {
  yafGraph->disconnect(source,this);
  yafGraph->getSources(this,sources);  
}


Edges* NodeDevice::getListeners() {
  return listeners;
}


Edges* NodeDevice::getSources() {
  return sources;
}

int NodeDevice::hasListener() {
  int n=listeners->getElements();
  return (n > 0);
}



void NodeDevice::writeIn(NodeDevice* source,DeviceConfig* buf) {
}


void NodeDevice::writeOut(DeviceConfigArray* buf) {
  int i;
  int j;
  NodeDevice* nodeDevice;
  DeviceConfig* config;
  int m=buf->getEntries();
  Edges* edges=getListeners();
  int n=edges->getElements();

  for(j=0;j<m;j++) {
    config=buf->getDeviceConfigAt(j);
    for(i=0;i<n;i++) {
      nodeDevice=edges->getNodeDevice(i);
      nodeDevice->writeInLock();
      nodeDevice->writeIn(this,config);
      nodeDevice->writeInUnlock();
    }
  }
}


void NodeDevice::writeInLock() {
  
  pthread_mutex_lock(&changeMut);
  wakeUpThread(true);
  pthread_mutex_lock(&writeInMut);
}


void NodeDevice::writeInUnlock() {
  wakeUpThread(false);
  pthread_mutex_unlock(&changeMut);
  pthread_cond_signal(&changeCond);
  pthread_mutex_unlock(&writeInMut);
}

 
void NodeDevice::writeOutLock() {
  pthread_mutex_lock(&writeOutMut);
}


void NodeDevice::writeOutUnlock() {
  pthread_mutex_unlock(&writeOutMut);
}



void NodeDevice::wakeUpThread(int lLockInProgress) {
}


void NodeDevice::doSleep(int msec) {
  struct timeval time;
  time.tv_sec=0;
  time.tv_usec=msec;

  select(0,NULL,NULL,NULL,&time);
}




void NodeDevice::deliverToListener(DeviceConfig* config) {
  int i; 
  NodeDevice* nodeDevice;
  Edges* edges=getListeners();
  int n=edges->getElements();

  for (i=0;i<n;i++) {
    nodeDevice=edges->getNodeDevice(i);
    nodeDevice->writeInLock();
    nodeDevice->writeIn(this,config);
    nodeDevice->writeInUnlock();
  }
}

