/*
  stores deviceConfig data in a fifo. (it makes a deep copy  of the data)
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#include <deviceConfig/deviceConfigFifo.h>



DeviceConfigFifo::DeviceConfigFifo(NodeDevice* id) {
  int i;

  freePos=0;
  lWaitForStoppedSource=false;
  lWaitForPausedSource=false;
  lDequeued=false;
  this->id=id;
  status=new StatusInfo();

  for(i=0;i<_MAX_FIFO_ENTRIES;i++) {
    configQueue[i]=new DeviceConfig();
  }
}


DeviceConfigFifo::~DeviceConfigFifo() {
  int i;
  
  clear();
  for(i=0;i<_MAX_FIFO_ENTRIES;i++) {
    delete configQueue[i];
  }
  delete status;
  DaisyChain* daisyChain=(DaisyChain*)NodeDevice::getDaisyChain();
  daisyChain->removeAllLocks(this);
}


  

void DeviceConfigFifo::enqueue(DeviceConfig* config) {
  DeviceConfig* freePosConfig;

  if (freePos == _MAX_FIFO_ENTRIES) {
    cout << "fifo overflow! ignore enqueue:"<<getpid()<<endl;
    return;
  }
  StatusInfo* statusInfo=config->getStatusInfo();

  if (statusInfo->getChange()) {
    statusInfo->copyTo(status);
    return;
  }
  freePosConfig=configQueue[freePos];
  config->copyTo(freePosConfig);
  freePosConfig->addLock();
  
  freePos++;
  if (overflowDanger() == true) {
    setOverflowDangerLock();
  }

}


DeviceConfig* DeviceConfigFifo::dequeue() {
  lDequeued=true;
  return configQueue[0];
}


void DeviceConfigFifo::forwardQueue() {
  int i;
  DeviceConfig* config;

  // first we define the status of the fifo. by default a fifo
  // is active but if we forward the queue and the deviceConfig
  // say that the status is stopped the fifo gets inactive
  // it stays inactive until a deviceConfig is inserted with
  // playing status

  if (freePos == 0) {
    cout << "forward queue on empty queue called!"<<endl;
    exit(0);
  }
  if (lDequeued == false) {
    cout << "error forward without dequeue"<<endl;
    exit(0);
  }
  lDequeued=false;
  config=configQueue[0];
  config->removeLock();


  for(i=0;i<=freePos-1;i++) {
    configQueue[i]=configQueue[i+1];
  }
  configQueue[freePos]=config;
  
  freePos--;
  if (overflowDanger() == false) {
    removeOverflowDangerLock();
  } 
}


/**
   Overflow danger exists if the shared memory is
   exausted or we did not have more entries in the 
   array.
   The first is true for very small ringbuffers.
*/

int DeviceConfigFifo::overflowDanger() {
  if (freePos == (_MAX_FIFO_ENTRIES-2)) {
    return true;
  }
  //return false;
  if (freePos==0) {
    return false;
  }


  DeviceConfig* config=configQueue[freePos-1];
  // now check if there is enough room in the
  // memory to store one another entry here
  AudioStream* audioStream=config->getAudioStream();
  AudioBuffer* buffer=audioStream->getAudioBuffer();
  MemChunk* memChunk=buffer->getMemChunk();
  if (memChunk == NULL) {
    return false;
  }
  int back=memChunk->overflowDanger();
  return back;


}


int DeviceConfigFifo::getFillgrade() {
  return freePos;
}


int DeviceConfigFifo::getWaitForStoppedSource() {
  return lWaitForStoppedSource;
}


void DeviceConfigFifo::setWaitForStoppedSource(int lWaitForStopped) {
  this->lWaitForStoppedSource=lWaitForStopped;
}


int DeviceConfigFifo::getWaitForPausedSource() {
  return lWaitForPausedSource;
}


void DeviceConfigFifo::setWaitForPausedSource(int lWaitForPaused) {
  this->lWaitForPausedSource=lWaitForPaused;
}


int DeviceConfigFifo::getDequeue() {
  return lDequeued;
}


int DeviceConfigFifo::getStatus() {
  return status->getStatus(); 
}


NodeDevice* DeviceConfigFifo::getID() {
  return id;
}


void DeviceConfigFifo::setOverflowDangerLock() {
  if (freePos > 0) {
    DeviceConfig* config=configQueue[0];
    StreamProducer* producer=(StreamProducer*)config->getStreamProducer();
    if (producer == NULL) {
      cout << "fatal Error in DeviceConfigFifo::setOverflowDangerLock!"
	   << " StreamProducer is NULL!"<<endl;
      exit(-1);
    }
    DaisyChain* daisyChain=(DaisyChain*)(producer->getDaisyChain());
    daisyChain->addLock(this,producer);
  }
}


void DeviceConfigFifo::removeOverflowDangerLock() {
  if (freePos > 0) {
    DeviceConfig* config=configQueue[0];
    StreamProducer* producer=(StreamProducer*)config->getStreamProducer();
    if (producer == NULL) {
      cout << "fatal Error in DeviceConfigFifo::setOverflowDangerLock!"
	   << " StreamProducer is NULL!"<<endl;
      exit(-1);
    }   
    DaisyChain* daisyChain=(DaisyChain*)(producer->getDaisyChain());
    daisyChain->removeLock(this,producer);
  }
}

  

void DeviceConfigFifo::clear() {
  while(getFillgrade() > 0) {
    dequeue();
    forwardQueue();
  }
  return;
}


void DeviceConfigFifo::print() {
  cout << "DeviceConfigFifo [START]"<<endl;
  status->print();
  cout << "entries:"<<getFillgrade()<<endl;
  cout << "DeviceConfigFifo [END]"<<endl;
 
}
