/*
  reads paket input data
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


#include "packet.h"



static int getByteDirect(InputStream* input) {
  unsigned char back=0;
  if (input->read((char*)&back,1) != 1) {
    return EOF;
  }
  return (int)back;
}


Packet::Packet() {
  sys_layer=-1;
  floatTimeStamp=0.0;
  ptsTimeStamp=0.0;
  dtsTimeStamp=0.0;
  
}


Packet::~Packet() {
}

/*
 *-----------------------------------------------------------------
 *
 *  ReadPackHeader
 *
 *      Parses out the PACK header
 *
 *  Returns: 1 on error, 0 on success
 *
 *-------------------------------------------------------------------
 */
int Packet::ReadPackHeader(InputStream* input) {
  int numRead;
  unsigned char inputBuffer[PACK_HEADER_SIZE];
  
  numRead = input->read((char*)inputBuffer, PACK_HEADER_SIZE);

  unsigned char hiBit;
  unsigned long low4Bytes;
  ReadTimeStamp((unsigned char*)inputBuffer,&hiBit,&low4Bytes);
  MakeFloatClockTime(hiBit,low4Bytes,&floatTimeStamp);

  unsigned long muxRate;
  ReadRate(&inputBuffer[5], &muxRate);
  muxRate *= MUX_RATE_SCALE_FACTOR;


  if (numRead < PACK_HEADER_SIZE) {
    // if eof is set
    return 1;
  }
  return 0;
}


/*
 *------------------------------------------------------------------
 *
 *   ReadSystemHeader
 *
 *      Parse out the system header, setup out stream IDs for parsing packets
 *
 *   Results:  Returns 1 on error, 0 on success.
 *             Sets gAudioStreamID and gVideoStreamID
 *
 *------------------------------------------------------------------
 */
int Packet::ReadSystemHeader(InputStream* input) { 
  unsigned char *inputBuffer = NULL;
  int numRead;
  unsigned short headerSize;
  unsigned int ftellPos;
  
  numRead = input->read((char *)&headerSize, 2); 
  headerSize = ntohs(headerSize);
  ftellPos=input->getBytePosition();
  if (numRead != 2) {
    // eof is set
    return 1;
  }
  inputBuffer = new(unsigned char[headerSize+1]);
  
  if (inputBuffer == NULL) {
    return 1;
  }
  inputBuffer[headerSize]=0;
  numRead = input->read((char*)inputBuffer, headerSize); 


  delete inputBuffer;
  return 0;
}

 
/*
 *-----------------------------------------------------------------
 *
 *  ReadPacket
 *
 *      Reads a single packet out of the stream, and puts it in the
 *      buffer if it is video.
 *
 *  Results:
 *      Changes the value of *length_ptr to be the new length (plus old)
 *      If the buffer is too small, can change *bs_ptr, *max_length, and 
 *      buf_ptr to be correct for a newly allocated buffer.
 *
 *  State:  
 *      The buffer is in ints, but the packets can be an arbitrary number
 *      of bytes, so leftover bytes are kept in the VidStream variable and
 *      are added on the next call.
 *
 *-----------------------------------------------------------------
 */   
/* Returns:
   0 - no error, but not video packet we want
   -1 - error
   >0 - length of packet
*/
int Packet::ReadPacket(unsigned char& packetID,InputStream* input) {    

  int ioBytes;
  unsigned short packetLength;
  unsigned int packetDataLength;

  /* Leftovers from previous video packets */
  
  if (packetID == NOT_PACKET_ID) {
    cout << "(vid_stream->bitwindow)->makeEnd()"<<endl;
  } else if (packetID==KILL_BUFFER) {
    printf("packetID==KILL_BUFFER\n");
  }
  
  ioBytes = input->read((char*)&packetLength, 2);
  packetLength = htons(packetLength);
  if (ioBytes < 2) {
    return -1;
  }
  switch (packetID) {
  case PAKET_ID_VIDEO:
    break;
  case PAKET_ID_AUDIO_1:
  case PAKET_ID_AUDIO_2:
    packetID=PAKET_ID_AUDIO;
    break;
  case PADDING_STREAM_ID:
    break;
  case RESERVED_STREAM_ID:
    break;
  case PRIVATE_STREAM_1_ID:
    break;
  case PRIVATE_STREAM_2_ID:
    break;
  default:
    printf("\nUnknown packet type. P'bly audio? (%x) at %8x\n",
	   packetID,input->getBytePosition());
  }
  if (packetID != PAKET_ID_VIDEO) {
    if (packetID != PAKET_ID_AUDIO) {
      // nuke it
      while(packetLength > 0) {
	getByteDirect(input);
	packetLength--;
      }
      return 0;
    }
  }


  packetDataLength = packetLength-readPacketTypeStd(input);
  return packetDataLength;

}


double Packet::getPTSTimeStamp() {
  return ptsTimeStamp;
}


double Packet::getDTSTimeStamp() {
  return dtsTimeStamp;
}
  

int Packet::readPacketTypeStd(InputStream* input) {
  unsigned char nextByte;
  int pos;
  unsigned char scratch[10];


  nextByte=getByteDirect(input);

  pos = 1;
  while (nextByte & 0x80) {
    ++pos;
    scratch[0]=getByteDirect(input);
    nextByte=scratch[0];
  }
  if ((nextByte >> 6) == 0x01) {
    pos += 2;
    scratch[1]=getByteDirect(input);
    scratch[2]=getByteDirect(input);
    nextByte=scratch[2];
  } 
  if ((nextByte >> 4) == 0x02) {
    scratch[0] = nextByte;                     
    input->read((char*)&scratch[1],4);
    /* presentation time stamps */
    unsigned char hiBit;
    unsigned long low4Bytes;
    ReadTimeStamp((unsigned char*)scratch,&hiBit,&low4Bytes);
    MakeFloatClockTime(hiBit,low4Bytes,&ptsTimeStamp);
    dtsTimeStamp=0.0;
    pos += 4;
  }
  else if ((nextByte >> 4) == 0x03) {
    scratch[0] = nextByte;                      
    input->read((char*)&scratch[1],9);
    /* presentation and decoding time stamps */
    unsigned char hiBit;
    unsigned long low4Bytes;
    ReadTimeStamp((unsigned char*)scratch,&hiBit,&low4Bytes);
    MakeFloatClockTime(hiBit,low4Bytes,&ptsTimeStamp);

    ReadTimeStamp((unsigned char*)&(scratch[5]),&hiBit,&low4Bytes);
    MakeFloatClockTime(hiBit,low4Bytes,&dtsTimeStamp);


    pos += 9;
  } 

  return pos;
}

/*
  ARGH!!!!!!!!!!
*/
double Packet::getFloatTimeStamp() {
  return floatTimeStamp;
}


int Packet::readNextLayer(unsigned char& packetID,InputStream* input) {
  /* A system layer stream (called after the 1st time), call the specialist */
  int errorCode;
  unsigned int startCode;
  errorCode = ReadStartCode(&startCode,input);
  if (input->eof()) {
    cout << "eof flag set"<<endl;
    return 0;
  }
  if (errorCode != 0) {
    fprintf(stderr, "Unable to read initial pack start code\n");
    return 0;
  }
  return readNextLayer(startCode,packetID,input);
}


/*
 *----------------------------------------------------------
 *
 *  read_sys
 *
 *      Parse out a packet of the system layer MPEG file.
 *
 *  Results:  Returns -1 if error or EOF
 *            Returns 1 if more data read (could be just one int)
 *
 *  Side Effects:  ReadPacket can change *bs_ptr to be a new buffer
 *                 buf_ptr will remain pointing at *length_ptr (at input)
 *                         into the buffer
 *                 *length_ptr will be changed to the new size
 *                 *max_length can be changed if a new buffer is alloc'd
 *
 *----------------------------------------------------------
 */
/* start is either a start code or 0 to indicate continued parsing */
int Packet::readNextLayer(unsigned int startCode,unsigned char& packetID,
			  InputStream* input) {
  int errorCode, PacketReply;


  BOOLEAN match;
  errorCode = 0;
  
  while (1) {
    match=false;
    if (startCode==PACK_START_CODE) {
      match=true;
      errorCode=ReadPackHeader(input);
      if (errorCode != 0) {
        printf("Error in reading pack header\n");
        return -1;
      }
      errorCode=ReadStartCode(&startCode,input);
      if (errorCode != 0) {
        printf("Error in reading start code\n");
        return -1;
      }
    }
    if (startCode==SYSTEM_HEADER_START_CODE) {
      cout << "SYSTEM_HEADER_START_CODE"<<endl;
      match = true;
      errorCode = ReadSystemHeader(input);
      if (errorCode != 0) {
        fprintf(stderr, "Error in reading system header\n");
        return -1;
      }
      errorCode = ReadStartCode(&startCode,input);
      if (errorCode != 0) {
        printf("Error in reading start code after system header\n");
        return -1;
      }
    }
    packetID=startCode & 0xff;

    while (((startCode & PACKET_START_CODE_MASK)==PACKET_START_CODE_PREFIX) &&
           (packetID >= 0xbc)) {
      match=true;
      packetID=startCode & 0xff;
      PacketReply=ReadPacket(packetID,input);
      if (PacketReply > 0) {
	return PacketReply;
      }
      if (PacketReply == -1) {
	return PacketReply;
      }      

      errorCode = ReadStartCode(&startCode,input);
      if (errorCode != 0) {
        printf("Error in start code after packet\n");
        return -1;
      }
      if (startCode==PACK_START_CODE || startCode==ISO_11172_END_CODE) {
        break;
      }
    }
    
    if (startCode == ISO_11172_END_CODE) {
      //match=true;
      cout << "should do makeEnd"<<endl;
      return -1;
    }
    
    if (errorCode != 0)
      return 1;
    if (!match) {
      //      printf("no match: %8x skipping to start code\n",startCode);
      errorCode=find_start_code_raw(&startCode,input);
      if (errorCode==-1) {
	return -1;
      }
    }
  }
}


/*
 *-----------------------------------------------------------
 *
 *  find_start_code
 *
 *  Parses a start code out of the stream by tossing bytes until it gets one
 *
 *  Results/Side Effects:  Parses bytes of the stream, returns code
 *                         Returns EOF in case of end of file
 *
 *-----------------------------------------------------------
 */
int Packet::find_start_code_raw(unsigned int* startCode,InputStream* input) {
 int byte; 
 int maxSearch=1024*1024*6;
 int cnt=0;
 NO_ZEROS:
  byte=getByteDirect(input);
  cnt++;
  if (cnt >= maxSearch) return true;
  switch(byte) {
  case 0:    goto ONE_ZERO;
  case EOF:  goto EOF_FOUND;
  default:   goto NO_ZEROS;
  }

 ONE_ZERO:
  byte=getByteDirect(input);
  cnt++;
  if (cnt >= maxSearch) return true;
  switch(byte) {
  case 0:    goto TWO_ZEROS;
  case EOF:  goto EOF_FOUND;
  default:   goto NO_ZEROS;
  }

 TWO_ZEROS:
  byte=getByteDirect(input);
  cnt++;
  if (cnt >= maxSearch) return true;
  switch(byte) {
  case 0x01:  goto CODE_FOUND;
  case 0x00:  goto TWO_ZEROS;
  case EOF:  goto EOF_FOUND;
  default:    goto NO_ZEROS;
  }

 CODE_FOUND:
  *startCode=0x00000100+getByteDirect(input);
  return 0;

 EOF_FOUND:   /* received EOF */
  if (input->eof()) {
    return true;
  }
  goto NO_ZEROS;
}



/*
 *-----------------------------------------------------------
 *
 *  ReadStartCode
 *
 *      Parses a start code out of the stream
 *
 *  Results/Side Effects:  Sets *startCode to the code, returns
 *     1 on error, 0 on success
 *
 *-----------------------------------------------------------
 */
int Packet::ReadStartCode(unsigned int* startCode,InputStream* input){
  int numRead;
  unsigned int readCode=0;

  numRead = input->read((char*)&readCode, 4);
  *startCode = htonl(readCode);

  if (numRead < 4) {
    // eof is set
    return 1;
  }


  if ((*startCode&0xfffffe00) != 0) {
   
    int error = find_start_code_raw(startCode,input);
    if (error) {
      // eof is set
      return 1;
    }
  }

  return 0;
}


int Packet::syncStream(unsigned int& syncCodeFound,InputStream* input) {


  
  /* Time to init ourselves */
  while (1) {
    int error = find_start_code_raw(&syncCodeFound,input);
    if (error) {
      cout << "find_start_code_raw -eeror"<<endl;
      return false;
    }
    
    switch (syncCodeFound) {

    case PACK_START_CODE:
    case SYSTEM_HEADER_START_CODE:{
      printf("This is an MPEG System Layer Stream.  ");
      printf("Audio is not played.\n");
      sys_layer = _PACKET_SYSLAYER;
      return true;
    }
    case SEQ_START_CODE: {
      printf("This is a raw MPEG video Stream.\n");
      sys_layer = _PACKET_NO_SYSLAYER;
      return true;
    }    

    }
  }
  return false;
}




int Packet::getSysLayer() {
  return sys_layer;
}


void Packet::setSysLayer(int sysLayer) {
  this->sys_layer=sysLayer;
}


void Packet::ReadTimeStamp(unsigned char* inputBuffer,
			   unsigned char* hiBit,
			   unsigned long* low4Bytes) {
  *hiBit = ((unsigned long)inputBuffer[0] >> 3) & 0x01;
  *low4Bytes = (((unsigned long)inputBuffer[0] >> 1) & 0x03) << 30; 
  *low4Bytes |= (unsigned long)inputBuffer[1] << 22; 
  *low4Bytes |= ((unsigned long)inputBuffer[2] >> 1) << 15; 
  *low4Bytes |= (unsigned long)inputBuffer[3] << 7; 
  *low4Bytes |= ((unsigned long)inputBuffer[4]) >> 1; 
}


void Packet::ReadSTD(unsigned char* inputBuffer,unsigned char* stdBufferScale,
		     unsigned long* stdBufferSize) {
  *stdBufferScale = ((int)(inputBuffer[0] & 0x20) >> 5); 
  *stdBufferSize = ((unsigned long)inputBuffer[0] & 0x1f) << 8;
  *stdBufferSize |= (unsigned long)inputBuffer[1];
}


int Packet::MakeFloatClockTime(unsigned char hiBit, unsigned long low4Bytes,
			       double * floatClockTime) {
  if (hiBit != 0 && hiBit != 1) {
    *floatClockTime = 0.0;
    return 1;
  }
  *floatClockTime 
    = (double)hiBit*FLOAT_0x10000*FLOAT_0x10000 + (double)low4Bytes;
  *floatClockTime /= (double)STD_SYSTEM_CLOCK_FREQ;
  return 0;
}



void Packet::ReadRate(unsigned char* inputBuffer,unsigned long* rate) {
  *rate = (inputBuffer[0] & 0x7f) << 15;
  *rate |= inputBuffer[1] << 7;
  *rate |= (int) (inputBuffer[2] & 0xfe) >> 1;
}




