/*
  this class manages all instances of the players
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */
#include <kmpgServer.h>


static int instance=0;

KmpgServer::KmpgServer() {
  mixerDevice=new MixerDevice();
  nasDevice=new NASDevice();
  audioDevice=new AudioDevice("/dev/dsp");
  wavFileDevice=new WavFileDevice();
  esdDevice=new ESDDevice();


  mixerDevice->addListener(audioDevice);
  mixerDevice->addListener(wavFileDevice);

  selectOutputDialog=new SelectOutputDialog();
  connect(selectOutputDialog,SIGNAL(applyEvent()),this,SLOT(outputEnable()));

  selectOutputDialog->setHaveNATIVE(true); 
  selectOutputDialog->setHaveFILE(true); 
  if (nasDevice->isCompiledIn()){
    mixerDevice->addListener(nasDevice);
    selectOutputDialog->setHaveNAS(true);
  }
  if (esdDevice->isCompiledIn()){
    mixerDevice->addListener(esdDevice);
    selectOutputDialog->setHaveESD(true);
  }
  // now what we want
  selectOutputDialog->setWantNATIVE(true);
  

  GarbageCollector::init();

 
  preferences=new Preferences();
  audioDeviceConfig=new AudioDeviceConfig("AudioDeviceConfig",preferences);
  connect(audioDeviceConfig,SIGNAL(applyEvent()),
	  this,SLOT(applyAudioDevice()));

  preferences->addTab(audioDeviceConfig,audioDeviceConfig->getGroupName());


  wavFileDeviceConfig=new WavFileDeviceConfig("WavFileDeviceConfig",
					      preferences);
  connect(wavFileDeviceConfig,SIGNAL(applyEvent()),
	  this,SLOT(applyWavFileDevice()));
  connect(wavFileDeviceConfig,SIGNAL(updateEvent()),
	  this,SLOT(updateWavWriter()));
  preferences->addTab(wavFileDeviceConfig,wavFileDeviceConfig->getGroupName());


  nasDeviceConfig=new NASDeviceConfig("NAS",preferences);
  connect(nasDeviceConfig,SIGNAL(applyEvent()),this,SLOT(applyNASDevice()));  
  preferences->addTab(nasDeviceConfig,nasDeviceConfig->getGroupName());
  applyNASDevice();

  esdDeviceConfig=new ESDDeviceConfig("esd",preferences);

  if (esdDevice->isCompiledIn()){
    esdDeviceConfig->setCompiledIn(true);
    connect(esdDeviceConfig,SIGNAL(applyEvent()),
	    this,SLOT(applyESDDevice()));  
  }

  preferences->addTab(esdDeviceConfig,esdDeviceConfig->getGroupName());

  openTimer=new QTimer(this);
  connect(audioDevice->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(processAudioDeviceEvent(char)));
  connect(esdDevice->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(processAudioDeviceEvent(char)));
  connect(nasDevice->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(processAudioDeviceEvent(char)));

  connect(wavFileDevice->getEventQueue(),SIGNAL(processEvent(char)),
	  this,SLOT(processWavFileDeviceEvent(char)));

  
  connect(openTimer,SIGNAL(timeout()),this,SLOT(openOutputs()));

  stopTimer=new QTimer(this);
  connect(stopTimer,SIGNAL(timeout()),this,SLOT(stopOutputs()));
  lStopRequest=false;
  lOpenRequest=false;
  // init audio things
  // first set the outputs to enable/disable state
  outputEnable();
  // then open them
}


KmpgServer::~KmpgServer() {
  audioDevice->close();
  delete mixerDevice;
  delete audioDevice;
  delete preferences;
  delete wavFileDevice;
  delete nasDevice;
  delete esdDevice;
  delete selectOutputDialog;
}



void KmpgServer::newPlayer() {
  if (instance==5) {
    // someday I must find out where the switch
    // is. its in the source, but I do not find it anymore.
    // there are a few other restrictions as well
    // max entries in grapfh max connection between nodes.
    // these are all static arrays.
    
    QString msg2;
    msg2 = i18n("You have reached the compiled limit of 5 instances");
    msg2 += "\n\n";
    msg2 += i18n("yaflib v0.1");
    KMsgBox::message(NULL,i18n("yaf-lib limit"),msg2.data());
    return;
  }
  DecoderStarter* decoderStarter=new DecoderStarter();
  connect(this,SIGNAL(stopAllDecoders()),decoderStarter,SLOT(stop()));
  KmpgMainWindow* mainWindow=new KmpgMainWindow();

  if (mainWindow->getPreferences()->isVisible()){
    mainWindow->getPreferences()->show();
  }

  // tell the gui from where the stream comes
  mainWindow->setInputNode(decoderStarter);

  // connect the output from the gui to our mixer
  mainWindow->getOutputNode()->addListener(mixerDevice);
  
  decoderStarter->setRefCount(1);

  connect(mainWindow,SIGNAL(closeIPCHandleForever()),
	  this,SIGNAL(closeIPCHandleForever()));
	

  connect(mainWindow,SIGNAL(removeDecoder(KmpgMainWindow*)),
	  this,SLOT(deleteDecoder(KmpgMainWindow*)));

  connect(mainWindow,SIGNAL(newPlayerEvent()),this,SLOT(newPlayer()));
  connect(mainWindow,SIGNAL(outputConfig()),this,SLOT(configure()));
  connect(mainWindow,SIGNAL(inputConfig()),decoderStarter,SLOT(configure()));
  connect(mainWindow,SIGNAL(outputSelectEvent()),
	  selectOutputDialog,SLOT(makeVisible()));
  
  
  instance++;
}


void KmpgServer::quit() {

  emit(closeIPCHandleForever());

  (kapp->getConfig())->sync();

  kapp->quit();

}



void KmpgServer::deleteDecoder(KmpgMainWindow* win) {
  DecoderStarter* decoderStarter;
  decoderStarter=win->getInputNode();
  if (decoderStarter == NULL) {
    cout << "already closes"<<endl;
  }
  decoderStarter->setRefCount(decoderStarter->getRefCount()-1);
  win->removeInputNode();
 
  if (decoderStarter->getRefCount() == 0) {
    delete decoderStarter;
  }

  win->getOutputNode()->removeListener(mixerDevice);

  win->removeInputNode();
  // connect the output from the gui to our mixer



  instance--;
  disconnect(win,0,this,0);
  if (instance == 0) {
    this->quit();
  }
}


void KmpgServer::processCommandLine(CommandLineCreator* cmdLine) {
  KmpgCommandLine::parseIPC(cmdLine);
}


void KmpgServer::configure() {
  preferences->show();
}

   

void KmpgServer::processAudioDeviceEvent(char eventId) {
  switch(eventId) {
  case _AUDIODEVICE_CLOSE_REQUEST: {
    if (lStopRequest==false) {
      stopTimer->start(2000,true);
      lStopRequest=true;
    }
    break;
  }
  case _AUDIODEVICE_CANNOT_OPEN: {
    if (lOpenRequest== false) {
      openTimer->start(1000,true);
      lOpenRequest=true;
    }
    break;
  }
  case _NAS_STREAM_CHANGE:
  case _NAS_DATA_TO_WRITE:
  case _NAS_VOLUME_TO_SET:
  case _NAS_INIT_TO_DO: {

    // ignore these handled by device
    break;
  }

  default:
    cout << "unknown msg: "<<eventId<<" in kmpgserver::processEvent-1"<<endl;
  }
}
  

void KmpgServer::applyAudioDevice() {

  if (lStopRequest==false) {
    stopTimer->start(2000,true);
    lStopRequest=true;
  }
  int autoInit=audioDeviceConfig->getAutoInit();
  audioDevice->setAutoInit(autoInit);
  if (autoInit==false) {
    int speed=audioDeviceConfig->getSampleFreq();
    int size=audioDeviceConfig->getSampleSize();
    int stereo=audioDeviceConfig->getStereo();
    audioDevice->init(size,speed,stereo);
  }
    
}


void  KmpgServer::stopOutputs() {
  if (audioDevice->isOpen() == true) {
    int enabled=audioDevice->getEnabled();
    if (enabled == false) {
      audioDevice->close();
      lStopRequest=false;
      return;
    }
    StatusInfo* statusInfo=audioDevice->getStatusInfo();
    if (statusInfo->getStatus()==_STATUS_STOPPED) {
      if (audioDeviceConfig->getAutoClose()) {
	audioDevice->close();
      }
    }
  }
  lStopRequest=false;
}


int KmpgServer::openOutputs() {
  int back=true;
  QString names;
  if (nasDevice->getEnabled()) {
    char* display=nasDeviceConfig->getDisplay();
    nasDevice->close();
    nasDevice->open(display);
    if (nasDevice->isOpen() == false) {
      names += i18n("Output");
      names += ":";
      names += "NAS";
      names += "\n";
     
      back=false;
    }
  }
  if (audioDevice->getEnabled()) {
    audioDevice->open();
    if (audioDevice->isOpen() == false) {
      names += i18n("Output");
      names += ":";
      names += i18n("Native");
      names += "\n";
     
      back=false;
    }
  }

  if (esdDevice->getEnabled()) {
    esdDevice->open();
    if (esdDevice->isOpen() == false) {
      names += i18n("Output");
      names += ":";
      names += "ESD";
      names += "\n";
      
      back=false;
    }
  }
  if (names.length() > 0) {
    outputError(&names);
  }
  lOpenRequest=false;
  return back;

}

void KmpgServer::disableAllOutputs() {
  nasDevice->setEnabled(false);
  esdDevice->setEnabled(false);
  audioDevice->setEnabled(false);
  wavFileDevice->setEnabled(false);
 
  
}


void KmpgServer::outputError(QString* deviceName) {
  QString msg2;
 

  emit(stopAllDecoders());
  msg2 += i18n("The following outputs cannot be opened:");
  msg2 += "\n\n";
  msg2 += *deviceName;
  msg2 += "\n\n";
  msg2 += i18n("All decoders stopped");
  msg2 += "\n\n";
  msg2 += i18n("Further information can be found in the online help");
  msg2 += "\n\n";
  KMsgBox::message(NULL,i18n("AudioPanic "),msg2.data());
  
  selectOutputDialog->show();
}


void KmpgServer::applyWavFileDevice() {
  int stopIsEOF=wavFileDeviceConfig->getStopIsEOF();
  const char* dirName=wavFileDeviceConfig->getDirectory();
  
  wavFileDevice->setTargetDirectory((char*)dirName);
  wavFileDevice->setStopIsEOF(stopIsEOF);
}


void KmpgServer::applyNASDevice() {
  char* display=nasDeviceConfig->getDisplay();
  nasDevice->close();
  nasDevice->open(display);
  
  const char* serverString=nasDevice->getServerString();
  const char* vendorString=nasDevice->getVendorString();
  int majorVersion=nasDevice->getMajorProtocolVersion();
  int minorVersion=nasDevice->getMinorProtocolVersion();

  nasDeviceConfig->setServerString(serverString);
  nasDeviceConfig->setVendorString(vendorString);
  nasDeviceConfig->setMajorVersion(majorVersion);
  nasDeviceConfig->setMinorVersion(minorVersion);

}


void KmpgServer::processWavFileDeviceEvent(char eventId) {
  switch(eventId) {
  case _WAVFILEDEVICE_SIGNAL_ERROR: {
    QString msg2;
    msg2 += i18n("An Error during write occured!");
    msg2 += "\n\n";
    msg2 += i18n("Descripton:");
    msg2 += strerror(wavFileDevice->getErrno());
    msg2 += "\n\n";
    msg2 += i18n("Writer disabled!");
    msg2 += "\n\n";
    selectOutputDialog->setWantFILE(false);
    outputEnable();
    KMsgBox::message(NULL,i18n("Wav File Writer Panic"),msg2.data());
    break;
  }
  default:
    cout << "unknown msg: "<<eventId<<" in kmpgserver::processEvent-2"<<endl;
  }
}


void KmpgServer::updateWavWriter() {
  long bytes=wavFileDevice->getByteCounter();
  char* filename=wavFileDevice->getFileName();
  wavFileDeviceConfig->setCurrentFile(filename);
  wavFileDeviceConfig->setByteCounter(bytes);
}


void KmpgServer::outputEnable() {
  int outputMask=selectOutputDialog->getValidOutputMask();

  nasDevice->setEnabled(false);
  esdDevice->setEnabled(false);
  audioDevice->setEnabled(false);
  wavFileDevice->setEnabled(false);
  wavFileDeviceConfig->update(false);

  if (outputMask & _SELECT_OUPUT_NAS) {
    nasDevice->setEnabled(true);
  } else {
    nasDevice->close();
  }
  if (outputMask & _SELECT_OUPUT_ESD) {
    esdDevice->setEnabled(true);
  } else {
    esdDevice->close();
  }
  if (outputMask & _SELECT_OUPUT_NATIVE) {
    audioDevice->setEnabled(true);
  } else {
    audioDevice->close();
  }
  if (outputMask & _SELECT_OUPUT_FILE) {
    wavFileDeviceConfig->update(true);
    wavFileDevice->setEnabled(true);
  }

}
