/*
 *	      PlayList - playlist dialog for kmikmod
 *	(C)1998, 1999 Roope Anttinen - roope.anttinen@ntc.nokia.com
 *	
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#include <iostream.h>
#include <qlayout.h>
#include <qgrpbox.h>
#include <qfileinfo.h>
#include <qframe.h>
#include <qtooltip.h>
#include <qdir.h>
#include <qfile.h>
#include <qregexp.h>
#include <kapp.h>
#include <kmsgbox.h>
#include <stdlib.h>
#include "PlayList.h"

PlayList::PlayList(QWidget *parent, const char *name)
	: QDialog(parent, name)
{
	QSize s_tmp;	// For getting widget sizes for different locales.
	pthread_mutex_init (&PL, NULL);
	randomize=false;
	item_count=max_idx=0;
	
	plDir = KApplication::localkdedir().data();
	plDir += "/share/apps/kmikmod";

	QGroupBox* group = new QGroupBox(this);
	listbox = new QListView(group);

	QBoxLayout* vbox1 = new QVBoxLayout (this,5,-1,"toplevel");
	QBoxLayout* vbox2 = new QVBoxLayout (group,10,-1,"playlist vbox");
	QBoxLayout* vbox3 = new QVBoxLayout (5, "list buttons");
	QBoxLayout* hbox1 = new QHBoxLayout (15, "dialog buttons");
	QBoxLayout* hbox2 = new QHBoxLayout (10, "playlist hbox");
	
	vbox1->addWidget(group,AlignBottom);
	vbox1->addLayout (hbox1);
	vbox2->addSpacing(15);	// This is the reason this vbox is needed...
	vbox2->addLayout(hbox2);
	hbox2->addWidget (listbox,10);
	hbox2->addLayout(vbox3);

	group->setTitle(i18n("PlayList"));
	
	listbox->addColumn(i18n("ID"), 30);
	listbox->addColumn(i18n("File"), 270);
	listbox->addColumn(i18n("Type"), 120);
	listbox->addColumn(i18n("Info"), 80);
	listbox->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	listbox->setAllColumnsShowFocus(true);
	
	PB_add = new QPushButton(group);
	PB_remove = new QPushButton(group);
	PB_load = new QPushButton(group);
	PB_save = new QPushButton(group);
	PB_clear = new QPushButton(group);
	PB_add->setText(i18n("Add.."));
	PB_remove->setText(i18n("Remove"));
	PB_load->setText(i18n("Load..."));
	PB_save->setText(i18n("Save..."));
	PB_clear->setText(i18n("Clear"));
	s_tmp = PB_add->sizeHint().width() > PB_remove->sizeHint().width() ?
		PB_add->sizeHint() : PB_remove->sizeHint();
	s_tmp = PB_load->sizeHint().width() > s_tmp.width() ?
		PB_load->sizeHint() : s_tmp;
	s_tmp = PB_save->sizeHint().width() > s_tmp.width() ?
		PB_save->sizeHint() : s_tmp;
	s_tmp = PB_clear->sizeHint().width() > s_tmp.width() ?
		PB_clear->sizeHint() : s_tmp;

	PB_add->setFixedSize(s_tmp);
	PB_remove->setFixedSize(s_tmp);
	PB_load->setFixedSize(s_tmp);
	PB_save->setFixedSize(s_tmp);
	PB_clear->setFixedSize(s_tmp);

	TB_rand = new QCheckBox(i18n("Random"), group);
	TB_loop = new QCheckBox(i18n("Loop"), group);
	s_tmp = TB_rand->sizeHint().width()>TB_loop->sizeHint().width()?
		TB_rand->sizeHint():TB_loop->sizeHint();
	TB_rand->setFixedSize(s_tmp);
	TB_loop->setFixedSize(s_tmp);
		
	vbox3->addWidget(PB_add);
	vbox3->addWidget(PB_remove);
	vbox3->addWidget(PB_load);
	vbox3->addWidget(PB_save);
	vbox3->addWidget(PB_clear);
	vbox3->addSpacing(20);
	vbox3->addWidget(TB_rand);
	vbox3->addWidget(TB_loop);
	vbox3->addStretch();

	PB_ok = new QPushButton(this);
	PB_cancel = new QPushButton(this);
	PB_apply = new QPushButton(this);
	PB_ok->setText(i18n("OK"));
	PB_cancel->setText(i18n("Cancel"));
	PB_apply->setText(i18n("Apply"));

	s_tmp = PB_ok->sizeHint().width() > PB_cancel->sizeHint().width() ?
		PB_ok->sizeHint() : PB_cancel->sizeHint();
	s_tmp = PB_apply->sizeHint().width() > s_tmp.width() ?
		PB_apply->sizeHint() : s_tmp;
	PB_ok->setFixedSize(s_tmp);
	PB_cancel->setFixedSize(s_tmp);
	PB_apply->setFixedSize(s_tmp);
	
	PB_ok->setDefault(true);

	hbox1->addStretch();
	hbox1->addWidget(PB_ok);
	hbox1->addWidget(PB_apply);
	hbox1->addWidget(PB_cancel);
	
	QToolTip::add (listbox, i18n("Drop files here from kfm"));
	QToolTip::add (PB_add, i18n("Add files to the list"));
	QToolTip::add (PB_remove, i18n("Remove selected item from the list"));
	QToolTip::add (PB_load, i18n("Load saved playlist"));
	QToolTip::add (PB_save, i18n("Save this playlist"));
	QToolTip::add (PB_clear, i18n("Clear the playlist"));
	QToolTip::add (TB_rand, i18n("Randomize playing order"));
	QToolTip::add (TB_loop, i18n("Loop list forever"));

//	fileDLG = new KFileDialog(0);
	
	KDNDDropZone *dnd = new KDNDDropZone(listbox, DndURL);

	connect (dnd, SIGNAL(dropAction(KDNDDropZone *)),
		SLOT(Dropped(KDNDDropZone *)));
	connect(PB_ok, SIGNAL(clicked()), SLOT(CB_ok()));
	connect(PB_cancel, SIGNAL(clicked()), SLOT(CB_cancel()));
	connect(PB_apply, SIGNAL(clicked()), SLOT(CB_apply()));
	connect(PB_add, SIGNAL(clicked()), SLOT(CB_add()));
	connect(PB_remove, SIGNAL(clicked()), SLOT(CB_remove()));
	connect(PB_load, SIGNAL(clicked()), SLOT(CB_load()));
	connect(PB_save, SIGNAL(clicked()), SLOT(CB_save()));
	connect(PB_clear, SIGNAL(clicked()), SLOT(CB_clear()));
	connect(listbox, SIGNAL(doubleClicked(QListViewItem *)), SLOT(CB_apply()));
	
	TB_rand->setEnabled(false);	// Not yet implemented, so disabled

	setCaption(i18n("Playlist editor"));
	vbox1->activate();
}

PlayList::~PlayList()
{
}

void PlayList::Dropped(KDNDDropZone *p)
{
	QStrList tmp_l = p->getURLList();
	QString tmp;
	int count = tmp_l.count();
	for (int idx=0; idx < count; idx++) {
		if (strncmp(tmp_l.at(idx), "file:", 5)==0) {
			tmp = &tmp_l.at(idx)[5];	 // eat 'file:' away
			tmp.replace(QRegExp("%20")," "); // take care of holes
			addToList(tmp, getType(tmp), getInfo(tmp));
		}
		else cerr << "kmikmod: invalid data -> " << tmp_l.at(idx) << " <-" << endl;
	}
}

const char *PlayList::getCurrent()
{
	if (item_count && listbox->currentItem()) {
		return ((const char *)(listbox->currentItem()->text(1)));
	}
	cerr << "kmikmod: empty or none selected" << endl;
	return NULL;
}

bool PlayList::next()
{
	bool ret = false;
	if (item_count) {
		QListViewItem *tmp = listbox->currentItem();
		if (item_count > 1 && tmp->itemBelow()) {
			listbox->setSelected(tmp->itemBelow(), true);
			ret = true;
		} else if (TB_loop->isChecked()) {
			listbox->setSelected(listbox->firstChild(), true);
			ret = true;
		}
		listbox->ensureItemVisible(listbox->currentItem());
	}
	return ret;
}

bool PlayList::previous()
{
	bool ret = false;
	if (item_count) {
		QListViewItem *tmp = listbox->currentItem();
		if (tmp->itemAbove()) {
			listbox->setSelected(tmp->itemAbove(), true);
			ret = true;
		} else if (TB_loop->isChecked()) {
			while ((tmp = tmp->itemBelow()))
				listbox->setCurrentItem(tmp);
			listbox->setSelected(listbox->currentItem(), true);
			ret = true;
		}
		listbox->ensureItemVisible(listbox->currentItem());
	}
	return ret;
}

void PlayList::remove()
{
	if (item_count) {
		QListViewItem *tmp = listbox->currentItem();
		if (tmp->itemBelow())
			listbox->setCurrentItem(tmp->itemBelow());
		else if(tmp->itemAbove())
			listbox->setCurrentItem (tmp->itemAbove());
		else {
			cerr << "kmikmod: no item below nor above -- weird??" << endl;
			if (listbox->firstChild())
				listbox->setCurrentItem(listbox->firstChild());
			else
				cerr << "kmikmod: no first child????" << endl;
		}
		listbox->setSelected(listbox->currentItem(), true);
		delete (tmp);
		item_count--;
		/* And now reindex the table */
		if (item_count) {
			int i=1;
			char txt[4] = "";
			sprintf(txt, "%03i", i);
			tmp = listbox->firstChild();
			tmp->setText(0, txt);
			while(tmp->itemBelow()) {
				sprintf(txt, "%03i", ++i);
				tmp = tmp->itemBelow();
				tmp->setText(0, txt);
			}
			max_idx = i--;
		} else max_idx = 0;
	}
	emit itemsChanged();
}

void PlayList::addToList(const char *n, const char *t, const char *i)
{
	char *tmp = 0;
	char index [4];
	int len = strlen(n);
	QListViewItem *i_tmp;
	tmp = new char [len+1];
	strncpy (tmp, n, len);
	tmp[len]='\0';
	sprintf(index, "%03i", max_idx+1);
	i_tmp = new QListViewItem(listbox, index, tmp, t, i);
	delete [] tmp;
	if(!item_count) listbox->setSelected (i_tmp, true);
	item_count++;
	max_idx++;
	emit itemsChanged();
}

void PlayList::setInfo(const char *info)
{
	listbox->currentItem()->setText(3,info);
}

void PlayList::setType(const char *type)
{
	listbox->currentItem()->setText(2, type);
}

const int PlayList::current()
{
	int ret=0;
	if (item_count && listbox->currentItem())
		ret = atoi(listbox->currentItem()->text(0));
	return ret;
}

char *PlayList::getType (const QString item)
{
	static char ret[10];
	QFileInfo info(item);
	strncpy (ret, info.extension(),9);
	return ret;
}

char *PlayList::getInfo (const QString)
{
	static char info[] = "- - -";
	return info;
}

void PlayList::CB_ok()
{
	CB_apply();
	CB_cancel();
}

void PlayList::CB_cancel()
{
	this->hide();
}

void PlayList::CB_apply()
{
	emit applyClicked();
}

void PlayList::CB_add()
{
// This really sucks... getOpenFileURLList isn't implemented in
// KDE libraries even though it appears in kfiledialog.h
// This must be a bug ;(
//	QStrList files = fileDLG->getOpenFileURLList();
//	int count = files.count();
//	for (int idx=0; idx < count; idx++) addToList(files.at(idx));
	static QString pos(".");
	QString file = KFileDialog::getOpenFileName(pos);
	if (file.contains("/"))
		pos = file.right((file.length() - file.findRev("/")) - 1);
	else pos = ".";
	if (!file.isEmpty()) addToList(file);
}

void PlayList::CB_remove()
{
	remove();
}

void PlayList::CB_load()
{
	QString file = KFileDialog::getOpenFileName(plDir);
	if (!file.isEmpty() && QFileInfo(file).isFile()) loadList(file);
}

void PlayList::CB_save()
{
	bool gotIt = false;
	if (gotPLDir()) gotIt = true;
	else if (createPLDir()) gotIt = true;
	if (gotIt) {
		QString file = KFileDialog::getSaveFileName(plDir);
		if (!file.isEmpty() && !QFileInfo(file).isDir()) saveList(file);
	}
}

bool PlayList::gotPLDir()
{
	QDir tmp;
	return (tmp.exists(plDir));
}

bool PlayList::createPLDir()
{
	QDir tmp;
	return (tmp.mkdir(plDir));
}

void PlayList::emptyList()
{
	listbox->clear();
	max_idx=item_count=0;
	emit itemsChanged();
}

void PlayList::loadList(const char *list)
{
	char tmp1[256], tmp2[64], tmp3 [32];	// maybe enough??
	int len=0;
	pthread_mutex_lock (&PL);
	if (item_count) emptyList();
	QFile f_tmp(list);
	if (f_tmp.open(IO_ReadOnly)) {
		while (!f_tmp.atEnd()) {
			len = f_tmp.readLine(tmp1, 255);
			tmp1[len-1] = 0;
			len = f_tmp.readLine(tmp2, 63);
			tmp2[len-1] = 0;
			len = f_tmp.readLine(tmp3, 31);
			tmp3[len-1] = 0;
			addToList(tmp1,tmp2,tmp3);
		}
		f_tmp.close();
	} else KMsgBox::message(NULL,
				i18n("Error"),
				i18n("Can't open file for reading"));
	pthread_mutex_unlock(&PL);
}

void PlayList::saveList(const char *list)
{
	QString s_tmp;
	pthread_mutex_lock (&PL);
	QListViewItem *tmp = listbox->firstChild();
	QFile f_tmp(list);
	if (f_tmp.open(IO_WriteOnly)) {
		for (int i=0; i<item_count; i++) {
			for (int y=1; y<4; y++) {
				s_tmp = tmp->text(y);
				f_tmp.writeBlock(s_tmp, s_tmp.length());
				f_tmp.writeBlock("\n", 1);
			}
			tmp = tmp->itemBelow();
		}
		f_tmp.close();
		KMsgBox::message(NULL,
				 i18n("Save"),
				 i18n("Playlist saved."));
	} else KMsgBox::message(NULL,
			i18n("Error"),
			i18n("Can't open file for writing.\nPlaylist not saved."));
	pthread_mutex_unlock (&PL);
}
