    /*

    Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "midi.h"
#include "interfaces.h"
#include "synth_impl.h"
#include "debug.h"
#include "sequenceutils.h"
#include <math.h>
#include <list>

void MidiChannel_impl::noteOn( CORBA::Octet channel, CORBA::Octet note,
														CORBA::Octet volume )
{
	list<MidiReceiver *>::iterator i;
	for(i=receivers.begin();i != receivers.end();i++)
	{
		(*i)->noteOn(channel,note,volume);
	}
}

void MidiChannel_impl::noteOff( CORBA::Octet channel, CORBA::Octet note )
{
	list<MidiReceiver *>::iterator i;
	for(i=receivers.begin();i != receivers.end();i++)
		(*i)->noteOff(channel,note);
}

void MidiChannel_impl::subscribe(MidiReceiver *receiver)
{
	receivers.push_back(receiver);
}

void MidiChannel_impl::unSubscribe(MidiReceiver *receiver)
{
	receivers.remove(receiver);
}

//---------------------------------------------------------------------------

class MidiEvent :public Interface_MIDI_NOTE
{
	int _channel, _note, _velocity;
	float outfrequency, outvelocity, outpressed;
	bool init;
public:

	long StructureID;

	int channel();
	int note();
	void off();
	void Initialize();
	void CalculateBlock(unsigned long samples);
	void setNote(int channel, int note, int velocity);
};

int MidiEvent::note()
{
	return _note;
}

int MidiEvent::channel()
{
	return _channel;
}

void MidiEvent::off()
{
	if(init)
	{
		outpressed = 0;
	}
	else
	{
		printf("MidiEvent: Humm? Got noteoff before being initialized?\n");
	}
}

void MidiEvent::setNote(int channel, int note, int velocity)
{
	_velocity = velocity;
	_note = note;
	_channel = channel;
	init = false;
}

void MidiEvent::Initialize()
{
/*
    float freq[]  = { 261.7,277.2,293.7,311.2,329.7,349.3,370.0,392.0,
					  415.3,440.0, 466.2,493.9, 0 };
    float zhoch[24];
	int i,octave = _note/12;

	for(i=0;i<24;i++) zhoch[i] = (1 << i);
	//outfrequency = freq[_note%12];
	outfrequency *= zhoch[_note/12] / zhoch[4];
*/
	outfrequency = 261.63 * pow(2,(float)_note/12)/32.0;
	outpressed = 1;
	outvelocity = (float)_velocity/127.0;
	init = true;
	haveCalculateBlock = true;
	artsdebug("pitch: %d, frequency: %2.2f\n",_note,outfrequency);
}

void MidiEvent::CalculateBlock(unsigned long samples)
{
	float *sigfrequency = out[FREQUENCY], *sigpressed = out[PRESSED],
          *sigvelocity = out[VELOCITY], *sigend = sigfrequency + samples;

	while(sigfrequency < sigend)
	{
		*sigfrequency++ = outfrequency;
		*sigpressed++   = outpressed;
		*sigvelocity++  = outvelocity;
	}	
}

//---------------------------------------------------------------------------
class GMRRange
{
protected:
	Arts::StructureDesc_var _structure;
	int from,to;
public:
	// be careful - GMRRange eats the structure on delete
	GMRRange(int from, int to,Arts::StructureDesc_ptr structure);
	~GMRRange();

	Arts::StructureDesc_ptr structure();
	bool match(int note);
};

GMRRange::GMRRange(int from, int to, Arts::StructureDesc_ptr structure)
{
	this->_structure = structure;
	this->from = from;
	this->to = to;

	_structure->incRef();
}

Arts::StructureDesc_ptr GMRRange::structure()
{
	return _structure;
}

GMRRange::~GMRRange()
{
	_structure->decRef();
}

bool GMRRange::match(int note)
{
	return ((note >= from) && (note <= to));
}

//---------------------------------------------------------------------------
class GenericMidiRouterModule :public SynthModule, MidiReceiver
{
private:
	string structurename;

protected:
	list<GMRRange *> gmrRanges;

	bool startGuiNow,restoreGui;
	int mychannel,maxvoices;
	int i;
	long guiID,oldGuiID;
	float count;
	float guiParent,guiX,guiY;

	long OldStructChangeCount;
	static long instanceNumber;
	string paramPrefix;

	list<MidiEvent *> activeEvents;

	// inputs
	enum { CHANNEL, MAXVOICES, PARENT, X, Y, PROP_TARGET, PROP_BUS};

	// outputs
	enum { COUNT };

	Arts::StructureDesc_ptr makeStructure(string bus,
								list<pair<string,string> >& extraParams);
	void startGui();

	/*
	 * When the GenericMidiRouterModule initializes, it expects the derived
	 * class to make some initialization, that is:
	 *
	 *  - setting the structurename using gmrSetStructureName
	 *  - adding ranges that should be routed with gmrAddRange();
	 *
	 * The derived class must perform this initialization in initGMR();
	 */

	virtual void initGMR() = 0;

	void gmrSetStructureName(const char *name)
	{
		structurename = name;
	}

	void gmrAddRange(int from, int to, list<pair<string,string> >& extraParams)
	{
		Arts::StructureDesc_ptr structure;
		structure = makeStructure(getStringProperty(PROP_BUS), extraParams);
		if(structure)
		{
			gmrRanges.push_back(new GMRRange(from,to,structure));

			// the GMRRange object owns the structure now
			structure->decRef();
		}
	}

	// extraParams contains param=value assignments

public:
	void cleanUp();
	void noteOn(int channel, int note, int velocity);
	void noteOff(int channel, int note);
	void CallBack();
	void FirstInitialize();
	void Initialize();
	void CalculateBlock(unsigned long samples);
	void DeInitialize();
	void Calculate();

// -- session management

	Arts::StringSeq *saveSessionParameters(list<long>& IDs);
	void restoreSessionParameters(const Arts::StringSeq& params);
};

long GenericMidiRouterModule::instanceNumber = 0;

void GenericMidiRouterModule::Initialize()
{
	char pprefix[1024];

	sprintf(pprefix,"_GenericMidiRouterModule_%ld_",instanceNumber++);
	paramPrefix = pprefix;

	OldStructChangeCount = 0;

	structurename = "";

	count = 0;
	maxvoices = 0;
	i = 0;
	Synthesizer->getMidiChannel()->subscribe(this);
	haveCalculateBlock = true;

	initGMR();

	startGuiNow = true;
	guiID = 0;
}

void GenericMidiRouterModule::FirstInitialize()
{
	restoreGui = false;
	oldGuiID = 0;
}

void GenericMidiRouterModule::DeInitialize()
{
	while(gmrRanges.size())
	{
		GMRRange *g = *gmrRanges.begin();
		delete g;
		gmrRanges.pop_front();
	}
	Synthesizer->getMidiChannel()->unSubscribe(this);

	// this will kill the remaining notes that are still playing the
	// hard way (just free their structures) - otherwise, it may happen
	// that some of them remain playing forever, since they never have
	// a chance to see the midi-release event without their MIDI_ROUTER
	list<MidiEvent *>::iterator i;
	
	for(i=activeEvents.begin();i != activeEvents.end();i++)
	{
		MidiEvent *ev = (*i);
		if(Synthesizer->isExecuting(ev->StructureID))
			Synthesizer->freeStructure(ev->StructureID);
	}

	if(Synthesizer->isExecuting(guiID))
		Synthesizer->freeStructure(guiID);
}

void GenericMidiRouterModule::Calculate()
{
}

void GenericMidiRouterModule::CallBack()
{
	startGui();
}

void GenericMidiRouterModule::CalculateBlock(unsigned long samples)
{
	if(startGuiNow == true)
	{
		assert(samples); // would be foolish to call with samples == 0 anyway

		guiParent = in[PARENT][0];
		guiX = in[X][0];
		guiY = in[Y][0];
		// it's not possible to start new structures while the scheduler
		// is running
		Synthesizer->requestCallBack(this);
		startGuiNow = false;
	}
	i+=samples;
	if(i>500)
	{
		if(Synthesizer->StructChangeCount() != OldStructChangeCount)
			cleanUp();

		OldStructChangeCount = Synthesizer->StructChangeCount();
		i = 0;
	}
	mychannel = (int)(in[CHANNEL][samples-1] + 0.5);
	maxvoices = (int)(in[MAXVOICES][samples-1] + 0.5);

	float *outcount = out[COUNT], *outend = &out[COUNT][samples];

	while(outcount < outend) *outcount++ = count;
}

void GenericMidiRouterModule::noteOn(int channel, int note, int velocity)
{
	// simply cut voices that are over the given maximum
	// could certainly be done more elegantly

	if(mychannel == channel && (count < maxvoices))
	{
		list<GMRRange *>::iterator i;
		for(i=gmrRanges.begin(); i != gmrRanges.end();i++)
		{
			if((*i)->match(note))
			{
				MidiEvent *event = new MidiEvent;
				event->setNote(channel,note,velocity);
				event->setClassName("Interface_MIDI_NOTE");

				activeEvents.push_back(event);

				list<SynthModule *> interfaces;
				interfaces.push_back(event);

				if((*i)->structure())
				{
					Arts::ArtsServerSeq preferredservers;
					event->StructureID =
					Synthesizer->createConnectedStructure((*i)->structure(), 
									preferredservers, interfaces);
				}	
			}
		}
		count = activeEvents.size();
	}
}

void GenericMidiRouterModule::noteOff(int channel, int note)
{
	list<MidiEvent *>::iterator i;
	
	for(i=activeEvents.begin();i != activeEvents.end();i++)
	{
		MidiEvent *ev = (*i);
		if(ev->channel() == channel && ev->note() == note) ev->off();
	}
}

Arts::StructureDesc_ptr GenericMidiRouterModule::makeStructure(string bus,
									list<pair<string, string> >& extraParams)
{
	artsdebug("building instrument structure");
	Arts::ModuleBroker_var mb = Synthesizer->moduleBroker();
	assert(mb);

	Arts::ModuleInfo_var info = mb->lookupModule(structurename.c_str());
	if(!info)
	{
		fprintf(stderr,"FATAL!!-> can't find structure %s for midi synth\n",
							structurename.c_str());
		return 0;
	}

	Arts::ModuleInfo_var pgetinfo = mb->lookupModule("Synth_PARAM_GET");
	assert(pgetinfo);

	Arts::StructureDesc_var newstructure = Synthesizer->createStructureDesc();
	assert(newstructure);

	newstructure->incRef();

	Arts::ModuleDesc_var instrumentmod = newstructure->createModuleDesc(info);
	assert(instrumentmod);

	Arts::PortDescSeq_var ports = instrumentmod->Ports();

	unsigned long i;
	for(i=0;i<ports->length();i++)
	{
		Arts::PortDesc_ptr p = (*ports)[i];
		Arts::PortType pt = p->Type();

		if(pt.Direction == Arts::input)
		{
			CORBA::String_var name = p->Name();
			bool paramassigned = false;

			list<pair<string,string> >::iterator i;

			for(i=extraParams.begin();i != extraParams.end();i++)
			{
				if(strcmp((*i).first.c_str(),(const char *)name) == 0)
				{
					if(pt.DataType == Arts::audio_data)
					{
						p->FloatValue(atof((*i).second.c_str()));
					}
					else if(pt.DataType == Arts::string_data)
					{
						p->StringValue((*i).second.c_str());
					}
					else assert(false);

					paramassigned = true;
				}
			}
			if(paramassigned)
			{
				// fine
			}
			else if(strcmp((const char *)name,"bus") == 0)
			{
				p->StringValue(bus.c_str());
			}
			else if(pt.DataType == Arts::audio_data)
			{
				Arts::ModuleDesc_var propmod =
					newstructure->createModuleDesc(pgetinfo);
				Arts::PortDescSeq_var pgetports = propmod->Ports();
				string paramName = paramPrefix + string(name);

				for(unsigned long j=0;j<pgetports->length();j++)
				{
					Arts::PortDesc_ptr pgp = (*pgetports)[j];
					CORBA::String_var pgname = pgp->Name();

					if(strcmp(pgname,"value") == 0)
						pgp->connectTo(p);
					else if(strcmp(pgname,"name") == 0)
						pgp->StringValue(paramName.c_str());
				}
			}
		}
	}

	/* FIXME! ==> should take the preferredservers we've been executed
		with to appear on the "right" GUI-Server */

	Arts::ArtsServerSeq preferredservers;

	Arts::StructureDesc_ptr result =
		Synthesizer->expandStructureDesc(newstructure);

	result->incRef();
	newstructure->decRef();

	return result;
}

void GenericMidiRouterModule::startGui()
{
	artsdebug("trying to start GUI now...");
	Arts::ModuleBroker_var mb = Synthesizer->moduleBroker();

	string guiname = structurename + string("_GUI");
	Arts::ModuleInfo_var info = mb->lookupModule(guiname.c_str());
	if(!info)
	{
		artsdebug("MIDI_ROUTER: the structure %s seems to have no"
		         	" visual part (which should be called %s)\n",
				 	structurename.c_str(),guiname.c_str());
		return;
	}

	Arts::ModuleInfo_var psetinfo = mb->lookupModule("Synth_PARAM_SET");
	assert(psetinfo);

	Arts::StructureDesc_var gui = Synthesizer->createStructureDesc();
	assert(gui);

	Arts::ModuleDesc_var guimod = gui->createModuleDesc(info);
	assert(guimod);

	Arts::PortDescSeq_var ports = guimod->Ports();

	unsigned long i;
	for(i=0;i<ports->length();i++)
	{
		Arts::PortDesc_ptr p = (*ports)[i];
		Arts::PortType pt = p->Type();
		CORBA::String_var name = p->Name();

		if(pt.Direction == Arts::input)
		{
			if(strcmp((const char *)name,"parent") == 0)
			{
				assert(guiParent > 0.5);
				printf("guiParent is %f\n",guiParent);
				p->FloatValue(guiParent);
			}

			if(strcmp((const char *)name,"x") == 0)
				p->FloatValue(guiX);

			if(strcmp((const char *)name,"y") == 0)
				p->FloatValue(guiY);
		}
		if(pt.Direction == Arts::output && pt.DataType == Arts::audio_data)
		{
			Arts::ModuleDesc_var propmod = gui->createModuleDesc(psetinfo);
			Arts::PortDescSeq_var psetports = propmod->Ports();
			string paramName = paramPrefix + string(name);

			for(unsigned long j=0;j<psetports->length();j++)
			{
				Arts::PortDesc_ptr psp = (*psetports)[j];
				CORBA::String_var psname = psp->Name();

				if(strcmp(psname,"value") == 0)
					psp->connectTo(p);
				else if(strcmp(psname,"name") == 0)
					psp->StringValue(paramName.c_str());
			}
		}
	}

	/* FIXME! ==> should take the preferredservers we've been executed
		with to appear on the "right" GUI-Server */

	Arts::ArtsServerSeq preferredservers;

	artsdebug("********************* createStructure NOW ******************\n");
	if(restoreGui)
	{
		guiID =
		 Synthesizer->restoreStructure(gui,preferredservers,restoreID,oldGuiID);
	}
	else
	{
		guiID = Synthesizer->createStructure(gui, preferredservers);
	}
	assert(guiID);                                                         

	artsdebug("... done startGUI: id = %ld\n",guiID);
	// this only frees the StructureDesc of the GUI (we don't need that any
	// longer), the Structure remains alive on the server
	gui->destroy();
}

void GenericMidiRouterModule::cleanUp()
{
	list<MidiEvent *>::iterator i;
	
	for(i=activeEvents.begin();i != activeEvents.end();i++)
	{
		MidiEvent *ev = (*i);
		if(!Synthesizer->isExecuting(ev->StructureID))
		{
			artsdebug("purged a module\n");
			activeEvents.erase(i);
			//remove(ev);
			//delete ev;  FIXME?
			i = activeEvents.begin();
			count = activeEvents.size();
		}
	}
}

// session management

Arts::StringSeq *GenericMidiRouterModule::saveSessionParameters(list<long>& IDs)
{
	Arts::StringSeq *result = new Arts::StringSeq;

	if(guiID)
	{
		sqprintf(result,"gui_id=%ld\n",guiID);
		IDs.push_back(guiID);
	}
	return result;
}

void GenericMidiRouterModule::restoreSessionParameters(const
														Arts::StringSeq& params)
{
   	char *cmd,*param;
	unsigned long i;

	restoreGui = false;
   	for(i=0;i<params.length();i++)
   	{
   		if(parse_line(params[i],cmd,param))   // otherwise: empty or comment
   		{
   			if(strcmp(cmd,"gui_id") == 0) {
				oldGuiID = atol(param);

				restoreGui = true;
			}
		}
   	}
}

//--------------------------------------------------------------------------

class Synth_MIDI_ROUTER :public GenericMidiRouterModule
{
public:
	void initGMR()
	{
		list<pair<string,string> > extraParams;
		gmrSetStructureName(getStringProperty(PROP_TARGET));
		gmrAddRange(0,127,extraParams);
	}
	string getParams() {
		return ("channel,maxvoices,parent,x,y,_structure,_bus;count");
	}
	static void *Creator() { return new Synth_MIDI_ROUTER; };
};

ModuleClient MC_Synth_MIDI_ROUTER(SynthModule::get_MS,"Synth_MIDI_ROUTER",
													Synth_MIDI_ROUTER::Creator);

//--------------------------------------------------------------------------

class Synth_MIDI_MAP_ROUTER :public GenericMidiRouterModule
{
public:

	enum parserstate { stNeedStruct, stNeedKeygroup, stNeedOpen, stInKeygroup };

	void initGMR()
	{
		CORBA::String_var filename =
			Synthesizer->artsFileName("maps",getStringProperty(PROP_TARGET));
		
		FILE *mapfile = fopen(filename,"r");

		list<pair<string,string> > extraParams;

		int kgFrom = 0, kgTo = 0;

		gmrSetStructureName("_mapper_failed");
		if(!mapfile)
		{
			printf("Couldn't load mapfile %s\n",getStringProperty(PROP_TARGET));
			return;
		}

		parserstate state = stNeedStruct;
		char buffer[1024];
		while(fgets(buffer,1024,mapfile))
		{
			char *keyword = strtok(buffer,"=\n");
			if(keyword)
			{
				char *value = strtok(NULL,"\n");

				switch(state) {
					case stNeedStruct:
						if(strcmp(keyword,"structure") == 0)
						{
							if(value)
								gmrSetStructureName(value);
							state = stNeedKeygroup;
						}
						else
						{
							printf("mapfile: expected structure!\n");
							return;
						}
						break;
					case stNeedKeygroup:
						if(strcmp(keyword,"keygroup") == 0)
						{
							if(value)
							{
								char *fr = strtok(value,"-");
								char *to = strtok(NULL,"-\n");

								if(fr && to)
								{
									kgFrom = atoi(fr);
									kgTo = atoi(to);
									state = stNeedOpen;
								}
								else
								{
									printf("keygroup: missing from/to value\n");
									return;
								}
							}
						}
						break;
					case stNeedOpen:
						if(strcmp(keyword,"{") == 0)
						{
							state = stInKeygroup;
						}
						else
						{
							printf("need a { to begin keygroup values\n");
							return;
						}
						break;
					case stInKeygroup:
						if(strcmp(keyword,"}") == 0)
						{
							gmrAddRange(kgFrom,kgTo,extraParams);
							extraParams.clear();
							state = stNeedKeygroup;
						}
						else
						{
							if(value)
							{
								extraParams.push_back(make_pair(keyword,value));
							}
							else
							{
								printf("keygroup must contain param=value"
										"entries\n");
							}
						}
						break;
				}
			}
		}
	}
	string getParams() {
		return ("channel,maxvoices,parent,x,y,_mapfile,_bus;count");
	}
	static void *Creator() { return new Synth_MIDI_MAP_ROUTER; };
};

ModuleClient MC_Synth_MIDI_MAP_ROUTER(SynthModule::get_MS,
						"Synth_MIDI_MAP_ROUTER",Synth_MIDI_MAP_ROUTER::Creator);

//--------------------------------------------------------------------------
class Synth_MIDI_DEBUG :public SynthModule, MidiReceiver
{
protected:
	int event;
	float time;
public:
	void noteOn(int channel, int note, int velocity)
	{
		printf("%d %f on %d %d %d\n",event++,time*1000,channel,note,velocity);
	}
	void noteOff(int channel, int note)
	{
		printf("%d %f off %d %d\n",event++,time*1000,channel,note);
	}
	void Initialize()
	{
		event = 1;
		time = 0.0;
		Synthesizer->getMidiChannel()->subscribe(this);
		haveCalculateBlock = true;
	}
	void CalculateBlock(unsigned long samples)
	{
		time += (float)samples / samplingRate;
	}
	void DeInitialize()
	{
		Synthesizer->getMidiChannel()->unSubscribe(this);
	}
	void Calculate() { }
	string getParams() { return(";"); }
	static void *Creator() { return new Synth_MIDI_DEBUG; };
};

ModuleClient MC_Synth_MIDI_DEBUG(SynthModule::get_MS,"Synth_MIDI_DEBUG",
												Synth_MIDI_DEBUG::Creator);
