    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#if 1
void debug_printf(const char *arg, ...)
{
}
#else
#define debug_printf printf
#endif

/* Defines */

#define AKAI_BLOCKSIZE 0x2000
#define AKAI_VOLUMES   100

/* Structures */

typedef struct {
	char name[13];
	int type;
	int start;
} akaivolume;

typedef struct {
	char name[13];
	char type;		// 0x70 (p) = PROGRAM, 0x73 (s) = SAMPLE
	long size;
	int start;
	int id;
} akaifile;

typedef struct {
	FILE *file;
	long absolute_offset;	/* offset in the file (mainly for debugging) */

	akaivolume *volume;
	long volumecount;

	char partitionname[2];
	long partitionoffset;
	long partitionsize;

	unsigned int *fat;
	int fatsize;
} akaisrc;

//* Prototypen */


void akaisrc_alloc(akaisrc *src);
const char *akai_type(char c);
akaifile *akai_read_directory(akaisrc *src, akaivolume *volume, int *count);

void akai_read_junk(akaisrc *src, int bytes);
void akai_read_volume_map(akaisrc *src);
void akai_read_name(akaisrc *src, int len, char *name);
void akai_read_seek(akaisrc *src, int offset);
int akai2asc(int c);
int akai_read_uint(akaisrc *src);
char akai_read_char(akaisrc *src);
long akai_read_long3(akaisrc *src);
void akai_read_file(akaisrc *src, akaifile *file, FILE *outfile);
void debug_allocation_table(akaisrc *src, akaivolume *volume, int fcnt);
void akai_read_allocation_table(akaisrc *src);
void akai_list(akaisrc *src, char *directory);
void akai_get(akaisrc *src, char *pattern);
void akai_first_partition(akaisrc *src);
void akai_next_partition(akaisrc *src);

/* Code */

void exit_usage(char *name)
{
fprintf(stderr,"AKAI sample CD parser - (c) 1999 Stefan Westerfeld\n");
fprintf(stderr,"This program is distributed under the terms of the GNU General Public License.\n");
fprintf(stderr,"It is free software with ABSOLUTELY NO WARRANTY.\n");
fprintf(stderr,"\n");
fprintf(stderr,"Listing the contents of a partition (or specific directory):\n\n");
fprintf(stderr,"  - %s <file> <partition> ls\n",name);
fprintf(stderr,"  - %s <file> <partition> ls <directory>\n\n",name);
fprintf(stderr,"Extracting the samples of a partition (or specific directory):\n\n");
fprintf(stderr,"  - %s <file> <partition> get\n",name);
fprintf(stderr,"  - %s <file> <partition> get <directory>\n\n",name);
fprintf(stderr,"<file> may be either a filename of a AKAI cdrom disk image\n");
fprintf(stderr,"or a CDRom device (like /dev/cdrom, /dev/scd0 or /dev/scr0).\n\n");
fprintf(stderr,"<partition> may be either a partition letter like A or B or\n");
fprintf(stderr,"the special 'all' to list (extract) all partitions.\n");
exit(1);
}

int main(int argc, char **argv)
{
	akaisrc src;
	int i,ok;

	if(argc < 4) exit_usage(argv[0]);

	src.file = fopen(argv[1],"r");
	if(!src.file)
	{
		fprintf(stderr,"can't open '%s' for reading\n",argv[1]);
		exit(1);
	}

	// ensure that partitions are given as big letters
	for(i=0;i<strlen(argv[2]);i++)
		argv[2][i] = toupper(argv[2][i]);

	// --- initialize akaisrc structure ---

	akaisrc_alloc(&src);
	akai_first_partition(&src);
	
	while(src.partitionsize != 0)
	{
		debug_printf("absolute_offset = %ld (%lx)\n",
				src.absolute_offset,src.absolute_offset);
		debug_printf("partitionsize = %ld (%lx)\n",
				src.partitionsize,src.partitionsize);

		if(strcmp(src.partitionname,argv[2]) == 0 || strcmp(argv[2],"ALL") == 0)
		{
			printf("Partition %s\n",src.partitionname);
			ok = 0;

			akai_read_junk(&src,200);
			akai_read_volume_map(&src);
			akai_read_allocation_table(&src);

			// --- now handle commands ---

			if(strcmp(argv[3],"ls") == 0 || strcmp(argv[3],"list") == 0)
			{
				switch(argc)
				{
					case 4: akai_list(&src,"");
							ok = 1;
						break;
					case 5: akai_list(&src,argv[4]);
							ok = 1;
						break;
				}
			}
		
			if(strcmp(argv[3],"get") == 0) // get: either all samples or by mask
			{
				switch(argc)
				{
					case 4: akai_get(&src,"");
							ok = 1;
						break;
					case 5: akai_get(&src,argv[4]);
							ok = 1;
						break;
				}
			}

			if(!ok) exit_usage(argv[0]);
		}
		akai_next_partition(&src);
	}
	return 0;
}

void akai_first_partition(akaisrc *src)
{
	strcpy(src->partitionname,"A");

	akai_read_seek(src, src->partitionoffset * AKAI_BLOCKSIZE);
	src->partitionsize = akai_read_uint(src);
}

void akai_next_partition(akaisrc *src)
{
	src->partitionname[0]++;
	src->partitionoffset += src->partitionsize;

	akai_read_seek(src, src->partitionoffset * AKAI_BLOCKSIZE);
	src->partitionsize = akai_read_uint(src);
}

void akaisrc_alloc(akaisrc *src)
{
	src->absolute_offset = 0;

	src->volume = (akaivolume *)malloc(sizeof(akaivolume)*AKAI_VOLUMES);
	src->volumecount = AKAI_VOLUMES;

	src->partitionoffset = 0;
	src->partitionsize = 0;

	src->fat = 0;
	src->fatsize = 0;
}
	
void akai_list(akaisrc *src, char *directory)
{
	int i;

	if(strlen(directory) == 0)
	{
		for(i=0;i<src->volumecount;i++)
		{
			if(src->volume[i].type != 0 || src->volume[i].start != 0)
			{
				printf("$%04x  %s  type %d\n",src->volume[i].start,
						src->volume[i].name,src->volume[i].type);
			}
		}
	}
	else
	{
		for(i=0;i<src->volumecount;i++)
		{
			if(strcmp(src->volume[i].name,directory) == 0)
			{
				akaifile *file;
				int filecount,j;

				file = akai_read_directory(src,&src->volume[i],&filecount);
			
				for(j=0;j<filecount;j++)
				{
					printf("%s  %s   %7ld\n",file[j].name,
									akai_type(file[j].type),file[j].size);
				}
			}
		}
	}
}

void akai_get(akaisrc *src, char *pattern)
{
	int i;
	for(i=0;i<src->volumecount;i++)
	{
		akaifile *file;
		int filecount,j;

		file = akai_read_directory(src,&src->volume[i],&filecount);
		
		for(j=0;j<filecount;j++)
		{
			char fname[26];

			sprintf(fname,"%s/%s",src->volume[i].name,file[j].name);
			if(strncmp(pattern,fname,strlen(pattern)) == 0)
			{
				FILE *outfile;
				mkdir(src->volume[i].name,0777);
				printf("%s ... ",fname);
				fflush(stdout);

				outfile= fopen(fname,"w");
				if(!outfile)
				{
					fprintf(stderr,"can't open file %s for writing\n",fname);
					exit(1);
				}
				akai_read_file(src,&file[j],outfile);
				fclose(outfile);

				printf("done!\n");
			}
		}
	}
}

const char *akai_type(char c)
{
	if(c == 'p') return "PROGRAM";
	if(c == 's') return "SAMPLE ";
	return              "unknown";
}

void akai_read_file(akaisrc *src, akaifile *file, FILE *outfile)
{
	long remaining = file->size, readsize;
	unsigned int block = file->start;
	char buffer[AKAI_BLOCKSIZE];

	assert(block <= src->fatsize);

	while(remaining)
	{
		debug_printf("reading block %d (%x), %d bytes remaining\n",block,block,remaining);
		akai_read_seek(src, (block + src->partitionoffset) * AKAI_BLOCKSIZE);

		readsize = remaining;
		if(readsize > AKAI_BLOCKSIZE) readsize = AKAI_BLOCKSIZE;

		assert(fread(buffer,1,readsize,src->file) == readsize);
		assert(fwrite(buffer,1,readsize,outfile) == readsize);

		remaining-=readsize;

		if(remaining)
		{
			block = src->fat[block];
			assert(block <= src->fatsize);
		}
		else
		{
			debug_printf("file end block (at $%4x) is %d = $%4x\n",block,src->fat[block],
														src->fat[block]);
			if(src->fat[block] != 0xc000)
			{
				fprintf(stderr,"WARNING: bad eof block ($c000 expected, got $%04x)\n",src->fat[block]);
				fprintf(stderr,"this may mean your file is corrupt now\n");
			}
		}
	}
}

void akai_read_seek(akaisrc *src, int offset)
{
	fseek(src->file,offset,SEEK_SET);
	src->absolute_offset = offset;
}

void akai_read_allocation_table(akaisrc *src)
{
	unsigned int x;
	unsigned int *table = 0;
	int l;

	l = 0;
	do {
		x = (unsigned int)akai_read_uint(src);
		table = (unsigned int *)realloc(table,sizeof(unsigned int)*(l+1));
		table[l++] = x;
	}	while (x != 0xffff);

	debug_printf("read allocation table, len = %d\n",l);

	src->fatsize = l;
	src->fat = table;
}

void debug_allocation_table(akaisrc *src, akaivolume *volume, int fcnt)
{
	int i,x,j, following = 0, blocks = 0;
	for(i=0;src->absolute_offset != 0x6000;i++)
	{
		x = akai_read_uint(src);
		for(j=0;j<fcnt;j++)
		{
			if(i != 0 && volume[j].start == i)
			{
				debug_printf("=> VOLUME %s\n",volume[j].name);
			}
		}
		if(x == 0x4000)
		{
			debug_printf("SYS\n");
			following = 0;
		}
		else
		{
			if(following)
			{
				blocks++;
				if(x == 0xc000)
				{
					debug_printf("FILE, len %d blocks\n",blocks);
					following = 0;
				}
				else if(x != i+1)
				{
					debug_printf("? with chaining to block %d\n",x);
					following = 0;
				}
			}
			else
			{
				if(x == i+1)
				{
					blocks = 1;
					following = 1;
				}
				else
				{
					if(x != 0) debug_printf("%d\n",x);
				}
			}
		}
	}
}


void akai_read_junk(akaisrc *src, int bytes)
{
	if(fseek(src->file,bytes,SEEK_CUR))
	{
		debug_printf("READ_JUNK(%d) failed\n",bytes);
		exit(1);
	}
	else
	{
		src->absolute_offset += bytes;
		debug_printf("READ %d bytes of junk.\n",bytes);
	}
}

akaifile *akai_read_directory(akaisrc *src, akaivolume *volume, int *count)
{
	char name[13];
	char xchars[5], xend[5] = {0,0,0,0,0}, *xcont="    ";
	int x,offset,c = 0;
	akaifile *result = 0;

	akai_read_seek(src,(volume->start+src->partitionoffset)*AKAI_BLOCKSIZE);
	akai_read_seek(src,(volume->start+src->partitionoffset)*AKAI_BLOCKSIZE);

	xchars[4] = 0;

	for(;;)
	{
		offset = src->absolute_offset;
		akai_read_name(src,12,name);

		for(x=0;x<4;x++)
			xchars[x] = akai_read_char(src);

		if(strcmp(xchars,xcont) != 0)
		{
			if(memcmp(xchars,xend,5) == 0)
			{
				debug_printf("END\n");
				*count = c;
				return result;
			}
			else
			{
				debug_printf("expected $20202020, got %d %d %d %d=$%2x%2x%2x%2x\n",
								xchars[0],xchars[1],xchars[2],xchars[3],
								xchars[0],xchars[1],xchars[2],xchars[3]);
				debug_printf("BREAK\n");
				*count = 0;
				return 0;
			}
		}
		result = (akaifile *)realloc(result, sizeof(akaifile)*(c+1));

		debug_printf("%06x: ",offset);
		debug_printf("FILE %s\n",name);
		strcpy(result[c].name,name);

		result[c].type = akai_read_char(src);
		switch(result[c].type) {
			case 'p': debug_printf("        PROGRAM\n");	/* 0x70 */
				  break;
			case 's': debug_printf("        SAMPLE\n");	/* 0x73 */
				  break;
			default:  debug_printf("        ??? UNKOWN ??? (type %2x)\n",
							  								result[c].type);
		}
		result[c].size = akai_read_long3(src);
		debug_printf("        size %ld $0x%06lx\n",result[c].size,
											result[c].size);
		result[c].start = akai_read_uint(src);
		debug_printf("        start block: %d $0x%04x => offset $%06x\n",
		   result[c].start, result[c].start,result[c].start*AKAI_BLOCKSIZE);

		result[c].id = akai_read_uint(src);
		if(result[c].id != 0x41e)	// S1000 ID?
		{
			debug_printf("expected 0x41e here, got $%04x\n",result[c].id);
		}

		c++;
	}
}

void akai_read_volume_map(akaisrc *src)
{
	int i;

	for(i=0;i<src->volumecount;i++)
	{
		akai_read_name(src,12,src->volume[i].name);
		src->volume[i].type = akai_read_uint(src);
		src->volume[i].start = akai_read_uint(src);

		debug_printf("start %7d $%04x (block): %s type %d\n",
					src->volume[i].start,src->volume[i].start,
					src->volume[i].name,src->volume[i].type);
	}
}

void akai_read_name(akaisrc *src, int len, char *name)
{
	int i;

	for(i=0;i<len;i++)
	{
		src->absolute_offset++;
		name[i] = akai2asc(fgetc(src->file));
	}
	name[len] = 0;
}

int akai2asc(int c)
{
	if(c >= 0 && c <= 9)	// AKAI 0-9
	{
		c += '0';
	}
	else if(c >= 11 && c <= 36)
	{
		c += 'a' - 11;
	}
	else
	{
		switch(c)
		{
			case 10: c = '_';		// normally mapped to space
				break;
			case 37: c = '#';
				break;
			case 38: c = '+';
				break;
			case 39: c = '-';
				break;
			case 40: c = '.';
				break;
			default: c = '?';
		}
	}
	return c;
}

int akai_read_uint(akaisrc *src)
{
	int low = fgetc(src->file);
	int high = fgetc(src->file);
	assert(low >= 0 && high >= 0);

	src->absolute_offset+=2;
	return (low + high*256);
}

char akai_read_char(akaisrc *src)
{
	int c = fgetc(src->file);
	assert(c >= 0);
	src->absolute_offset++;
	return c;
}

long akai_read_long3(akaisrc *src)
{
	int low = fgetc(src->file);
	int high = fgetc(src->file);
	int highest = fgetc(src->file);

	assert(low >= 0 && high >= 0 && highest >= 0);
	src->absolute_offset+=3;

	return (low + high*256 + highest*256*256);
}
