// KreateCD - CD recording software for the K desktop environment
//
// 1999 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#include "IsoOptions.h"
#include "IsoImage.h"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>

#include <qlayout.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qlabel.h>
#include <qbuttongroup.h>
#include <qlineedit.h>

#include <strings.h>
#include <stdio.h>

extern KLocale *locale;

ISOOptions::ISOOptions( QWidget *parent, const char *name,ISOImage *track):QWidget(parent,name)
 {
  KConfig *config;
  QRect DefaultRect(100,50,500,300);
  QVBoxLayout *mainlayout;
  QHBoxLayout *cbutlay;
  QButtonGroup *isoopt,*presetopt,*stringopt;
  QGridLayout *isogrid,*presetgrid,*stringgrid;
  QPushButton *okbutton,*applybutton,*cancelbutton;

  if (track!=0) 
   {
    iso=track;
    configMode=false;
   }
   else
   {
    iso=new ISOImage();
    configMode=true;
   }

  this->setCaption(locale->translate("KreateCD - ISO image options"));

  config=kapp->getConfig();

  mainlayout=new QVBoxLayout(this);

  mainlayout->addStretch(1);

  // build string lines

  stringopt=new QButtonGroup(locale->translate("ISO strings"),this);
  stringopt->setAlignment(AlignHCenter);
  stringopt->setMinimumSize(300,60);
  mainlayout->addWidget(stringopt);
  mainlayout->addStretch(1);

  stringgrid=new QGridLayout(stringopt,10,2,20,0);



  appIDLabel=new QLabel(locale->translate("Application ID"),stringopt);
  appIDLabel->setFixedSize(appIDLabel->sizeHint());
  appIDEdit=new QLineEdit(stringopt,0);
  appIDEdit->setMaxLength(128);
  appIDEdit->setMinimumSize(appIDEdit->sizeHint());
  stringgrid->addWidget(appIDLabel,1,0,AlignCenter);
  stringgrid->addWidget(appIDEdit,1,1);
#if QT_VERSION >= 200
  connect(appIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setApplicationID(const QString &)));
#else
  connect(appIDEdit,SIGNAL(textChanged(const char *)),
          iso,SLOT(setApplicationID(const char *)));
#endif

  sysIDLabel=new QLabel(locale->translate("System ID"),stringopt);
  sysIDLabel->setFixedSize(sysIDLabel->sizeHint());
  sysIDEdit=new QLineEdit(stringopt,0);
  sysIDEdit->setMaxLength(32);
  sysIDEdit->setMinimumSize(sysIDEdit->sizeHint());
  stringgrid->addWidget(sysIDLabel,3,0,AlignCenter);
  stringgrid->addWidget(sysIDEdit,3,1);
#if QT_VERSION >= 200
  connect(sysIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setSystemID(const QString &)));
#else
  connect(sysIDEdit,SIGNAL(textChanged(const char *)),
          iso,SLOT(setSystemID(const char *)));
#endif

  volIDLabel=new QLabel(locale->translate("Volume ID"),stringopt);
  volIDLabel->setFixedSize(volIDLabel->sizeHint());
  volIDEdit=new QLineEdit(stringopt,0);
  volIDEdit->setMaxLength(32);
  volIDEdit->setMinimumSize(volIDEdit->sizeHint());
  stringgrid->addWidget(volIDLabel,5,0,AlignCenter);
  stringgrid->addWidget(volIDEdit,5,1);
#if QT_VERSION >= 200
  connect(volIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setVolumeID(const QString &)));
#else
  connect(volIDEdit,SIGNAL(textChanged(const char *)),
          iso,SLOT(setVolumeID(const char *)));
#endif

  preIDLabel=new QLabel(locale->translate("Preparer"),stringopt);
  preIDLabel->setFixedSize(preIDLabel->sizeHint());
  preIDEdit=new QLineEdit(stringopt,0);
  preIDEdit->setMaxLength(128);
  preIDEdit->setMinimumSize(preIDEdit->sizeHint());
  stringgrid->addWidget(preIDLabel,7,0,AlignCenter);
  stringgrid->addWidget(preIDEdit,7,1);
#if QT_VERSION >= 200
  connect(preIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setPreparerID(const QString &)));
#else
  connect(preIDEdit,SIGNAL(textChanged(const char *)),
          iso,SLOT(setPreparerID(const char *)));
#endif

  pubIDLabel=new QLabel(locale->translate("Publisher"),stringopt);
  pubIDLabel->setFixedSize(pubIDLabel->sizeHint());
  pubIDEdit=new QLineEdit(stringopt,0);
  pubIDEdit->setMaxLength(128);
  pubIDEdit->setMinimumSize(pubIDEdit->sizeHint());
  stringgrid->addWidget(pubIDLabel,9,0,AlignCenter);
  stringgrid->addWidget(pubIDEdit,9,1);
#if QT_VERSION >= 200
  connect(pubIDEdit,SIGNAL(textChanged(const QString &)),
          iso,SLOT(setPublisherID(const QString &)));
#else
  connect(pubIDEdit,SIGNAL(textChanged(const char *)),
          iso,SLOT(setPublisherID(const char *)));
#endif


  // build ISO options presets

  presetopt=new QButtonGroup(locale->translate("ISO option presets"),this);
  presetopt->setAlignment(AlignHCenter);
  presetopt->setMinimumSize(300,60);
  mainlayout->addWidget(presetopt);
  mainlayout->addStretch(1);

  presetgrid=new QGridLayout(presetopt,5,5,20,0);

  unixButton=new QRadioButton(locale->translate("Unix RockRidge extensions"),presetopt);
  unixButton->setFixedSize(unixButton->sizeHint());
  presetgrid->addWidget(unixButton,1,1);
  connect(unixButton,SIGNAL(clicked()),this,SLOT(doUnix()));

  unixWinButton=new QRadioButton(locale->translate("Unix RockRidge + Windows joliet"),presetopt);
  unixWinButton->setFixedSize(unixWinButton->sizeHint());
  presetgrid->addWidget(unixWinButton,3,1);
  connect(unixWinButton,SIGNAL(clicked()),this,SLOT(doUnixWin()));

  winButton=new QRadioButton(locale->translate("Windows joliet extensions"),presetopt);
  winButton->setFixedSize(winButton->sizeHint());
  presetgrid->addWidget(winButton,1,3);
  connect(winButton,SIGNAL(clicked()),this,SLOT(doWin()));

  msDOSButton=new QRadioButton(locale->translate("MS-DOS standard"),presetopt);
  msDOSButton->setFixedSize(msDOSButton->sizeHint());
  presetgrid->addWidget(msDOSButton,3,3);
  connect(msDOSButton,SIGNAL(clicked()),this,SLOT(doMSDos()));

   {
    int i;
    for (i=0;i<5;i+=2)
     {
      presetgrid->setColStretch(i,(i!=0)?4:1);
     }
    for (i=0;i<5;i+=2)
     {
      presetgrid->setRowStretch(i,1);
     }
   }

  // build ISO options checkboxes

  isoopt=new QButtonGroup(locale->translate("ISO options"),this);
  isoopt->setAlignment(AlignHCenter);
  isoopt->setMinimumSize(300,60);
  mainlayout->addWidget(isoopt);


  isogrid=new QGridLayout(isoopt,11,5,20,0);

  tdotButton=new QCheckBox(locale->translate("Omit trailing dots"),isoopt);
  tdotButton->setFixedSize(tdotButton->sizeHint());
  isogrid->addWidget(tdotButton,1,1);
  connect(tdotButton,SIGNAL(toggled(bool)),iso,SLOT(setOmitTrailingDot(bool)));

  noDeepButton=new QCheckBox(locale->translate("No deep directory relocation"),isoopt);
  noDeepButton->setFixedSize(noDeepButton->sizeHint());
  isogrid->addWidget(noDeepButton,3,1);
  connect(noDeepButton,SIGNAL(toggled(bool)),iso,SLOT(setNoDeepRelocation(bool)));

  longNameButton=new QCheckBox(locale->translate("Long ISO-9660 names"),isoopt);
  longNameButton->setFixedSize(longNameButton->sizeHint());
  isogrid->addWidget(longNameButton,1,3);
  connect(longNameButton,SIGNAL(toggled(bool)),iso,SLOT(setLongISONames(bool)));

  leadDotButton=new QCheckBox(locale->translate("Allow leading dots"),isoopt);
  leadDotButton->setFixedSize(leadDotButton->sizeHint());
  isogrid->addWidget(leadDotButton,3,3);
  connect(leadDotButton,SIGNAL(toggled(bool)),iso,SLOT(setAllowLeadingDot(bool)));

  omitVerButton=new QCheckBox(locale->translate("Omit ISO-9660 versions"),isoopt);
  omitVerButton->setFixedSize(omitVerButton->sizeHint());
  isogrid->addWidget(omitVerButton,5,1);
  connect(omitVerButton,SIGNAL(toggled(bool)),iso,SLOT(setOmitVersions(bool)));

  transTabButton=new QCheckBox(locale->translate("Create TRANS.TBL"),isoopt);
  transTabButton->setFixedSize(transTabButton->sizeHint());
  isogrid->addWidget(transTabButton,5,3);
  connect(transTabButton,SIGNAL(toggled(bool)),iso,SLOT(setMakeTransTab(bool)));

  rockButton=new QCheckBox(locale->translate("Full RockRidge extensions"),isoopt);
  rockButton->setFixedSize(rockButton->sizeHint());
  isogrid->addWidget(rockButton,7,1);
  connect(rockButton,SIGNAL(toggled(bool)),iso,SLOT(setWithRockRidge(bool)));

  anoRockButton=new QCheckBox(locale->translate("Anonymous RockRidge extensions"),isoopt);
  anoRockButton->setFixedSize(anoRockButton->sizeHint());
  isogrid->addWidget(anoRockButton,7,3);
  connect(anoRockButton,SIGNAL(toggled(bool)),iso,SLOT(setWithAnonymousRockRidge(bool)));

  jolietButton=new QCheckBox(locale->translate("Joliet extensions"),isoopt);
  jolietButton->setFixedSize(jolietButton->sizeHint());
  isogrid->addWidget(jolietButton,9,1);
  connect(jolietButton,SIGNAL(toggled(bool)),iso,SLOT(setWithJoliet(bool)));

  allButton=new QCheckBox(locale->translate("Include ALL files"),isoopt);
  allButton->setFixedSize(allButton->sizeHint());
  isogrid->addWidget(allButton,9,3);
  connect(allButton,SIGNAL(toggled(bool)),iso,SLOT(setAllFiles(bool)));

   {
    int i;
    for (i=0;i<5;i+=2)
     {
      isogrid->setColStretch(i,(i!=0)?4:1);
     }
    for (i=0;i<11;i+=2)
     {
      isogrid->setRowStretch(i,1);
     }
   }


  // make the rest
 
  mainlayout->addSpacing(5);
  mainlayout->addStretch(1);

  if (!configMode)
   {
    okbutton=new QPushButton(locale->translate("OK"),this);
    okbutton->setFixedSize(okbutton->sizeHint());
    mainlayout->addWidget(okbutton,1,AlignCenter);
    connect(okbutton,SIGNAL(clicked()),this,SLOT(hide()));
   }
   else
   {
    cbutlay=new QHBoxLayout();
    mainlayout->addLayout(cbutlay,1);

    cbutlay->addSpacing(5);
    cbutlay->addStretch(1);

    okbutton=new QPushButton(locale->translate("OK"),this);
    okbutton->setFixedSize(okbutton->sizeHint());
    cbutlay->addWidget(okbutton,1,AlignCenter);
    connect(okbutton,SIGNAL(clicked()),this,SLOT(configOK()));

    cbutlay->addStretch(1);
    applybutton=new QPushButton(locale->translate("Apply"),this);
    applybutton->setFixedSize(applybutton->sizeHint());
    cbutlay->addWidget(applybutton,1,AlignCenter);
    connect(applybutton,SIGNAL(clicked()),this,SLOT(configApply()));

    cbutlay->addStretch(1);
    cancelbutton=new QPushButton(locale->translate("Cancel"),this);
    cancelbutton->setFixedSize(cancelbutton->sizeHint());
    cbutlay->addWidget(cancelbutton,1,AlignCenter);
    connect(cancelbutton,SIGNAL(clicked()),this,SLOT(configCancel()));
    
    cbutlay->addStretch(1);
    cbutlay->addSpacing(1);
   }

  isogrid->activate();
  presetgrid->activate();
  mainlayout->activate();

  updateWindow();
  connect(iso,SIGNAL(imageChanged()),this,SLOT(updateWindow()));

  config->setGroup("Geometry");
  this->setGeometry(config->readRectEntry("IsoOptWindowGeo",&DefaultRect));
 }


ISOOptions::~ISOOptions(void)
 {
  KConfig *config;
  if (configMode==true)
   {
    delete iso;
   }
   else
   {
    QRect SaveRect(this->pos(),this->size());
    config=kapp->getConfig();
    config->setGroup("Geometry");
    config->writeEntry("IsoOptWindowGeo",SaveRect);
   }
 }

void ISOOptions::hide()
 {
  emit closed();
 }

void ISOOptions::updateWindow(void)
 {
  appIDEdit->setText(iso->getApplicationID());
  preIDEdit->setText(iso->getPreparerID());
  pubIDEdit->setText(iso->getPublisherID());
  volIDEdit->setText(iso->getVolumeID());
  sysIDEdit->setText(iso->getSystemID());

  switch (iso->getOptionsDefault())
   {
    case ISOImage::Unix:
      unixButton->setChecked(true);
      break;
    case ISOImage::UnixWin:
      unixWinButton->setChecked(true);
      break;
    case ISOImage::Win:
      winButton->setChecked(true);
      break;
    case ISOImage::MSDos:
      msDOSButton->setChecked(true);
      break;
    default:
      unixButton->setChecked(false);
      unixWinButton->setChecked(false);
      winButton->setChecked(false);
      msDOSButton->setChecked(false);
      break;
   }

  tdotButton->setChecked(iso->getOmitTrailingDot());
  noDeepButton->setChecked(iso->getNoDeepRelocation());
  longNameButton->setChecked(iso->getLongISONames());
  leadDotButton->setChecked(iso->getAllowLeadingDot());
  omitVerButton->setChecked(iso->getOmitVersions());
  rockButton->setChecked(iso->getWithRockRidge());
  anoRockButton->setChecked(iso->getWithAnonymousRockRidge());
  transTabButton->setChecked(iso->getMakeTransTab());
  jolietButton->setChecked(iso->getWithJoliet());
  allButton->setChecked(iso->getAllFiles());
 }

void ISOOptions::doUnix(void)
 {
  iso->setOptionsDefault(ISOImage::Unix);
 }

void ISOOptions::doUnixWin(void)
 {
  iso->setOptionsDefault(ISOImage::UnixWin);
 }

void ISOOptions::doWin(void)
 {
  iso->setOptionsDefault(ISOImage::Win);
 }

void ISOOptions::doMSDos(void)
 {
  iso->setOptionsDefault(ISOImage::MSDos);
 }

void ISOOptions::configOK(void)
 {
  configApply();
  hide();
 }

void ISOOptions::configApply(void)
 {
  KConfig *config;
  config=kapp->getConfig();
  config->setGroup("ISOOptions");
  iso->saveConfig(config);
 }

void ISOOptions::configCancel(void)
 {
  hide();
 }
