

#include "includes.h"

#include "kgo_widget.moc"
#include "go.h"
#include "go_view.h"
#include "go_board.h"
#include "go_player.h"
#include "computer_player.h"
#include "network_player.h"
#include "view_player.h"
#include "view_2player.h"
#include "dialog.h"
#include "netdialog.h"
#include "count.h"


/* ##########################
 * #	class KGoWidget     #
 * ##########################
 */



void KGoWidget::newgame()
{
	#ifdef DEBUG
	printf("KGoWidget - newgame()\n") ;
	#endif

	if ( endgame() )
	{
		NewGame *newGameDialog = new NewGame( this ) ;

		if ( newGameDialog->exec() )
		{
			int g = newGameDialog->getGameType() ;
			int xS = newGameDialog->getXSize() ;
			int yS = newGameDialog->getYSize() ;
		
			board->newGame( xS, yS ) ;

			switch( g )
			{
				case COMPUTER_GAME:	computerGame();
							break ;
				case TWO_PLAYER_GAME:	twoplayerGame();
							break ;
				case NETWORK_GAME:	networkGame() ;
							break ;
				default:
				{
					KMsgBox::message( this,
						i18n("New game"),
						i18n(
							"No such game type.")
							 );
					break ;
				}
 			}

			board->startGame() ;

			gameType( g ) ;
			// TODO: make this general using KConfig*
			if ( view != 0 )
				view->legenda( blegenda ) ;
 		}

		delete newGameDialog ;
  	}

	board->pushPlayer() ;
	emit boardStateChange( NEWGAME ) ;
	emit boardStateChange( PLAYING ) ;
}

void KGoWidget::gameEnded()
{
	#ifdef DEBUG
	printf("KGoWidget - gameEnded()\n") ;
	#endif

	GameOver *g = new GameOver( this, 0 ) ;
	
	switch ( g->exec() )
	{
		case COUNT:	countScores() ;
				break ;
		case NEWGAME:	newgame() ;
				break ;
		default:	break;
	} 
}

void KGoWidget::countScores()
{
	static CountDialog *d=0 ;

	if (d!=0)
		delete d ;

	d = new CountDialog( board, this, 0 ) ;

	switch( gameType() )
	{
		case COMPUTER_GAME:
			connect( view,
				SIGNAL(positionClicked(int,int,int)),
				d,
				SLOT(receiveInput(int,int,int)) ) ;
		 	break ;
		case TWO_PLAYER_GAME:	twoplayerGame();
			connect( view,
				SIGNAL(positionClicked(int,int,int)),
				d,
				SLOT(receiveInput(int,int,int)) ) ;
			break ;
		case NETWORK_GAME:
			notImpl() ;
			break ;
		default:	error( "unknown game type" ) ;
				break ;
	}

	d->exec() ;

}

bool KGoWidget::endgame()
{
	#ifdef DEBUG
	printf("KGoWidget - endgame()\n") ;
	#endif

	if( view == 0 )
	{
		// no game has been started yet...
		return true ;
	} 

	int q = QMessageBox::information(this,
		klocale->translate("Start new game"),
		klocale->translate(
		"Starting a new game will destroy your old game.\n"
		"Are you sure?\n" ),
		klocale->translate("Yes"),
		klocale->translate("No")
			) ;

	if ( q==0 )
	{
		// TODO: fix
	
		moveList->removeAllMoves() ;

		// we start a new game, so we want to destroy
		// view, white and black properly
		if( view!=0)
			delete view ; 
		view = 0 ;

		board->removeWhitePlayer() ;
		board->removeBlackPlayer() ;

		delete white ;
		delete black ;
		white = 0 ; black = 0 ;

		return true ;
	}
	else	return false ;
}

bool KGoWidget::saveAbleGame()
{
	return board->saveAbleGame() ;
}

void KGoWidget::setPlayers( KGoPlayer *w, KGoPlayer *b )
{
	board->setWhitePlayer( w ) ;
	board->setBlackPlayer( b ) ;
}

void KGoWidget::computerGame()
{
	#ifdef DEBUG
	printf("KGoWidget - computerGame()\n") ;
	#endif

	int vraag =
		QMessageBox::information(this,
			klocale->translate("Color selection" ),
			klocale->translate(
				"Please choose a color to play:\n" ),
			klocale->translate("&White"),
			klocale->translate("&Black"),
			0 );

	if ( vraag==0 )
	{
		computerCGame( BLACK ) ;
	}
	else	computerCGame( WHITE ) ;
}

void KGoWidget::computerCGame( int comp_player )
{
	computer_color = comp_player ;

	if( comp_player==BLACK )
	{
		black = new KGoComputer( BLACK, board, this, 0) ;
		view = new KViewPlayer( WHITE, board, this, 0 ) ;
		white = ((KViewPlayer*)view)->getPlayer() ;
	}
	else
	{
		white = new KGoComputer( WHITE, board, this, 0) ;
		view = new KViewPlayer( BLACK, board, this, 0 ) ;
		black = ((KViewPlayer*)view)->getPlayer() ;
	}

	((KViewPlayer*)view)->setInformation( i18n("Computer game") ) ;
	setPlayers( white, black ) ; 
	setView( view ) ;
}

void KGoWidget::twoplayerGame()
{
	#ifdef DEBUG
	printf("KGoWidget - twoplayerGame()\n") ;
	#endif

	view = new KView2Player( board, this, 0 ) ;
	white = ((KView2Player *)view)->getWhitePlayer() ;
	black = ((KView2Player *)view)->getBlackPlayer() ;

	setPlayers( white, black ) ;
	setView( view ) ;
}
 

void KGoWidget::networkGame()
{
	NetDialog *d = new NetDialog( this ) ;

	if ( d->exec() )
	{
		KNetObject *n = d->getNetObject() ;

		if ( d->isServer() )
		{
			char s[MAX_MESSAGE_SIZE] ;
			snprintf( s, MAX_MESSAGE_SIZE,
				"%3d%3d%3d", BOARDSIZE,
				board->getXWidth(), board->getYWidth() ) ;
			n->sendMessage( s ) ;

			view = new KViewPlayer( WHITE,
					board, this, 0 ) ;
			white = ((KViewPlayer*)view)->getPlayer() ;
			black = new KNetworkPlayer( d->getNetObject(),
					BLACK, board, this, 0) ;
			((KViewPlayer*)view)->setInformation(
				i18n("Network game - Server") ) ;
			setPlayers( white, black ) ;
			setView( view ) ;
		}
		else
		{
			connect( n, SIGNAL(messageArrived()),
				SLOT(network_client_receiver()) ) ;
 			nobject = n ;

			TimeDialog *d = new TimeDialog( i18n(
				"Waiting for server to send data."
					),
					4000, this ) ;
			d->exec() ;
		}
 	}
	else
	{
	}
 	delete d ;
}


void KGoWidget::network_client_receiver()
{
	disconnect( nobject, 0, this, 0 ) ;
	int command, xS=19, yS=19 ;
	char *s = 0 ;

	s = nobject->getMessage() ;
	sscanf( s, "%d", &command ) ;
	if ( command!=BOARDSIZE )
	{
		KMsgBox::message(this, CAPTION,
			i18n("Network error, cannot fix this right now.")  );
		return ;
  	}

	sscanf( s, "%d%d%d", &command, &xS, &yS ) ;

	board->newGame( xS, yS ) ;
	view = new KViewPlayer( BLACK,
			board, this, 0 ) ;
	white = new KNetworkPlayer( nobject,
			WHITE, board, this, 0) ;
	black = ((KViewPlayer*)view)->getPlayer() ;
	((KViewPlayer*)view)->setInformation(
		i18n("Network game - Client") ) ;

	setPlayers( white, black ) ;
	setView( view ) ;

	white->opponentMoved( PASS, PASS ) ;
	emit boardStateChange( NEWGAME ) ;
	emit boardStateChange( PLAYING ) ;
}

void KGoWidget::loadGame()
{
	//debug("KGoWidget-loadGame()");

	if( ! endgame() )
		return ;

	QString s = KFileDialog::getOpenFileName( 0, "*.go", this, 0 ) ;
	QFile f( s.data() ) ;
	QDataStream d ;

	if ( f.open( IO_ReadOnly ) )
	{
		d.setDevice( &f ) ;

		/* check for kgo file */
		char *str ;
		d >> str ;

		if ( strcmp( str, FILE_KGOWIDGET) != 0 )
		{
			KMsgBox::message(this,
				klocale->translate("Unable to load game"),
				klocale->translate(
					"Sorry, this file is no Kgo save file."
						)
					);
			return ;
		}	
	
		/* version number */
		d >> str ;
		/* game type */
		d >> game_type ;

		switch( game_type )
		{
			case COMPUTER_GAME:	loadComputerGame( d ) ;
						break ;
			case TWO_PLAYER_GAME:	loadTwoPlayerGame( d ) ;
						break ;
			case NETWORK_GAME:	loadNetworkGame( d ) ;
						break ;

		}

		f.close() ;
	}

}

void KGoWidget::loadComputerGame( QDataStream &d )
{
	d >> computer_color ;
	computerCGame( computer_color ) ;

	/* load game data */
	board->load( d ) ;
	white->load( d, board ) ;
	black->load( d, board ) ;

	board->resumeGame() ;
}

void KGoWidget::loadTwoPlayerGame( QDataStream &d )
{
	// debug("loadTwoPlayerGame()") ;

	twoplayerGame() ;
	/* load game data */
	board->load( d ) ;
	white->load( d, board ) ;
	black->load( d, board ) ;

	board->resumeGame() ;
}

void KGoWidget::loadNetworkGame( QDataStream & )
{
//	networkgame() ;
	/* load game data */
//	board->load( d ) ;
//	white->load( d, board ) ;
//	black->load( d, board ) ;
	notImpl() ;
}

void KGoWidget::saveComputerGame( QDataStream &d )
{
	d << computer_color ;

	/* save game data */
	board->save( d ) ;
	white->save( d ) ;
	black->save( d ) ;
}

void KGoWidget::saveTwoPlayerGame( QDataStream &d )
{
	/* save game data */
	board->save( d ) ;
	white->save( d ) ;
	black->save( d ) ;
}

void KGoWidget::saveNetworkGame( QDataStream & )
{
	/* save game data */
//	board->save( d ) ;
//	white->save( d ) ;
//	black->save( d ) ;
	notImpl() ;
}

void KGoWidget::saveGame()
{
	if ( ! saveAbleGame() )
	{
		KMsgBox::message(this,
			klocale->translate("Unable to save game"),
			klocale->translate(
				"Sorry, this type of game cannot be saved."
					)
				);
 		return ;
 	}
	
	QString s = KFileDialog::getSaveFileName( 0, "*.go", this, 0 ) ;

	if ( s.isEmpty() )
		return ;

	QFile f( s.data() ) ;

	if ( f.open( IO_WriteOnly ) )
	{
		// printf("opened file %s for writing\n", s.data() ) ;

		QDataStream d( &f ) ;
		d << (const char *)FILE_KGOWIDGET ;
		/* version number */
		d << (const char *)VERSION ;
		/* game type */
		d << (int)game_type ;

		switch( game_type )
		{
			case COMPUTER_GAME:	saveComputerGame( d ) ;
						break ;
			case TWO_PLAYER_GAME:	saveTwoPlayerGame( d ) ;
						break ;
			case NETWORK_GAME:	saveNetworkGame( d ) ;
						break ;

		}

		f.close() ;
	}

	KMsgBox::message(this,
		klocale->translate("Saving game"),
		klocale->translate(
		"This is still experimental. Don't expect this\n"
		"to have worked.\n\n"
		"Your game has been saved"
				) 
		) ;
}

void KGoWidget::toggleLegenda()
{
	if ( blegenda )
		blegenda = false ;
	else	blegenda = true ;

	if ( view != 0 )
		view->legenda( blegenda ) ;
}

void KGoWidget::legenda( bool b )
{
	blegenda = b ;
	if ( view != 0 ) 
		view->legenda( b ) ;
	// resize the widget
	resize() ;
}

bool KGoWidget::legenda()
{
	return blegenda ;
}

void KGoWidget::toggleMovesList()
{
	bool b = movesList() ;
	movesList( ! b ) ;
}

void KGoWidget::movesList( bool b )
{
	moves_list = b ;
	// resize the widget
	resize() ;
}

bool KGoWidget::movesList()
{
	return moves_list ;
}

void KGoWidget::loadconfig( KConfig *config )
{
	config->setGroup( FILE_KGOWIDGET ) ;

	legenda( config->readNumEntry( LEGENDA_STRING, 1) == 1 ) ;
	movesList( config->readNumEntry( MOVE_LIST_STRING, 0 )==1 ) ;
}

void KGoWidget::saveconfig( KConfig *config )
{
	config->setGroup( FILE_KGOWIDGET ) ;

	config->writeEntry( LEGENDA_STRING, legenda() ) ;
	config->writeEntry( MOVE_LIST_STRING, movesList() ) ;
}

void KGoWidget::quitapp()
{
}


void KGoWidget::setView( KGoView *w )
{
	connect( this, SIGNAL( boardStateChange(int) ),
			view, SLOT( boardStateChange(int) ) ) ;
	connect( board, SIGNAL( playerMoved(int,int,int)),
			view, SLOT( receiveMove(int,int,int)) ) ;
	view = w ;
	view->show() ;
	resizeEvent( 0 ) ;
}

void KGoWidget::setBoard( KGoBoard *b )
{
	board = b ;
	connect( board, SIGNAL( boardStateChange(int)),
			this, SIGNAL( boardStateChange(int)) ) ;
	connect( board, SIGNAL( boardStateChange(int)),
			this, SLOT( status(int)) ) ;
	connect( board, SIGNAL(playerMoved(int, int, int)),
			this, SLOT(player_moved(int,int,int)) ) ;
}

void KGoWidget::paintEvent( QPaintEvent* )
{
	if( view!=0 )
		view->repaint() ;
	if( moveList!=0 )
		moveList->repaint() ;
}

void KGoWidget::resize()
{
	resizeEvent( 0 ) ;
}

void KGoWidget::resizeEvent( QResizeEvent * )
{
	QSize s = size() ;
	int ml_width = 100 ;

	if( movesList() )
	{
		list_title->show() ;
		moveList->show() ;
		ml_width= 100 ;
	}
	else
	{
		list_title->hide() ;
		moveList->hide() ;
		ml_width= 0 ;
	}

	if ( view !=0 )
		view->setGeometry( 0, 0,
			s.width()-ml_width, s.height() ) ;

	list_title->setGeometry( s.width()-ml_width+5,
			5, ml_width-10, 20 ) ;
	moveList->setGeometry( s.width()-ml_width+5, 30, 
			ml_width-10, s.height() - 40 ) ;
}

void KGoWidget::status( int s ) 
{
	switch( s )
	{
		case ENDED:	gameEnded() ;
				break ;
		default:	break ;
	}
}

void KGoWidget::player_moved( int x, int y, int )
{
	QString tr = KGoBoard::translateMove( x, y ) ;
	moveList->addMove( tr ) ;
}

KGoWidget:: KGoWidget ( QWidget *parent, const char* name )
	: QWidget ( parent, name ),
	board(0), white( 0 ), black ( 0 ),  game_type( NO_GAME ), 
	view( 0 ), moves_list( false )
{
	board = new KGoBoard() ;
	setBoard( board ) ;
 
	list_title = new QLabel( klocale->translate("Moves"),
			this, "list_title" ) ;  
	moveList = new KMoveList( 12, this, 0 ) ;

	show() ;
}

KGoWidget::~KGoWidget()
{
	// ?????
	delete board ;
	delete view ;

	delete list_title ;
	delete moveList ;
}

void KGoWidget::gameType(int g )
{
	game_type = g ;
}

int KGoWidget::gameType()
{
	return game_type ;
}

KGoBoard* KGoWidget::getGoBoard()
{
	return board ;
}

KGoView* KGoWidget::getGoView()
{
	return view ;
}

void KGoWidget::error( char * s )
{
	QString tmp = "Internal error: " ;
	tmp.append( s ) ;

	KMsgBox::message(0, CAPTION,
		tmp.data() ) ;
}

void KGoWidget::notImpl()
{
	KMsgBox::message(this, CAPTION,
	klocale->translate("Sorry, this function hasn't been implemented yet...")  );
}
