/* board_view.cpp
 *
 * Pieter Eendebak <pte@ddsw.nl>
 */

#include "includes.h"
#include "board_view.moc"
#include "go_board.h"

#define LEGENDA_BORDER 30 
#define FIELD 50 
#define BORDER (int)(FIELD/2) 

KGoBoardView::KGoBoardView( KGoBoard *b=0, QWidget *parent, const char *name )
	: QWidget ( parent, name, 0 ),
		xSize(0), ySize(0),
		field( 100 ), border( 25 ),
		lastmove_x( PASS ), lastmove_y( PASS ),
		lastmove_p( WHITE )
{
	painter = new QPainter() ;
	map = new QPixmap() ;

/*	if( parent!=0 )
	{
		QFontInfo finfo = parent->fontInfo() ;
		QFont paint_font = QFont( finfo.family(),
			40, QFont::Bold ) ;
	}
*/

	legenda_border = LEGENDA_BORDER ;
	border = BORDER ;
	field = FIELD ;

	line_color = QColor( black ) ;

	setBoard( b ) ;

	legenda( false ) ;
	redraw() ;
	show() ;
} 

KGoBoardView::~KGoBoardView()
{

}    

void KGoBoardView::setBoard( KGoBoard *b )
{
	if ( b != 0 )
		board = b ;
	else b = new KGoBoard() ;

	connect( b, SIGNAL(boardSetup()),
			this, SLOT(setupBoardData()) ) ;
	setupBoardData() ;
	redraw() ;
}

void KGoBoardView::setupBoardData()
{
	xSize = board->getXWidth() ;
	ySize = board->getYWidth() ;
}

void KGoBoardView::mouseReleaseEvent( QMouseEvent *mouse )
{
	QPoint p = screenToPosition( mouse->x(),
			mouse->y() ) ;

	//printf("position: %d %d\n", p.x(), p.y() ) ;
	emit positionClicked( p.x(), p.y(), mouse->button() ) ;
}

void KGoBoardView::legenda( bool b )
{
	blegenda = b ;
	if ( blegenda==true )
		legenda_border = 30 ;
	else	legenda_border = 0 ;

	redraw() ;
}

bool KGoBoardView::legenda()
{
	return blegenda ;
}

void KGoBoardView::lastMove( int x, int y, int p )
{
	
	lastmove_x = x ;
	lastmove_y = y ;
	lastmove_p = p ;
	redraw() ;
} 

void KGoBoardView::redraw()
{
	map->resize( size() ) ;
	paintEvent( 0 ) ;
}

int KGoBoardView::xLogicalSize()
{
	return legenda_border + 2*border + xBoardSize()*field ;
}

int KGoBoardView::yLogicalSize()
{
	return legenda_border + 2*border + yBoardSize()*field ;
}

void KGoBoardView::screenToLogicalRatios( double &x, double &y )
{
	QSize real = size() ;
	double xratio = ((double)xLogicalSize()) /
				(double)real.width() ;
	double yratio = ((double)yLogicalSize()) /
				(double)real.height() ;

	x = xratio ;
	y = yratio ;
}


QPoint KGoBoardView::screenToLogical( int x, int y )
{
	double xratio=0, yratio=0 ;
	screenToLogicalRatios( xratio, yratio ) ;

	double lx = x * xratio ;
	double ly = y * yratio ;

	QPoint p = QPoint( int(lx), int(ly) ) ;
	return p ;
}

QPoint KGoBoardView::logicalToScreen( int x, int y )
{
	double xratio=0, yratio=0 ;
	screenToLogicalRatios( xratio, yratio ) ;

	double lx = x * (1/xratio) ;
	double ly = y * (1/yratio) ;

	QPoint p = QPoint( int(lx), int(ly) ) ;
	return p ;
}

int KGoBoardView::xPositionToLogical( int xp )
{
	return (int)(legenda_border + border + field*xp) ;
}

int KGoBoardView::yPositionToLogical( int yp )
{
	return (int)(legenda_border + border + field*yp) ;
}

int KGoBoardView::xLogicalToPosition( int v )
{
	v -=legenda_border ;
	v -=border ;
	int half = (int)(field/2) ;

	int num = (int)((v+half)/field) ;
	// could be used for border problems...
	// int precision = (int)(v%field) ;
	if ( num < 0 || num>=xBoardSize() )
		return -1 ;
	return num ;
}
int KGoBoardView::yLogicalToPosition( int v )
{
	v -=legenda_border ;
	v -=border ;
	int half = (int)(field/2) ;

	int num = (int)((v+half)/field) ;
	// could be used for border problems...
	// int precision = (int)(v%field) ;
	if ( num < 0 || num>=yBoardSize() )
		return -1 ;
	return num ;
}

QPoint KGoBoardView::screenToPosition( int x, int y )
{
	QPoint p = screenToLogical( x, y ) ;
	return QPoint( xLogicalToPosition( p.x() ),
			yLogicalToPosition( p.y() ) ) ;
}

int KGoBoardView::logicalPieceSize()
{
	return (int)((field/4)*3) ;
}

int KGoBoardView::logicalFieldHeight()
{
	return logicalPieceSize() ;
}
int KGoBoardView::logicalFieldWidth()
{
	return logicalPieceSize() ;
}

void KGoBoardView::drawPiece( int x, int y )
{
	int r = xPositionToLogical( x ) ;
	int s = yPositionToLogical( y ) ;
	int u = logicalFieldWidth() ;
	int v = logicalFieldHeight() ;

	painter->drawEllipse( r-(u/2), s-(v/2), u, v ) ;
}

void KGoBoardView::drawZone( int x, int y )
{
	int r = xPositionToLogical( x ) ;
	int s = yPositionToLogical( y ) ;
	int u = logicalFieldWidth() ;
	int v = logicalFieldHeight() ;

	painter->drawEllipse( r-(u/4), s-(v/4), u/2, v/2 ) ;
}

void KGoBoardView::drawLegenda()
{
	// we use this because we cannot draw text in logical 
	// coordinates :-(
	const int HSHIFT = 5 ;
	const int VSHIFT = 7 ;

	int i, x, y ;
	QString tmp ;
	QPoint p ;

	int xS = xBoardSize() ;
	int yS = yBoardSize() ;

	// we cannot draw with setWindow on logical mode
	// because fonts get screwed
	painter->setWindow( 0, 0, width(), height() );

	for( i=0;i<xS; i++)
	{	
		// horizontal row
		tmp.sprintf("%c", i+65 ) ;
		x= xPositionToLogical( i ) - HSHIFT ;
		y= legenda_border ;

		p = logicalToScreen( x, y ) ;
		painter->drawText( p, tmp.data(), tmp.length() ) ;
	}

	for( i=0;i<yS; i++)
	{	
		// vertical column
		tmp.sprintf("%d", i+1 ) ;
		x=(int)((1/5)*legenda_border) ;
		y=yPositionToLogical( i )+VSHIFT ;


		p = logicalToScreen( x, y ) ;
		painter->drawText( p, tmp.data(), tmp.length() ) ;
	}

	painter->setWindow( 0, 0, xLogicalSize(), yLogicalSize() );
}

void KGoBoardView::drawHoshi()
{
	int xS = xBoardSize() ;
	int yS = yBoardSize() ;

	if( xS!=yS )	return ;

	QPoint *pt ;
	QList<QPoint> p ;
	p.setAutoDelete( true ) ;
	
	switch( xS )
	{
		case 9:		p.append( new QPoint(3,3) ) ;
				p.append( new QPoint(3,7) ) ;
				p.append( new QPoint(7,3) ) ;
				p.append( new QPoint(7,7) ) ;
				break ;
		case 13:	p.append( new QPoint(4,4) ) ;
				p.append( new QPoint(4,10) ) ;
				p.append( new QPoint(10,4) ) ;
				p.append( new QPoint(10,10) ) ;
				p.append( new QPoint(7,7) ) ;
				break ;
		case 19:	p.append( new QPoint(5,5) ) ;
				p.append( new QPoint(5,10) ) ;
				p.append( new QPoint(5,15) ) ;
				p.append( new QPoint(10,5) ) ;
				p.append( new QPoint(10,10) ) ;
				p.append( new QPoint(10,15) ) ;
				p.append( new QPoint(15,5) ) ;
				p.append( new QPoint(15,10) ) ;
				p.append( new QPoint(15,15) ) ;
				break ;
		default:	break ;
	}

	while( ! p.isEmpty() )
	{
		pt = p.getFirst() ;
		drawHoshiPoint( pt->x()-1, pt->y()-1 ) ;
		p.removeFirst() ;
	} 
}

void KGoBoardView::drawHoshiPoint( int x, int y )
{
	painter->setBrush( black ) ;
	drawZone( x, y ) ;
}

void KGoBoardView::drawGrid()
{
	int xS = xBoardSize() ;
	int yS = yBoardSize() ;

	int offset=0 ;
	int begin = yPositionToLogical( 0 ) ;
	int end = yPositionToLogical( yBoardSize()-1 ) ;

	for( int i = 0 ; i<xS; i++ )
	{
		offset = xPositionToLogical( i ) ;
		painter->drawLine( offset, begin, offset,
			end ) ;
 	}

	begin = xPositionToLogical( 0 ) ;
	end = xPositionToLogical( xBoardSize()-1 ) ;

	for( int i = 0 ; i<yS; i++ )
	{
		offset = yPositionToLogical( i ) ;
	
 		painter->drawLine( begin, offset,
			end, offset ) ;
 	}
}

void KGoBoardView::resizeEvent( QResizeEvent *r )
{
	map->resize( size() ) ;
	QWidget::resizeEvent( r ) ;
}

void KGoBoardView::paintEvent( QPaintEvent * )
{
	//debug("KGoBoardView - paintEvent()") ;
 
	// start painting in widget 
	painter->begin( map );
        // define coordinate system
	painter->setWindow( 0, 0, xLogicalSize(), yLogicalSize() );

	// clear the screen
	painter->fillRect( 0,0, xLogicalSize(), 
			yLogicalSize(), backgroundColor() ) ;
	// set pen color
	painter->setPen( line_color );
	// set Font
	painter->setFont( paint_font );

	if ( legenda() )
		drawLegenda() ;
	drawGrid() ;
	drawHoshi() ;

	int x = EMPTY ;
	int xS = xBoardSize() ;
	int yS = yBoardSize() ;

	for( int i=0;i<xS; i++ )
	for( int j=0; j<yS; j++ )
	{
		x = board->get( i, j ) ;
		switch( x )
		{
			case EMPTY:	continue ;
					break ;
			case BLACK:	painter->setBrush( black ) ;
					drawPiece( i, j ) ;
					break ;
			case WHITE:	painter->setBrush( white ) ;
					drawPiece( i, j ) ;
					break ;
			case WHITE_ZONE:
					painter->setBrush( white ) ;
					drawZone( i, j ) ;
					break ;
			case BLACK_ZONE:
					painter->setBrush( black ) ;
					drawZone( i, j ) ;
					break ;
			case NEUTRAL:
					painter->setBrush( gray ) ;
					drawZone( i, j ) ;
					break ;
			default:	debug("internal error") ;
					break ;
		}
	}

	// highlight last move
	if ( lastmove_x!=PASS )
	{
		painter->setBrush( blue ) ;
		drawZone( lastmove_x, lastmove_y ) ;
	}

	painter->end();
	// painting finished
	// now copy the image
	bitBlt( this, 0, 0, map ) ;
}
