#include "Obstacle.h"
#include <cstdio>
#include <cstring>

Obstacle::Obstacle(const int _sizeX,
		   const int _sizeY,
		   const int _sizeZ) : CubeContainer (_sizeX, _sizeY, _sizeZ)
{
	borderColor=Color (128, 128, 128);

	planeZBuffer=new unsigned char [size];
	memset (planeZBuffer, 0, sizeof (unsigned char)*size);

	mergedCubes=new const Color*[size];
	memset (mergedCubes, 0, sizeof(Color*)*size);
	
	zBufferNeedsUpdate=false;

	viewX=_sizeX/2;
	viewY=_sizeY/2;
}

Obstacle::~Obstacle()
{
	delete[] mergedCubes;
	delete[] planeZBuffer;
}

void Obstacle::draw (Figure* fig)
{
	drawBorder();

	// merge cubes of figure and obstacle

	// copy obstacle cubes
	memcpy (mergedCubes, cubes, sizeof(Color*)*size);
	
	if (fig)
	{
		// insert figure cubes (they don't overlay)
	
		insertFigure (*fig, mergedCubes, false);
	}
	
		// draw merged cubes
	drawCubes();
}

void Obstacle::addCube (const Position& pos, const Color* color, const Color** buffer)
{
	CubeContainer::addCube (pos, color, buffer);

	zBufferNeedsUpdate=true;
}

bool Obstacle::isInsideBorder (const Figure& _fig) const
{
	if (_fig.getOrigin().getX() < 0 ||
	    _fig.getOrigin().getY() < 0 ||
	    _fig.getOrigin().getZ() < 0 ||
	    _fig.getOrigin().getX()+_fig.getSizeX() > sizeX ||
	    _fig.getOrigin().getY()+_fig.getSizeY() > sizeY)
		return false;
	else
		return true;
}

int Obstacle::removeLevelsWherePossible()
{
	/* Algorithm:
	 * Count all cubes within a level. If count==sizeX*sizeY,
	 * the level is full and can be removed, so higher levels
	 * drop down. Does this for all levels.
	 */

	int level=0;
	int levelsDroppedInLine=0;
	int points=0;
	
	for (int i=0;i<sizeZ;i++)
	{
		int cubes=countCubesOnLevel (level);
		if (cubes==sizeXY)
		{
			removeCubesOnLevel (level);
			levelsDroppedInLine++;
		} else {
			if (levelsDroppedInLine) // at least one level has been dropped, accumulate points
				points+=sizeXY;
			
			level++;
		}
	}

	return points;
}

void Obstacle::removeCubesOnLevel (int level)
{
	const Color **cube=cubes+sizeXY*level; // start on level


	// delete cubes to be removed
	for (int i=0; i<sizeXY; i++)
	{
		if (*cube)
		{
			delete *cube;
			*cube=0;
		}
		cube++;
	}
	
	cube=cubes+sizeXY*level;	// now drop higher levels by one
	const Color **cubeOneLevelHigher=cube+sizeXY;

	memcpy (cube, cubeOneLevelHigher, (sizeZ-level-1)*sizeXY*sizeof (Color *));
	
	memset (cubes+sizeXY*(sizeZ-2), 0, sizeof (Color *)*sizeXY);

	zBufferNeedsUpdate=true;
}

int Obstacle::countCubesOnLevel (int level)
{
	const Color **cube=cubes+sizeXY*level; // start on level
	int count=0;
		
	for (int i=0; i<sizeXY; i++)	// check all posititions within level
	{
		if (*cube)
			count++;
		cube++;
	}
	
	
	return count;
}

void Obstacle::drawBorder()
{
	Draw3D::getGlobalInstance()->setForegroundColor (borderColor);

	Position points[4];
	
	int i;

	// horizontal separators of levels

	for (i=0; i<=sizeZ; i++)	// n levels=n+1 separators
	{
		points [0]=Position (0, 0, i);
		points [1]=Position (sizeX, 0, i);
		points [2]=Position (sizeX, sizeY, i);
		points [3]=Position (0, sizeY, i);

		Draw3D::getGlobalInstance()->drawPolygon (points, 4);
	}
	
	// vertical slices (constant x) (scoping different in Visual C++ 5 and gcc 2.8)
	for (i=0;i<=sizeX;i++)
	{
		points [0]=Position (i, 0, sizeZ);
		points [1]=Position (i, 0, 0);
		points [2]=Position (i, sizeY, 0);
		points [3]=Position (i, sizeY, sizeZ);

		Draw3D::getGlobalInstance()->drawLineStrip (points, 4);
	}
	
	// vertical slices (constant y)
	for (i=1;i<sizeY;i++)
	{

		points [0]=Position (0, i, sizeZ);
		points [1]=Position (0, i, 0);
		points [2]=Position (sizeX, i, 0);
		points [3]=Position (sizeX, i, sizeZ);

		Draw3D::getGlobalInstance()->drawLineStrip (points, 4);
		}
}


bool Obstacle::areCollisions (const Figure& fig)
{
	const Color** cube=fig.getCubes();
	int x=0,y=0,z=0;
	
	for (int i=0; i<fig.getSize(); i++) // move all cubes from fig to this
	{
		if (*cube)		// check for cube at this
		{
			Position p(x,y,z);
			
			p+=fig.getOrigin();	// calculate absolute position

			if (p.getZ() < sizeZ && 
			    cubes [get3DIndex (p)]) // collisiion with cube
				return true;
		}

		fig.incrementPosition (&x, &y, &z);
		cube++;
	}

	return false;
}

void Obstacle::insertFigure (Figure& fig, const Color **buffer, bool remove)
{
	const Color** cube=fig.getCubes();
	int x=0,y=0,z=0;
	
	for (int i=0; i<fig.getSize(); i++)	// move all cubes from fig to this
	{
		if (*cube)
		{
			zBufferNeedsUpdate=true;

			Position p(x,y,z);
			
			p+=fig.getOrigin();	// calculate absolute position

			addCube (p, *cube, buffer);

			if (remove)
				*cube=NULL;	// is now managed by this
		}

		fig.incrementPosition (&x, &y, &z);

		if (z+fig.getOrigin().getZ()>=sizeZ) // not visible cubes are following
			return;

		cube++;
	}

}


void Obstacle::clear()
{
	CubeContainer::clear();
	
	memset (planeZBuffer, 0, sizeof (unsigned char)*size);

	zBufferNeedsUpdate=false;
}


void Obstacle::drawCubes()
{
	const double zFactor=0.06;		// color is reduce by zFactor*z

	if (zBufferNeedsUpdate)
		calculateZBuffer();

	// For all cubes of one plane the sides can be displayed independent
	// because of z-Buffer

	// draw cube beginning with first plane and then the higher ones

	const Color** cubeTmp=mergedCubes;
	unsigned char *zBufferTmp=planeZBuffer;
	int z,y,x;
	
	for (z=0; z<sizeZ; z++)
	{
		const Color** cube=cubeTmp;
		unsigned char *zBuffer=zBufferTmp;
		const double depthColorFactor=zFactor*(getSizeZ()-1-z);
		
		for (y=0; y<sizeY; y++)
		{
			for (x=0; x<sizeX; x++)
			{
				if (*zBuffer & Left)
				{
					Position p0 (x, y, z);
					Position p1 (x, y+1, z+1);

					Draw3D::getGlobalInstance()->setForegroundColor (**cube*(0.9-depthColorFactor));
					Draw3D::getGlobalInstance()->drawFilledRectangleX (p0, p1);
				}
				if (*zBuffer & Right)
				{
					Position p0 (x+1, y, z);
					Position p1 (x+1, y+1, z+1);

					Draw3D::getGlobalInstance()->setForegroundColor (**cube*(0.9-depthColorFactor));
					Draw3D::getGlobalInstance()->drawFilledRectangleX (p0, p1);
				}

				if (*zBuffer & Front)
				{
					Position p0 (x, y, z);
					Position p1 (x+1, y, z+1);
					
					Draw3D::getGlobalInstance()->setForegroundColor (**cube*(0.8-depthColorFactor));
					Draw3D::getGlobalInstance()->drawFilledRectangleY (p0, p1);
				}

				if (*zBuffer & Back)
				{
					Position p0 (x, y+1, z);
					Position p1 (x+1, y+1, z+1);

					Draw3D::getGlobalInstance()->setForegroundColor (**cube*(0.8-depthColorFactor));
					Draw3D::getGlobalInstance()->drawFilledRectangleY (p0, p1);
				}
				cube++;
				zBuffer++;
			}
		}

		// draw top side at last, so side of neighbor cubes don't overlay

		cube=cubeTmp;
		zBuffer=zBufferTmp;

		for (y=0; y<sizeY; y++)
		{
			for (x=0; x<sizeX; x++)
			{

				if (*zBuffer & Top)
				{
					Position p0 (x, y, z+1);
					Position p1 (x+1, y+1, z+1);

					Draw3D::getGlobalInstance()->setForegroundColor (**cube*(1-depthColorFactor));
					Draw3D::getGlobalInstance()->drawFilledRectangleZ (p0, p1);
				}


				cube++;
				zBuffer++;
			}
		}

		cubeTmp=cube;
		zBufferTmp=zBuffer;
	}
}


void Obstacle::calculateZBuffer()
{
	const Color** cube=mergedCubes;
	unsigned char *zBuffer=planeZBuffer;

	// create simple z-Buffer, consider only the sight of line of one cube
	// checks which sides of cubes can be visible

	int z,y,x;			// Visual C++ 5 doesn't know about ANSI Scope
	memset (planeZBuffer, 0, sizeof (unsigned char)*size); // empty zBuffer

	for (z=0; z<sizeZ; z++)
	{
		for (y=0; y<sizeY; y++)
		{
			for (x=0; x<sizeX; x++)
			{
				if (*cube)
				{
					unsigned char visibility=Top;

					// check if sides of cube could visible
					// because in line of sight

					if (x > viewX)
						visibility|=Left;
					else
						if (x < viewX)
							visibility|=Right;

					if (y > viewY)
						visibility|=Front;
					else
						if (y < viewY)
							visibility|=Back;

					*zBuffer=visibility;
				}
				
				cube++;
				zBuffer++;
			}
		}
	}

	// optimize z-Buffer, remove all sides which are shared by two cubes

	// remove left and right sides

	for (z=0; z<sizeZ; z++)
	{
		for (y=0; y<sizeY; y++)
		{
			unsigned char *zBufferOneCubeLeft=planeZBuffer+sizeX*sizeY*z+sizeX*y;
			zBuffer=zBufferOneCubeLeft+1;

			// if anything is right of the current cube (left of view center), don't draw right side
			for (x=1; x<=viewX; x++)
			{
				if (*zBuffer) // there is one cube
				{
					*zBufferOneCubeLeft&=~Right;
					*zBuffer&=~Left;
				}
				
				zBuffer++;
				zBufferOneCubeLeft++;
			}

			// if anything is left of the current cube (right of view center), don't draw left side
			for (;x<sizeX; x++)
			{
				if (*zBufferOneCubeLeft)
				{
					*zBuffer&=~Left;
					*zBufferOneCubeLeft&=~Right;
				}
				
				zBuffer++;
				zBufferOneCubeLeft++;
			}
		}
		
	}

	// remove front and back sides

	for (z=0; z<sizeZ; z++)
	{
		zBuffer=planeZBuffer+sizeX*sizeY*z;
		unsigned char *zBufferOneCubeBehind=zBuffer+sizeX;
		
		// if anything is in front of cube, don't draw front side

		for (y=0; y<viewY; y++)
		{
			for (int x=0; x<sizeX; x++)
			{
				if (*zBufferOneCubeBehind)
				{
					*zBufferOneCubeBehind&=~Front;
					*zBuffer&=~Back;
				}
				
				zBuffer++;
				zBufferOneCubeBehind++;
			}
		}

		for (; y<sizeY; y++)
		{
			for (int x=0; x<sizeX; x++)
			{
				if (*zBuffer)
				{
					*zBuffer&=~Back;
					*zBufferOneCubeBehind&=~Front;
				}
				
				zBuffer++;
				zBufferOneCubeBehind++;
			}
		}
		
	}

	// remove top sides

	unsigned char *zBufferOnePlaneLower=planeZBuffer;
	zBuffer=planeZBuffer+sizeX*sizeY; // start on plane two


	for (z=1; z<sizeZ; z++)
	{
		for (y=0; y<sizeY; y++)
		{
			for (int x=0;x<sizeX;x++)
			{
				if ((*zBuffer & Top) && (*zBufferOnePlaneLower & Top))
					*zBufferOnePlaneLower&=~Top; // top side not visible
				

				zBuffer++;
				zBufferOnePlaneLower++;
			}
		}
		
	}

	zBufferNeedsUpdate=false;
}

// Local Variables:
// compile-command: "make Obstacle.o"
// End:
