/***************************************************************************
    ksqlplusdlg.cpp  -  SQL*Plus executable name setting dialog
                        implementation

    begin                : Mon Apr 24 2000
    copyright            : (C) 2000 by Yury Lebedev
    email                : yurylebedev@mail.ru
    home page            : http://devel-home.kde.org/~ksqlplus
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     * *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for Qt
#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qbuttongroup.h>

// include files for KDE
#include <kapp.h>
#include <kconfig.h>
#include <kmsgbox.h>

// include system headers
#include <stdlib.h>

#include "ksqlplusdlg.h"

KSqlPlusDlg::KSqlPlusDlg(QWidget* parent, const char* name) :
  QDialog(parent, name, true)
{
  // read the local ORACLE_HOME variable
  const char* s = getenv("ORACLE_HOME");
  if( s && *s )
  {
    // set the path of local SQL*Plus process
    localSqlplusPath = s;
    localSqlplusPath += "/bin/sqlplus";
  }

  // set dialog caption
  setCaption(i18n("SQL*Plus location settings"));

  // calculate widgets size
  QFontMetrics metrics = fontMetrics();
  int h = metrics.height() + 8;
  int w = metrics.width(i18n("Cancel")) + 32;

  // create main layouts
  QVBoxLayout* topLayout = new QVBoxLayout(this, 12, 12);
  QHBoxLayout* buttonLayout = new QHBoxLayout(6);

  // init groups
  QButtonGroup* remoteGroup =
    new QButtonGroup(i18n("SQL*Plus location"), this);
  remoteGroup->setMinimumSize(w * 5, h * 6);

  // add components to top layout
  topLayout->addWidget(remoteGroup);
  topLayout->addLayout(buttonLayout);

  // create group layout
  QVBoxLayout* groupLayout = new QVBoxLayout(remoteGroup, 12, 6);
  groupLayout->addSpacing(h / 3);

  // init choices
  QString ss = i18n("Local SQL*Plus");
  if( !localSqlplusPath.isEmpty() )
  {
    // add the SQL*Plus process location
    ss += " (";
    ss += localSqlplusPath;
    ss += ')';
  }
  localRadio = new QRadioButton(ss, remoteGroup);
  localRadio->setMinimumSize(metrics.width(ss) + 16, h);
  groupLayout->addWidget(localRadio);

  s = i18n("Remote SQL*Plus via RSH/SSH :");
  remoteRadio = new QRadioButton(s, remoteGroup);
  remoteRadio->setMinimumSize(metrics.width(s) + 16, h);
  groupLayout->addWidget(remoteRadio);

  // create layout for remote SQL*Plus parameters
  QGridLayout* remoteLayout = new QGridLayout(2, 2, 6);
  groupLayout->addLayout(remoteLayout);
  remoteLayout->addColSpacing(0, 12);

  // create subgroups
  QGroupBox* remoteShellGroup =
    new QGroupBox(i18n("Remote shell parameters"), remoteGroup);
  remoteShellGroup->setMinimumSize(w * 4, h * 2);
  remoteLayout->addWidget(remoteShellGroup, 0, 1, AlignCenter);
  QGroupBox* remoteEnvGroup =
    new QGroupBox(i18n("Remote environment"), remoteGroup);
  remoteEnvGroup->setMinimumSize(w * 4, h * 3);
  remoteLayout->addWidget(remoteEnvGroup, 1, 1, AlignCenter);

  // create layouts for subgroup
  QGridLayout* remoteShellLayout =
    new QGridLayout(remoteShellGroup, 2, 6, 12, 6);
  remoteShellLayout->addRowSpacing(0, h / 3);
  QGridLayout* remoteEnvLayout =
    new QGridLayout(remoteEnvGroup, 3, 4, 12, 6);
  remoteEnvLayout->addRowSpacing(0, h / 3);

  // init widgets for remote parameters
  s = i18n("Shell");
  QLabel* l = new QLabel(s, remoteShellGroup);
  l->setMinimumSize(metrics.width(s), h);
  remoteShellLayout->addWidget(l, 1, 0, AlignLeft);

  shellCombo = new QComboBox(remoteShellGroup);
  shellCombo->setMinimumSize(metrics.width("rsh") + 36, h);
  shellCombo->insertItem("rsh");
  shellCombo->insertItem("ssh");
  remoteShellLayout->addWidget(shellCombo, 1, 1, AlignCenter);

  s = i18n("User");
  l = new QLabel(s, remoteShellGroup);
  l->setMinimumSize(metrics.width(s), h);
  remoteShellLayout->addWidget(l, 1, 2, AlignLeft);

  userEdit = new QLineEdit(remoteShellGroup);
  userEdit->setMinimumSize(w, h);
  remoteShellLayout->addWidget(userEdit, 1, 3, AlignLeft);

  s = i18n("Host");
  l = new QLabel(s, remoteShellGroup);
  l->setMinimumSize(metrics.width(s), h);
  remoteShellLayout->addWidget(l, 1, 4, AlignLeft);

  hostEdit = new QLineEdit(remoteShellGroup);
  hostEdit->setMinimumSize(w, h);
  remoteShellLayout->addWidget(hostEdit, 1, 5, AlignLeft);

  s = "ORACLE_HOME=";
  l = new QLabel(s, remoteEnvGroup);
  l->setMinimumSize(metrics.width(s), h);
  remoteEnvLayout->addWidget(l, 1, 0, AlignLeft);

  oracleHomeEdit = new QLineEdit(remoteEnvGroup);
  oracleHomeEdit->setMinimumSize(w, h);
  remoteEnvLayout->addMultiCellWidget(oracleHomeEdit, 1, 1, 1, 3, AlignLeft);

  s = "ORACLE_SID=";
  l = new QLabel(s, remoteEnvGroup);
  l->setMinimumSize(metrics.width(s), h);
  remoteEnvLayout->addWidget(l, 2, 0, AlignLeft);

  oracleSidEdit = new QLineEdit(remoteEnvGroup);
  oracleSidEdit->setMinimumSize(w, h);
  remoteEnvLayout->addWidget(oracleSidEdit, 2, 1, AlignLeft);

  s = "NLS_LANG=";
  l = new QLabel(s, remoteEnvGroup);
  l->setMinimumSize(metrics.width(s), h);
  remoteEnvLayout->addWidget(l, 2, 2, AlignLeft);

  nlsLangEdit = new QLineEdit(remoteEnvGroup);
  nlsLangEdit->setMinimumSize(w, h);
  remoteEnvLayout->addWidget(nlsLangEdit, 2, 3, AlignLeft);

  // init buttons
  QPushButton* okButton = new QPushButton(this);
  okButton->setText(i18n("OK"));
  okButton->setMinimumSize(w + 6, h + 6);
  okButton->setDefault(true);

  QPushButton* cancelButton = new QPushButton(this);
  cancelButton->setText(i18n("Cancel"));
  cancelButton->setMinimumSize(w + 6, h + 6);

  // add components to button layout
  buttonLayout->addWidget(okButton);
  buttonLayout->addWidget(cancelButton);

  // restore saved parameters
  readOptions();

  // center dialog inside parent
  if( parent )
    move(parent->x() + (parent->width() - width()) / 2,
      parent->y() + (parent->height() - height()) / 2);

  // setup slots
  connect(cancelButton, SIGNAL(clicked()), SLOT(reject()));
  connect(okButton, SIGNAL(clicked()), SLOT(ok()));
  connect(remoteRadio, SIGNAL(toggled(bool)), SLOT(remoteRadioToggled(bool)));
}

KSqlPlusDlg::~KSqlPlusDlg()
{
}

void KSqlPlusDlg::remoteRadioToggled(bool on)
{
  // disable or enable remote SQL*Plus settings
  shellCombo->setEnabled(on);
  userEdit->setEnabled(on);
  hostEdit->setEnabled(on);
  oracleHomeEdit->setEnabled(on);
  oracleSidEdit->setEnabled(on);
  nlsLangEdit->setEnabled(on);
}

void KSqlPlusDlg::ok()
{
  if( remoteRadio->isOn() )
  {
    // check filling the required parameters for remote SQL*Plus
    QString s(hostEdit->text());
    if( s.isEmpty() )
    {
      KMsgBox::message(this,
        i18n("Error"),
        i18n("You must set up host of the remote SQL*Plus."),
        KMsgBox::EXCLAMATION);
      hostEdit->setFocus();
      return;
    }
    s = oracleHomeEdit->text();
    if( s.isEmpty() )
    {
      KMsgBox::message(this,
        i18n("Error"),
        i18n("You must set up ORACLE_HOME variable -\n"
             "root directory of the installed ORACLE software\n"
             "on the remote host."),
        KMsgBox::EXCLAMATION);
      oracleHomeEdit->setFocus();
      return;
    }
  }

  // save new parameters
  saveOptions();
  // end dialog
  accept();
}

int KSqlPlusDlg::localRemote()
{
  if( localRadio->isOn() ) return 0;
  // TODO: if this dialog'll tune several remote SQL*Plus sessions
  // it'll be index of them
  return 1;
}

void KSqlPlusDlg::setLocalRemote(int localRemote, bool writeConfig)
{
  // turn on the required radio
  if( localRemote > 0 )
  {
    // TODO: if this dialog'll tune several remote SQL*Plus sessions
    // it'll be index of them
    remoteRadio->setChecked(true);
  }
  else
  {
    localRadio->setChecked(true);
  }
  // enable/disable remote parameter's group
  remoteRadioToggled(remoteRadio->isOn());
  // set focus
  //if( remoteRadio->isOn() ) userEdit->setFocus();

  // save local/remote sign
  if( writeConfig )
  {
    KConfig* config = kapp->getConfig();
    QString prev(config->group());
    config->setGroup("Remote Options");
    config->writeEntry("LocalRemote", localRemote);
    config->setGroup(prev);
  }
}

void KSqlPlusDlg::readOptions()
{
  KConfig* config = kapp->getConfig();
  QString prev(config->group());
  config->setGroup("Remote Options");

  setLocalRemote(config->readNumEntry("LocalRemote"), false);
  shellCombo->setCurrentItem(
    (config->readEntry("Shell", "rsh") == "ssh") ? 1 : 0);
  userEdit->setText(config->readEntry("User", "oracle"));
  hostEdit->setText(config->readEntry("Host", "localhost"));
  oracleHomeEdit->setText(config->readEntry("OracleHome", ""));
  oracleSidEdit->setText(config->readEntry("OracleSid", "orcl"));
  nlsLangEdit->setText(config->readEntry("NlsLang", ""));

  config->setGroup(prev);
}

void KSqlPlusDlg::saveOptions()
{
  KConfig* config = kapp->getConfig();
  QString prev(config->group());
  config->setGroup("Remote Options");

  config->writeEntry("LocalRemote", localRemote());
  config->writeEntry("Shell", shellCombo->currentText());
  config->writeEntry("User", userEdit->text());
  config->writeEntry("Host", hostEdit->text());
  config->writeEntry("OracleHome", oracleHomeEdit->text());
  config->writeEntry("OracleSid", oracleSidEdit->text());
  config->writeEntry("NlsLang", nlsLangEdit->text());

  config->setGroup(prev);
}

void KSqlPlusDlg::setupSqlplus(KProcess& proc, const char* connectString)
{
  // clean old process' arguments
  proc.clearArguments();
  if( localRemote() == 0 )
  {
    // local SQL*Plus
    if( localSqlplusPath.isEmpty() )
    {
      // path to local SQL*Plus is not found -
      // try to find it in PATH
      proc << "sqlplus" << connectString;
    }
    else
    {
      // call the SQL*Plus by its absolute path
      proc << localSqlplusPath << connectString;
    }
  }
  else
  {
    // remote SQL*Plus
    // set the shell...
    proc << shellCombo->currentText();
    // user...
    QString s(userEdit->text());
    if( !s.isEmpty() )
    {
      proc << "-l" << s;
    }
    // host...
    proc << hostEdit->text();
    // ...and SQL*Plus starting command with environment
    QString cmd("/bin/sh -c \"ORACLE_HOME=");
    cmd += oracleHomeEdit->text();
    s = oracleSidEdit->text();
    if( !s.isEmpty() )
    {
      cmd += " ORACLE_SID=";
      cmd += s;
    }
    s = nlsLangEdit->text();
    if( !s.isEmpty() )
    {
      cmd += " NLS_LANG=";
      cmd += s;
    }
    cmd += ' ';
    cmd += oracleHomeEdit->text();
    cmd += "/bin/sqlplus ";
    cmd += connectString;
    cmd += '\"';
    proc << cmd;
  }
}
