/***************************************************************************
    ksqlplus.cpp  -  Application class implementation

    begin                : Fri Dec 10 16:02:01 MSK 1999
    copyright            : (C) 1999 by Yury Lebedev
    email                : yurylebedev@mail.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for QT
#include <qdir.h>
#include <qfile.h>
#include <qstrlist.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qsplitter.h>

// include files for KDE
#include <kiconloader.h>
#include <kmsgbox.h>
#include <kfiledialog.h>
#include <kfontdialog.h>
#include <kkeydialog.h>

// application specific includes
#include "ksqlplus.h"
#include "klogindlg.h"
#include "khistorydlg.h"
#include "resource.h"


KSqlPlusApp::KSqlPlusApp()
{
  // create application configuration
  config = kapp->getConfig();

  // call inits to invoke all other construction parts
  initStatusBar();
  initView();
  initMenuBar();
  initToolBar();
  initKeyAccel();

  // restore saved options
  readOptions();

  // initiate SQL*Plus process
  initSqlplus();

  // init command history
  commandHistory.setAutoDelete(true);

  // disable menu and toolbar items at startup
  disableCommand(ID_FILE_PRINT);
  disableCommand(ID_FILE_PRINT_OUTPUT);
  disableCommand(ID_EDIT_CUT);
  disableCommand(ID_EDIT_COPY);
  disableCommand(ID_COMMAND_HISTORY);
  disableCommand(ID_COMMAND_PREV);
  disableCommand(ID_COMMAND_NEXT);

  // update application caption
  updateCaption();

  // shedule logging on to Oracle
  startTimer(0);
}

KSqlPlusApp::~KSqlPlusApp()
{
  // try to finish SQL*Plus correctly (with implicit commit)
  if( sqlplus.isRunning() )
  {
    sqlplusExecute("\nEXIT\n", false);
  }
}

void KSqlPlusApp::timerEvent(QTimerEvent* e)
{
  // destroy timer first
  killTimer(e->timerId());
  // force logging on to Oracle
  slotFileLogin();
}

void KSqlPlusApp::initKeyAccel()
{
  keyAccel = new KAccel(this, "keyAccel");

  // create application-specific keys
  keyAccel->insertItem(i18n("Execute command"), KEY_COMMAND_EXECUTE, "CTRL+Return");
  keyAccel->insertItem(i18n("Previous command"), KEY_COMMAND_PREV, "ALT+Left");
  keyAccel->insertItem(i18n("Next command"), KEY_COMMAND_NEXT, "ALT+Right");

  // read default settings
  keyAccel->readSettings(config);

  // fileMenu accelerators
  keyAccel->connectItem(KAccel::New, this, SLOT(slotFileNew()));
  keyAccel->connectItem(KAccel::Open, this, SLOT(slotFileOpen()));
  keyAccel->connectItem(KAccel::Save, this, SLOT(slotFileSave()));
  keyAccel->connectItem(KAccel::Close, this, SLOT(slotFileClose()));
  keyAccel->connectItem(KAccel::Print, this, SLOT(slotFilePrint()));
  keyAccel->connectItem(KAccel::Quit, this, SLOT(slotFileQuit()));

  // editMenu accelerators
  keyAccel->connectItem(KAccel::Cut, inputView, SLOT(cut()));
  keyAccel->connectItem(KAccel::Copy, inputView, SLOT(copyText()));
  keyAccel->connectItem(KAccel::Paste, inputView, SLOT(paste()));

  // helpMenu
  keyAccel->connectItem(KAccel::Help, kapp, SLOT(appHelpActivated()));

  // commandMenu
  keyAccel->connectItem(KEY_COMMAND_EXECUTE, this, SLOT(slotCommandExecute()));
  keyAccel->connectItem(KEY_COMMAND_PREV, this, SLOT(slotCommandPrev()));
  keyAccel->connectItem(KEY_COMMAND_NEXT, this, SLOT(slotCommandNext()));

  // change menu accelerators
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_NEW, KAccel::New);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_OPEN, KAccel::Open);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_SAVE, KAccel::Save);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_CLOSE, KAccel::Close);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_PRINT, KAccel::Print);
  keyAccel->changeMenuAccel(fileMenu, ID_FILE_QUIT, KAccel::Quit);

  keyAccel->changeMenuAccel(editMenu, ID_EDIT_CUT, KAccel::Cut);
  keyAccel->changeMenuAccel(editMenu, ID_EDIT_COPY, KAccel::Copy);
  keyAccel->changeMenuAccel(editMenu, ID_EDIT_PASTE, KAccel::Paste);

  keyAccel->changeMenuAccel(commandMenu, ID_COMMAND_EXECUTE, KEY_COMMAND_EXECUTE);
  keyAccel->changeMenuAccel(commandMenu, ID_COMMAND_PREV, KEY_COMMAND_PREV);
  keyAccel->changeMenuAccel(commandMenu, ID_COMMAND_NEXT, KEY_COMMAND_NEXT);
}

void KSqlPlusApp::initMenuBar()
{
  ///////////////////////////////////////////////////////////////////
  // MENUBAR
  recentFilesMenu = new QPopupMenu();

  ///////////////////////////////////////////////////////////////////
  // menuBar entry fileMenu
  fileMenu = new QPopupMenu();
  fileMenu->insertItem(kapp->getMiniIcon(), i18n("New &Window"), ID_FILE_NEW_WINDOW);
  fileMenu->insertSeparator();
  fileMenu->insertItem(Icon("connect.xpm"), i18n("&Connect..."), ID_FILE_LOGIN);
  fileMenu->insertSeparator();
  fileMenu->insertItem(Icon("filenew.xpm"), i18n("&New"), ID_FILE_NEW);
  fileMenu->insertItem(Icon("fileopen.xpm"), i18n("&Open..."), ID_FILE_OPEN);
  fileMenu->insertItem(i18n("Open &recent"), recentFilesMenu, ID_FILE_OPEN_RECENT);
  fileMenu->insertItem(i18n("&Close"), ID_FILE_CLOSE);
  fileMenu->insertSeparator();
  fileMenu->insertItem(Icon("filefloppy.xpm") ,i18n("&Save"), ID_FILE_SAVE);
  fileMenu->insertItem(i18n("Save &As..."), ID_FILE_SAVE_AS);
  fileMenu->insertItem(i18n("Save &output as..."), ID_FILE_SAVE_OUTPUT);
  fileMenu->insertSeparator();
  fileMenu->insertItem(Icon("fileprint.xpm"), i18n("&Print..."), ID_FILE_PRINT);
  fileMenu->insertItem(Icon("fileprint.xpm"), i18n("Print o&utput..."), ID_FILE_PRINT_OUTPUT);
  fileMenu->insertSeparator();
  fileMenu->insertItem(i18n("E&xit"), ID_FILE_QUIT);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry editMenu
  editMenu = new QPopupMenu();
  editMenu->insertItem(Icon("editcut.xpm"), i18n("Cu&t"), ID_EDIT_CUT);
  editMenu->insertItem(Icon("editcopy.xpm"), i18n("&Copy"), ID_EDIT_COPY);
  editMenu->insertItem(Icon("editpaste.xpm"), i18n("&Paste"), ID_EDIT_PASTE);
  editMenu->insertItem(Icon("delete.xpm"), i18n("C&lear"), ID_EDIT_CLEAR);
  editMenu->insertSeparator();
  editMenu->insertItem(Icon("editcopy.xpm"), i18n("Copy &output"), ID_EDIT_COPY_OUTPUT);
  editMenu->insertItem(Icon("delete.xpm"), i18n("Clear o&utput"), ID_EDIT_CLEAR_OUTPUT);
  editMenu->insertItem(i18n("Select &all output"), ID_EDIT_SELECT_ALL_OUTPUT);
  editMenu->insertSeparator();
  editMenu->insertItem(i18n("&Keys..."), ID_EDIT_KEYS);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry viewMenu
  viewMenu = new QPopupMenu();
  viewMenu->setCheckable(true);
  viewMenu->insertItem(i18n("&Toolbars"), ID_VIEW_TOOLBAR);
  viewMenu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR);
  viewMenu->setCheckable(false);
  viewMenu->insertSeparator();
  viewMenu->insertItem(Icon("text.xpm"), i18n("&Font..."), ID_VIEW_FONT);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry commandView
  commandMenu = new QPopupMenu();
  commandMenu->insertItem(Icon("configure.xpm"), i18n("&Execute"), ID_COMMAND_EXECUTE);
  commandMenu->insertItem(Icon("stop.xpm"), i18n("&Break"), ID_COMMAND_BREAK);
  commandMenu->insertSeparator();
  commandMenu->insertItem(Icon("back.xpm"), i18n("&Previous command"), ID_COMMAND_PREV);
  commandMenu->insertItem(Icon("forward.xpm"), i18n("&Next command"), ID_COMMAND_NEXT);
  commandMenu->insertItem(i18n("Command &history..."), ID_COMMAND_HISTORY);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry helpMenu
  QString about = kapp->appName() + " " + VERSION +
    i18n("\n\nKDE front-end for ORACLE SQL*Plus.\n"
    "Developed by Yury Lebedev (yurylebedev@mail.ru).\n\n"
    "This is free software covered by GPL.\n\n"
    "ORACLE and SQL*Plus are registered marks of ORACLE Corporation\n"
    "and these are commercial software.\n");
  helpMenu = kapp->getHelpMenu(true, about);

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // insert your popup menus with the according menubar entries in the order
  // they will appear later from left to right
  menuBar()->insertItem(i18n("&File"), fileMenu);
  menuBar()->insertItem(i18n("&Edit"), editMenu);
  menuBar()->insertItem(i18n("&View"), viewMenu);
  menuBar()->insertItem(i18n("&Command"), commandMenu);
  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), helpMenu);

  ///////////////////////////////////////////////////////////////////
  // CONNECT THE MENU SLOTS WITH SIGNALS
  // for execution slots and statusbar messages

  connect(fileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(fileMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(recentFilesMenu, SIGNAL(activated(int)), SLOT(slotFileOpenRecent(int)));

  connect(editMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(editMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(viewMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(viewMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(commandMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(commandMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
}

void KSqlPlusApp::initToolBar()
{
  // create two toolbars - one for output view, other - for input view

  // create toolbar for output view
  toolBar(0)->setBarPos(KToolBar::Top);
  toolBar(0)->setTitle(i18n("Tools for output"));

  toolBar(0)->insertButton(Icon("filefloppy.xpm"), ID_FILE_SAVE_OUTPUT, true, i18n("Save output"));
  toolBar(0)->insertButton(Icon("fileprint.xpm"), ID_FILE_PRINT_OUTPUT, true, i18n("Print output"));
  toolBar(0)->insertSeparator();
  toolBar(0)->insertButton(Icon("editcopy.xpm"), ID_EDIT_COPY_OUTPUT, true, i18n("Copy output"));
  toolBar(0)->insertButton(Icon("delete.xpm"), ID_EDIT_CLEAR_OUTPUT, true, i18n("Clear output"));

  // create toolbar for input view
  toolBar(1)->setBarPos(KToolBar::Bottom);
  toolBar(1)->setTitle(i18n("Tools for input"));

  toolBar(1)->insertButton(Icon("fileopen.xpm"), ID_FILE_OPEN, true, i18n("Open File"));
  toolBar(1)->insertButton(Icon("filefloppy.xpm"), ID_FILE_SAVE, true, i18n("Save File"));
  //toolBar(1)->insertButton(Icon("fileprint.xpm"), ID_FILE_PRINT, true, i18n("Print"));
  toolBar(1)->insertSeparator();
  toolBar(1)->insertButton(Icon("editcut.xpm"), ID_EDIT_CUT, true, i18n("Cut"));
  toolBar(1)->insertButton(Icon("editcopy.xpm"), ID_EDIT_COPY, true, i18n("Copy"));
  toolBar(1)->insertButton(Icon("editpaste.xpm"), ID_EDIT_PASTE, true, i18n("Paste"));
  toolBar(1)->insertButton(Icon("delete.xpm"), ID_EDIT_CLEAR, true, i18n("Clear"));
  toolBar(1)->insertSeparator();
  toolBar(1)->insertButton(Icon("configure.xpm"), ID_COMMAND_EXECUTE, true, i18n("Execute command"));
  toolBar(1)->insertButton(Icon("stop.xpm"), ID_COMMAND_BREAK, true, i18n("Break SQL*Plus"));
  toolBar(1)->insertButton(Icon("back.xpm"), ID_COMMAND_PREV, true, i18n("Previous command"));
  toolBar(1)->insertButton(Icon("forward.xpm"), ID_COMMAND_NEXT, true, i18n("Next command"));

  // connect for invoking the slot actions
  connect(toolBar(0), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
  // connect for the status help on holing icons pressed with the mouse button
  connect(toolBar(0), SIGNAL(pressed(int)), SLOT(statusCallback(int)));
  // connect for invoking the slot actions
  connect(toolBar(1), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
  // connect for the status help on holing icons pressed with the mouse button
  connect(toolBar(1), SIGNAL(pressed(int)), SLOT(statusCallback(int)));
}

void KSqlPlusApp::initStatusBar()
{
  statusBar()->setInsertOrder(KStatusBar::RightToLeft);
  // input view cursor position message
  QString s;
  s.sprintf(i18n("Line: %d Column: %d "), 9999, 999);
  statusBar()->insertItem(s, ID_STATUS_ROW_COL);
  s.sprintf(i18n("Line: %d Column: %d "), 1, 1);
  statusBar()->changeItem(s, ID_STATUS_ROW_COL);
  // status message
  statusBar()->insertItem(i18n("Ready"), ID_STATUS_MSG);
}

void KSqlPlusApp::initView()
{
  // create main view splitter
  QSplitter* splitter = new QSplitter(QSplitter::Vertical, this, "viewsSplitter");

  // create view for SQL*Plus output
  outputView = new QMultiLineEdit(splitter, "outputView");
  outputView->setReadOnly(true);

  // create view for user input
  inputView = new KEdit(kapp, splitter, "inputView");
  // connect slot
  connect(inputView, SIGNAL(CursorPositionChanged()), SLOT(slotInputViewCursorChanged()));

  splitter->setOpaqueResize(true);
  // set splitter as main view
  setView(splitter);

  // focus item
  inputView->setFocus();
}

void KSqlPlusApp::initSqlplus()
{
  // init SQL*Plus process

  // connect signals to slots
  connect(&sqlplus, SIGNAL(processExited(KProcess*)), this, SLOT(slotSqlplusExited(KProcess*)));
  connect(&sqlplus, SIGNAL(receivedStdout(KProcess*,char*,int)), this, SLOT(slotSqlplusStdout(KProcess*,char*,int)));
  connect(&sqlplus, SIGNAL(receivedStderr(KProcess*,char*,int)), this, SLOT(slotSqlplusStdout(KProcess*,char*,int)));
}

void KSqlPlusApp::updateCaption()
{
  // get original application caption
  QString caption = kapp->getCaption();
  // add Oracle 'username@database'
  if( !lastOracleIdentity.isEmpty() )
    caption += " - " + lastOracleIdentity;
  // add current filename
  QString filename;
  if( isNew() ) filename = i18n("New SQL script");
  else filename = inputView->getName();
  caption += " - " + filename;
  // set main window caption
  setCaption(caption);
}

void KSqlPlusApp::appendToOutputView(const char* text)
{
  // turn off auto update
  outputView->setAutoUpdate(false);
  // get last line
  QString line = outputView->textLine(outputView->numLines() - 1);
  // remove last line
  outputView->removeLine(outputView->numLines() - 1);
  // append saved line and new text
  outputView->append(line + text);
  // turn on auto update
  outputView->setAutoUpdate(true);
  outputView->repaint();
  // scroll down output view
  outputView->setCursorPosition(outputView->numLines(), 1);
}

void KSqlPlusApp::enableCommand(int id)
{
  ///////////////////////////////////////////////////////////////////
  // enable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id, true);
  for( int i = 0; i < MAX_TOOLBARS; i++ )
    toolBar(i)->setItemEnabled(id, true);
}

void KSqlPlusApp::disableCommand(int id)
{
  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id, false);
  for( int i = 0; i < MAX_TOOLBARS; i++ )
    toolBar(i)->setItemEnabled(id, false);
}

void KSqlPlusApp::saveOptions()
{
  // save general settings
  config->setGroup("General Options");
  config->writeEntry("Geometry", size());
  config->writeEntry("Show Toolbar", toolBar()->isVisible());
  config->writeEntry("Show Statusbar",statusBar()->isVisible());
  config->writeEntry("MenuBar Position", (int)menuBar()->menuBarPos());
  config->writeEntry("OutputToolBar Position", (int)toolBar(0)->barPos());
  config->writeEntry("InputToolBar Position", (int)toolBar(1)->barPos());
  config->writeEntry("Recent Files", recentFiles);
  config->writeEntry("Font", viewsFont);
  // save accelerators
  keyAccel->writeSettings(config);
}

void KSqlPlusApp::readOptions()
{
  config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
  viewMenu->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
  if( !bViewToolbar )
  {
    for( int i = 0; i < MAX_TOOLBARS; i++ )
      enableToolBar(KToolBar::Hide, i);
  }
  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  viewMenu->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
  if( !bViewStatusbar )
  {
    enableStatusBar(KStatusBar::Hide);
  }

  // bar position settings
  menuBar()->setMenuBarPos((KMenuBar::menuPosition)
    config->readNumEntry("MenuBar Position", KMenuBar::Top));
  toolBar(0)->setBarPos((KToolBar::BarPosition)
    config->readNumEntry("OutputToolBar Position", KToolBar::Top));
  toolBar(1)->setBarPos((KToolBar::BarPosition)
    config->readNumEntry("InputToolBar Position", KToolBar::Bottom));

  // initialize the recent file list
  recentFiles.setAutoDelete(true);
  config->readListEntry("Recent Files", recentFiles);
  for(int i = 0; i < (int)recentFiles.count(); i++ )
  {
    recentFilesMenu->insertItem(recentFiles.at(i));
  }

  QSize size = config->readSizeEntry("Geometry");
  if( !size.isEmpty() )
  {
    resize(size);
  }

  // initialize default font for input/output views
  QFont font("courier");
  viewsFont = config->readFontEntry("Font", &font);
  // set restored font for input/output views
  outputView->setFont(viewsFont);
  inputView->setFont(viewsFont);
}

const QString KSqlPlusApp::readUsername()
{
  config->setGroup("ORACLE Options");
  return config->readEntry("Username");
}

const QString KSqlPlusApp::readDatabase()
{
  config->setGroup("ORACLE Options");
  return config->readEntry("Database");
}

void KSqlPlusApp::saveUsernameAndDatabase(const char* username, const char* database)
{
  config->setGroup("ORACLE Options");
  config->writeEntry("Username", username);
  config->writeEntry("Database", database);
}

void KSqlPlusApp::saveProperties(KConfig *cfg)
{
  if( isNew() || !inputView->isModified() )
  {
    // saving to tempfile not necessary
  }
  else
  {
    QString filename = inputView->getName();
    cfg->writeEntry("filename", filename);
    cfg->writeEntry("modified", inputView->isModified());

    QString tempname = kapp->tempSaveName(filename);
    inputView->doSave(tempname);
  }
}

void KSqlPlusApp::readProperties(KConfig* cfg)
{
  QString filename = cfg->readEntry("filename", "");
  bool modified = cfg->readBoolEntry("modified", false);
  if( modified )
  {
    bool canRecover;
    QString tempname = kapp->checkRecoverFile(filename, canRecover);

    if( canRecover )
    {
      inputView->loadFile(tempname, KEdit::OPEN_READWRITE);
      inputView->setName(filename);
      QFile::remove(tempname);
    }
  }
  else
  {
    if( !filename.isEmpty() )
      inputView->loadFile(filename, KEdit::OPEN_READWRITE);
  }

  updateCaption();
}

bool KSqlPlusApp::isNew()
{
  QString n = inputView->getName();
  return (n.isEmpty() || n == i18n("Untitled"));
}

bool KSqlPlusApp::saveModified()
{
  if( inputView->isModified() )
  {
    int want_save = KMsgBox::yesNoCancel(this, i18n("Warning"),
      i18n("The current SQL script has been modified.\nDo you want to save it?"),
      KMsgBox::QUESTION, i18n("Yes"), i18n("No"), i18n("Cancel"));
    switch( want_save )
    {
      case 1:
        if( isNew() ) slotFileSaveAs();
        else inputView->doSave();
        return true;
      case 2:
        inputView->toggleModified(false);
        return true;
      default:
        return false;
    }
  }

  return true;
}

void KSqlPlusApp::addRecentFile(const QString &file)
{
  if( recentFiles.find(file) == -1 )
  {
    // update files list
    if( recentFiles.count() >= MAX_RECENT_FILES )
      recentFiles.removeLast();
    recentFiles.insert(0, file);
    // recreate files menu
    recentFilesMenu->clear();
    for( int i = 0 ; i < (int)recentFiles.count(); i++ )
      recentFilesMenu->insertItem(recentFiles.at(i));
  }
}

void KSqlPlusApp::removeRecentFile(int idx)
{
  if( idx < 0 || idx >= (int)recentFiles.count() ) return;
  // update files list
  recentFiles.remove((unsigned)idx);
  // recreate files menu
  recentFilesMenu->clear();
  for( int i = 0 ; i < (int)recentFiles.count(); i++ )
    recentFilesMenu->insertItem(recentFiles.at(i));
}

void KSqlPlusApp::openDocumentFile(const char* cmdl)
{
  slotStatusMsg(i18n("Opening SQL script..."));

  QString file = cmdl;
  // Remove possible "file:" prefix if 'cmdl' is an URL
  if( file.left(5) == "file:" ) file.remove(0, 5);

  QFileInfo fi(file);

  // check file is readable
  if( !fi.isFile() || !fi.isReadable() )
  {
    QString msg(file);
    msg += i18n(":\nfile does not exist.");
    KMsgBox::message(this, i18n("Error"), msg, KMsgBox::EXCLAMATION);
  }
  else
  {
    // load file into the input area
    inputView->loadFile(file, KEdit::OPEN_READWRITE);
    // check file is writable
    if( !fi.isWritable() )
    {
      // set name as for new file
      inputView->setName(i18n("Untitled"));
      // inform user
      QString msg(file);
      msg += i18n(":\nfile is write-protected. You've got the copy of it.");
      KMsgBox::message(this, i18n("Warning"), msg, KMsgBox::INFORMATION);
    }
    else
    {
      // add to recent files
      addRecentFile(file);
    }

    updateCaption();
    // set path of this file as current directory
    QDir::setCurrent(fi.dirPath(true));
  }

  slotStatusMsg(i18n("Ready"));
}

bool KSqlPlusApp::queryClose()
{
  return saveModified();
}

bool KSqlPlusApp::queryExit()
{
  saveOptions();
  return true;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void KSqlPlusApp::slotFileNewWindow()
{
  slotStatusMsg(i18n("Opening a new application window..."));

  KSqlPlusApp* new_window = new KSqlPlusApp();
  new_window->show();

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileLogin()
{
  slotStatusMsg(i18n("Logging on into ORACLE..."));

  // login parameters
  QString connect;
  QString username;
  QString database;
  bool do_login = false;

  // process Login dialog
  // (fixed 'core dump' after cancel dialog and close application)
  {
    KLoginDlg dlg(this, "loginDlg", readUsername(), readDatabase());
    if( dlg.exec() )
    {
      // OK is pressed
      do_login = true;
      connect = dlg.connectString();
      username = dlg.username();
      database = dlg.database();
      // save 'username@database'
      lastOracleIdentity = dlg.identityString();
    }
  }

  if( do_login )
  {
    // do real login
    if( !sqlplus.isRunning() )
    {
      // first time logon - pass connect string as argument
      sqlplus << "sqlplus" << connect;
      // start SQL*Plus
      if( !sqlplus.start(KProcess::NotifyOnExit, KProcess::All) )
      {
        KMsgBox::message(this, i18n("Error"),
          i18n("Cannot start SQL*Plus.\n"
          "Do you really have installed ORACLE software and\n"
          "configured ORACLE environment?\n"
          "Application will be closed now."),
          KMsgBox::STOP);

        close(true);
      }
    }
    else
    {
      // next time logon - pass as command "CONNECT"
      QString cmd("CONNECT ");
      cmd += connect;
      cmd += '\n';
      sqlplusExecute(cmd, true);
      // and set it as new argument for SQL*Plus process
      sqlplus.clearArguments();
      sqlplus << "sqlplus" << connect;
    }

    // write connect parameters to config
    saveUsernameAndDatabase(username, database);
    // update window title
    updateCaption();
  }
  else if( !sqlplus.isRunning() )
  {
    KMsgBox::message(this, i18n("Warning"),
      i18n("You have canceled login into ORACLE at first time.\n"
      "Application will be closed now."),
      KMsgBox::EXCLAMATION);

    close(true);
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileNew()
{
  slotStatusMsg(i18n("Creating new SQL script..."));

  if( !saveModified() )
  {
     // here saving wasn't successful
  }
  else
  {
    inputView->newFile();
    updateCaption();
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening SQL script..."));

  if( !saveModified() )
  {
     // here saving wasn't successful
  }
  else
  {
    QString file = KFileDialog::getOpenFileName(QDir::currentDirPath(),
      i18n("*.sql|SQL scripts\n*|All files"), this, i18n("Open File..."));

    if( !file.isEmpty() ) openDocumentFile(file);
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileOpenRecent(int id)
{
  slotStatusMsg(i18n("Opening SQL script..."));

  if( !saveModified() )
  {
     // here saving wasn't successful
  }
  else
  {
    // check existence of this file
    QFileInfo fi(recentFiles.at(id));
    if( fi.isFile() && fi.isReadable() )
    {
      inputView->loadFile(recentFiles.at(id), KEdit::OPEN_READWRITE);
      // check file for writing access
      if( !fi.isWritable() )
      {
        // set name as for new file
        inputView->setName(i18n("Untitled"));
        // inform user
        QString msg(recentFiles.at(id));
        msg += i18n(":\nfile is write-protected. You've got the copy of it.");
        KMsgBox::message(this, i18n("Warning"), msg, KMsgBox::INFORMATION);
      }
      updateCaption();
      // set path of this file as current directory
      QDir::setCurrent(fi.dirPath(true));
    }
    else
    {
      QString msg(recentFiles.at(id));
      msg += i18n(":\nfile does not exist.");
      KMsgBox::message(this, i18n("Error"), msg, KMsgBox::EXCLAMATION);
      removeRecentFile(id);
    }
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileSave()
{
  slotStatusMsg(i18n("Saving SQL script..."));

  if( isNew() ) slotFileSaveAs();
  else inputView->doSave();

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileSaveAs()
{
  slotStatusMsg(i18n("Saving SQL script with a new filename..."));

  QString file = KFileDialog::getSaveFileName(QDir::currentDirPath(),
    i18n("*.sql|SQL scripts\n*|All files"), this, i18n("Save as..."));
  if( !file.isEmpty() )
  {
    inputView->doSave(file);
    inputView->setName(file);
    addRecentFile(file);
    updateCaption();
    // set path of this file as current directory
    QFileInfo fi(file);
    QDir::setCurrent(fi.dirPath(true));
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileSaveOutput()
{
  slotStatusMsg(i18n("Saving SQL*Plus output into a new file..."));

  QString file = KFileDialog::getSaveFileName(QDir::currentDirPath(),
    i18n("*|All files"), this, i18n("Save output as..."));
  if( !file.isEmpty() )
  {
    // open file
    QFile f(file);
    if( !f.open(IO_WriteOnly) )
    {
      KMsgBox::message(this, i18n("Error"),
        file + i18n(":\ncannot open file for writing."),
        KMsgBox::EXCLAMATION);
    }
    else
    {
      // get text buffer by line and write it to the file
      QString s;
      for( int i = 0; i < outputView->numLines(); i++ )
      {
        s = outputView->textLine(i);
        s += '\n';
        f.writeBlock(s, s.length());
      }
      f.flush();
    }
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing SQL script..."));
	
  close();

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));

  QPrinter printer;
  if( printer.setup(this) )
  {
    // TODO: implement this
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFilePrintOutput()
{
  slotStatusMsg(i18n("Printing output..."));

  QPrinter printer;
  if( printer.setup(this) )
  {
    // TODO: implement this
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));

  saveOptions();
  // close the first window, the list makes the next one the first again.
  // This ensures that queryClose() is called on each window to ask for closing
  KTMainWindow* w;
  if( memberList )
  {
    for( w = memberList->first(); w != 0; w = memberList->first() )
    {
      // only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
      // the window and the application stay open.
      if( !w->close() ) break;
    }
  }	

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotEditKeys()
{
  slotStatusMsg(i18n("Configuring application keys..."));

  // open KDE key dialog
  KKeyDialog::configureKeys(keyAccel);

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggle the toolbar..."));

  // turn toolbars on or off
  bool checked = viewMenu->isItemChecked(ID_VIEW_TOOLBAR);
  for( int i = 0; i < MAX_TOOLBARS; i++)
    enableToolBar((checked ? KToolBar::Hide : KToolBar::Show), i);
  viewMenu->setItemChecked(ID_VIEW_TOOLBAR, !checked);

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));

  //turn statusbar on or off
  bool checked = viewMenu->isItemChecked(ID_VIEW_STATUSBAR);
  enableStatusBar(checked ? KStatusBar::Hide : KStatusBar::Show);
  viewMenu->setItemChecked(ID_VIEW_STATUSBAR, !checked);

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotViewFont()
{
  slotStatusMsg(i18n("Configuring font for views..."));

  // open KDE font dialog
  if( KFontDialog::getFont(viewsFont) )
  {
    // set choosed font
    outputView->setFont(viewsFont);
    inputView->setFont(viewsFont);
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotStatusMsg(const char* text)
{
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}

void KSqlPlusApp::slotStatusHelpMsg(const char* text)
{
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}

void KSqlPlusApp::slotInputViewCursorChanged()
{
  // show new cursor position
  QString s;
  s.sprintf(i18n("Line: %d Column: %d "),
    inputView->currentLine() + 1, inputView->currentColumn() + 1);
  statusBar()->changeItem(s, ID_STATUS_ROW_COL);
  // watch for text selection for enable/disable cut/copy operations
  if( inputView->markedText().isEmpty() )
  {
    disableCommand(ID_EDIT_CUT);
    disableCommand(ID_EDIT_COPY);
  }
  else
  {
    enableCommand(ID_EDIT_CUT);
    enableCommand(ID_EDIT_COPY);
  }
}

void KSqlPlusApp::sqlplusExecute(const char* cmd, bool addToHistory)
{
  slotStatusMsg(i18n("Sending command to SQL*Plus..."));

  QString s(cmd);

  // append command to output area
  if( addToHistory ) appendToOutputView(s);

  // send command to stdin of SQL*Plus
  sqlplus.writeStdin(s.data(), s.length());

  // append command to command history
  if( addToHistory && commandHistory.find(s) == -1 )
  {
    if( commandHistory.count() >= MAX_COMMAND_HISTORY )
      commandHistory.removeFirst();
    commandHistory.append(s);
    enableCommand(ID_COMMAND_HISTORY);
    enableCommand(ID_COMMAND_PREV);
    enableCommand(ID_COMMAND_NEXT);
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotCommandExecute()
{
  QString s;

  // get command from user input area
  s = inputView->markedText();
  if( s.isEmpty() ) s = inputView->text();

  // normalize command
  s = s.stripWhiteSpace();

  // check empty command
  if( s.isEmpty() )
  {
    slotStatusHelpMsg(i18n("Nothing to execute in SQL*Plus"));
    return;
  }

  // append command terminator
  s += '\n';

  // execute command
  sqlplusExecute(s);
}

void KSqlPlusApp::slotCommandBreak()
{
  // kill SQL*Plus
  sqlplus.kill();
}

void KSqlPlusApp::slotCommandPrev()
{
  QString cmd = commandHistory.prev();
  if( cmd.isEmpty() ) cmd = commandHistory.first();
  inputView->setText(cmd);
}

void KSqlPlusApp::slotCommandNext()
{
  QString cmd = commandHistory.next();
  if( cmd.isEmpty() ) cmd = commandHistory.last();
  inputView->setText(cmd);
}

void KSqlPlusApp::slotCommandHistory()
{
  slotStatusMsg(i18n("Showing SQL command history..."));

  // show command history dialog
  KHistoryDlg dlg(&commandHistory, this, "historyDlg");

  if( dlg.exec() )
  {
    inputView->setText(commandHistory.at(dlg.selected()));
  }

  slotStatusMsg(i18n("Ready"));
}

void KSqlPlusApp::slotSqlplusStdout(KProcess *proc, char* buffer, int buflen)
{
  // got SQL*Plus output - write it to output area
  QString s(buffer, buflen + 1);
  appendToOutputView(s);
}

void KSqlPlusApp::slotSqlplusExited(KProcess *proc)
{
  // check shutdown
  if( QApplication::closingDown() ) return;

  QString msg(i18n("SQL*Plus exited"));
  if( !proc->normalExit() )
  {
    msg += i18n(" abnormally");
  }
  msg += i18n(".\nDo you want to restart SQL*Plus again or to close application?");

  if( KMsgBox::yesNo(this, i18n("Warning"), msg,
    KMsgBox::EXCLAMATION | KMsgBox::DB_SECOND,
    i18n("Restart SQL*Plus"), i18n("Close application")) == 1 )
  {
    // start SQL*Plus again
    if( sqlplus.start(KProcess::NotifyOnExit, KProcess::All) )
      return;
  }

  close(true);
}


void KSqlPlusApp::commandCallback(int id)
{
  switch( id )
  {
    case ID_FILE_NEW_WINDOW:
         slotFileNewWindow();
         break;

    case ID_FILE_LOGIN:
    	     slotFileLogin();
         break;

    case ID_FILE_NEW:
    	     slotFileNew();
         break;

    case ID_FILE_OPEN:
         slotFileOpen();
         break;

    case ID_FILE_SAVE:
         slotFileSave();
         break;

    case ID_FILE_SAVE_AS:
         slotFileSaveAs();
         break;

    case ID_FILE_SAVE_OUTPUT:
         slotFileSaveOutput();
         break;

    case ID_FILE_CLOSE:
         slotFileClose();
         break;

    case ID_FILE_PRINT:
         slotFilePrint();
         break;

    case ID_FILE_PRINT_OUTPUT:
         slotFilePrintOutput();
         break;

    case ID_FILE_QUIT:
         slotFileQuit();
         break;

    case ID_EDIT_CUT:
         inputView->cut();
         break;

    case ID_EDIT_COPY:
         inputView->copyText();
         break;

    case ID_EDIT_PASTE:
         inputView->paste();
         break;
  
    case ID_EDIT_CLEAR:
         inputView->clear();
         break;

    case ID_EDIT_COPY_OUTPUT:
         outputView->copyText();
         break;

    case ID_EDIT_CLEAR_OUTPUT:
         outputView->clear();
         break;

    case ID_EDIT_SELECT_ALL_OUTPUT:
         outputView->selectAll();
         break;

    case ID_EDIT_KEYS:
         slotEditKeys();
         break;

    case ID_VIEW_TOOLBAR:
         slotViewToolBar();
         break;

    case ID_VIEW_STATUSBAR:
         slotViewStatusBar();
         break;

    case ID_VIEW_FONT:
         slotViewFont();
         break;

    case ID_COMMAND_EXECUTE:
         slotCommandExecute();
         break;

    case ID_COMMAND_BREAK:
         slotCommandBreak();
         break;

    case ID_COMMAND_PREV:
         slotCommandPrev();
         break;

    case ID_COMMAND_NEXT:
         slotCommandNext();
         break;

    case ID_COMMAND_HISTORY:
         slotCommandHistory();
         break;

    default:
         break;
  }
}

void KSqlPlusApp::statusCallback(int id)
{
  switch( id )
  {
    case ID_FILE_NEW_WINDOW:
         slotStatusHelpMsg(i18n("Opens a new application window"));
         break;

    case ID_FILE_LOGIN:
         slotStatusHelpMsg(i18n("Connect to ORACLE"));
         break;

    case ID_FILE_NEW:
         slotStatusHelpMsg(i18n("Creates a new SQL script"));
         break;

    case ID_FILE_OPEN:
         slotStatusHelpMsg(i18n("Opens an existing SQL script"));
         break;

    case ID_FILE_OPEN_RECENT:
         slotStatusHelpMsg(i18n("Opens a recently used file"));
         break;

    case ID_FILE_SAVE:
         slotStatusHelpMsg(i18n("Saves the actual SQL script"));
         break;

    case ID_FILE_SAVE_AS:
         slotStatusHelpMsg(i18n("Saves the actual SQL script as..."));
         break;

    case ID_FILE_SAVE_OUTPUT:
         slotStatusHelpMsg(i18n("Saves the output area in file"));
         break;

    case ID_FILE_CLOSE:
         slotStatusHelpMsg(i18n("Closes the actual SQL script"));
         break;

    case ID_FILE_PRINT:
         slotStatusHelpMsg(i18n("Prints out the actual SQL script"));
         break;

    case ID_FILE_PRINT_OUTPUT:
         slotStatusHelpMsg(i18n("Prints out the output area"));
         break;

    case ID_FILE_QUIT:
         slotStatusHelpMsg(i18n("Quits the application"));
         break;

    case ID_EDIT_CUT:
         slotStatusHelpMsg(i18n("Cuts the selected section and puts it to the clipboard"));
         break;

    case ID_EDIT_COPY:
         slotStatusHelpMsg(i18n("Copies the selected section to the clipboard"));
         break;

    case ID_EDIT_PASTE:
         slotStatusHelpMsg(i18n("Pastes the clipboard contents to actual position"));
         break;

    case ID_EDIT_CLEAR:
         slotStatusHelpMsg(i18n("Clear the input area"));
         break;

    case ID_EDIT_COPY_OUTPUT:
         slotStatusHelpMsg(i18n("Copies the selected section in the output area to the clipboard"));
         break;

    case ID_EDIT_CLEAR_OUTPUT:
         slotStatusHelpMsg(i18n("Clear the output area"));
         break;

    case ID_EDIT_SELECT_ALL_OUTPUT:
         slotStatusHelpMsg(i18n("Select all text in the output area"));
         break;

    case ID_EDIT_KEYS:
         slotStatusHelpMsg(i18n("Configure application accelerators"));
         break;

    case ID_VIEW_TOOLBAR:
         slotStatusHelpMsg(i18n("Enables/disables the toolbars"));
         break;

    case ID_VIEW_STATUSBAR:
         slotStatusHelpMsg(i18n("Enables/disables the statusbar"));
         break;

    case ID_VIEW_FONT:
         slotStatusHelpMsg(i18n("Configure font for input/output views"));
         break;

    case ID_COMMAND_EXECUTE:
         slotStatusHelpMsg(i18n("Execute the marked or full content of input area in SQL*Plus"));
         break;

    case ID_COMMAND_BREAK:
         slotStatusHelpMsg(i18n("Break current command execution - kill SQL*Plus process"));
         break;

    case ID_COMMAND_PREV:
         slotStatusHelpMsg(i18n("Get previous command from command history into input view"));
         break;

    case ID_COMMAND_NEXT:
         slotStatusHelpMsg(i18n("Get next command from command history into input view"));
         break;

    case ID_COMMAND_HISTORY:
         slotStatusHelpMsg(i18n("Display command history dialog"));
         break;

    default:
         break;
  }
}
