/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *	 Adapted for BibleTime by Torsten Uhlmann, TUhlmann@gmx.de
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <kapp.h>
#include "advancedprinter.h"
#include "printerdialog.h"


AdvancedPrinter::AdvancedPrinter( void )
  : QPrinter()
{
  mPosition.curPage = 1;
  mPosition.maxPage = 1;
  mPosition.now = time( 0 );
}

AdvancedPrinter::~AdvancedPrinter( void )
{
}

bool AdvancedPrinter::setup ( QWidget *parent )
{
	CPrinterDialog *printerDialog = new CPrinterDialog(this, i18n("BibleTime - print"), parent, "printerDialog", true);
	int result = printerDialog->exec();

	if (printerDialog)
		delete printerDialog;
		
	//return QPrinter::setup(parent);
	return (result == QDialog::Rejected) ? false : true;
}

void AdvancedPrinter::setAsText( bool state )
{
  mAsText = state;
}


void AdvancedPrinter::setAll( bool state )
{
  mAll = state;
}


void AdvancedPrinter::setSelection( bool state )
{
  mInSelection = state;
}


void AdvancedPrinter::setRange( bool state, uint start, uint stop )
{
  mInRange = state;
  if( mInRange == false ) 
  { 
    start = stop = 0; 
  }
  else
  {
    if( stop < start ) { stop = start; }
    
    mStartOffset = start;
    mStopOffset  = stop;
  }
}


void AdvancedPrinter::setOutputToStdout( bool state )
{
  mOutputToStdout = state;
}


void AdvancedPrinter::setScaleToFit( bool state )
{
  mScaleToFit = state;
}


void AdvancedPrinter::setPrintBackWhite( bool state )
{
  mPrintBlackWhite = state;
}


void AdvancedPrinter::setPageMarginMM( uint top, uint bottom, int right, int left )
{
  mPageMargin.top = top;
  mPageMargin.left = left;
  mPageMargin.bottom = bottom;
  mPageMargin.right = right;
}

void AdvancedPrinter::setTopMarginMM( uint value )
{
  mPageMargin.top = value;
}

void AdvancedPrinter::setLeftMarginMM( uint value )
{
  mPageMargin.left = value;
}

void AdvancedPrinter::setBottomMarginMM( uint value )
{
  mPageMargin.bottom = value;
}

void AdvancedPrinter::setRightMarginMM( uint value )
{
  mPageMargin.right = value;
}



void AdvancedPrinter::setPageHeader( bool enable, uint left, uint center,
				 uint right, uint line )
{
  if( left > SPageHeader::FreeString ) 		{ left = SPageHeader::NoString; }
  if( center > SPageHeader::FreeString )	{ center = SPageHeader::NoString; }
  if( right > SPageHeader::FreeString ) 	{ right = SPageHeader::NoString; }
  if( line > SPageHeader::Rectangle ) 		{ line = SPageHeader::NoLine; }

  mHeader.enable = enable;
  mHeader.pos[0] = (SPageHeader::EHeaderString)left; 
  mHeader.pos[1] = (SPageHeader::EHeaderString)center;
  mHeader.pos[2] = (SPageHeader::EHeaderString)right;
  mHeader.line = (SPageHeader::EHeaderLine)line;

  if( mHeader.pos[0] == SPageHeader::NoString && 
      mHeader.pos[1] == SPageHeader::NoString &&
      mHeader.pos[2] == SPageHeader::NoString &&
      mHeader.line == SPageHeader::NoLine )
  {
    mHeader.enable = false;
  }
}

void AdvancedPrinter::setPageFooter( bool enable, uint left, uint center,
				 uint right, uint line )
{
  if( left > SPageHeader::FreeString ) 		{ left = SPageHeader::NoString; }
  if( center > SPageHeader::FreeString ) 	{ center = SPageHeader::NoString; }
  if( right > SPageHeader::FreeString ) 	{ right = SPageHeader::NoString; }
  if( line > SPageHeader::Rectangle ) 		{ line = SPageHeader::NoLine; }

  mFooter.enable = enable;
  mFooter.pos[0] = (SPageHeader::EHeaderString)left; 
  mFooter.pos[1] = (SPageHeader::EHeaderString)center;
  mFooter.pos[2] = (SPageHeader::EHeaderString)right;
  mFooter.line = (SPageHeader::EHeaderLine)line;

  if( mFooter.pos[0] == SPageHeader::NoString && 
      mFooter.pos[1] == SPageHeader::NoString &&
      mFooter.pos[2] == SPageHeader::NoString &&
      mFooter.line == SPageHeader::NoLine )
  {
    mFooter.enable = false;
  }
}


SPageMargin AdvancedPrinter::pageMargin( void )
{
  QPaintDeviceMetrics metric( this );
  float f = (float)metric.width()/(float)metric.widthMM();
    
  SPageMargin margin;
  margin.top = (uint) (f*(float)mPageMargin.top);
  margin.right = (uint) (f*(float)mPageMargin.right);
  margin.bottom = (uint) (f*(float)mPageMargin.bottom);
  margin.left = (uint) (f*(float)mPageMargin.left);

  return( margin );
}


SPageMargin AdvancedPrinter::pageMarginMM( void )
{
  return( mPageMargin );
}


SPageSize AdvancedPrinter::pageFullSize( void )
{
  QPaintDeviceMetrics metric( this );
  SPageSize size;
  size.width = metric.width();
  size.height = metric.height();
  
  return( size );
}


SPageSize AdvancedPrinter::pageUsableSize( void )
{
  QPaintDeviceMetrics metric( this );
  SPageMargin margin = pageMargin();
  SPageSize size;

  uint mw = margin.left + margin.right;
  if( metric.width() <= (int)mw )
  {
    size.width = 1;
  }
  else
  {
    size.width = metric.width() - mw;
  }

  uint mh = margin.top + margin.bottom;
  if( metric.height() <= (int)mh )
  {
    size.height = 1;
  }
  else
  {
    size.height = metric.height() - mh;
  }

  return( size );
}

/** prints the defined header on the page */
bool AdvancedPrinter::printHeader(QPainter *pDev){
	if (!mHeader.enable) return false;
	
	int sx = mPageMargin.left;
	int width = pageUsableSize().width;
	int y = mPageMargin.top - 8;
	
	return drawHeader(pDev, sx, width, y, false, &mHeader, &mPosition);
}

/** prints the defined footer on the page */
bool AdvancedPrinter::printFooter(QPainter *pDev){
	if (!mFooter.enable) return false;
	
	int sx = mPageMargin.left;
	int width = pageUsableSize().width;
	int y = pageFullSize().height - mPageMargin.bottom + 8;
//	  debug("Page is: %d x %d",pageFullSize().width,pageFullSize().height);
	return drawHeader(pDev, sx, width, y, true, &mFooter, &mPosition);
}

bool AdvancedPrinter::drawHeader( QPainter *paint, int sx, int width, int y,
     bool isFooter, const SPageHeader *header, const SPagePosition *position )
{
  QFont font( paint->font() );
  paint->setFont( "helvetica" );
  const QFontMetrics &fm = paint->fontMetrics();

	QString dummy;
	debug(dummy.setNum(y));
	debug(dummy.setNum(sx));
	debug(dummy.setNum(width));
	if ( !isFooter )
	{
		// check if top space is big enough
		y -= fm.height();
		if (y < 0) return false;
	} else {
		// check if bottom space is big enought
		if ((uint)(y + fm.height()) > pageFullSize().height) return false;
	}
	
  paint->setPen( black );
  paint->fillRect( sx, y, width, fm.height(), white );

  if( header->line == SPageHeader::SingleLine )
  {
    if( !isFooter )
    {
      paint->drawLine( sx, y+fm.height(), sx+width, y+fm.height() );
    }
    else
    {
      paint->drawLine( sx, y, sx+width, y );
    }
  }
  else if( header->line == SPageHeader::Rectangle )
  {
    paint->drawRect( sx, y, width, fm.height() );
  }

  int pos[3] = { AlignLeft, AlignHCenter, AlignRight };
  for( int i=0; i<3; i++ )
  {
    if( header->pos[i] == SPageHeader::DateTime )
    {
      char *text = ctime( &position->now );
      paint->drawText( sx, y, width, fm.height(), pos[i], text, strlen(text)-1);
    }
    else if( header->pos[i] == SPageHeader::PageNumber )
    {
      QString msg;
      //msg.sprintf( "%s %u %s %u", i18n("Page"), position->curPage,i18n("of"), position->maxPage );
      msg.sprintf( "%s %u", i18n("Page"), position->curPage);
      paint->drawText( sx, y, width, fm.height(), pos[i], msg );
    }
    else if( header->pos[i] == SPageHeader::FreeString )
    {
      paint->drawText( sx, y, width, fm.height(), pos[i], mFreeString );
    }
  }
  paint->setFont( font );

  return true;
}

/** includes own functionality to the derived newPage function */
bool AdvancedPrinter::newPage(){
	bool result = QPrinter::newPage();
	if (result)
		mPosition.curPage++;
		
	return result;
}
/** give a free formatted string which can 
be printed in Header and Footer */
void AdvancedPrinter::setHeaderFreeString(const QString &s){
	mFreeString = s.copy();
}
/** do we preview or print really */
void AdvancedPrinter::setPreview(bool preview){ mPreview = preview; }

/** get the preview value */
bool AdvancedPrinter::getPreview(){ return mPreview; }

/** set the application that handles previews */
void AdvancedPrinter::setPreviewApplication(const QString &app)
{
	mPreviewApplication = app.copy();
}

/** get the application name that handles 
previews */
QString AdvancedPrinter::getPreviewApplication()
{
	return mPreviewApplication;
}
