/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <sys/utsname.h>

#include <qdatetime.h>
#include <qpixmap.h>

#include <kapp.h>
#include <kiconloader.h>

#include "generaltab.h"


GeneralTab::GeneralTab(QWidget *parent, const char *name) : QWidget(parent, name)
{
  KIconLoader     *loader       = kapp->getIconLoader();
  static QPixmap  tuxscreen_xpm = loader->loadIcon("tuxscreen.xpm");
  
  _tuxlabel = new QLabel(this, "tuxlabel");
  _tuxlabel->setPixmap(tuxscreen_xpm);
  _tuxlabel->adjustSize();

  _syslabel = new QLabel(this, "syslabel");
  _syslabel->setText(getSystemInformation().data());
  _syslabel->adjustSize();

  _complabel = new QLabel(this, "complabel");
  _complabel->setText(getComputerInformation().data());
  _complabel->adjustSize();
}


/*
 * Protected methods
 ********************/
 
void  GeneralTab::resizeEvent(QResizeEvent *)
{
  uint  margin = 16;
  uint  infox  = 5*margin+_tuxlabel->width();
  
  _tuxlabel->move(3*margin, 3*margin);
  _syslabel->move(infox,    3*margin); 
  _complabel->move(infox,   5*margin+_syslabel->height()); 
}


/*
 * Private methods
 ******************/
 
QString  GeneralTab::getComputerInformation()
{
  QString  compinfo = "Computer:";
  QString  value, mhzstr, cpustr, memstr;
  FILE     *fhd;
  char     buffer[256];
  int      pos = -1, mhzint;
  int      cpucount = 0;
  float    mhz;
  ulong    mem;
  
  if ((fhd = fopen("/proc/cpuinfo", "r")))
  {
    while (fgets(buffer, sizeof(buffer), fhd))
    {
      value = QString(buffer);
      
      if ((value.find("vendor_id") >= 0) || (value.find("model name") >= 0))
      {
        if (!cpucount)
	{
	  pos	= value.find(':');
	  value = value.right(value.length()-pos-1);	
	  value = value.stripWhiteSpace();
	
	  compinfo += QString("\n")+value;
        }
      } 
      else if (value.find("cpu MHz") >= 0)
      {
        if (!cpucount)
	{
	  pos	 = value.find(':');
	  value  = value.right(value.length()-pos-1);
	  value  = value.stripWhiteSpace();
 
	  mhz	 = value.toFloat(); 
	  mhzint = (int) mhz;
	  
	  if (mhzint % 10 == 8) mhzint += 2;
	  if (mhzint % 10 == 9) mhzint += 1;
	
	  mhzstr.setNum(mhzint);
	  
	  mhzstr   += " MHz";
          compinfo += QString("\n")+mhzstr;	  
        }
      }
      else if (value.find("flags") >= 0)
      {
        if (!cpucount)
	{
	  pos	 = value.find(':');
	  value  = value.right(value.length()-pos-1);
	  value  = value.stripWhiteSpace();
	  
	  if (value.find("mmx") >= 0)
            compinfo += i18n("\nIntel MMX(TM) Technology");
	}
	
        cpucount++;
      }      
    }
      
    if (cpucount > 1)
    {
      cpustr.setNum(cpucount);
      compinfo += QString("\n")+cpustr+i18n(" processors");
    }
    
    fclose(fhd);

    if ((fhd = fopen("/proc/meminfo", "r")))
    {
      while (fgets(buffer, sizeof(buffer), fhd))
      {
        value = QString(buffer);
	
	if ((pos = value.find("Mem:")) >= 0) break;
      }
      
      fclose(fhd);
      
      if (pos >= 0)
      {
        value = value.right(value.length()-4);
	value = value.stripWhiteSpace();
	
	if ((pos = value.find(' ')) >= 0)
	  value.truncate(pos);
	
	if ((mem = value.toULong()))
	{
	  mem += 8388607;
	  mem /= (8*1024*1024);
	  mem *= 8;
	  
	  memstr.setNum(mem);
	  
	  compinfo += QString("\n\n")+memstr+",0 MB RAM";
	}
      }
    }
    
    return compinfo;
  }

  return compinfo+"\nUnknown";
}

 
QString  GeneralTab::getSystemInformation()
{
  QString         sysinfo = "System:\n";
  QString         version, monthname, datestr;
  struct utsname  sysname;
  char            dayofweek[7][4] = { "Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun" };
  int             day, month, year, pos;
  QDate           date(99, 1, 1);
  
  if (!uname(&sysname))
  {
    sysinfo += QString(sysname.sysname)+' '+sysname.release+' '+sysname.machine;
    version  = QString(sysname.version);

    if (version.contains("SMP")) sysinfo += " SMP";
    
    for (day = 0; day < 7; day++)
    {
      if ((pos = version.find(dayofweek[day])) >= 0)
      {
        monthname = version.mid(pos+4, version.length()-pos-4);
        day       = (monthname.mid(4, 2)).toInt();
        year      = (monthname.right(4)).toInt();
	
	monthname.truncate(3);
	
        for (month = 0; month < 12; month++)
	{
	  if (date.monthName(month) == monthname)
	  {
	    datestr.sprintf("\n%i.%i.%i", day, month, year);
	    sysinfo += datestr;
	  }
	}
      
        break;
      }
    }
    
    
    return sysinfo;
  }
  
  return sysinfo+"Unknown";
}


